/***************************************************************************/
/*                                                                         */
/* driver.c - program to test the prttest() function                       */
/*                                                                         */
/* Command line syntax:  DRIVER [1|2 d]                                    */
/*                                                                         */
/*        where:  1|2 is the printer port (1=LPT1)                         */
/*                      if none specified, default is LPT1                 */
/*                                                                         */
/*                d   enables "debug" mode - printer port status is        */
/*                      displayed to the screen on errors                  */
/*                                                                         */
/*                                                                         */
/*       Examples:  DRIVER 1 d - print to LPT1, debug active               */
/*                  DRIVER     - print to LPT1, debug not active           */
/*                  DRIVER 2   - print to LPT2, debug not active           */
/*                                                                         */
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>                  /* for clrscr()                        */
#include <ctype.h>                  /* for toupper()                       */
#include <bios.h>                   /* for bioskey() - see gethex()        */

int port = 0;                       /* Printer port used - Default LPT1    */
int ptrstat;                        /* Status of line printer port         */
int saveptrstat;                    /* Saved version for debug mode        */

unsigned char debug;                /* Used to trace printer problems      */
unsigned char digit;

int prttest(int port);              /* Function declaration                */

void main(char argc,char **argv)
    {
/*9 comment: ****************************************************************
 *9 Handle the command line parms, if any.
 ****************************************************************************/

    if ( argc > 1 )
        {
        if ( ((char)*argv[1] == '1') || ((char)*argv[1] == '2') )
            {
            port = (atoi(argv[1]))-1;
            }
        else
            {
            printf("\n\nImproper LPT port designation!\n\n");
            printf("Invocation:  %s [1|2 d] \n",argv[0]);
            printf("    where 1=LPT1, 2=LPT2\n");
            printf("          d - enables debug mode\n");
            exit(2);
            }
        if ( argc == 3 )
            {
            debug = *argv[2];
            }
        }


/*9 comment: ****************************************************************
 *9 Test the printer port to see if it's ready.
 ****************************************************************************/

    while (  (ptrstat = (prttest(port))) != 0 )
        {
        clrscr();
        printf("\n\n\n\n\n\n\n\n\n\n");
        switch(ptrstat)
            {
            case 1:
                {
                printf("          Printer is off line!  ");
                break;
                }
            case 2:
                {
                printf("          Printer is out of paper!  ");
                break;
                }
            case 3:
                {
                printf("          Printer is powered off!  ");
                break;
                }
            case 4:
                {
                printf("          Printer is not attached!  ");
                break;
                }
            case 5:
                {
                printf("    Printer port is invalid!  ");
                printf("Try again with a valid port specification.\n");
                if ( toupper(debug) == 'D' )
                    {
                    printf("    (ptrstat = %02X)\n",saveptrstat);
                    }
                exit(2);
                }
            case 6:
                {
                printf("          Unknown printer problem!  ");
                break;
                }
            }

        printf("Correct and press a key...");


/*9 comment: ****************************************************************
 *9 If "debug" mode is active, display the printer port's status
 ****************************************************************************/
        
        if ( toupper(debug) == 'D' )
            {
              printf(" (ptrstat = %02X)",saveptrstat);
            }

        while ( kbhit() )
            {
            getch();
            }
        digit = getch();
        }

/*9 comment: ****************************************************************
 *9 Print the message
 ****************************************************************************/

    fprintf(stdprn,"This is a test message.\r\n");
    }

/***************************** D I E B O L D ********************************
 *
 *1  FUNCTION:  prttest
 *
 *2  SUMMARY:  This function tests the printer port to see if it is ready
 *
 *3  INVOCATION: int prttest(int port)
 *
 *4  INPUTS:  Port to be tested (0 = LPT1:, 1 = LPT2:)
 *
 *5  OUTPUTS: Returns 0 if printer is ready.
 *5                   1 if printer is off-line
 *5                   2 if printer is out of paper
 *5                   3 if printer is powered off
 *5                   4 if no printer is attached
 *5                   5 if printer port is unavailable
 *5                   6 if printer status is not recognized
 *
 *6  CAVEATS:
 *
 *7  AUTHOR: Brad Stephenson     DATE:  04/07/90
 *            
 *8  REVISION:  05/18/90 BSTE - Replaced int86() call with biosprint().
 *8             05/29/90 BSTE - Added "debug" mode
 *8                             Added ability to check LPT2
 *
 ****************************************************************************/

int prttest(int port)
    {

    int ptrstatval;

/*9 comment: ****************************************************************
 *9 Check Int 17 return for specific known problem returns
 ****************************************************************************/
    
    ptrstatval = biosprint( 2, 0x00, port );

/*9 comment: ****************************************************************
 *9 If we are in debug mode, save the returned value.
 ****************************************************************************/

    if ( toupper(debug) == 'D' )
        {
        saveptrstat = ptrstatval;
        }


/*9 comment: ****************************************************************
 *9 Based on the port status, return an appropriate value to the caller
 ****************************************************************************/
    
    switch(ptrstatval)
        {
        case 0x90:
            {
            return(0);      /* Printer ready and on-line */
            }
        case 0x18:
        case 0x08:
            {
            return(1);      /* Printer off-line          */
            }
        case 0x28:
        case 0x38:
            {
            return(2);      /* Printer out of paper      */
            }
        case 0x48:
        case 0xC8:
            {
            return(3);      /* Printer powered off       */
            }
        case 0x30:
        case 0xB8:
        case 0xB0:
            {
            return(4);      /* No printer attached       */
            }
        case 0x10:
        case 0x4A:
        case 0x02:
            {
            return(5);      /* Port selected is invalid  */
            }
        default:
            {
            return(6);      /* Unknown printer status    */
            }
        }
    } /* End of prttest()  */

