//----------------------------------------------------------------------------
/* Add-on for ObjectWindows 2 - (C) Dieter Menne, Biomed Tuebingen
   Defines templates and template classes for number validators
   supporting transfer to native variables types.

   Version 1.1, July 94, with improved overflow checking
   Version 1.2, Sept 94, with Chuck Gerson's const char[];
                         (no more valtempl.cpp needed)
   Version 2.0, Oct  94, Needs BC 4.02, because it uses PictureValidators
                         now that were broken in BC 4.00. Separate Versions
                         for fixed and scientific float/doubles.
                         Uses right-justified, multiline edit fields.
                         As option, empty fields are treated as zero.
                         User of previous versions must make Edit fields
                         right-justified, multiline.
*/
//----------------------------------------------------------------------------
#if !defined(__VALTEMPL_H)
#define __VALTEMPL_H

#if !defined(__OWL_VALIDATE_H)
  #include <owl\validate.h>
#endif

#if !defined(__STRING_H)
  #include <string.h>
#endif

#define BUFSIZE 30

/*
  Defines the template for a number validator, and creates instances
  for float, double, int, long, unsigned int, unsigned long.
  The template delegates conversion to the overloaded string streams.

  Thanks to:
    Eric Nagler  TeamB
    Chuck Gerson BCPPWIN
    Bill Hines   BCPPWIN

  If you should require a validator for bytes, you must override the
  template by a special class, because normally BYTE is an unsigned
  char, which cannot be converted to a number using default stream
  operators.
*/

//#define EMPTY_IS_ZERO   // if #defined, empty fields are zero

/* Template for number validation classes */
// No range checking is done for default values of min/max,
// only overflow tests via errno
template <class Num, const char *ValS>
class TNumberValidator: public TPXPictureValidator
{
  public:
 	  TNumberValidator(int width=10,int precision=2,Num min=0,Num max=0):
		  TPXPictureValidator(ValS),
      Precision(precision),
      Width(width),
      Min(min),
      Max(max)
	  {
      PRECONDITION(width > (precision +4)); // Could be relaxed a little
		  SetOption(voTransfer);
		  CheckRange=Min < Max;
      ScienceFloat=strchr(Pic.c_str(),'E') != 0;
	  }

    #ifdef EMPTY_IS_ZERO
    BOOL IsValidInput(char far* str, BOOL /*suppressFill*/)
    {
     if (*str=='\0') // Allow empty fields
       return TRUE;
     else
       return TPXPictureValidator::IsValidInput(str,TRUE);
    }
    #endif

    BOOL IsValid(const char far* s)
    {
      char *st=const_cast<char *>(s);
      #ifdef EMPTY_IS_ZERO
      // Treat empty field as zero (if you want this...)
      // Spaces in fields are not treated as empty.
      if (*st=='\0') {return TRUE;}
      #endif

      if (TPXPictureValidator::IsValid(s))
      {
        Num value;
        istrstream str(st);
        errno=0;
        str >> value;       // Convert non-zero strings
        CHECK(!str.fail()); // Probably bad mask
        RangeOk=
          ! errno      &&  // Overflow
         (! CheckRange ||  // Check range at all?
          value <= Max && value >= Min); // or out of range
        return RangeOk;
      }
      return FALSE;
    }

    void Error() // Improve this to use resources!!
    {
      if (!RangeOk)  // Out of range soft error
      {
        char Msg[80];
        ostrstream MsgSt(Msg,sizeof(Msg)-1);
        if (CheckRange)
          MsgSt << "Value not in range " << Min << ".." << Max << ends;
        else
          MsgSt << "Value too big" << ends;
        ::MessageBox(0, Msg,GetApplicationObject()->GetName(),
          MB_ICONEXCLAMATION | MB_OK | MB_TASKMODAL);
      }
       else
         TPXPictureValidator::Error(); // Default validator
    }

    UINT Transfer(char far* s, void* buffer,TTransferDirection direction)
    {
      if (Options & voTransfer)
      {
        if (direction == tdGetData)
        {
          #ifdef EMPTY_IS_ZERO
          if (*s=='\0') *(Num *) buffer=0; else
          #endif
          istrstream(s) >> *(Num *) buffer;
        }
        else if (direction == tdSetData)
        {
          // First, try to convert without E's
          ostrstream str(s,BUFSIZE);
          str.precision(Precision);
          str.setf(ios::showpoint);
          str.setf(ios::fixed, ios::floatfield);
          str <<  *(Num *) buffer << ends;
          // If this give a long number, try E-format
          if (ScienceFloat && strlen(s) > Width)
          {
            ostrstream str1(s,BUFSIZE);
            str1.precision(Width-Precision);
            str1.setf(ios::scientific, ios::floatfield);
            str1 <<  *(Num *) buffer << ends;
          }
        }
        return sizeof(Num);
      }
      return 0;  // Let Edit do the work
    };

    protected:
      BOOL CheckRange,RangeOk,ScienceFloat;
      int Precision,Width;
      Num Min, Max;
};

// The Paradox-Style Pictures for the fields
#pragma warn -use   // Avoid "not used" warning.
// Science: optional E-field
const char ScienceValidatorVal[]="[* ][-][*#][.][*#][{{E,e}[{-,+}]*#}]";
// Normal float, no E-field accepted
const char FloatValidatorVal[]="[* ][-][*#][.][*#]";
// Up to 99 trailing cents
const char Float2ValidatorVal[]="[* ][-][*#][.][#][#]";
// Guess what comes here...
const char IntValidatorVal[]="[* ][-][*#]";
const char UintValidatorVal[]="[* ][*#]";
#pragma warn .use


// The templates classes
typedef TNumberValidator<int,IntValidatorVal> TIntValidator;
typedef TNumberValidator<UINT,UintValidatorVal> TUintValidator;
typedef TNumberValidator<LONG,IntValidatorVal> TLongValidator;
typedef TNumberValidator<float,Float2ValidatorVal> TFloat2Validator;
typedef TNumberValidator<float,FloatValidatorVal> TFloatValidator;
typedef TNumberValidator<double,ScienceValidatorVal> TDScienceValidator;
typedef TNumberValidator<float,ScienceValidatorVal> TFScienceValidator;
#endif

