// msg.hpp - Dan Rempel, June 1994
//
// iostream-based message classes, designed to simplify reporting of
// status, errors, etc.  The msgbuf class is the interesing part; it
// allows a user-supplied character sink function to deliver output to
// a status line, dialog box, etc.  It's based on Borland's filebuf
// class; examining the internals of one of their streambuf-based classes
// was the only way I could figure out to make this one work.

// Note that this thing is buggy: as long as the string is shorter than
// the buffer size, all is well.  If the streambuf internals have to
// flush the buffer with a call to sync() and accept more input,
// characters are lost.  It is quite clear I don't know what I'm doing here;
// still, the class is useful for output shorter than the buffer size until
// I get this sorted out.
// Note as well that the msgbuf class is designed for insertion, i.e.
// output only.

#include	<stdlib.h>
#include	<msg.hpp>

	//--------------------------------------------------------------

msgbuf::msgbuf( void ( *outFunc )( const char *const str, int n ) ) :
		streambuf(), OutFunction( outFunc )
{
	// This constructor creates a msgbuf with a DEFAULT_BUF_SIZE
	// buffer.  Originally I assumed (as Steve Teale suggests on page 54)
	// that the default streambuf() ctor would dynamically allocate
	// memory; this apparently isn't the case in the Borland
	// implementation; I allocate my own buffer and tell streambuf
	// about it with setb() and setp().  The extra argument to setb
	// apparently means ~streambuf will delete the buffer (and according
	// to Bounds Checker, it works).
	// outFunc is the final sink for whatever is inserted to the stream
	// using a msgbuf; it's default value is 0, allowing output to be shut
	// off if desired.
	// Note that buf will be deleted by ~streambuf: see setbuf().

	char	*buffer = new char[ DEFAULT_BUF_SIZE ];
	if ( buffer )
		setbuf( buffer, DEFAULT_BUF_SIZE );
//	dbp();	// this provides debugging output
}

	//--------------------------------------------------------------

msgbuf::msgbuf( int buflen,
				void ( *outFunc )( const char *const str, int n ) ) :
		streambuf(), OutFunction( outFunc )
{
	// As above, but creates a buflen buffer.
	char	*buffer = new char[ buflen ];
	if ( buffer )
		setbuf( buffer, buflen );
}

	//--------------------------------------------------------------

msgbuf::msgbuf( char *buffer, int buflen,
				void ( *outFunc )( const char *const str, int n ) ) :
	streambuf( buffer, buflen ), OutFunction( outFunc )
{
	// In this case, the streambuf ctor is initalized with buffer and
	// buflen, and it should do all the work; however it doesn't do a
	// setp() and I have no idea why.
	setp( buffer, buffer + buflen );
}

	//--------------------------------------------------------------

msgbuf::msgbuf( unsigned char *buffer, int buflen,
				void ( *outFunc )( const char *const str, int n ) ) :
	streambuf( ( char *)buffer, buflen ), OutFunction( outFunc )
{
	setp( ( char * )buffer, ( char *)buffer + buflen );
}

	//--------------------------------------------------------------

streambuf 	*msgbuf::setbuf( char *buffer, int buflen )
{
	if ( base() )						// already buffered
		return 0;

	setb( buffer, buffer + buflen, 1 );	// ~streambuf will delete buffer
	setp( buffer, buffer + buflen );
	return this;
}

	//--------------------------------------------------------------

int		msgbuf::sync()
{
	int		count = out_waiting();
	if ( count )
	{
		char	*p = pbase();
		if ( OutFunction )
			OutFunction( p, count );
		// have to reset the put area
		char	*b = base();
		setp( b, b + blen() );
	}
	return 0;
}

	//--------------------------------------------------------------

int		msgbuf::overflow( int )
{
	if ( out_waiting() )
		sync();
	return 0;
}

	//--------------------------------------------------------------

msgstream::msgstream( void ( *outFunc )( const char *const str, int n ) ) :
		Buf( outFunc )
{
	ios::init( &Buf );
}

	//--------------------------------------------------------------

msgstream::msgstream( int buflen,
					  void ( *outFunc )( const char *const s, int n ) ) :
		Buf( buflen, outFunc )
{
	ios::init( &Buf );
}

	//--------------------------------------------------------------

msgstream::msgstream( char *buffer, int buflen,
					  void ( *outFunc )( const char *const str, int n ) ) :
		Buf( buffer, buflen, outFunc )
{
	ios::init( &Buf );
}

	//--------------------------------------------------------------

msgstream::msgstream( unsigned char *buffer, int buflen,
					  void ( *outFunc )( const char *const str, int n ) ) :
		Buf( buffer, buflen, outFunc )
{
	ios::init( &Buf );
}

	//--------------------------------------------------------------

#ifdef	TEST_MSG

#include	<conio.h>

// Simple class to demonstrate passing an ostream & for output.
class	testClass
{
public:
	testClass( ostream &status, ostream &msg, ostream &error );

	void	showStatus();
	void	showMsg();
	void	showError();

private:
	ostream		&Status;
	ostream		&Msg;
	ostream		&Error;
};

testClass::testClass( ostream &status, ostream &msg, ostream &error )
			: Status( status ), Msg( msg ), Error( error ) {}

void	testClass::showStatus()
{
	Status << "This is a status message" << endl;
}
void	testClass::showMsg()
{
	Msg << "This is msg message" << endl;
}
void	testClass::showError()
{
	Error << "This is an error message" << endl;
}

	//--------------------------------------------------------------

void	outFunc( const char *const str, int n )
{
	// This isn't very interesting; it gets better when cout.write()
	// is replaced with output to a Windows status line.
	cout.write( str, n  );
}

	//--------------------------------------------------------------

const				BUF_LEN = 60;
const char *const	LONG_MSG =	// lines of 40 characters
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789"
	"0123456789012345678901234567890123456789";

	//--------------------------------------------------------------

void	main()
{
	cout << "\nmsgstream test: default buffer\n" << endl;
	msgstream	msg1( outFunc );
	msg1 << LONG_MSG << endl;

	cout << "\nHit a key for a " << BUF_LEN << " byte long buffer " << endl;
	getch();

	msgstream	msg2( BUF_LEN, outFunc );
	msg2 << LONG_MSG << endl;

	cout << "\nHit a key for an externally allocated buffer " << BUF_LEN
		 << " bytes long" << endl;
	getch();

	char		*buffer = new char[ BUF_LEN ];
	msgstream	msg3( buffer, BUF_LEN, outFunc );
	msg3 << LONG_MSG << endl;
	delete [] buffer;

	testClass	test( msg1, msg2, msg3 );
	test.showStatus();
	test.showMsg();
	test.showError();

	msg1 << "That is all" << endl;
}

#endif	// #ifdef	TEST_MSG

// EOF
