/***************************************************************************/
/*                                                                         */
/*                               MASKLINE.C                                */
/*                                                                         */
/***************************************************************************/
/*	(c) 1996 Zephyr Software, Stephen L. Balkum and Daniel A. Sill	       */
/***************************************************************************/
/* The content of this file is released to the public domain.  It is       */
/* provided AS-IS with the simple statement, "We tried it and it worked."  */
/***************************************************************************/

/*--------------------------------------------------------------------------\
| Provide a styled line function and demonstrate the efficient use of 
| the library internals.
|
| Revision History:
|
| Feb 12, 1995: First pass complete
\--------------------------------------------------------------------------*/


#include <conio.h>
#include <stdio.h>

#include "zsvga.h"


/*---------------------------------------------------------------------------
| Common macros.
*/
#define zABS(x) (((x) < 0) ? -(x) : (x))
#define SWAP(x,y) ((x)^=(y)^=(x)^=(y))


/*---------------------------------------------------------------------------
| Allow access to a couple of ZSVGA internals.
|
| zXwidth     - width of screen in pixels
| zCurbk      - current bank setting
| zScrAddr    - pointer to video memory window
| _zNewBank() - changes current bank as specified in EAX.
*/
#ifdef __cplusplus
extern "C" {
#endif
extern zdword zXwidth;
extern zdword zCurbk;
extern zbyte *zScrAddr;
extern void _zNewBank(void);
#ifdef __cplusplus
}
#endif
#ifdef __WATCOMC__
#pragma aux zXwidth "_*"
#pragma aux zCurbk "_*"
#pragma aux zScrAddr "_*"
#pragma aux _zNewBank "_*"
#endif



/*---------------------------------------------------------------------------
| zSetBank() (Watcom only)
|
| Define a routine for setting a new bank.
*/
#ifdef __WATCOMC__
extern void zSetBank(dword bank);
#pragma aux zSetBank =	\
    "MOV  EAX, bank"    \
    "CALL _zNewBank";
#endif


/*---------------------------------------------------------------------------
| zDrawMaskedLine
|
| Draws a line and determines pixel existence by a test between a walking bit
| the mask argument.  The algorithm only has one special case, vertical
| line.  The other special cases that exist in a line routine, horizontal
| line and single point, are handled automatically.  There may be a bit of
| extraneous math occurring for the single point case, but how often will
| this actually happen and we save the compares the remainder of the time.
*/
void zDrawMaskedLine(zPixelMode mode, zbyte color, zdword mask, z2DPoint* p0, z2DPoint *p1)
{
    zlong x1, y1, x2, y2;
    zlong deltaX, deltaY;
    zbyte code1, code2;
	zlong slope;
	zdword pxlInMask;

	// values will likely be modified, so work on copies
    x1 = p0->x;
    y1 = p0->y;
    x2 = p1->x;
    y2 = p1->y;

	//do any nessessary clipping
    do
	{
		code1 = 0;
		code2 = 0;
        if(x1 > (zlong)zViewX2)
            code1 |= 4;
        if(x1 < (zlong)zViewX1)
            code1 |= 8;
        if(x2 > (zlong)zViewX2)
            code2 |= 4;
        if(x2 < (zlong)zViewX1)
            code2 |= 8;
        if(y1 > (zlong)zViewY2)
            code1 |= 1;
        if(y1 < (zlong)zViewY1)
            code1 |= 2;
        if(y2 > (zlong)zViewY2)
            code2 |= 1;
        if(y2 < (zlong)zViewY1)
            code2 |= 2;

		// are the lines totally out of bounds?
        if(code1 & code2)
			return;
		// do we have to fix them?
        if(code1 | code2)
		{
            if(code1 == 0)
			{
                SWAP(x1, x2);
                SWAP(y1, y2);
                SWAP(code1, code2);
			}
            deltaX = x2 - x1;
            deltaY = y2 - y1;
            if(code1 & 1)
            {
                x1 += (((zlong)zViewY2 - y1) * deltaX / deltaY);
                y1 = (zlong)zViewY2;
            }
            else if(code1 & 2)
			{
                x1 += (((zlong)zViewY1 - y1) * deltaX / deltaY);
                y1 = (zlong)zViewY1;
			}
            else if(code1 & 4)
			{
                y1 += (((zlong)zViewX2 - x1) * deltaY / deltaX);
                x1 = (zlong)zViewX2;
			}
            else if(code1 & 8)
			{
                y1 += (((zlong)zViewX1 - x1) * deltaY / deltaX);
                x1 = (zlong)zViewX1;
			}
		}
        else
        {
            break; // no clipping needed, proceed to drawing
        }
    } while(1);

    deltaX = x2 - x1;
    deltaY = y2 - y1;
    if( zABS(deltaX) < zABS(deltaY) )
	{
		// y major line, (dy == 0) not possible
		if(deltaY < 0)
		{
			// want to step +1 along y axis
			SWAP(x1, x2);
			SWAP(y1, y2);
		}
		deltaY = y1 * zXwidth + x1;

		// shift into fixed point math and add 1/2 to cover round-off
		x1 <<= 16;
		x1 += 0x8000;
		x2 <<= 16;
		x2 += 0x8000;
		slope = (x2 - x1) / (y2 - y1);
		x1 &= 0xFFFF; // rolling over 0xFFFF used as a flag to bump x
		deltaX = 1;
		if(slope < 0)
		{
			slope = -slope;
			deltaX = -1;
		}

		// x2 now used as temporary
		pxlInMask = 1;
		do
		{
			if(pxlInMask & mask)
			{
				x2 = deltaY >> 16; // bank number in top 16 bits
				if(x2 != zCurbk)
				{
					// need a new bank setting
					#ifdef __SC__
						asm MOV  EAX, x2
						asm CALL _zNewBank
					#endif
					#ifdef __WATCOMC__
						zSetBank(x2);
					#endif
				}
				x2 = deltaY & 0xFFFF;
				*(zbyte*)(zScrAddr + x2) = color;
			}

			// increment points
			x1 += slope;
			if(x1 > 0xFFFF)
			{
				x1 -= 0x10000;
				deltaY += deltaX;
			}
			deltaY += zXwidth;
			y1++;

			// increment mask tracking bit
			pxlInMask <<= 1;
			if(pxlInMask == 0)
				pxlInMask = 1;
        } while(y1 <= y2);
	}
	else
	{
		// x major line, must check for (dx==0) which only occurs when (*p0==*p1)

		if(deltaX < 0)
		{
			// want to step +1 along x axis
			SWAP(x1, x2);
			SWAP(y1, y2);
		}
		deltaX = y1 * zXwidth + x1;

		// shift into fixed point math and add 1/2 to cover round-off
		y1 <<= 16;
		y1 += 0x8000;
		y2 <<= 16;
		y2 += 0x8000;
		slope = (y2 - y1);
		y2 = x2 - x1; // y2 now used as temporary
		if(0 == y2) // watch for div by 0
			slope = 0x10000;
		else
			slope /= y2;
		deltaY = zXwidth;
		y1 &= 0xFFFF; // rolling over 0xFFFF used a flag to bump y
		if(slope < 0)
		{
			slope = -slope;
			deltaY = -deltaY;
		}

		pxlInMask = 1;
		do
		{
			if(pxlInMask & mask)
			{
				y2 = deltaX >> 16; // bank number in top 16 bits
				if(y2 != zCurbk)
				{
					// need a new bank setting
					#ifdef __SC__
						asm MOV  EAX, y2
						asm CALL _zNewBank
					#endif
					#ifdef __WATCOMC__
						zSetBank(y2);
					#endif
				}
				y2 = deltaX & 0xFFFF;
				*(zbyte*)(zScrAddr + y2) = color;
			}

			// increment points
			y1 += slope;
			if(y1 > 0xFFFF)
			{
				y1 -= 0x10000;
				deltaX += deltaY;
			}
			x1++;
			deltaX++;

			// increment mask tracking bit
			pxlInMask <<= 1;
			if(pxlInMask == 0)
				pxlInMask = 1;
        } while(x1 <= x2);
	}
}


void Example(void)
{
	zangle a;
	z2DPoint p1={320,240}, p2i={220,240}, p2f;

	p2f = p2i;
	zRes640();
	for(a=0;a<=0x10000;a+=0x100)
	{
		zDrawMaskedLine(zSET, 0, 0xCCCCCCCC, &p1, &p2f);
		z2DRotate(1, &p1, a, &p2i, &p2f);
		zDrawMaskedLine(zSET, 15, 0xCCCCCCCC, &p1, &p2f);
		zSDelay(2);
	}
	getch();
	zResText();
}


