/*----------------------------------------------------------------------------
** File:
**	ls.c
**
** Description:
**	The UNIX "ls" command for MS-DOS.  May be specific to Turbo-C 2.0
**
**	compile with "tcc -ml ls"
**
** Author:
**	Evan J. Mortimore
**
**	This source code and the executable are hereby placed in the public
**	domain by me.  You may do ANYTHING you want with this code, except
**	complain to me about it.  Really!  I mean it. If you don't like it,
**	don't use it.
**
** UNIX is a trademark of AT&T.
**---------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <alloc.h>

/*
** The total number of entries any one directory may contain.  I don't
** know what will happen if a disk has exceeded this, but it probably won't
** be what you want.
*/
#define MAX_ENTRIES	1000


#define NUM_COLUMNS	5

/* Why doesn't TC define these somewhere? */
#define FALSE	0
#define TRUE 	1

/* 
** The various flags that determine what the progam does, and the appearance
** of the output.
*/
int aflag = FALSE;	/* -a all files listed flag */
int cflag = FALSE;	/* -c list only one column if no -l */
int lflag = FALSE;	/* -l long listing flag */
int rflag = FALSE;	/* -r reverse sort */
int sflag = FALSE;	/* -s size listing flag */
int tflag = FALSE;	/* -t sort by time rather than name */
int uflag = FALSE;	/* -u print usable space left on disk */
int Sflag = FALSE;	/* -S sort by size rather than name */
int Rflag = FALSE;	/* -R recursively list all subdirectories */

int num_columns = NUM_COLUMNS;
int drive_num = 0;


/*-----------------------------------------------------------------------------
** Function:
**	main()
**
** Description:
**	Parse the command line, setting global flags accordingly.  Pass results
**	to MakeList(), which does the real work.
**---------------------------------------------------------------------------*/
main(argc,argv)
int argc;
char *argv[];
{
	int i;
	char *path;
	char path_buff[MAXPATH];

	if( argc > 3 || argc < 1 )
	{
		Usage();
		exit(0);
	}

	/* This is pretty braindead command line parsing. */
	if( (argc == 3) || (argc == 2) )
	{
		if( argv[1][0] == '-' )
		{
			for(i=1; i<strlen(argv[1]); i++)
			{
				switch( argv[1][i] )
				{
					case 'a':
						aflag = TRUE;
						break;

					case 'c':
						cflag = TRUE;
						break;

					case 'l':
						lflag = TRUE;
						break;
	
					case 'r':
						rflag = TRUE;
						break;
	
					case 's':
						sflag = TRUE;
						break;

					case 't':
						tflag = TRUE;
						break;

					case 'u':
						uflag = TRUE;
						break;

					case 'R':
						Rflag = TRUE;
						break;
	
					case 'S':
						Sflag = TRUE;
						break;

					case '?':
						Usage();
						exit(0);
						break;

					default:
						printf("Unknown option '-%c'\n",(argv)[1][i]);
						break;
				}
			}
			if( argc == 3)
				path = argv[2];
			else
				path = getcwd(path_buff,MAXPATH);
		}
		else
			path = argv[1];
	}
	else if( argc == 1 )
		path = getcwd(path_buff,MAXPATH);

	strupr(path);
	
	MakePath(path,path);
	MakeList(path);

	/* Print the free count on this drive if -u specified */
	if( uflag )
	{
		struct dfree df;
		unsigned long remain;
		getdfree(drive_num+1,&df);
		remain = (long)df.df_avail*(long)df.df_bsec*(long)df.df_sclus;
		printf("Bytes remaining on drive %c: %ld",'A'+drive_num,remain);
	}
	exit(0);
}


/*----------------------------------------------------------------------------
** Function:
**	MakeList()
**
** Synopsis:
**	MakeList(path)
**	char *path;
**
** Description:
**	Scans the directory specified by "path" and builds a list of all
**	matching files.  PrintList() is then called to print the results.
**	MakeList() is recursive if the "-R" command line option is specified.
**---------------------------------------------------------------------------*/
MakeList(path)
char *path;
{
	char new_path[MAXPATH],*s;
	int i,mask,count,DirCompare();
	struct ffblk **fblks;
	struct ffblk fblk;

	count = 0;

	if( (fblks = farmalloc(MAX_ENTRIES * sizeof( char *))) == NULL )
	{
		printf("ls: out of memory.\n");
		return(-1);
	}

	mask = aflag?(FA_DIREC|FA_HIDDEN|FA_SYSTEM):(FA_DIREC);

	/* Look for the specified path, return if not found */
	if( findfirst(path, &fblk, mask) != 0)
	{
		printf("%s: file or path not found.\n",path);
		return(-1);
	}

	/* If "path" is a directory, append wildcard spec, then look again */
	if( fblk.ff_attrib & FA_DIREC )
	{
		strcpy(new_path,path);
		strcat(new_path,"\\*.*");
		findfirst(new_path, &fblk, mask);
	}

	if( (fblks[count] = (struct ffblk *)farmalloc((long)sizeof(fblk))) == NULL )
	{
		printf("ls: out of memory\n");
		return(-1);
	}
	memcpy(fblks[count],&fblk,sizeof(fblk));
	strlwr(fblks[count]->ff_name);

	for(count=1; findnext(&fblk)==0; count++)
	{
		if( (fblks[count] = (struct ffblk *)farmalloc((long)sizeof(fblk))) == NULL )
		{
			printf("ls: out of memory\n");
			for(i=count; i>=0; i--)
				farfree(fblks[i]);
			return(-1);
		}
		memcpy(fblks[count],&fblk,sizeof(fblk));
		strlwr(fblks[count]->ff_name);
	}

	/* Use qsort() to print in alphabetical (or whatever) order */
	qsort(fblks,count,sizeof(char *),DirCompare);

	/* Now actually print the list to stdout */
	PrintList(fblks,count);

	/* If recursive flag specified */
	if( Rflag )
	{
		/* go through every entry in this dir */
		for(i=0; i<count; i++)
		{
			/* If this is a DIR, build new pathname and search it */
			if( (fblks[i]->ff_attrib & FA_DIREC) &&
			    (fblks[i]->ff_name[0] != '.') )
			{
				if( strstr(path,"*.*") == NULL )
				{
					strcpy(new_path,path);
					strcat(new_path,"\\");
					strcat(new_path,fblks[i]->ff_name);
					strupr(new_path);
					printf("\n%s:\n",new_path);
					strcat(new_path,"\\");
					strcat(new_path,"*.*");
				}
				else
				{
					strcpy(new_path,path);
					s = strstr(new_path,"*.*");
					*s = '\0';
					strcat(new_path,fblks[i]->ff_name);
					strupr(new_path);
					printf("\n%s:\n",new_path);
					strcat(new_path,"\\*.*");
				}
				MakeList(new_path);
			}
		}
	}

	for(i=count-1; i>=0; i--)
		farfree(fblks[i]);

	farfree(fblks);
	return(0);
}

/*-----------------------------------------------------------------------------
** Function:
**	PrintList()
**
** Synopsis:
**	PrintList(fblks,count)
**	struct ffblk *fblks[];
**	int count;
**
** Description:
**	PrintList() prints the files from the list of ffblks specified by
**	"fblks".  "Count" contains the number of entries in "fblks".
**	The format and content of the listing is specified by the global
**	flags defined at the top of this file. 
**----------------------------------------------------------------------------*/
PrintList(fblks,count)
struct ffblk *fblks[];
int count;
{
	unsigned long t;
	unsigned long size = 0L;
	int i,j,k,day,year,hours,minutes,seconds,rows;
	char *month;
	char name[15];
	static char *months[] = {
		" ",
		"Jan", "Feb", "Mar", "Apr", "May", "Jun",
		"Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
	};

	rows = (count/num_columns) + ((count%num_columns)?1:0);

	/* Long (ie: verbose) listing */
	if( lflag )
	{
		for(i=0; i<count; i++)
		{
			size += fblks[i]->ff_fsize;
			hours = (fblks[i]->ff_ftime >> 11) & 0x1F;
			minutes = (fblks[i]->ff_ftime >> 5) & 0x3F;
			seconds = (fblks[i]->ff_ftime & 0x1F) * 2;
			year = (fblks[i]->ff_fdate >> 9) & 0x7F;
			month = months[(fblks[i]->ff_fdate >> 5)&0xF];
			day = (fblks[i]->ff_fdate & 0x1F);
			printf("%c%c%c%c%c %6ld   %02d:%02d:%02d   %s %2d, %d   %s\n",
				(fblks[i]->ff_attrib&FA_DIREC)?'D':'-',
				(fblks[i]->ff_attrib&FA_RDONLY)?'R':'-',
				(fblks[i]->ff_attrib&FA_SYSTEM)?'S':'-',
				(fblks[i]->ff_attrib&FA_HIDDEN)?'H':'-',
				(fblks[i]->ff_attrib&FA_ARCH)?'A':'-',
				fblks[i]->ff_fsize,
				hours,minutes,seconds,
				month,day,1980+year,
				fblks[i]->ff_name);
		}
		if(sflag)
			printf("Total size in bytes: %ld\n", size);
	}
	/* Just bytes in dir total */
	else if( sflag )
	{
		for(i=0; i<count; i++)
			size += fblks[i]->ff_fsize;
		printf("Total size in bytes: %ld\n", size);
	}
	/* brief listing, but only one column */
	else if( cflag )
	{
		for(i=0; i<count; i++)
		{
			if( fblks[i]->ff_attrib & FA_DIREC )
			{
			    strcpy(name,fblks[i]->ff_name);
			    strcat(name,"\\");
			    printf("%s\n",name);
			}
			else
			    printf("%s\n",fblks[i]->ff_name);
		}
	}
	/* Do default multicolumn brief listing */
	else
	{
		for(i=0; i<rows; i++)
		{
			for(j=0; j<num_columns; j++)
			{
				k = rows * j + i;
				if( k < count )
				{
					if( fblks[k]->ff_attrib & FA_DIREC )
					{
					    strcpy(name,fblks[k]->ff_name);
					    strcat(name,"\\");
					    printf("%-15s",name);
					}
					else
					    printf("%-15s",fblks[k]->ff_name);
				}
			}
			printf("\n");
		}
	}

	return(0);
}

/*----------------------------------------------------------------------------
** Function:
**
**	DirCompare()
**
** Synopsis:
**
**	DirCompare(f1,f2)
**	struct ffblk *f1,*f2;
**
** Description:
**
**	This function is called by qsort() to sort directory entries.  The
**	actual comparison of the two arguments is done according to the
**	global flags defined at the top of the file.
**----------------------------------------------------------------------------*/
DirCompare(f1,f2)
struct ffblk **f1,**f2;
{
	struct ffblk *tf1,*tf2;
	unsigned long t1,t2;
	int compare;

	/* if reverse sort is specified */
	if( rflag )
	{
		tf1 = *f2;
		tf2 = *f1;
	}
	else
	{
		tf1 = *f1;
		tf2 = *f2;
	}

	/* Compare according to <t>ime flag or <S>ize flag */
	if( !tflag && !Sflag )
		compare = strcmp(tf1->ff_name,tf2->ff_name);
	else if( Sflag )
	{
		if( tf1->ff_fsize > tf2->ff_fsize )
			compare = 1;
		else if( tf1->ff_fsize < tf2->ff_fsize )
			compare = -1;
		else
			compare = strcmp(tf1->ff_name,tf2->ff_name);
	}
	else if( tflag )
	{
		t1 = ((unsigned long)tf1->ff_fdate << 16) | tf1->ff_ftime;
		t2 = ((unsigned long)tf2->ff_fdate << 16) | tf2->ff_ftime;
		if( t1 < t2 )
			compare = -1;
		else if( t1 > t2 )
			compare = 1;
		else
			compare = 0;
	}
	return(compare);
}

/*----------------------------------------------------------------------------
** Function:
**	MakePath()
**
** Synopsis:
**	MakePath(name,path)
**	char *name;
**	char *path;
**
** Description:
**	MakePath() builds a more or less complete path specification from
**	"name" and stores the results in "path".  Any missing parts of the
**	specification (ie: drive, dir, or file) are filled in.
**--------------------------------------------------------------------------*/
MakePath(name,path)
char *name;
char *path;
{
	char drive[MAXDRIVE],dir[MAXDIR],file[MAXFILE],ext[MAXEXT];

	fnsplit(name,drive,dir,file,ext);

	if( drive[0] == '\0' )
	{
		drive[0] = 'A' + getdisk();
		drive[1] = ':';
		drive[2] = '\0';
	}

	drive_num = drive[0] - 'A';

	if( file[0] == '\0' )
		strcpy(file,"*.*");

	fnmerge(path,drive,dir,file,ext);
}

/*-----------------------------------------------------------------------------
** Function:
**	Usage()
**
** Synopsis:
**
**	Usage()
**
** Description:
**	Usage() simply prints the command line arguments and a description of
**	what they do.
**--------------------------------------------------------------------------*/
Usage()
{
	static char *usage[] = 
	{
		"Usage: ls [-{aclrstuRS}] [pathname]\n",
    		"  -a : list all files (HIDDEN and SYSTEM)",
		"  -c : print only a single column",
    		"  -l : long (verbose) listing",
    		"  -r : reverse the sort order",
    		"  -s : print only size of all files in directory (unless -l)",
    		"  -t : sort by time",
		"  -u : print usable disk space left on drive",
		"  -R : Recursively list all subdirectories below \"pathname\"",
		"  -S : sort by file size",
    		"  -? : print this usage message\n",
		"Placed in the public domain by the author, Evan J. Mortimore",
		NULL,
	};
	char **s;

	for(s = usage; *s != NULL; s++)
		printf("%s\n",*s);
}
