//	=========================================================================
//	Project...:	kscxms.lib
//	File Name.:	kscxms.h
// Date......: March 14, 1995
// Version...: 6.3
// COPYRIGHT (c) 1995 by Kenneth S. Cascio.  All Rights Reserved
// =========================================================================

#ifndef __LARGE__
	#error This program MUST be compiled in the LARGE memory model!
#endif	// __LARGE__

#ifndef		__KSCXMS_H
#define		__KSCXMS_H

#include <iostream.h>

// Define all golbal macros if not previously defined

#ifndef TRUE
	#define	TRUE			 1
#endif

#ifndef FALSE
	#define	FALSE			 0
#endif

#ifndef NULL
	#define	NULL			 0
#endif

#define MAXCONTROLLERS	 5
#define MAPARRAYSIZE		 5
#define FRAMESIZE			 128
#define LOADMETHOD		 Smart

// Define enumeration variables for XMSOBJS/M real memory load style

enum ORIENTATION {Check, Left, Right, Center, Smart};

// Define enumeration variables for XMSACCESS scope function

enum SCOPECODE {Normal, Alter};

// Incomplete class designations for early referencing

class XMSACCESS;	// Phantom class definition for XMSACCESS
class XMSCONTROL;	// Phantom class definition for XMSCONTROL

// ParmsStruct Structure of XMS access

struct XMS_PARAMETERS {
	unsigned long byteCount;	// Number of bytes to move to/from XMS
	unsigned int sourceHandle;	// 0 for real memory, xmsHandle for XMS
	far void *sourcePtr;	// Pointer to start of source object
	unsigned int destHandle;	// 0 for real memory, xmsHandle for XMS
	far void *destPtr;	// Pointer to start of destination object
};

// XMS Map Structure

struct XMSMAP {
	int map_ID;	// ID number of this map object
	char inUse;		// Boolean, is this map in use
	char mapDirty;	// Boolean, must write before reading to this location
	char frameDirty;	// Boolean, must write before reading to frame
	XMSACCESS far *owner;	// Pointer to the object using this map
	unsigned int sizeOne;	// Size of one occurance of T
	unsigned long firstItem;	// First item loaded in the frame
	unsigned long lastItem;	// Last item loaded in the frame
	unsigned long XMS_Offset;	// Offset from XMS_Address for current function
	unsigned long totalBytes;	// Total bytes used by this object
};

//	XMS_STATS, structure for XMS_Verify()

struct XMS_STATS {
	char XMS_Present;	// Boolean, is XMS accessable
	int KB_Available;	// Available amount of XMS Memory in K bytes
	int KB_Largest;	// Largest continuous XMS block available in K bytes
};

// CONTROLSTATS, XMSCONTROL statistics structure

struct CONTROLSTATS {
	char status;	// Success/Failure status of initialization
	int KB_Available;	// Available XMS memory in K bytes
	int KB_Largest;	// Largest continous XMS block available
	int KB_Requested;	// Kilo-bytes of XMS requested by application
	int KB_Allocated;	// Requested amount of XMS memory
	int mapSize;	// Size of one occurance of the XMSMAP structure
	int lastMap;	// Last XMS map location currently allowed
	unsigned int mapArraySize;	// Dynamic size of mapArray area
	unsigned long XMS_Free;	// Amount of free XMS memory available
	unsigned long XMS_Largest;	// Largest free chunk of XMS in bytes
	unsigned long XMS_Used;	// Amount of XMS memory currently in use
	unsigned long XMS_Top;	// Highest XMS byte allowed - sizeof(XMSMAP)
	unsigned long XMS_Bottom;	// Lowest unallocated byte in XMS
	int MapArraySize;	// Displays default map array size
	int FrameSize;	// Displays default frame size
	ORIENTATION LoadMethod;	// Displays default load method
};

// XMSSHARED default value alteration routines

XMS_STATS XMS_Verify(void);	// Verify XMS memory and return stats
int setMaxControllers(int newSize = -1);	// Change default value
int setMapArraySize(int newSize = -1);	// Change default value
int setFrameSize(int newSize = -1);	// Change default value
int setLoadMethod(ORIENTATION newMethod = Check);	// Change default

// XMSSHARED base class declaration

class XMSSHARED {
	friend int setMaxControllers(int newSize);	// Change default
	friend int setMapArraySize(int newSize);	// Change default
	friend int setFrameSize(int newSize);	// Change default
	friend int setLoadMethod(ORIENTATION newMethod); // Change default

private:
	static unsigned long checkSum;	// Check text character sum for copyRight()

protected:
	static int MaxControllers;	// Maximum number of controllers allowed
	static int MapArraySize;	// Default map array size
	static int FrameSize;	// Default XMSOBJS/M frame size
	static ORIENTATION LoadMethod;	// Default load method

protected:
	static int numActive;	// Number of controllers currently active
	static int maxLoc;	// Maximum subscript for ctrlArray
	static int selectedLoc;	// Location of current active controller
	static XMSCONTROL far **ctrlArray;	// Array of active controllers

private:
	static char far *decode(char far *text, int length);	// Decode text array
	static void copyRight(void);	// Display copyRight information

protected:
	static char install(XMSCONTROL far *controller);	// Install new ctrl'r
	static char remove(XMSCONTROL far *controller);	// Remove ctrl'r from list
	static char shutDown(void);	// Perform complete XMS shut down and exit

public:
	XMSSHARED();	// Default class constructor
	~XMSSHARED();	// Default class destructor

protected:
	static void error(int err = 0);	// Signal an error via cerr and halt
};

// XMSCONTROL class declaration; Controls all XMS access

class XMSCONTROL : protected XMSSHARED {
	friend XMSACCESS;	// Give XMSOBJ access to private members via XMSACCESS

private:
	char status;	// Success/Failure status of initialization
	unsigned int XMS_Handle;	// XMS handle assigned by XMS manager
	int KB_Available;	// Available XMS memory in K bytes
	int KB_Largest;	// Largest continous XMS block available
	int KB_Requested;	// Kilo-bytes of XMS requested by application
	int KB_Allocated;	// Requested amount of XMS memory
	int mapSize;	// Size of one occurance of the XMSMAP structure
	int lastMap;	// Last XMS map location currently allowed
	XMSMAP far *mapArray;	// Real memory store for XMSMAP objects
	unsigned int mapArraySize;	// Dynamic size of mapArray area
	unsigned long XMS_Free;	// Amount of free XMS memory available
	unsigned long XMS_Largest;	// Largest free chunk of XMS in bytes
	unsigned long XMS_Used;	// XMS memory currently in use by this controller
	unsigned long XMS_Top;	// Highest XMS byte allowed - sizeof(XMSMAP)
	unsigned long XMS_Bottom;	// Lowest unallocated byte in XMS

private:
	XMSMAP far *getMap(int mapLocation);	// Retrun pointer within mapArray
	char XMS_Compress(void);	// Compress all used XMS memory for max avail.
	char XMS_StartUp(void);	// Initialize xmsobject
	void XMS_ClearMaps(void);	// Clear all maps in mapArray
	unsigned int XMS_CheckMap(int map_ID, char allocating = FALSE); //ID oK
	void XMS_WriteMap(int map_ID, void huge *source = NULL);	// Write map_ID
	void XMS_ReadMap(int map_ID, void huge *destination = NULL);	// Read map_ID

public:
	XMSCONTROL();	// Default constructor, use all available XMS
	XMSCONTROL(unsigned int KB);	// Alternate constructor, KB of XMS
	~XMSCONTROL();	// Class destructor, release XMS memory and reset

public:
	CONTROLSTATS getStats(void);	// Load statistics structure
	char success(void);	// Return creation status
	int sizeMapArray(int newSize = -1); // Resize mapArray
	char select(void);	// Make this XMS controller active for new xmsobjects
	unsigned long XMS_Available(int objects = 1);	// Return net XMS resources
	void release(char shutDown = FALSE);	// Release allocated XMS memory
};

// XMSACCESS class declaration; Allows access to all XMS Control functions

class XMSACCESS : protected XMSSHARED {
	friend XMSCONTROL;	// Allow control functions to access private members

public:
	char status;	// Creation status of this object
	XMSCONTROL far *master;	// XMS Controller owning this object
	int map_ID;	// XMS map ID number used by this object

public:
	ORIENTATION orientation;	// How objects are fetched into the frame
	unsigned int frameSize;	// Size of the real memory window to the XMS
	void huge *frame;	// Pointer to the type <T> real memory frame
	unsigned long elements;	// Number of elements in array

public:
	void checkFrame(int sizeT);	// Verify and adjust for legal frame size
	char allocate(unsigned int size); // Claim space on XMS for this object
	char release(void);	// Release hold on XMS researved for this XMSOBJ
	void saveFrame(void);	// Save frame to XMS
	unsigned int scope(unsigned long newItem, SCOPECODE code = Normal); // Scope
	char copy(int direction, void huge *memPtr, unsigned long bytes,
		unsigned long offset);	// Move x number of bytes from/to XMS

public:
	XMSACCESS();	// Default class constructor
	~XMSACCESS();	// Default class destructor

};

// XMSOBJS class declaration (structures, no math), template<class T>

template <class T> class XMSOBJS : protected XMSSHARED {
private:
	unsigned int tVars;	// Number of array dimensions
	unsigned long far (*tSubs)[2];	// Subscripts array
	unsigned long tItem;	// Item selected for temporary variables

private:
	XMSACCESS far *access;	// Allows access to XMS via controller
	T far *object;	// Pointer to this object's real memory frame

private:
	void verify(int level);	// Check validity of operation

public:
	XMSOBJS();	// Default class contructor
	XMSOBJS(unsigned long m);	// 1D
	XMSOBJS(unsigned long m, unsigned long n);	// 2D
	XMSOBJS(unsigned long m, unsigned long n, unsigned long o);	// 3D
	XMSOBJS(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p);	// 4D
	XMSOBJS(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p, unsigned long q);	// 5D
	~XMSOBJS();	// Class destructor

	T far *operator[](unsigned long m);	// Direct Array Access

	T far *operator()(unsigned long m);
	T far *operator()(unsigned long m, unsigned long n);
	T far *operator()(unsigned long m, unsigned long n, unsigned long o);
	T far *operator()(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p);
	T far *operator()(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p, unsigned long q);

public:
	char success(void);	// Return creation status
	int loadMethod(ORIENTATION method = Check);  // Alter load method
	int frameSize(int newSize = -1); // Resize real-memory frame
	char copy(int direction, void huge *memPtr, unsigned long bytes = 0,
		unsigned long offset = 0);	// Move x number of bytes from/to XMS
	void kill(void);	// Reclaim all HEAP and XMS memory used by this object
};

// XMSOBJM class declaration (includes math), template<class T>

template <class T> class XMSOBJM : protected XMSSHARED {
private:
	unsigned int tVars;	// Number of array dimensions
	unsigned long far (*tSubs)[2];	// Subscripts array
	XMSOBJM<T> far *tOwner;	// Owner access for temporary variables
	int tLevel;	// Temporary variable creation level
	unsigned long tItem;	// Item selected for temporary variables

private:
	XMSACCESS far *access;	// Allows access to XMS via controller
	T far *object;	// Pointer to this object's real memory frame

private:
	void verify(int level);	// Check validity of operation

public:
	XMSOBJM();	// Default class contructor
	XMSOBJM(XMSOBJM<T> &pass);	// Copy constructor
	XMSOBJM(unsigned long m);	// 1D
	XMSOBJM(unsigned long m, unsigned long n);	// 2D
	XMSOBJM(unsigned long m, unsigned long n, unsigned long o);	// 3D
	XMSOBJM(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p);	// 4D
	XMSOBJM(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p, unsigned long q);	// 5D
	~XMSOBJM();	// Class destructor

	T operator*(void);	// De-reference operator

	XMSOBJM<T> operator[](unsigned long m);	// Direct Array Access

	XMSOBJM<T> operator()(unsigned long m);
	XMSOBJM<T> operator()(unsigned long m, unsigned long n);
	XMSOBJM<T> operator()(unsigned long m, unsigned long n, unsigned long o);
	XMSOBJM<T> operator()(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p);
	XMSOBJM<T> operator()(unsigned long m, unsigned long n, unsigned long o,
		unsigned long p, unsigned long q);

	T operator=(T t);
	T operator=(XMSOBJM<T> &pass);

	char operator!(void);

	char operator==(T t);
	char operator||(T t);
	char operator&&(T t);
	char operator!=(T t);
	char operator< (T t);
	char operator> (T t);
	char operator<=(T t);
	char operator>=(T t);

	char operator==(XMSOBJM<T> &pass);
	char operator||(XMSOBJM<T> &pass);
	char operator&&(XMSOBJM<T> &pass);
	char operator!=(XMSOBJM<T> &pass);
	char operator< (XMSOBJM<T> &pass);
	char operator> (XMSOBJM<T> &pass);
	char operator<=(XMSOBJM<T> &pass);
	char operator>=(XMSOBJM<T> &pass);

	T operator+(T t);
	T operator-(T t);
	T operator*(T t);
	T operator/(T t);
	T operator%(T t);

	T operator+(XMSOBJM<T> &pass);
	T operator-(XMSOBJM<T> &pass);
	T operator*(XMSOBJM<T> &pass);
	T operator/(XMSOBJM<T> &pass);
	T operator%(XMSOBJM<T> &pass);

	T operator+=(T t);
	T operator-=(T t);
	T operator*=(T t);
	T operator/=(T t);
	T operator%=(T t);

	T operator+=(XMSOBJM<T> &pass);
	T operator-=(XMSOBJM<T> &pass);
	T operator*=(XMSOBJM<T> &pass);
	T operator/=(XMSOBJM<T> &pass);
	T operator%=(XMSOBJM<T> &pass);

	T operator++(void);	// Prefix ++ operator
	T operator--(void);	// Prefix -- operator
	T operator++(int);		// Postfix ++ operator
	T operator--(int);		// Postfix -- operator

	friend ostream &operator<<(ostream &out_data, XMSOBJM<T> &pass);
	friend istream &operator>>(istream &in_data, XMSOBJM<T> &pass);

public:
	char success(void);	// Return creation status
	int loadMethod(ORIENTATION method = Check);  // Alter load method
	int frameSize(int newSize = -1); // Resize real-memory frame
	char copy(int direction, void huge *memPtr, unsigned long bytes = 0,
		unsigned long offset = 0);	// Move x number of bytes from/to XMS
	void kill(void);	// Reclaim all HEAP and XMS memory used by this object
};

#ifndef	NOXMSOBJ
	#include <xmsobjs.h>
	#include <xmsobjm.h>
#endif

#endif	// __KSCXMS_H
