/****************************************************************************

    Module  : DockUtil.c

    Version : v1.0alpha

    Date    : 20/8/93

    Changes : 20/8/93 - Added use of RePaintSlot function to selectively
                        repaint a single slot instead of using invalidate
                        clientrect to paint whole window when only a single
                        slot changed.



*****************************************************************************/


#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <mem.h>
#include <shellapi.h>
#include "freedock.h"


/****************************************************************
   ReadINIFile()

   Read in and process all slot data, also carry out any setup
   which is required.
****************************************************************/


int 
ReadINIFile(HANDLE hInst, SLOT_ENTRY * Slot, int DockSize)
{
    char            SlotSection[MAX_FPATH_LEN], TmpBuffer[MAX_FPATH_LEN];
    int             StartSlot, SlotNum;
    int             i;

    /**********************************************************************
           Start by initialising the first slot to be the title icon
    ***********************************************************************/

    Slot[0].SlotType = SLOT_SPECIAL_TITLE;
    Slot[0].IconHandle = LoadIcon(hInst, "0_DockIcon");


    /**********************************************************************
       Set the index to the first free slot to the slot after the title
       slot
    ***********************************************************************/

    StartSlot = 1;


    /**********************************************************************
        Setup the special built in options in their own slots if configured
        and adjust StartSlot if required
    ***********************************************************************/

    GetPrivateProfileString("Main Options", "WinExitActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.WinExitActive = TRUE;
        Slot[StartSlot].SlotType = SLOT_SPECIAL_EXIT;
        Slot[StartSlot].IconHandle = LoadIcon(hInst, "EXITICON");
        StartSlot++;
        GetPrivateProfileString("Main Options", "WinExitConfirm", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            DockOptions.WinExitConfirm = FALSE;
        else
            DockOptions.WinExitConfirm = TRUE;
    } else
        DockOptions.WinExitActive = FALSE;

    GetPrivateProfileString("Main Options", "ClockActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.ClockActive = TRUE;
        Slot[StartSlot].SlotType = SLOT_SPECIAL_CLOCK;
        Slot[StartSlot].IconHandle = LoadIcon(hInst, "CLOCKICON");
        StartSlot++;
    } else
        DockOptions.ClockActive = FALSE;

    GetPrivateProfileString("Main Options", "MailActive", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y') {
        DockOptions.MailActive = TRUE;
        GetPrivateProfileString("Main Options", "MailSound", "Yes", TmpBuffer, MAX_FPATH_LEN, IniFile);
        if (toupper(TmpBuffer[0]) == 'N')
            DockOptions.MailSound = FALSE;
        else
            DockOptions.MailSound = TRUE;

        GetPrivateProfileString("Main Options", "MailPath", "c:\\mail.box", DockOptions.MailPath, MAX_FPATH_LEN, IniFile);
        /* GetPrivateProfileString Cannot handle long, so do it manually */
        GetPrivateProfileString("Main Options", "MailBoxSize", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &DockOptions.MailBoxSize);

        GetPrivateProfileString("Main Options", "MailBoxDateTime", "0", TmpBuffer, MAX_FPATH_LEN, IniFile);
        sscanf(TmpBuffer, "%lu", &DockOptions.MailBoxDateTime);

        DockOptions.MailFreq = GetPrivateProfileInt("Main Options", "MailFreq", 10, IniFile);

        Slot[StartSlot].SlotType = SLOT_SPECIAL_MAIL;
        Slot[StartSlot].IconHandle = LoadIcon(hInst, "NOMAILICON");
        StartSlot++;
    } else
        DockOptions.MailActive = FALSE;

    GetPrivateProfileString("Main Options", "AlwaysOnTop", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        DockOptions.AlwaysOnTop = TRUE;
    else
        DockOptions.AlwaysOnTop = FALSE;

    GetPrivateProfileString("Main Options", "MaxView", "No", TmpBuffer, MAX_FPATH_LEN, IniFile);
    if (toupper(TmpBuffer[0]) == 'Y')
        DockOptions.MaxView = TRUE;
    else
        DockOptions.MaxView = FALSE;


    /*******************************************************************************
       SlotNum is used to index the sections in the .INI file, each slot is
       uniquely numbered in a section of the type [Slot xx] where xx is the slot
       number.
       The for loop attempts to read the AppName field from each slot, if this
       fails the slot is empty so the slot type is set to SLOT_FREE and the next
       slot is read. If the reading of AppName succeeds, the slot is used and
       the rest of the details are read in.
    *******************************************************************************/
    SlotNum = 1;

    for (i = StartSlot; i < DockSize; i++) {

        sprintf(SlotSection, "Slot %d", SlotNum++);

        GetPrivateProfileString(SlotSection, "AppName", "", Slot[i].AppName, MAX_FPATH_LEN, IniFile);
        if (Slot[i].AppName[0] != '\0') {
            GetPrivateProfileString(SlotSection, "CmdLine", "", Slot[i].CmdLine, MAX_FPATH_LEN, IniFile);
            GetPrivateProfileString(SlotSection, "RunTimeDir", "", Slot[i].RunTimeDir, MAX_FPATH_LEN, IniFile);

            Slot[i].StartState = GetPrivateProfileInt(SlotSection, "StartState", -1, IniFile);
            Slot[i].WinX = GetPrivateProfileInt(SlotSection, "WinX", -1, IniFile);
            Slot[i].WinY = GetPrivateProfileInt(SlotSection, "WinY", -1, IniFile);
            Slot[i].WinWidth = GetPrivateProfileInt(SlotSection, "WinWidth", -1, IniFile);
            Slot[i].WinHeight = GetPrivateProfileInt(SlotSection, "WinHeight", -1, IniFile);

            GetPrivateProfileString(SlotSection, "IconFile", "", Slot[i].IconFile, MAX_FPATH_LEN, IniFile);

            Slot[i].IconPos = GetPrivateProfileInt(SlotSection, "IconPos", 0, IniFile);

            Slot[i].IconHandle = ExtractIcon(hInst, Slot[i].IconFile, Slot[i].IconPos);

            Slot[i].SlotType = SLOT_USED;
        } else
            Slot[i].SlotType = SLOT_FREE;
    }

    return 1;
}


int 
WriteINIFile(SLOT_ENTRY * Slot, int nDockSize)
{
    char            TmpBuffer[MAX_FPATH_LEN], SlotName[MAX_FPATH_LEN];
    char            PosStr[4][7] =
    {
        {"Left"},
        {"Right"},
        {"Top"},
        {"Bottom"}};
    int             i, SlotNum;

    /* Flush any cached .INI file to disk */
    WritePrivateProfileString(NULL, NULL, NULL, IniFile);

    WritePrivateProfileString("Main Options", "Position", PosStr[DockOptions.Position], IniFile);
    WritePrivateProfileString("Main Options", "MaxView", (DockOptions.MaxView) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "AlwaysOnTop", (DockOptions.AlwaysOnTop) ? "Yes" : "No", IniFile);

    WritePrivateProfileString("Main Options", "ClockActive", (DockOptions.ClockActive) ? "Yes" : "No", IniFile);

    WritePrivateProfileString("Main Options", "MailActive", (DockOptions.MailActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "MailPath", DockOptions.MailPath, IniFile);
    WritePrivateProfileString("Main Options", "MailSound", (DockOptions.MailSound) ? "Yes" : "No", IniFile);
    sprintf(TmpBuffer, "%d", DockOptions.MailFreq);
    WritePrivateProfileString("Main Options", "MailFreq", TmpBuffer, IniFile);
    sprintf(TmpBuffer, "%lu", DockOptions.MailBoxDateTime);
    WritePrivateProfileString("Main Options", "MailBoxDateTime", TmpBuffer, IniFile);
    sprintf(TmpBuffer, "%lu", DockOptions.MailBoxSize);
    WritePrivateProfileString("Main Options", "MailBoxSize", TmpBuffer, IniFile);


    WritePrivateProfileString("Main Options", "WinExitActive", (DockOptions.WinExitActive) ? "Yes" : "No", IniFile);
    WritePrivateProfileString("Main Options", "WinExitConfirm", (DockOptions.WinExitConfirm) ? "Yes" : "No", IniFile);

    SlotNum = 1;                /* Number of first slot in .INI file */

    for (i = 0; i < nDockSize; i++) {
        if (Slot[i].SlotType == SLOT_FREE) {
            sprintf(SlotName, "Slot %d", SlotNum++);
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
        } else if (Slot[i].SlotType == SLOT_USED) {
            sprintf(SlotName, "Slot %d", SlotNum++);
            WritePrivateProfileString(SlotName, NULL, NULL, IniFile);
            WritePrivateProfileString(SlotName, "AppName", Slot[i].AppName, IniFile);
            WritePrivateProfileString(SlotName, "CmdLine", Slot[i].CmdLine, IniFile);
            WritePrivateProfileString(SlotName, "RunTimeDir", Slot[i].RunTimeDir, IniFile);

            sprintf(TmpBuffer, "%d", Slot[i].StartState);
            WritePrivateProfileString(SlotName, "StartState", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot[i].WinX);
            WritePrivateProfileString(SlotName, "WinX", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot[i].WinY);
            WritePrivateProfileString(SlotName, "WinY", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot[i].WinWidth);
            WritePrivateProfileString(SlotName, "WinWidth", TmpBuffer, IniFile);
            sprintf(TmpBuffer, "%d", Slot[i].WinHeight);
            WritePrivateProfileString(SlotName, "WinHeight", TmpBuffer, IniFile);

            WritePrivateProfileString(SlotName, "IconFile", Slot[i].IconFile, IniFile);

            sprintf(TmpBuffer, "%d", Slot[i].IconPos);
            WritePrivateProfileString(SlotName, "IconPos", TmpBuffer, IniFile);
        }
    }
    return 1;
}

int FindSlotHit(int DockPosition, int XHit, int YHit)
{
    if ((DockPosition == DOCK_BOTTOM) || (DockPosition == DOCK_TOP)) {
        return (XHit / SLOT_BUTTON_WIDTH);
    } else {
        return (YHit / SLOT_BUTTON_HEIGHT);
    }


}

BOOL IsCursorOutOfDock(int X, int Y)
{
    if ((DockOptions.Position == DOCK_BOTTOM) || (DockOptions.Position == DOCK_TOP)) {
        if( (X < (0-(SLOT_BUTTON_HEIGHT/2))) ||
            (Y < (0-(SLOT_BUTTON_WIDTH/2))) ||
            (Y > (SLOT_BUTTON_HEIGHT+(SLOT_BUTTON_HEIGHT/2)) ) ||
            (X > (DockOptions.ScreenWidth+(DockOptions.ScreenWidth/2))) ){
            return TRUE;
        }
    } else {
        if( (X < (0-(SLOT_BUTTON_HEIGHT/2))) ||
            (Y < (0-(SLOT_BUTTON_WIDTH/2))) ||
            (X > (SLOT_BUTTON_WIDTH+(SLOT_BUTTON_WIDTH/2)) ) ||
            (Y > (DockOptions.ScreenHeight+(DockOptions.ScreenHeight/2))) ){
            return TRUE;
        }
    }

  return( FALSE );
}


#pragma argsused

BOOL FAR PASCAL
AppOptionsDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{
    static BOOL     Grabbing;
    static int      NumIcons;
    static FARPROC  lpfnChooseIconDlgProc;
    RECT            rect;
    HWND            hwndGrab, hwndTmp;
    POINT           pt;
    int             Status;
    static HCURSOR  hGrabCur;

    switch (iMessage) {
    case WM_INITDIALOG:

        SendDlgItemMessage(hDlg, QX_FILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
        SendDlgItemMessage(hDlg, QX_CMDLINE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
        SendDlgItemMessage(hDlg, QX_RUNTIMEDIR, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
        SendDlgItemMessage(hDlg, QX_ICONFILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);

        lpfnChooseIconDlgProc = MakeProcInstance((FARPROC) ChooseIconDlgProc, hAppInst);

        memcpy(&TmpSlot, &Slot[CurSlot], sizeof(SLOT_ENTRY));

        switch (TmpSlot.StartState) {

        case START_MINIMUM:
            CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MINIMUM);
            break;

        case START_MAXIMUM:
            CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MAXIMUM);
            break;

        case START_STORE:
            CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
            break;

        default:
            CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_NORMAL);
            break;
        }

        NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);

        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);

        /* Set starting values. */
        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
        SetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine);
        SetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir);
        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);

        SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);

        Grabbing = FALSE;

        return (TRUE);

    case WM_HSCROLL:
        if (wParam == SB_LINEDOWN)
            if (NumIcons > 0)
                TmpSlot.IconPos = min(NumIcons - 1, TmpSlot.IconPos + 1);

        if (wParam == SB_LINEUP)
            if (NumIcons > 0)
                if (TmpSlot.IconPos > 0)
                    TmpSlot.IconPos = TmpSlot.IconPos - 1;

        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
        TmpSlot.IconHandle = ExtractIcon(hAppInst, TmpSlot.IconFile, TmpSlot.IconPos);

        SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);

        return (TRUE);

    case WM_COMMAND:
        switch (wParam) {
        case QX_OK:
            if (SendDlgItemMessage(hDlg, QX_ICONFILE, EM_GETMODIFY, 0, 0L)) {
                SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
                GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);
                if (NumIcons == 0) {
                    MessageBox(hDlg, "This file contains no icons",
                               "FreeDock", MB_OK | MB_ICONSTOP);
                    TmpSlot.IconPos = Slot[CurSlot].IconPos;
                    TmpSlot.IconHandle = Slot[CurSlot].IconHandle;
                    strcpy(TmpSlot.IconFile, Slot[CurSlot].IconFile);
                    SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                    NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);
                } else {
                    TmpSlot.IconPos = 0;
                    TmpSlot.IconHandle = ExtractIcon(hAppInst, TmpSlot.IconFile, TmpSlot.IconPos);
                }

                SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);
                return (TRUE);
            }
            GetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName, MAX_FPATH_LEN);
            GetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine, MAX_FPATH_LEN);
            GetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir, MAX_FPATH_LEN);
            GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);

            if (IsDlgButtonChecked(hDlg, QX_MINIMUM))
                TmpSlot.StartState = START_MINIMUM;
            else if (IsDlgButtonChecked(hDlg, QX_MAXIMUM))
                TmpSlot.StartState = START_MAXIMUM;
            else if (IsDlgButtonChecked(hDlg, QX_STORE))
                TmpSlot.StartState = START_STORE;
            else
                TmpSlot.StartState = START_NORMAL;

            memcpy(&Slot[CurSlot], &TmpSlot, sizeof(SLOT_ENTRY));

            RePaintSlot(CurSlot, FALSE);
            FreeProcInstance(lpfnChooseIconDlgProc);
            EndDialog(hDlg, TRUE);
            break;

        case QX_GRABWIN:
            SetCapture(hDlg);
            hGrabCur = LoadCursor(hAppInst, "GrabberCur");
            SetCursor(hGrabCur);
            Grabbing = TRUE;

            return (TRUE);

        case QX_REMOVE:
                memset(&Slot[CurSlot], '\0', sizeof(SLOT_ENTRY) );
                RePaintSlot(CurSlot, TRUE);
                EndDialog(hDlg, TRUE);
            return (TRUE);

        case QX_BROWSEICON :
            Status = BrowseIconFile( hDlg, TmpSlot.IconFile );
            if(Status){
                NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);
                TmpSlot.IconPos = 0;
                TmpSlot.IconHandle = ExtractIcon(hAppInst, TmpSlot.IconFile, TmpSlot.IconPos);
                SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);
            }
            break;

        case QX_BROWSEFILE :
            BrowseFileName( hDlg, TmpSlot.AppName );
            if(Status){
                SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                NumIcons = ExtractIcon(hAppInst, TmpSlot.AppName, -1);
                if(NumIcons){
                    strcpy( TmpSlot.IconFile, TmpSlot.AppName );
                    TmpSlot.IconPos = 0;
                }
                else{
                    // File has no Icons so select PROGMAN.EXE : Icon (2)
                    GetWindowsDirectory(TmpSlot.IconFile, MAX_FPATH_LEN);
                    /**********************************************************
                       only add \ to path if it is not already there i.e. P:\
                    **********************************************************/
                    if (TmpSlot.IconFile[strlen(TmpSlot.IconFile) - 1] == '\\') {
                        strcat(TmpSlot.IconFile, "PROGMAN.EXE");
                    } else {
                        strcat(TmpSlot.IconFile, "\\PROGMAN.EXE");
                    }
                    TmpSlot.IconPos = 1;
                }
                TmpSlot.IconHandle = ExtractIcon(hAppInst, TmpSlot.IconFile, TmpSlot.IconPos);
                NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);
                SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);
            }
            break;

        case QX_CANCEL:
            FreeProcInstance(lpfnChooseIconDlgProc);
            EndDialog(hDlg, FALSE);
            break;

        case QX_CHOOSEICON :
            Status = DialogBox(hAppInst, "CHOOSEICON", hDlg, lpfnChooseIconDlgProc);
            if(Status){
                SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, TmpSlot.IconHandle, 0L);
            }
            break;
        }

    case WM_LBUTTONDOWN:
        if (!Grabbing)
            return FALSE;
        /********************************************************
           Button click positions are reported relative to the
           Dialog window, so correct for this by adding the x,y
           coordinates of the top left corner of the dialog to
           the click position. Also add a frig factor because
           for some unknown reason the cursor reports values
           which are out by (1,20) !!!!!!
        ********************************************************/
        GetWindowRect(hDlg, &rect);
        pt.x = rect.left + LOWORD(lParam) + 1;
        pt.y = rect.top + HIWORD(lParam) + 20;

        hwndGrab = WindowFromPoint(pt);

        if ((hwndGrab == NULL) || (hwndGrab == GetDesktopWindow())) {
            MessageBox(hDlg, "No Window there to grab !",
                       "FreeDock", MB_OK | MB_ICONSTOP);
            Grabbing = FALSE;
            return (TRUE);
        }
        /* Climb back up to the top of a parent-child list */
        hwndTmp = GetParent(hwndGrab);
        while (hwndTmp != NULL) {
            hwndGrab = hwndTmp;
            hwndTmp = GetParent(hwndGrab);
        }
        GetWindowRect(hwndGrab, &rect);
        TmpSlot.WinX = rect.left;
        TmpSlot.WinY = rect.top;
        TmpSlot.WinWidth = rect.right - rect.left;
        TmpSlot.WinHeight = rect.bottom - rect.top;

        MessageBox(hDlg, "Window Grabbed Successfully.",
                   "FreeDock", MB_OK | MB_ICONINFORMATION);

        CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);

        Grabbing = FALSE;
        ReleaseCapture();
        SetCursor(LoadCursor(NULL, IDC_ARROW));
        DestroyCursor(hGrabCur);

        return (TRUE);
    }

    return FALSE;
}


#pragma argsused
BOOL FAR PASCAL 
MainOptionsDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{

    switch (iMessage) {
        case WM_INITDIALOG:
        CheckDlgButton(hDlg, QX_MAILCHECK, DockOptions.MailActive);
        CheckDlgButton(hDlg, QX_WINEXIT, DockOptions.WinExitActive);
        CheckDlgButton(hDlg, QX_CLOCK, DockOptions.ClockActive);
        CheckDlgButton(hDlg, QX_ON_TOP, DockOptions.AlwaysOnTop);
        CheckDlgButton(hDlg, QX_MAX_VIEW, DockOptions.MaxView);
        CheckRadioButton(hDlg, QX_LEFT, QX_BOTTOM, QX_LEFT + DockOptions.Position);
        return (TRUE);

    case WM_COMMAND:
        switch (wParam) {
        case QX_SAVE:
        case QX_OK:
            DockOptions.MailActive = IsDlgButtonChecked(hDlg, QX_MAILCHECK);
            DockOptions.WinExitActive = IsDlgButtonChecked(hDlg, QX_WINEXIT);
            DockOptions.ClockActive = IsDlgButtonChecked(hDlg, QX_CLOCK);
            DockOptions.AlwaysOnTop = IsDlgButtonChecked(hDlg, QX_ON_TOP);
            DockOptions.MaxView = IsDlgButtonChecked(hDlg, QX_MAX_VIEW);

            /**************************************************************
                Handle Always On Top Option
            **************************************************************/
            if (DockOptions.AlwaysOnTop) {
                SetWindowPos(hwndDock, HWND_TOPMOST, 0, 0, 0, 0,
                             SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
            } else {
                SetWindowPos(hwndDock, HWND_BOTTOM, 0, 0, 0, 0,
                             SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
            }

            /****************************************************************
                Handle Dock Position Options
            ****************************************************************/
            if (IsDlgButtonChecked(hDlg, QX_LEFT)) {
                DockOptions.Position = DOCK_LEFT;
                SetWindowPos(hwndDock,
                             HWND_TOP,
                             0,
                             0,
                             SLOT_BUTTON_WIDTH + 2,
                             DockOptions.ScreenHeight,
                             SWP_SHOWWINDOW);
            } else if (IsDlgButtonChecked(hDlg, QX_TOP)) {
                DockOptions.Position = DOCK_TOP;
                SetWindowPos(hwndDock,
                             HWND_TOP,
                             0,
                             0,
                             DockOptions.ScreenWidth,
                             SLOT_BUTTON_HEIGHT + 2,
                             SWP_SHOWWINDOW);
            } else if (IsDlgButtonChecked(hDlg, QX_BOTTOM)) {
                DockOptions.Position = DOCK_BOTTOM;
                SetWindowPos(hwndDock,
                             HWND_TOP,
                             0,
                             DockOptions.ScreenHeight - SLOT_BUTTON_HEIGHT,
                             DockOptions.ScreenWidth,
                             SLOT_BUTTON_HEIGHT + 2,
                             SWP_SHOWWINDOW);
            } else {
                DockOptions.Position = DOCK_RIGHT;
                SetWindowPos(hwndDock,
                             HWND_TOP,
                             DockOptions.ScreenWidth - SLOT_BUTTON_WIDTH,
                             0,
                             SLOT_BUTTON_WIDTH + 2,
                             DockOptions.ScreenHeight,
                             SWP_SHOWWINDOW);
            }

            if (wParam == QX_SAVE)      /* Only write file if save was
                                         * selected */
                WriteINIFile(Slot, DockOptions.DockSize);
            EndDialog(hDlg, TRUE);
            break;

        case QX_CANCEL:
            EndDialog(hDlg, TRUE);
            break;
        }
        break;

    }

    return FALSE;
}


#pragma argsused
BOOL FAR PASCAL 
WinExitOptionsDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{

    switch (iMessage) {
        case WM_INITDIALOG:
        CheckDlgButton(hDlg, QX_WINEXITCONFIRM, DockOptions.WinExitConfirm);
        return (TRUE);

    case WM_COMMAND:
        switch (wParam) {
        case QX_OK:
            DockOptions.WinExitConfirm = IsDlgButtonChecked(hDlg, QX_WINEXITCONFIRM);
            EndDialog(hDlg, TRUE);
            break;

        case QX_CANCEL:
            EndDialog(hDlg, TRUE);
            break;
        }
        break;
    }
    return FALSE;
}


#pragma argsused
BOOL FAR PASCAL 
MailOptionsDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{
    BOOL            bTxlated;

    switch (iMessage) {
    case WM_INITDIALOG:
        SendDlgItemMessage(hDlg, QX_MAILPATH, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
        SetDlgItemText(hDlg, QX_MAILPATH, DockOptions.MailPath);
        SetDlgItemInt(hDlg, QX_MAILFREQ, DockOptions.MailFreq, TRUE);
        CheckDlgButton(hDlg, QX_MAILSOUND, DockOptions.MailSound);
        return (TRUE);

    case WM_COMMAND:
        switch (wParam) {
        case QX_OK:
            GetDlgItemText(hDlg, QX_MAILPATH, DockOptions.MailPath, MAX_FPATH_LEN);
            DockOptions.MailFreq = GetDlgItemInt(hDlg, QX_MAILFREQ, &bTxlated, FALSE);
            if (!bTxlated) {
                MessageBox(hDlg, "Not a valid integer!", "FreeDock", MB_OK | MB_ICONSTOP);
                SendDlgItemMessage(hDlg, QX_MAILFREQ, EM_SETSEL, 0, MAKELONG(0, -1));
                return (TRUE);
            }
            DockOptions.MailSound = IsDlgButtonChecked(hDlg, QX_MAILSOUND);
            KillTimer(hwndDock, MAIL_TIMER);
            SetTimer(hwndDock, MAIL_TIMER, DockOptions.MailFreq * 1000 * 60, NULL);
            EndDialog(hDlg, TRUE);
            break;

        case QX_CANCEL:
            EndDialog(hDlg, TRUE);
            break;
        }
        break;
    }
    return FALSE;
}


#pragma argsused
BOOL FAR PASCAL 
ClockOptionsDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{
    switch (iMessage) {
        case WM_INITDIALOG:
        return (TRUE);

    case WM_COMMAND:
        if (wParam == QX_OK)
            EndDialog(hDlg, TRUE);
        break;
    }
    return FALSE;
}


#pragma argsused
BOOL FAR PASCAL 
ExitDockDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{
    switch (iMessage) {
    case WM_INITDIALOG:
	SetDlgItemText(hDlg, QX_VERSION, VersionText);
	return (TRUE);

    case WM_COMMAND:
	switch (wParam) {
	case QX_EXIT:
//            nItem = MessageBox(hDlg, "Do you really want to exit FreeDock ?",
//                               "FreeDock", MB_OKCANCEL | MB_ICONSTOP);

//            if (nItem == IDOK) {
		WriteINIFile(Slot, DockOptions.DockSize);
		PostQuitMessage(0);
//            }
	    return (TRUE);

	case QX_OK:
	    EndDialog(hDlg, TRUE);
            break;

        }
    }
    return FALSE;
}


BOOL FAR PASCAL BrowseIconFile( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Icon Files (*.ICO)\0*.ICO\0DLL's (*.DLL)\0*.DLL\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select an Icon File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL BrowseFileName( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Batch Files (*.BAT)\0*.BAT\0PIF Files (*.PIF)\0*.PIF\0DOS .COM Files (*.COM)\0*.COM\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select a File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


#pragma argsused
BOOL FAR PASCAL 
ChooseIconDlgProc(HWND hDlg, WORD iMessage, WORD wParam, LONG lParam)
{
    // Slot info passed in via global TmpSlot struct
    // CurIcon and CurPage are 0 relative
    static int NumIcons, CurIcon, NumPages, CurPage;
    static HANDLE *Icon;          // pointer to array of icon handles
    RECT rect;
    DRAWITEMSTRUCT *ItemStruct;
    int i, ItemNum;

    switch (iMessage) {
        case WM_INITDIALOG:
             // find number of icons & create array to hold ALL handles
             NumIcons = ExtractIcon(hAppInst, TmpSlot.IconFile, -1);
             Icon = (HANDLE *)malloc( NumIcons * sizeof(HANDLE) );

             // Read all Icon handles
             for( i=0; i< NumIcons ; i++){
                 Icon[i] = ExtractIcon(hAppInst, TmpSlot.IconFile, i);
             }

             CurIcon = TmpSlot.IconPos;
             NumPages = (NumIcons / 50) + 1;
             CurPage = (CurPage / 50);

             SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
             SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
             SetDlgItemInt(hDlg, QX_PAGENUM, (CurPage+1), TRUE);
             SetDlgItemInt(hDlg, QX_PAGETOTAL, NumPages, TRUE);
             SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, Icon[CurIcon], 0L);
             return (TRUE);

        case WM_DRAWITEM :
             ItemStruct = (DRAWITEMSTRUCT *)lParam;
             ItemNum = (ItemStruct->CtlID-QX_ICON1) + (CurPage*50);
             if(ItemNum < NumIcons){
                 DrawIcon( ItemStruct->hDC, 0, 0, Icon[ItemNum]);
             }
             return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (CurPage < (NumPages-1)){
                    CurPage = CurPage + 1;
                    GetClientRect( hDlg, &rect );
                    InvalidateRect( hDlg, &rect, TRUE );
                }

            if (wParam == SB_LINEUP)
                if (CurPage > 0){
                    CurPage = CurPage - 1;
                    GetClientRect( hDlg, &rect );
                    InvalidateRect( hDlg, &rect, TRUE );
                }

            SetDlgItemInt(hDlg, QX_PAGENUM, CurPage + 1, TRUE);

            return (TRUE);

        case WM_COMMAND:
             if (wParam == QX_OK){
                 TmpSlot.IconPos = CurIcon;
                 TmpSlot.IconHandle = Icon[CurIcon];
                 free( Icon );
                 EndDialog(hDlg, TRUE);
             }
             else if (wParam == QX_CANCEL){
                 free( Icon );
                 EndDialog(hDlg, FALSE);
             } 
             else if( (wParam >= QX_ICON1) && (wParam <= QX_ICON1+50) ){
                 CurIcon = min( (NumIcons-1), (wParam - QX_ICON1) + (CurPage*50));
                 SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
                 SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, Icon[CurIcon], 0L);
             }
        break;
    }
    return FALSE;
}

/**************************************************************************
    ExecSlot.

    This function handles the execution of a program associated
    with a slot, it takes the Slot index so it can find the program
    to execute, it also takes the command line to pass, to enable
    this function to be used to execute a program as a result of
    a drag and drop action.
    In the case of drag&drop, the caller must create the command line
    before calling this function.
**************************************************************************/

void ExecSlot( int SlotHit, char *CmdLine ){

    int     Retry, Status;
    int     StartState;
    HANDLE  hExecWin, hWinTmp;

    switch (Slot[SlotHit].StartState) {

        case START_MINIMUM:
            StartState = SW_SHOWMINIMIZED;
            break;

        case START_MAXIMUM:
            StartState = SW_SHOWMAXIMIZED;
            break;

        case START_NORMAL:
        case START_STORE :
            StartState = SW_SHOWNORMAL;
            break;

        default: return;
    }

    
    SetCursor(LoadCursor(NULL, IDC_WAIT));

    ShellExecute(GetDesktopWindow(),
                    "open",
                    Slot[SlotHit].AppName,
                    CmdLine,
                    Slot[SlotHit].RunTimeDir,
                    StartState);

    if(Slot[SlotHit].StartState == START_STORE){
        /**************************************************
            Get the active window, checking it's not us.
            <Fix for Martin's 2nd Bug find.
        **************************************************/
        Retry = 0;
        hExecWin = GetActiveWindow();
        while ((hExecWin == hwndDock) && (Retry < 25)) {
            hExecWin = GetActiveWindow();
            Retry++;
        }
        /* Climb back up to the top of a parent-child list */
        hWinTmp = GetParent(hExecWin);
        while (hWinTmp != NULL) {
            hExecWin = hWinTmp;
            hWinTmp = GetParent(hExecWin);
        }
        /*************************************
            Re-position the window if we can
        *************************************/
        if (Retry < 25) {
            Status = SetWindowPos(hExecWin,
                                  HWND_NOTOPMOST,
                                  Slot[SlotHit].WinX,
                                  Slot[SlotHit].WinY,
                                  Slot[SlotHit].WinWidth,
                                  Slot[SlotHit].WinHeight,
                                  SWP_NOZORDER);
        }
    }

    SetCursor(LoadCursor(NULL, IDC_ARROW));
}
