/*******************************************************************************
** EXPRESS.HPP - Expression, ExpVariable, and ExpConstant classes
** Created 5/14/95 by John Stockton
** CIS: 76041,2534
** INTERNET: jstockton%inova@mcimail.com
**
** Copyright (c) 1995 INOVA Corporation.  May be copied, used and distributed, 
** as long as no charge is made for its use and this copyright notice is not
** altered or removed.
**
** These classes allow the construction of Expressions with any combination of
** ExpVariables and ExpConstants combined by simple arithmetic and logical
** relations.  They provide evaluation operations (although any expression that
** includes variables must be based on a ExpVariable-derived class which
** can provide values for each variable at evaluation-time).  They also provide
** stream input and output operations.
**
** These expressions can be converted into CANExprs used by the Borland Database
** Engine IDAPI's DbiAddFilter() function.  For example:
**
**    Expression exp = 
**       ExpVariable( "AMOUNT") <= ExpConstant( 1500.00) ||
**       ExpVariable( "CATEGORY") == ExpConstant( 'C');
**          // Note that pCANExpr's for dBase fields do not need the
**          // field #; ExpVariable has an optional second parameter
**          // to allow the field # to be specified for those databases
**          // that require it for filtering
**
**    DbiAddFilter( hCur, 0, 1, FALSE, exp, 0, &hFilter);
**          // Note that Expression has a cast operator that allows it to 
**          // be used as a const pCANExpr
**
** Notes: To suppress the iostream output operator during compilation,
**        #define EXPRESS_NO_IOSTREAM.
**        To suppress IDAPI support (which requires IDAPI.H and ARRAYS.H),
**        #define EXPRESS_NO_IDAPI.
**        To suppress debugging message box support (via Windows MessageBox()),
**        #define EXPRESS_NO_WINDOWS
**
** $Header: j:/library/express/rcs/express.hpp 1.4 1995/06/29 15:07:14 JOHN Exp $
*******************************************************************************/

#if !defined( EXPRESS_HPP)
#define EXPRESS_HPP

#if !defined( EXPRESS_NO_IDAPI)
   #include "outbuff.hpp"      // Borland Database Engine
   #include <idapi.h>
#endif

#if !defined( EXPRESS_NO_IOSTREAM)
   #include <iostream.h>
#endif

#include <classlib\time.h>

#ifndef INT16
   #if defined(__FLAT__)
      #define INT16 short
   #else
      #define INT16 int
   #endif
#endif

#ifndef UINT16
   #if defined(__FLAT__)
      #define UINT16 unsigned short
   #else
      #define UINT16 unsigned int
   #endif
#endif

#ifndef INT32
   #define INT32 long
#endif
#ifndef UINT32
   #define UINT32 unsigned long
#endif

#ifndef BOOL
   #if defined(__FLAT__)
      #define BOOL short
   #else
      #define BOOL int
   #endif
#endif

#ifndef TRUE
   #define TRUE 1
#endif
#ifndef FALSE
   #define FALSE 0
#endif

class Expression;

class ExpElement
{
public:
   enum eElementTypes {
         ExpEXPRESSION,
         ExpVARIABLE,
         ExpCONSTANT };
   enum eEvalTypes {
         ExpUNKNOWN,
         ExpINT16,
         ExpUINT16,
         ExpINT32,
         ExpUINT32,
         ExpDOUBLE,
         ExpCHAR,
         ExpSTRING,
         ExpBOOL,
         ExpDATE,
         ExpTIME };
   enum eEvalErrors {
         ExpNONE,
         ExpDATA_UNAVAILABLE,
         ExpTYPE_MISMATCH,
         ExpOVERFLOW,
         ExpERROR,
         ExpSYNTAX_ERROR };

   ExpElement( eElementTypes _eElementType, eEvalTypes _eEvalType = ExpUNKNOWN) :
         eElementType( _eElementType), eEvalType( _eEvalType)
         {}
      // Constructor
   virtual ~ExpElement( void) {}
      // Destructor

   eElementTypes GetElementType( void) const
         { return this->eElementType; }
      // Returns the type of this element (see eElementTypes, above)
   eEvalTypes GetEvalType( void) const
         { return this->eEvalType; }
      // Returns the C type to which this Expression element evaluates
   static eEvalErrors GetEvalError( void)
         { return ExpElement::eEvalError; }
      // Returns the error associated with the last evaluation of any expression

   operator INT16( void) const;
   operator UINT16( void) const;
   operator INT32( void) const;
   operator UINT32( void) const;
   operator double( void) const;
   operator char( void) const;
   operator const char *( void) const;
   operator TDate( void) const;
   operator TTime( void) const;
      // All Expression elements will have the ability to be evaluated and to
      // return at least one C type (use GetEvalType() to determine the type
      // of the Expression element

   virtual ExpElement *Duplicate( void) const = 0;
      // Create a copy of this object and return a pointer (caller must delete!)

   Expression operator==( const ExpElement &) const;  // ExpEQUAL
   Expression operator*=( const ExpElement &) const;  // ExpLIKE
   Expression operator!=( const ExpElement &) const;  // ExpNOT_EQUAL
   Expression operator>( const ExpElement &) const;   // ExpGREATER
   Expression operator<( const ExpElement &) const;   // ExpLESS
   Expression operator>=( const ExpElement &) const;  // ExpGREATER_OR_EQUAL
   Expression operator<=( const ExpElement &) const;  // ExpLESS_OR_EQUAL
   Expression operator&&( const ExpElement &) const;  // ExpAND
   Expression operator||( const ExpElement &) const;  // ExpOR
   Expression operator+( const ExpElement &) const;   // ExpPLUS
   Expression operator-( const ExpElement &) const;   // ExpMINUS
   Expression operator*( const ExpElement &) const;   // ExpMULTIPLY
   Expression operator/( const ExpElement &) const;   // ExpDIVIDE
      // Overload binary expression construcion operators

   operator Expression( void) const;                  // ExpTRUE
   Expression operator!( void) const;                 // ExpFALSE
   Expression IsBlank( void) const;                   // ExpBLANK
   Expression IsNotBlank( void) const;                // ExpNOT_BLANK
      // Overload unary expression construcion operators

#if !defined( EXPRESS_NO_IOSTREAM)
   virtual void Output( ostream &) const = 0;
      // Ostream operator
#endif

#if !defined( EXPRESS_NO_IDAPI)
   operator const pCANExpr( void);
      // Convert this expression/variable/literal into a pCANExpr to be used in
      // Borland Database Engine DbiAddFilter() call.  Do NOT delete this pointer:
      // it will be deleted when this object is destroyed!
   virtual void BuildCANBuffers(
         OutputBuffer &_bufNodes,
         OutputBuffer &_bufLiterals,
         UINT16 &_nNodes,
         UINT16 &_nNodeOffset,
         UINT16 &_nLiteralOffset) = 0;
      // Build a node buffer and a literal buffer for this ExpElement
      // (must be provided by all derived classes).
#endif

   static void EnableMessageBox( BOOL = TRUE);
      // Enables or disables message box errors (available only if compiled
      // without EXPRESS_NO_WINDOWS)

protected:
   void SetEvalType( eEvalTypes _eEvalType)
         { this->eEvalType = _eEvalType; }
      // Sets this eEvalType
   static void SetEvalError( eEvalErrors _eEvalError, const ExpElement *, eEvalTypes);
      // Sets the evaluation error status, if not already set (see eEvalErrors,
      // above; note that this is static and must be reset by ResetEvalError, below)
   static void OverrideEvalError( eEvalErrors _eEvalError)
         { ExpElement::eEvalError = _eEvalError; }
      // Sets the evaluation error status without regard to its current status, and
      // without reporting to the user
   static void ResetEvalError( void);
      // Resets the evaluation error status
   virtual INT16 EvalINT16( void) const = 0;
   virtual UINT16 EvalUINT16( void) const = 0;
   virtual INT32 EvalINT32( void) const = 0;
   virtual UINT32 EvalUINT32( void) const = 0;
   virtual double EvalDOUBLE( void) const = 0;
   virtual char EvalCHAR( void) const = 0;
   virtual const char *EvalSTRING( void) const = 0;
   virtual BOOL EvalBOOL( void) const = 0;
   virtual TDate EvalDATE( void) const = 0;
   virtual TTime EvalTIME( void) const = 0;
      // Evaluation functions; derived classes must provide overrides for all
      // of these (even if they just return 0).

private:
   eElementTypes eElementType;
   eEvalTypes eEvalType;
   OutputBuffer bufCANExpr;
      // CANExpr buffer returned by operator const pCANExpr(), above.
   static eEvalErrors eEvalError;
   static BOOL bMessageBoxEnabled;
};


class ExpVariable : public ExpElement
{
public:
   ExpVariable(
         const char *_szName,
         ExpElement::eEvalTypes = ExpElement::ExpUNKNOWN,
         const char *_szDescription = 0,
         UINT16 nFieldNumber = 0);
      // Constructor
   ExpVariable( const ExpVariable &);
      // Copy constructor
   ~ExpVariable( void);
      // Destructor

   virtual ExpElement *Duplicate( void) const
         { return new ExpVariable( *this); }
      // Create a copy of this object and return a pointer (caller must delete!)

#if !defined( EXPRESS_NO_IOSTREAM)
   void Output( ostream &) const;
      // Output to stream
   friend ostream &operator<<( ostream &, const ExpVariable &);
      // Output stream operator
#endif

#if !defined( EXPRESS_NO_IDAPI)
   virtual void BuildCANBuffers(
         OutputBuffer &bufNodes,
         OutputBuffer &bufLiterals,
         UINT16 &nNodes,
         UINT16 &_nNodeOffset,
         UINT16 &_nLiteralOffset);
      // Build a node buffer and a literal buffer for this ExpElement
#endif

protected:
   INT16 EvalINT16( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpINT16); return 0; }
   UINT16 EvalUINT16( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpUINT16); return 0; }
   INT32 EvalINT32( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpINT32); return 0; }
   UINT32 EvalUINT32( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpUINT32); return 0; }
   double EvalDOUBLE( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpDOUBLE); return 0; }
   char EvalCHAR( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpCHAR); return 0; }
   const char *EvalSTRING( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpSTRING); return 0; }
   BOOL EvalBOOL( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpBOOL); return 0; }
   TDate EvalDATE( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpDATE); return TDate(); }
   TTime EvalTIME( void) const
         { SetEvalError( ExpElement::ExpDATA_UNAVAILABLE, this, ExpElement::ExpTIME); return 0; }
      // Evaluation functions; derived classes must provide overrides for all
      // of these (even if they just return 0).

private:
   char *szName;
      // Field name
   char *szDescription;
      // Field description (optional; 0 = none)
   UINT16 nFieldNumber;
      // Field number (optional; 0 = none)
};


class ExpConstant : public ExpElement
{
public:
   ExpConstant( INT16);
   ExpConstant( UINT16);
   ExpConstant( INT32);
   ExpConstant( UINT32);
   ExpConstant( double);
   ExpConstant( char);
   ExpConstant( const char *);
   ExpConstant( const TDate &);
   ExpConstant( const TTime &);
      // Constructors: overloaded for any basic C type and TDate and TTime
   ExpConstant( const ExpConstant &);
      // Copy constructor
   ~ExpConstant( void);
      // Destructor

   virtual ExpElement *Duplicate( void) const
         { return new ExpConstant( *this); }
      // Create a copy of this object and return a pointer (caller must delete!)

#if !defined( EXPRESS_NO_IOSTREAM)
   void Output( ostream &) const;
      // Output to stream
   friend ostream &operator<<( ostream &, const ExpConstant &);
      // Output stream operator
#endif

#if !defined( EXPRESS_NO_IDAPI)
   virtual void BuildCANBuffers(
         OutputBuffer &bufNodes,
         OutputBuffer &bufLiterals,
         UINT16 &nNodes,
         UINT16 &_nNodeOffset,
         UINT16 &_nLiteralOffset);
      // Build a node buffer and a literal buffer for this ExpElement
#endif

protected:
   INT16 EvalINT16( void) const;
   UINT16 EvalUINT16( void) const;
   INT32 EvalINT32( void) const;
   UINT32 EvalUINT32( void) const;
   double EvalDOUBLE( void) const;
   char EvalCHAR( void) const;
   const char *EvalSTRING( void) const;
   BOOL EvalBOOL( void) const;
   TDate EvalDATE( void) const;
   TTime EvalTIME( void) const;
      // Evaluation functions; derived classes must provide overrides for all
      // of these (even if they just return 0).

private:
   void *pValue;
};


class Expression : public ExpElement
{
public:
   enum eOperators {
      // Binary logical operators (require operand):
         ExpEQUAL,
         ExpLIKE,
         ExpNOT_EQUAL,
         ExpGREATER,
         ExpLESS,
         ExpGREATER_OR_EQUAL,
         ExpLESS_OR_EQUAL,
         ExpAND,
         ExpOR,
      // Binary arithmetic operators
         ExpPLUS,
         ExpMINUS,
         ExpMULTIPLY,
         ExpDIVIDE,
      // Unary operators (no operand):
         ExpTRUE,
         ExpFALSE,
         ExpBLANK,
         ExpNOT_BLANK
      };

   Expression( const ExpElement &, eOperators, const ExpElement &);
      // Binary expression constructor
   Expression( const ExpElement &, eOperators);
      // Unary expression constructor
   Expression( const Expression &);
      // Copy constructor
   ~Expression( void);
      // Destructor

   eOperators GetOperator( void) const
         { return this->eOperator; }
      // Returns the operator for this Expression

   inline BOOL IsBinaryOperator( void) const;
      // Returns TRUE if a valid binary operator was specified for this Expression.
   inline BOOL IsUnaryOperator( void) const;
      // Returns TRUE if a valid unary operator was specified for this Expression.
   inline BOOL IsLogicalOperator( void) const;
      // Returns TRUE if this expression's operator is logical (evaluates to BOOL).
   inline BOOL IsArithmeticOperator( void) const;
      // Returns TRUE if this expression's operator is arithmetic.

#if !defined( EXPRESS_NO_IOSTREAM)
   void Output( ostream &) const;
      // Output to stream
   friend ostream &operator<<( ostream &, const Expression &);
   static void SetVerbose( BOOL bVerbose)
         { Expression::bVerbose = bVerbose; }
      // Turn verbose mode on (i.e. "is less than") or off (i.e. "<") (default is off)
   static BOOL bVerbose;
#endif

#if !defined( EXPRESS_NO_IDAPI)
   virtual void BuildCANBuffers(
         OutputBuffer &bufNode,
         OutputBuffer &bufLiteral,
         UINT16 &nNodes,
         UINT16 &_nNodeOffset,
         UINT16 &_nLiteralOffset);
      // Build a node buffer and a literal buffer for this ExpElement
   CANOp GetCANOp( void);
      // Returns the CANOp equivalent of this ExpElement's operator
#endif

protected:
   virtual ExpElement *Duplicate( void) const
         { return new Expression( *this); }
      // Create a copy of this object and return a pointer (caller must delete!)
   INT16 EvalINT16( void) const;
   UINT16 EvalUINT16( void) const;
   INT32 EvalINT32( void) const;
   UINT32 EvalUINT32( void) const;
   double EvalDOUBLE( void) const;
   char EvalCHAR( void) const;
   const char *EvalSTRING( void) const;
   BOOL EvalBOOL( void) const;
   TDate EvalDATE( void) const;
   TTime EvalTIME( void) const;
      // Evaluation functions; derived classes must provide overrides for all
      // of these (even if they just return 0).

private:
   ExpElement *pExpLeft, *pExpRight;
   eOperators eOperator;
};

#endif
