/*
        enforcer.cpp -- police memory
        (C) Copyright 1996  John Webster Small
        All rights reserved
*/

#include "enforcer.h"

#ifndef ENFORCER_NDEBUG

  cl * Enforcer_destructed::newInstance()
  {
    cl * B = new cl(CL_BIND_ONLY);
    if (B) B->setMaxNodes
        (ENFORCER_DESTRUCTED_MAXNODES);
    return B;
  }

#endif

void Enforcer:: release(Enforcer * E)
{
    assert(E);
    if (!E->unlink())  {
        if (E->storage() == DYNAMIC_STORAGE)
            delete E;
        else if (E->storage() == DYNAMIC_ARRAY_STORAGE)
            (*(E->deleteArrayFnc()))(E);
    }
}

#ifndef ENFORCER_NDEBUG

  Enforcer::HeapAudit::HeapAudit
    (void * V, size_t size, size_t sizeofType,
     ENFORCER_STORAGE storage_,
     const char * typeid_name,
     const char * FILE, unsigned LINE)
    : _V(V), _size(size), _sizeofType(sizeofType),
      _storage(storage_),
      _typeid_name(), _FILE(), _LINE(LINE)
  {
    cstr tmp;
    _typeid_name.link2(tmp.cpy(typeid_name));
    _FILE.link2(tmp.cpy(FILE));
  }

  ostream& Enforcer::HeapAudit::locStats
    (ostream& os, const char * typeid_name,
     const char * FILE, unsigned LINE)
  {
    return  os
      << setiosflags(ios::left)
      << setw(14)
      << typeid_name
      << setiosflags(ios::left)
      << setw(14)
      << FILE
      << setw(7)
      << LINE;
  }

  ostream& Enforcer::HeapAudit::logColumns
    (ostream& os)
  {
    return  os
      << setiosflags(ios::left)
      << setw(14)
      << "TYPEID_NAME"
      << setiosflags(ios::left)
      << setw(14)
      << "FILE"
      << setw(7)
      << "LINE"
      << setw(6) << "TSIZE"
      << setw(6) << "SIZE"
      << " @ ADDRESS     COMMENT:";
  }

  void Enforcer::HeapAudit::audit
    (ostream& os, const char * msg, const void * V)
     const
  {
    locStats(os,
      (_typeid_name.readReady()?
         ((const cstr&)_typeid_name).str() : "void"),
      (_FILE.readReady()?
          ((const cstr&)_FILE).str() : "unknown"),
      _LINE)
      << setw(6) << _sizeofType
      << setw(6) << _size
      << " @ " << _V;
      if (msg)
          os  << "  "  <<  msg;
      size_t idx = interiorCell(V);
      if (idx)
          os << "\n\tinterior cell[" << idx << ']';
      os << endl;
  }

  size_t Enforcer::HeapAudit::interiorCell
      (const void * V) const
  // versus base cell.
  // assert that V falls somewhere in allocation
  // i.e. cmp(V) == 0!
  {
    if (!V)
        return 0;
    ptrdiff_t offset = (ptrdiff_t)
      ((unsigned long int)V) - ((unsigned long int)_V);
    if (offset <= 0)
        return 0;
    return size_t(offset/_sizeofType);
  }

  int Enforcer::HeapAudit::cmp(const void * V) const
  {
    ptrdiff_t signum = (ptrdiff_t)
      ((unsigned long int)V) - ((unsigned long int)_V);
    if (signum > 0)
        if (((unsigned long int)signum)
            < ((unsigned long int)_size))
            return 0;
        else
            return -1;
    if (signum < 0)
        return 1;
    return 0;
  }

  Enforcer::HeapAuditor::~HeapAuditor()
  {
    if (nodes())
        audit(ENF_CERR,0,"memory leak!");
  }

  void Enforcer::HeapAuditor::open
    (void * V, size_t size,
    size_t sizeofType,
    ENFORCER_STORAGE storage,
    const char * typeid_name,
    const char * FILE, unsigned LINE)
  {
    HeapAudit * R = insSortDL
        (new HeapAudit
            (V,size,sizeofType,storage,
               typeid_name,FILE,LINE));
    if (_trace && R)
        R->audit(ENF_CERR,"new");
  }

  int Enforcer::HeapAuditor::close
    (void * V, ENFORCER_STORAGE storage)
  {
    if (CL_find<HeapAudit,void *>(*this,&V))
    {
        HeapAudit * HA = get();
        if (_trace)
            HA->audit(ENF_CERR,"delete",V);
        if (HA->interiorCell(V))  {
            HA->audit(ENF_CERR,"error: delete",V);
            return 0;
        }
        if (HA->storage() == DYNAMIC_ARRAY_STORAGE
            && storage != DYNAMIC_ARRAY_STORAGE)  {
            HA->audit(ENF_CERR,"error: use delete[]");
        }
        else if (HA->storage() == DYNAMIC_STORAGE
            && storage != DYNAMIC_STORAGE)  {
            HA->audit(ENF_CERR,"error: use delete");
        }
        (void) delBackup();
        return 1;
    }
    ENF_CERR << "error: attempt to delete static "
       << "storage @ " << V << endl;
    return 0;
  }

  void Enforcer::HeapAuditor::audit
    (ostream& os, const void * V, const char * msg)
  {
    if (V)  {
        if (CL_find<HeapAudit,const void *>(*this,&V))
            get()->audit(os,msg,V);
    }
    else  {
        if (nodes())
            HeapAudit::logColumns
                (os << "\nENFORCER AUDIT:\n") << endl;
        CL_forEach(*this)
            get()->audit(os,msg);
   }
  }

  void Enforcer::HeapAuditor::traceOn
    (ostream& os, const char * FILE, unsigned LINE)
  {
    _trace = 1;
    HeapAudit::locStats(os,"HeapAuditor",FILE,LINE)
        << "TRACE ON" << endl;
    HeapAudit::logColumns(os) << endl;
  }

  void Enforcer::HeapAuditor::traceOff
    (ostream& os, const char * FILE, unsigned LINE)
  {
    _trace = 0;
    HeapAudit::locStats(os,"HeapAuditor",FILE,LINE)
          << "TRACE OFF" << endl;
  }

#endif

ENFORCER_STORAGE Enforcer:: lastAlloc
    (const void * E, size_t & lastSize,
     size_t sizeofType, ENFORCER_STORAGE set)
{
    static size_t top = 0;
    static ENFORCER_STORAGE tokenStack[MAX_ENFORCER_STACK]
          = { STATIC_STORAGE };
    static size_t sizeStack[MAX_ENFORCER_STACK]
          = { 0 };
    static size_t sizeofTypeStack[MAX_ENFORCER_STACK]
          = { 0 };
    static const void * EStack[MAX_ENFORCER_STACK] = { 0 };
    if (E) if (set == READ_TOKEN)  {
       lastSize = 0;
       if (top)  {
          top--;
          if ((((unsigned long int)E) >=
             ((unsigned long int)EStack[top]))
            && (((unsigned long int)E)
             < (((unsigned long int)EStack[top])
            + sizeStack[top]))) {
            ptrdiff_t offset =
              ((unsigned long int)E) -
              ((unsigned long int)EStack[top]);
            if (offset >= 0)  {
                lastSize = sizeStack[top] - offset;
                if ((offset/sizeofTypeStack[top]) == 0)
                    return tokenStack[top];
                top++;
                return DYNAMIC_ARRAY_INTERIOR_STORAGE;
            }
          }
          top++;
       }
    }
    else {
       assert(top < MAX_ENFORCER_STACK);
       tokenStack[top] = set;
       sizeStack[top] = lastSize;
       sizeofTypeStack[top] = sizeofType;
       EStack[top++] = E;
    }
    return STATIC_STORAGE;
}

#ifndef ENFORCER_NDEBUG

  void * Enforcer:: alloc(size_t size,
    size_t sizeofType,
    ENFORCER_STORAGE storage,
    const char * typeid_name,
    const char * FILE, unsigned LINE)
  {
    void * V = ::operator new(size);
    if (V)  {
        static Singleton<Enforcer,HeapAuditor> trail;
        if (trail)
            trail->open
               (V,size,sizeofType,storage,
                  typeid_name,FILE,LINE);
        (void) lastAlloc(V,size,sizeofType,storage);
    }
    return V;
  }

  void Enforcer:: free(void * V,
    ENFORCER_STORAGE storage,
    const char * typeid_name)
  {
    static Singleton<Enforcer,HeapAuditor> trail;
    if (trail.ok())  {
        if (!trail->close(V,storage))
            return;
    }
    else
        ENF_CERR << setiosflags(ios::left) << setw(16)
             << typeid_name << "  " << V
             << "  deleted - not leaked!"
             << endl;
    ::operator delete(V);
  }

  Enforcer:: Enforcer() : Pedigree(), _links(0U)
  {
      _storage = lastAlloc(this,_size,0);
      if (destructed.ok())
          (void) destructed->atRmv
            (destructed->indexOf(this));
        // reuse address: discard destructed listing
  }

  Enforcer::~Enforcer()
  {
      if (_links)
          audit(ENF_CERR,this,"references remaining");
      if (destructed)  {
          if (!destructed->vacancy())
              for (unsigned i = 0;
                  i < destructed->delta(); i++)
                   (void) destructed->unQ();
          (void) destructed->push(this);
      }
  }

  void Enforcer::audit
    (ostream& os, const void * V, const char * msg)
  {
    static Singleton<Enforcer,HeapAuditor> trail;
    if (trail)
        trail->audit(os,V,msg);
  }

  void Enforcer::traceOn
    (ostream& os, const char * FILE, unsigned LINE)
  {
    static Singleton<Enforcer,HeapAuditor> trail;
    if (trail)
        trail->traceOn(os,FILE,LINE);
  }

  void Enforcer::traceOff
    (ostream& os, const char * FILE, unsigned LINE)
  {
    static Singleton<Enforcer,HeapAuditor> trail;
    if (trail)
        trail->traceOff(os,FILE,LINE);
  }

#endif

void pEnforcer::point2(Enforcer * E_)
{
    if (_E)
        Enforcer::release(_E);
    if ((_E = E_) != 0)  {
  #ifndef ENFORCER_NDEBUG
        if (destructed.ok() &&
           (destructed->indexOf(_E) != CL_NOTFOUND))  {
            ENF_CERR
                 << "Enforcer object already deleted: "
                 << _E << endl;
            _E = 0;
            return;
        }
  #endif
        if (!_E->link())
            _E = 0;
    }
}

#ifndef ENFORCER_NDEBUG

  void pEnforcer::assertE
      (const char * typeid_name) const
  {
      if (!_E)  {
          ENF_CERR << typeid_name
              << " attempted NULL access" << endl;
          int NullAccess = 1;
          assert(!NullAccess);
      }
      if (destructed.ok())
        if (destructed->indexOf(_E) != CL_NOTFOUND)  {
          ENF_CERR << typeid_name
              << " object already deleted: "
              << _E << endl;
          int ObjectDeleted = 1;
          assert(!ObjectDeleted);
        }
  }

  void pEnforcer::assertEIdx
      (const char * typeid_name, size_t n,
       const char * cellTypeid_name) const
  {
      assertE(typeid_name);
      if (n >= _E->cells())  {
          Enforcer::audit(ENF_CERR,_E,0);
          ENF_CERR << '\t' << typeid_name << '[' << n
              << "] range violation" << endl;
          int IndexOutOfBounds = 1;
          assert(!IndexOutOfBounds);
      }
      cstr actualCellType;
      if (strcmp(cellTypeid_name,
          _E->typeid_name(actualCellType)))
      {
          Enforcer::audit(ENF_CERR,_E,0);
          ENF_CERR << "\tattempt to access "
              << typeid_name << '[' << n
              << "].  Array is of type "
              << actualCellType.str() << endl;
          int WrongArrayType = 1;
          assert(!WrongArrayType);
      }
  }

#endif

void CL_EnforcerTraits:: _delete(Enforcer * I) throw()
{ pEnforcer deleteLeak(I); }

void CL_EnforcerTraits::detach(Enforcer& i, const cl *)
   throw()
{ (void) i.unlink(); }
