/*
        cow.h -- copy on write
        (C) Copyright 1996  John Webster Small
        All rights reserved
        Demo: CC cowdemo cstr

        Use COW<TYPE> to minimize duplicates.  For
        example, check writeReady() before modifying
        your private copy and fold() afterwards to
        reduce duplicates.

        TYPE must have:

           TYPE();  //i.e. default constructor
           stream insert/extraction operators
           int cmp(const TYPE &) const;

        If your type doesn't have a cmp() function you
        can define the following function instead:

           inline
           int COW_cmp(const TYPE& t1, const TYPE& t2)
           { return ??? }
*/

#ifndef cow_h
#define cow_h

/* LINTLIBRARY */

#ifndef cl_h
  #include "cl.h"
#endif

#ifndef single_h
  #include "single.h"
#endif

template <class TYPE>
inline int COW_cmp(const TYPE& t1, const TYPE& t2)
{ return t1.cmp(t2); }

template <class TYPE>
struct COW_OBJ {
    TYPE _t;
    unsigned links;
    COW_OBJ() : _t(), links(0) {}
    COW_OBJ(const TYPE & t) : _t(t), links(0) {}
    int cmp(const TYPE& t) const
        { return COW_cmp(_t,t); }
    int cmp(const COW_OBJ<TYPE >& c) const
        { return COW_cmp(_t,c._t); }
};

template <class TYPE>
class COW
{
  private:
    Singleton<COW<TYPE >,CL_cmp<COW_OBJ<TYPE > > >
        instances;
    COW_OBJ<TYPE > * C;
    COW_OBJ<TYPE > * new_COW_OBJ(const TYPE& t);
  public:
    COW() : C(0) {}
    COW(const TYPE & t) : C(0) { link2(t); }
    ~COW() { unlink(); }
    void link2(const TYPE & t);
    void operator()(const TYPE & t) { link2(t); }
    void unlink();
    unsigned links() const
        { return (C? C->links : 0); }
    void fold();
    TYPE * writeReady();
    const TYPE * readReady() const
        { return (C? &(C->_t) : 0); }
    operator TYPE *() const
        { return (C? &(C->_t) : 0); }
    operator const TYPE &() const
        { assert(C); return C->_t; }
    operator TYPE &()
        { TYPE * T = writeReady();
          assert(T); return *T; }
    friend ostream& operator<<
        (ostream& os, COW<TYPE >& c)
        { if (c.C) os << c.C->_t; return os; }
    friend istream& operator>>
        (istream& is, COW<TYPE >& c)
        { if (c.writeReady())
             { is >> c.C->_t; c.fold(); }
          return is; }
};

template <class TYPE>
void COW<TYPE >::link2(const TYPE & t)
{
    unlink();
    if (instances)  {
        C = CL_find<COW_OBJ<TYPE >,TYPE>
              (*instances,&t);
        if (!C)
            C = instances->insSortDL
              (new COW_OBJ<TYPE >(t));
        if (C)
            C->links++;
    }
}

template <class TYPE>
void COW<TYPE >::unlink()
{
    if (C)  {
        assert(C->links);
        if (!--C->links)  {
            if (instances.ok()
                && instances->findFirst(C))
                do {
                    if (C == instances->get())  {
                        instances->delBackup();
                        break;
                    }
                } while (instances->findNext(C));
        }
        C = 0;
    }
}

template <class TYPE>
void COW<TYPE >::fold()
{
    if (C && C->links == 1)  {
        assert(instances.ok());
        if (instances->findFirst(C))
            if (C == instances->get())  {
                unsigned discard
                    = instances->curNode();
                if (instances->findNext(C))  {
                    C = instances->get();
                    C->links++;
                    instances->atDel(discard);
                }
            }
            else  {
                unsigned save = instances->curNode();
                while (instances->findNext(C))
                    if (C == instances->get())  {
                        C = instances->atGet(save);
                        C->links++;
                        instances->delBackup();
                        break;
                    }
            }
    }
}

template <class TYPE>
TYPE * COW<TYPE >::writeReady()
{
    if (instances)  {
        if (!C)  {
            if ((C = instances->insSortDL
                (new COW_OBJ<TYPE >))
                == 0)
                return 0;
        }
        else if (C->links > 1)  {
            COW_OBJ<TYPE > * _C =
                instances->insSortDL
                (new COW_OBJ<TYPE >(C->_t));
            if (!_C)
                return 0;
            C->links--;
            C = _C;
        }
        C->links = 1;
        return &(C->_t);
    }
    return 0;
}

#endif
