/*
        single.h -- allow only a single instance
          per <SCOPE,TYPE,ID> tuple!
        (C) Copyright 1996  John Webster Small
        All rights reserved

        TYPE must selectable and have a default
        constructor. Singletons of arrays are not
        allowed!

          Singleton<GLOBAL_SCOPE,Foo> onlyFoo;
          if (onlyFoo)
              onlyFoo->???;
*/

#ifndef single_h
#define single_h

/* LINTLIBRARY */

#ifdef SINGLETON_DEBUG
  #include <iostream.h>
#endif

#ifndef assert_h
  #include <assert.h>
  #ifndef assert_h
    #define assert_h
  #endif
#endif

#ifndef limits_h
  #include <limits.h>
  #ifndef limits_h
    #define limits_h
  #endif
#endif

#ifndef GLOBAL_SCOPE
  #define SCOPE_TYPE(scope) scope ## _SCOPE
  #define SCOPE_TYPEDEF(scope) \
    struct SCOPE_TYPE(scope) {}
  SCOPE_TYPEDEF(global);
  #define GLOBAL_SCOPE SCOPE_TYPE(global)
#endif

template <class SCOPE, class TYPE, int ID = 0>
struct Singleton_traits
// struct Singleton_traits<SCOPE,TYPE,ID>
{
    TYPE * newInstance() { return new TYPE; }
};

enum SingletonOp
    { SGLTN_RELEASE, SGLTN_CREATE, SGLTN_PRESENT };

template <class SCOPE, class TYPE, class TRAITS, int ID = 0>
class Singleton_Base
{
  public:
    unsigned long refs(int inc = 0) const;
    TYPE * instance
        (SingletonOp op = SGLTN_CREATE) const;
    Singleton_Base() { (void) refs(1); }
    ~Singleton_Base() { if (!refs(-1)) (void)
          instance(SGLTN_RELEASE); }
};

template <class SCOPE, class TYPE, class TRAITS, int ID>
unsigned long
Singleton_Base<SCOPE,TYPE,TRAITS,ID>::refs(int inc) const
{
    static unsigned long _refs = 0;
    if (inc > 0)  {
        assert(_refs < ULONG_MAX);
        _refs++;
        #ifdef SINGLETON_DEBUG
           cerr << "Singleton<"
               << typeid(SCOPE).name() << ','
               << typeid(TYPE).name() << ',' << ID
               << ">.refs() = " << _refs << '\n';
        #endif
    }
    else if (inc < 0) {
        #ifdef SINGLETON_DEBUG
           cerr << "Singleton<"
               << typeid(SCOPE).name() << ','
               << typeid(TYPE).name() << ',' << ID
               << ">.refs() = " << _refs << '\n';
        #endif
        assert(_refs);
        _refs--;
    }
    return _refs;
}

template <class SCOPE, class TYPE, class TRAITS, int ID>
TYPE *
Singleton_Base<SCOPE,TYPE,TRAITS,ID>::
    instance(SingletonOp op) const
{
    static TYPE * _instance = 0;
    if (op != SGLTN_PRESENT)
        if (op == SGLTN_CREATE)  {
            if (!_instance && refs())
                _instance = TRAITS().newInstance();
        }
        else if (_instance)  { // SGLTN_RELEASE
            delete _instance;
            _instance = 0;
        }
    return _instance;
};

template <class SCOPE, class TYPE, class TRAITS, int ID = 0>
class Singleton_ : Singleton_Base<SCOPE,TYPE,TRAITS,ID>
{
  public:
    Singleton_() : Singleton_Base<SCOPE,TYPE,TRAITS,ID>() {}
    unsigned long refs() const { return
        Singleton_Base<SCOPE,TYPE,TRAITS,ID>::refs(); }
    operator TYPE * () const
        { return instance(SGLTN_CREATE); }
    TYPE * ok() const
        { return instance(SGLTN_PRESENT); }
    operator TYPE & () const
        { TYPE * T = instance(SGLTN_CREATE);
          assert(T);  return *T; }
    TYPE * operator->() const
        { TYPE * T = instance(SGLTN_CREATE);
          assert(T); return T; }
    void release() const
        { (void) instance(SGLTN_RELEASE); }
};

template <class SCOPE, class TYPE, int ID = 0>
class Singleton : public
  Singleton_<SCOPE,TYPE,
    Singleton_traits<SCOPE,TYPE,ID>,ID>
{
  public:
    Singleton()
      : Singleton_<SCOPE,TYPE,
          Singleton_traits<SCOPE,TYPE,ID>,ID>() {}
};

#endif

