/*
        enforcer.h -- police memory
        (C) Copyright 1996  John Webster Small
        All rights reserved
*/

#ifndef enforcer_h
#define enforcer_h

/* LINTLIBRARY */

#ifndef MAX_ENFORCER_STACK
  #define MAX_ENFORCER_STACK 10
#endif

#ifndef ENFORCER_DESTRUCTED_MAXNODES
  #define ENFORCER_DESTRUCTED_MAXNODES 1024
#endif

#ifdef NDEBUG
  #ifndef ENFORCER_NDEBUG
    #define ENFORCER_NDEBUG
  #endif
#endif

#ifdef ENFORCER_NDEBUG
  #define newAudit new
  #define HEAP_AUDIT(OS,V,MSG)
  #define HEAP_TRACE_ON(OS)
  #define HEAP_TRACE_OFF(OS)
  #define ENFORCER_DERIVED(DERIVED,BASE) \
    private: \
      static void deleteArray(Enforcer * E) \
        { delete[] dynamic_cast<DERIVED *>(E); } \
    protected: \
      virtual void (*deleteArrayFnc())(Enforcer *)  \
         { return deleteArray; } \
    public: \
      static void * operator new (size_t size) \
      { \
          void * V = ::operator new(size); \
          (void) lastAlloc(V,size,sizeof(DERIVED), \
              DYNAMIC_STORAGE); \
          return V; \
      } \
      static void * operator new[] (size_t size) \
      { \
          void * V = ::operator new(size); \
          (void) lastAlloc(V,size,sizeof(DERIVED), \
              DYNAMIC_ARRAY_STORAGE); \
          return V; \
      } \
      static void operator delete(void * E) \
      { ::operator delete(E); } \
      static void operator delete[](void * E) \
      { ::operator delete(E); } \
      PEDIGREE_DERIVED(DERIVED,BASE)
#else
  #define ENF_CERR cerr
  #ifndef cow_h
    #include "cow.h"
  #endif
  #define newAudit new(__FILE__,__LINE__)
  #define HEAP_AUDIT(OS,V,MSG)  \
    Enforcer::audit(OS,V,MSG);
  #define HEAP_TRACE_ON(OS)  \
    Enforcer::traceOn(OS,__FILE__,__LINE__);
  #define HEAP_TRACE_OFF(OS) \
    Enforcer::traceOff(OS,__FILE__,__LINE__);
  #define ENFORCER_DERIVED(DERIVED,BASE) \
    private: \
      static void deleteArray(Enforcer * E) \
        { delete[] dynamic_cast<DERIVED *>(E); } \
    protected: \
      virtual void (*deleteArrayFnc())(Enforcer *)  \
         { return deleteArray; } \
    public: \
      static void * operator new (size_t size) \
      { return Enforcer::alloc(size,sizeof(DERIVED), \
        DYNAMIC_STORAGE,typeid(DERIVED).name(), \
        "unknown",0); } \
      static void * operator new \
        (size_t size, const char * FILE, unsigned LINE) \
      { return Enforcer::alloc(size,sizeof(DERIVED), \
        DYNAMIC_STORAGE,typeid(DERIVED).name(), \
        FILE,LINE); } \
      static void * operator new[] (size_t size) \
      { return Enforcer::alloc \
          (size,sizeof(DERIVED),DYNAMIC_ARRAY_STORAGE, \
           typeid(DERIVED).name(),"unknown",0); } \
      static void * operator new[] \
        (size_t size, const char * FILE, unsigned LINE) \
      { return Enforcer::alloc \
          (size,sizeof(DERIVED),DYNAMIC_ARRAY_STORAGE, \
        typeid(DERIVED).name(),FILE,LINE); } \
      static void operator delete(void * E) \
      { Enforcer::free(E,DYNAMIC_STORAGE, \
         typeid(DERIVED).name()); } \
      static void operator delete[](void * E) \
      { Enforcer::free(E,DYNAMIC_ARRAY_STORAGE, \
         typeid(DERIVED).name()); } \
      PEDIGREE_DERIVED(DERIVED,BASE)
#endif

#ifndef cl_h
  #include  "cl.h"
#endif

#ifndef cstr_h
  #include "cstr.h"
#endif

#ifndef pedigree_h
  #include "pedigree.h"
#endif

enum ENFORCER_STORAGE
{
    READ_TOKEN,
    STATIC_STORAGE = READ_TOKEN,
    DYNAMIC_STORAGE,
    DYNAMIC_ARRAY_STORAGE,
    DYNAMIC_ARRAY_INTERIOR_STORAGE
};

#ifndef ENFORCER_NDEBUG
   struct Enforcer_destructed
      { cl * newInstance(); };
#endif

class pEnforcer;

struct CL_EnforcerTraits;

class Enforcer : public virtual Pedigree
{
    unsigned _links;
    size_t _size;
    ENFORCER_STORAGE _storage;
    friend CL_EnforcerTraits;
      unsigned unlink()
       { assert(_links); return --_links; }
      friend pEnforcer;
        unsigned link()
        { return ((_links < UINT_MAX)? ++_links : 0); }
        static void release(Enforcer * E);//assert(E)!
    #ifndef ENFORCER_NDEBUG
      Singleton_<Enforcer,cl,Enforcer_destructed>
        destructed;
      class HeapAudit
      {
          void * _V;
          size_t _size;
          size_t _sizeofType;
          ENFORCER_STORAGE _storage;
          COW<cstr> _typeid_name, _FILE;
          unsigned  _LINE;
        public:
          HeapAudit(void * V, size_t size,
            size_t sizeofType,
            ENFORCER_STORAGE storage_,
            const char * typeid_name,
            const char * FILE, unsigned LINE);
          void audit(ostream& os,
            const char * msg = "heap audit!",
            const void * V = 0) const;
          ENFORCER_STORAGE storage() const
              { return _storage; }
          size_t cells() const
              { return _size/_sizeofType; }
          size_t interiorCell(const void * V) const;
              // versus base cell.
              // assert V falls somewhere in allocation
              // i.e. cmp(V) == 0!
          int cmp(const void * V) const;
          int cmp(const HeapAudit& a) const
              { return cmp(a._V); }
          static ostream& locStats
              (ostream& os, const char * typeid_name,
               const char * FILE, unsigned LINE);
          static ostream& logColumns(ostream& os);
      };
      class HeapAuditor : CL_cmp<HeapAudit>
      {
          int _trace;
        public:
          HeapAuditor() : CL_cmp<HeapAudit>(), _trace(0) {}
          ~HeapAuditor();
          void open(void * V, size_t size,
             size_t sizeofType,
             ENFORCER_STORAGE storage,
             const char * typeid_name,
             const char * FILE, unsigned LINE);
          int close(void * V, ENFORCER_STORAGE storage);
          void audit(ostream& os, const void * V = 0,
            const char * msg = 0);
          void traceOn(ostream& os, const char * FILE,
            unsigned LINE);
          void traceOff(ostream& os, const char * FILE,
            unsigned LINE);
      };
    #endif
  protected:
    static ENFORCER_STORAGE lastAlloc
      (const void * E, size_t & lastSize,
       size_t sizeofType,
       ENFORCER_STORAGE set = READ_TOKEN);
    #ifndef ENFORCER_NDEBUG
      static void * alloc(size_t size,
        size_t sizeofType,
        ENFORCER_STORAGE storage,
        const char * typeid_name,
        const char * FILE, unsigned LINE);
      static void free(void * V,
        ENFORCER_STORAGE storage,
        const char * typeid_name);
    #endif
  public:
    #ifdef ENFORCER_NDEBUG
      Enforcer() : Pedigree(), _links(0U)
        { _storage = lastAlloc(this,_size,0); }
    #else
      Enforcer();
      virtual ~Enforcer();
    #endif
    ENFORCER_DERIVED(Enforcer,Pedigree)
    unsigned links() { return _links; }
    virtual size_t cells()
       { return (_size? _size/sizeofType() : 0); }
    ENFORCER_STORAGE storage()
        { return _storage; }
#ifndef ENFORCER_NDEBUG
    static void audit(ostream& os, const void * V = 0,
      const char * msg = 0);
    static void traceOn
      (ostream& os, const char * FILE, unsigned LINE);
    static void traceOff
      (ostream& os, const char * FILE, unsigned LINE);
#endif
};

class pEnforcer
{
  protected:
    Enforcer * _E;
    void point2(Enforcer * E_);
    #ifdef ENFORCER_NDEBUG
      void assertE(const char *) const {}
      void assertEIdx(const char *, size_t,
          const char *) const {}
    #else
      Singleton_<Enforcer,cl,Enforcer_destructed>
          destructed;
      void assertE(const char * typeid_name) const;
      void assertEIdx(const char * typeid_name,
          size_t n, const char * cellTypeid_name)
          const;
    #endif
  public:
    pEnforcer(Enforcer * E_ = 0)
        : _E(0) { point2(E_); }
    ~pEnforcer() { point2(0); }
    Enforcer * E() const { return _E; }
};

template <class TYPE>
class ENF : public pEnforcer
{
  protected:
    ENF<TYPE >& point2(Enforcer * E_);
  public:
    ENF(TYPE * E_ = 0) : pEnforcer()
        { (void) point2(E_); }
    ENF(TYPE& e) : pEnforcer()
        { (void) point2(&e); }
    ENF(const pEnforcer& p) : pEnforcer()
        { (void) point2(p.E()); }
    ENF<TYPE >& operator()(Enforcer * E_)
        { return point2(E_); }
    ENF<TYPE >& operator()(Enforcer& e)
        { return point2(&e); }
    ENF<TYPE >& operator()(pEnforcer& p)
        { return point2(p.E()); }
    ENF<TYPE >& operator=(Enforcer * E_)
        { return point2(E_); }
    ENF<TYPE >& operator=(Enforcer& e)
        { return point2(&e); }
    ENF<TYPE >& operator=(pEnforcer& p)
        { return point2(p.E()); }
    ENF<TYPE >& operator=(int i)
        { assert(!i); return point2(0); }
    operator TYPE * () const
        { return (TYPE *)_E; }
    operator TYPE & () const
        { assertE(typeid(ENF<TYPE >).name());
          return * (TYPE *)_E; }
    TYPE * operator->() const
        { assertE(typeid(ENF<TYPE >).name());
          return (TYPE *)_E; }
    TYPE& operator*() const
        { assertE(typeid(ENF<TYPE >).name());
          return * (TYPE *)_E; }
    TYPE& operator[](size_t n) const
        { assertEIdx(typeid(ENF<TYPE >).name(),n,
          typeid(TYPE).name());
          return ((TYPE *)_E)[n]; }
};

template <class TYPE>
ENF<TYPE >& ENF<TYPE >:: point2(Enforcer * E_)
{
    pEnforcer deleteLeak(E_);
    pEnforcer::point2
         (static_cast<Enforcer *>
              (dynamic_cast<TYPE *>(E_)));
    return *this;
}

struct CL_EnforcerTraits
{
  void * assign(Enforcer&, const Enforcer&, unsigned)
    { return 0; }
  void * newCopy(const Enforcer &)
    { return 0; }
  void _delete(Enforcer * I) throw();
  int attach(Enforcer& i, const cl *)
    { return (i.link() != 0); }
  void detach(Enforcer& i, const cl *) throw();
  int cmp(const Enforcer&, const Enforcer&)
    { assert(0); return 0; }
  int put(ostream&, Enforcer&)
      { return 0; }
  int print(ostream&, Enforcer&)
      { return 0; }
  void * get(istream&)
      { return 0; }
};

template <class ITEM>
struct CL_EnforcerDefaultTraits
    : public CL_EnforcerTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
};

CL_template(CL_EnforcerDefault,CL_EnforcerDefaultTraits)

template <class ITEM>
struct CL_EnforcerWellEndowedTraits
    : public CL_EnforcerTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    int cmp(const ITEM& i1, const ITEM& i2)
        { return i1.cmp(i2); }
    int put(ostream& os, ITEM& i)
        { return ((os << i << endm)? 1 : 0); }
    int print(ostream&, ITEM&)
        { return 0; }
    void * get(istream& is)
        {
          ITEM * I = new ITEM;
          if (I) if (!(is >> *I >> nextm))
              { delete I; I = 0; }
          return I;
        }
};

CL_template(CL_EnforcerWellEndowed,CL_EnforcerWellEndowedTraits)

template <class ITEM>
struct CL_EnforcerCmpTraits
    : public CL_EnforcerTraits
{
  int cmp(const ITEM& i1, const ITEM& i2)
    { return i1.cmp(i2); }
};

CL_template(CL_EnforcerCmp,CL_EnforcerCmpTraits)

template <class ITEM>
struct CL_EnforcerStreamTraits
    : public CL_EnforcerTraits
{
  int put(ostream& os, ITEM& i)
      { return ((os << i << endm)? 1 : 0); }
  void * get(istream& is)
      {
        ITEM * I = new ITEM;
        if (I) if (!(is >> *I >> nextm))
           { delete I; I = 0; }
        return I;
      }
};

CL_template(CL_EnforcerStream,CL_EnforcerStreamTraits)

template <class ITEM>
struct CL_EnforcerCmpStreamTraits
    : public CL_EnforcerTraits
{
  int cmp(const ITEM& i1, const ITEM& i2)
    { return i1.cmp(i2); }
  int put(ostream& os, ITEM& i)
      { return ((os << i << endm)? 1 : 0); }
  void * get(istream& is)
      {
        ITEM * I = new ITEM;
        if (I) if (!(is >> *I >> nextm))
           { delete I; I = 0; }
        return I;
      }
};

CL_template(CL_EnforcerCmpStream,CL_EnforcerCmpStreamTraits)


template <class ITEM>
struct CL_EnforcerAllTraits
    : public CL_EnforcerTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    int cmp(const ITEM& i1, const ITEM& i2)
        { return i1.cmp(i2); }
    int put(ostream& os, ITEM& i)
        { return ((os << i << endm)? 1 : 0); }
    int print(ostream&, ITEM& i)
        { return (i.printOn(os)? 1 : 0); }
    void * get(istream& is)
        {
          ITEM * I = new ITEM;
          if (I) if (!(is >> *I >> nextm))
              { delete I; I = 0; }
          return I;
        }
};

CL_template(CL_EnforcerAll,CL_EnforcerAllTraits)


#endif
