/* Public domain dual column sorted file/directory lister with with two line
   summary containing volume label, directory name, number of sub directories,
   current date/time, number of files, bytes in files, and free/total space in
   drive.  Without any parameters all files in the default directory are
   listed.  With one parameter all files are listed that correspond to the file
   spec.  Also has /options to /h display a help screen, /e sort by file
   extension, /s sort by file size, /n sort by file name, /u unsorted, and
   /i change sort default.  With redirection (> file) the file list is sent to
   the file or printer (i.e. LPT1) specified in the second parameter.  On a
   color screen, the files are displayed with different colors in the following
   type priority order: <= 2 days old: White, Directory: Blue, Hidden: Red,
   Executable: Green, Other: Yellow.

   Author:           John Haluska  74000.1106@compuserve.com
   Compiler:         Borland Turbo C V2.0 (Compact Model)
   Operating System: MS-DOS  */

/* 1.0   4/11/95   Initial release.  */

#include <stdio.h>  /* printf(), sprintf(), fopen(), fread(), fwrite(),
                       fseek(), fclose() */
#include <stdlib.h> /* exit(), malloc(), qsort() */
#include <string.h> /* strlen(), strcpy(), strcat(), strchr(), strrchr()
                       strupr(), memcmp() */
#include <ctype.h>  /* tolower()  */
#include <io.h>     /* ioctl()   */
#include <conio.h>  /* cprintf(), textattr(), wherex(), gettextinfo()
                       getch() */
#include <dir.h>    /* findfirst(), findnext(), getcwd(), fnsplit()   */
#include <dos.h>    /* getdfree(), getdate(), gettime() */

#define VIDMODE *(unsigned char far*) 0x449L  /* color, black/white, etc */
#define MAXFILES   10000
#define SCRNLINES     21
#define MAXLONGINT 2147483647

typedef struct {   /* elements of the file list */
  char name[9];
  char ext[5];
  long size;
  int date;
  int time;
  char attr;
} FILESTRUC;
FILESTRUC *fp[MAXFILES];

char *version = "V1.0";   /* program version number */
char *sortstr = "n";      /* default sort type (searchable in .EXE) */
char sortyp;              /* criteria to sort file list */
int redir;   /* 1 if list redirected to a file, 0 if list to screen */
int S_LIGHTGRAY = LIGHTGRAY,
    S_YELLOW    = YELLOW,
    S_GREEN     = GREEN,
    S_BLUE      = CYAN,
    S_RED       = RED,
    S_MAGENTA   = MAGENTA,
    S_NEWFILE   = WHITE;
int totfiles = 0,
    totdirs = 0;
long filebytes = 0L;
int savtextattr;
int datetst, timetst;  /* for defining new files */

void helpscreen(void);
void chg_sort_default(void);
int compn(const void *p1, const void *p2);
int comps(const void *p1, const void *p2);
int compd(const void *p1, const void *p2);
int compe(const void *p1, const void *p2);
void summary_info(char *path, int dirs, int files, long filebytes);
char *fnexpand(char *fname);
char *systime(void);
void packedrevsystime(int daydecr, int *date, int *time);
void showfiledata1(int n);
void showfiledata2(int n);
char *filetime(int date, int time);
char *filesize(long n);
int output_redirected(void);
long srch_file(FILE *f, long ofst, unsigned char *arg, int arglen);
char *comma_num(long n, char *result);
void strndel(char *s, int p, int n);

/*--------*/
void main (int argc, char *argv[])
{
  char fspec[80], fspec2[80], *cp, fext[6];
  struct ffblk fs;
  struct text_info ti;
  int status, screens, j, j1, j2, cumfiles = 0, c, filesleft, evn;

  if (VIDMODE == 7 || VIDMODE == 2) {  /* mono or bw80 screen */
    S_YELLOW  = LIGHTGRAY;
    S_GREEN   = LIGHTGRAY;
    S_BLUE    = LIGHTGRAY;
    S_RED     = LIGHTGRAY;
    S_MAGENTA = LIGHTGRAY;
  }
  sortyp = *sortstr;            /* assign default sort type */
  redir = output_redirected();  /* 1 if list redirected to file or printer */
  gettextinfo(&ti);
  savtextattr = ti.attribute;    /* for restoring screen colors when done */
  packedrevsystime(2,&datetst,&timetst);  /* for new files test */

  /* ---------- Parse command line ----------*/
  if (argc == 1)                        /* no command line args */
    strcpy(fspec,"*.*");
  else {                                /* 1 command line arg */
    strcpy(fspec,argv[1]);              /* filespec */
    strupr(fspec);
    if ((argc == 2) && ((*fspec == '-') || (*fspec == '/'))) {  /* option */
      strcpy(fspec2,argv[1]);
      switch (tolower(*(fspec2+1))) {
        case 'h': helpscreen();
                  exit(1);
        case 'i': chg_sort_default();
                  exit(1);
        default:  sortyp = tolower(*(fspec2+1));
                  break;
      }
      strcpy(fspec,"*.*");
    }
    else if (argc == 3) {   /* 2 args, filespec and options (except /h, /i) */
      strcpy(fspec2,argv[2]);
      sortyp = tolower(*(fspec2+1));
    }

    strcpy(fspec2,fspec);
    strcat(fspec2,"\\*.*");
    status = findfirst(fspec2,&fs,FA_DIREC);   /* directory only specified? */
    if (status >= 0) {  /* directory found */
      if (fspec[strlen(fspec)-1] == ':')
        strcat(fspec,"*.*");
      else
        strcat(fspec,"\\*.*");
    }
    else if ((fspec[strlen(fspec)-1]==':') || (fspec[strlen(fspec)-1]=='\\'))
      strcat(fspec,"*.*");
    else if (strchr(fspec,'.') == NULL)
      strcat(fspec,".*");
  }
  fnexpand(fspec);     /* add default drive and directory if not specified */

  /* ---------- Find matching files ---------- */
  status = findfirst(fspec,&fs,
                       FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_DIREC | FA_ARCH);
  if (status < 0) {
    printf(" Error: %d. Directory and/or file(s) not found.\n",errno);
    summary_info(fspec,totdirs,totfiles,filebytes);
    exit(1);
  }
  while (status >= 0) {
    if ((fp[totfiles] = malloc(sizeof(FILESTRUC))) == NULL) {
      printf(" Out of memory.\n");
      exit(1);
    }
    if ((cp = (char *)strchr(fs.ff_name,'.')) != NULL) {
      strcpy(fext,cp);          /* extract file name extension */
      *cp = '\0';               /* remove extension from file name */
    }
    else
      *fext = '\0';  /* file name doesn't contain an extension */
    sprintf(fp[totfiles]->name,"%-8s",fs.ff_name);
    sprintf(fp[totfiles]->ext,"%-4s",fext);
    fp[totfiles]->size = fs.ff_fsize;
    filebytes += fs.ff_fsize;
    fp[totfiles]->date = fs.ff_fdate;
    fp[totfiles]->time = fs.ff_ftime;
    if ((fs.ff_attrib & FA_DIREC) == FA_DIREC) {     /* directories */
      fp[totfiles]->size = MAXLONGINT;
      fp[totfiles]->attr = 'D';
      if ((strcmp(fp[totfiles]->ext,".   ") == 0) ||
                                 (strcmp(fp[totfiles]->ext,"..  ") == 0)) {
        strcpy(fp[totfiles]->name,fp[totfiles]->ext);
        strcat(fp[totfiles]->name,"    ");
        strcpy(fp[totfiles]->ext,"    ");
      }
      totdirs++;
    }
    else if ((fs.ff_attrib & FA_HIDDEN) == FA_HIDDEN)
      fp[totfiles]->attr = 'H';                      /* hidden files */
    else if ((strcmp(fp[totfiles]->ext,".EXE")==0) ||
             (strcmp(fp[totfiles]->ext,".COM")==0) ||
             (strcmp(fp[totfiles]->ext,".BAT")==0) ||
             (strcmp(fp[totfiles]->ext,".BIN")==0) ||
             (strcmp(fp[totfiles]->ext,".SYS")==0))
      fp[totfiles]->attr = 'E';                      /* executable files */
    else
      fp[totfiles]->attr = 'O';                      /* other files */
    if (fp[totfiles]->date > datetst ||
              (fp[totfiles]->date == datetst && fp[totfiles]->time > timetst))
      fp[totfiles]->attr = 'T';                      /* new files, dirs */
    ++totfiles;
    status = findnext(&fs);
  }

  /* ---------- Sort file list ---------- */
  switch (sortyp) {
    case 'n': qsort((void *)fp,totfiles,sizeof(fp[0]),compn); break;
    case 's': qsort((void *)fp,totfiles,sizeof(fp[0]),comps); break;
    case 'e': qsort((void *)fp,totfiles,sizeof(fp[0]),compe); break;
    case 'd': qsort((void *)fp,totfiles,sizeof(fp[0]),compd); break;
  }

  /* ---------- Send file list to display, file or printer ---------- */
  screens = totfiles / (SCRNLINES * 2);
  j1 = screens;
  while (j1 > 0) {                            /* full screens of file list */
    if (redir) {   /* to file or printer */
      for (j = 0; j < SCRNLINES; j++) {
        showfiledata1(cumfiles);
        printf(" |");
        showfiledata1(cumfiles+SCRNLINES);
        printf("\n");
        ++cumfiles;
      }
      printf("\n");
    }
    else          /* to screen */
      for (j = 0; j < SCRNLINES; j++) {
        showfiledata2(cumfiles);
        textattr(S_YELLOW);
        cprintf(" |");
        showfiledata2(cumfiles+SCRNLINES);
        cprintf("\r\n");
        ++cumfiles;
      }
    j1--;
    cumfiles += SCRNLINES;
    if (!redir) {
      textattr(S_MAGENTA);
      cprintf("   Press any key to continue (Esc: Cancel) ...");
      c = getch();
      if (c == 27) {
        cprintf(" Canceled.\n");
        summary_info(fspec,totdirs,totfiles,filebytes);
        exit(1);
      }
      else
        cprintf("\r\n");
    }
  } /* while (j1 > 0) */

  filesleft = totfiles - cumfiles;   /* partial screen of file list */
  j = filesleft / 2;
  j1 = cumfiles + j;
  if (filesleft % 2 == 1) {
    evn = 0;
    j1++;
  }
  else
    evn = 1;
  if (redir)
    for (j2=1; j2 <= j; j2++) {
      showfiledata1(cumfiles);
      printf(" |");
      showfiledata1(j1);
      printf("\n");
      cumfiles++;
      j1++;
    }
  else
    for (j2=1; j2 <= j; j2++) {
      showfiledata2(cumfiles);
      textattr(S_YELLOW);
      cprintf(" |");
      showfiledata2(j1);
      cprintf("\r\n");
      cumfiles++;
      j1++;
    }

  if ((evn == 0) || (filesleft == 1))   /* 1 file to list */
    if (redir) {
      showfiledata1(cumfiles);
      printf("\n");
    }
    else {
      showfiledata2(cumfiles);
      cprintf("\r\n");
    }
  summary_info(fspec,totdirs,totfiles,filebytes);
} /* main */
/*--------*/
void helpscreen(void)
{
  char path[80], drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];

  if (_osmajor >= 3) {
    strcpy(path,_argv[0]);
    fnsplit(path,drive,dir,file,ext);
    cprintf(" %s%s",file,ext);   /* actual name of this program */
  }
  cprintf(" %s Dual column sorted file lister. Public Domain.\r\n\r\n",version);
  cprintf(" Syntax: %s [drive:][path] [filename] [/options] [> file]\r\n\r\n",
                                                                         file);
  cprintf(" [drive:][path] [filename]  specifies drive, directory and/or\r\n");
  cprintf("                            files to list\r\n");
  cprintf(" [/options]  /h help, /e sort by file extension, /s sort by file size,\r\n");
  cprintf("             /d sort by file date, /n sort by file name,\r\n");
  cprintf("             /u unsorted, /i change sort default (currently /%c)\r\n",sortyp);
  cprintf(" [> file]  redirect output from screen to file or printer (PRN)\r\n\r\n");
  cprintf(" On a color screen, the files are displayed with different colors in the\r\n");
  cprintf(" following type priority order:  <= 2 days old: White;  Directory: Blue;\r\n");
  cprintf(" Hidden: Red;  Executable: Green;  Other: Yellow.\r\n\r\n");
  cprintf(" John Haluska, 74000.1106@compuserve.com\r\n");
} /* helpscreen */
/*--------*/
void chg_sort_default(void)
{
  FILE *f;
  long ofst;
  char fspec[80], ch;

  if (_osmajor < 3) {
    cprintf("DOS 3.0 or above required to change sort default.\r\n");
    return;
  }
  strcpy(fspec,_argv[0]);
  if ((f = fopen(fspec,"r+b")) == NULL) {
    cprintf(" Can't find %s.  Must be in current directory\r\n",fspec);
    cprintf(" or directory defined in DOS PATH statement.\r\n");
    return;
  }
  ofst = srch_file(f,20000, version, sizeof(version) + 1);
  if (ofst == -1L) {
    cprintf(" Not a valid file. Operation canceled.\r\n");
    return;
  }
  ofst += sizeof(version) + 1;   /* sortstr location in .EXE file */
  /* cprintf("%s: %s found at %ld  %X\r\n",path,version,ofst,ofst); */
  fseek(f,ofst,SEEK_SET);
  fread(&ch,1,1,f);
  cprintf(" Current sort default is /%c\r\n",tolower(ch));
  cprintf(" Sort by date, extension, name, size, or unsorted (d,e,n,s,u): ");
  ch = tolower(getch());
  cprintf("%c\r\n",ch);
  if (ch=='d' || ch=='e' || ch=='n' || ch=='s' || ch=='u') {
    fseek(f,ofst,SEEK_SET);
    if (fwrite(&ch,1,1,f) != 1)
      cprintf(" Write error. Operation canceled.\r\n");
    else
      cprintf(" New sort default is /%c\r\n",tolower(ch));
    fclose(f);
  }
  else
    cprintf(" Invalid sort type. Operation canceled.\r\n");
} /* chg_sort_default */
/*--------*/
/* compn() is the sorting criteria used by qsort() to sort the file list in
   order of file name and extension. */

int compn(const void *p1, const void *p2)
{
  int n;
  const FILESTRUC *ps1 = *(FILESTRUC **)p1;
  const FILESTRUC *ps2 = *(FILESTRUC **)p2;

  n = strcmp(ps1->name,ps2->name);
  if (n != 0) return n;
  return strcmp(ps1->ext,ps2->ext);
} /* compn */
/*--------*/
/* comps() is the sorting criteria used by qsort() to sort the file list in
   order of decreasing file size and increasing name/extension. */

int comps(const void *p1, const void *p2)
{
  int n;
  const FILESTRUC *ps1 = *(FILESTRUC **)p1;
  const FILESTRUC *ps2 = *(FILESTRUC **)p2;

  if (ps2->size > ps1->size) return 1;
  if (ps1->size > ps2->size) return -1;
  n = strcmp(ps1->name,ps2->name);
  if (n != 0) return n;
  return strcmp(ps1->ext,ps2->ext);
} /* comps */
/*--------*/
/* compd() is the sorting criteria used by qsort() to sort the file list in
   order of file date, name and extension. */

int compd(const void *p1, const void *p2)
{
  int n;
  const FILESTRUC *ps1 = *(FILESTRUC **)p1;
  const FILESTRUC *ps2 = *(FILESTRUC **)p2;

  n = ps1->date - ps2->date;
  if (n != 0) return n;
  n = ps1->time - ps2->time;
  if (n != 0) return n;
  n = strcmp(ps1->name,ps2->name);
  if (n != 0) return n;
  return strcmp(ps1->ext,ps2->ext);
} /* compd */
/*--------*/
/* compe() is the sorting criteria used by qsort() to sort the file list in
   order of file extension and name. */

int compe(const void *p1, const void *p2)
{
  int n;
  const FILESTRUC *ps1 = *(FILESTRUC **)p1;
  const FILESTRUC *ps2 = *(FILESTRUC **)p2;

  n = strcmp(ps1->ext,ps2->ext);
  if (n != 0) return n;
  return strcmp(ps1->name,ps2->name);
} /* compe */
/*--------*/
/* Display two lines of volume, directory, disk, etc. information. */

void summary_info(char *path, int dirs, int files, long filebytes)
{
  char filespec[10], s1[80], s2[20], s3[80], *p;
  char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
  struct ffblk fs;
  struct dfree d;
  int status;
  long n;

  *filespec = *path;                                /* volume label */
  *(filespec+1) = '\0';
  strcat(filespec,":\\*.*");
  status = findfirst(filespec,&fs,FA_LABEL);
  if (status == 0) {
    if ((p = (char *)strrchr(fs.ff_name,'.')) != NULL)
      strndel(fs.ff_name,(int)(p-fs.ff_name),1);
    sprintf(s3,"\r\n Vol: %s",fs.ff_name);
  }
  else
    strcpy(s3,"\r\n Vol: No Label");
  if (redir)
    printf("%s",s3);
  else {
    textattr(S_MAGENTA);
    cprintf("%s",s3);
  }

  fnsplit(path,drive,dir,file,ext);     /* directory name, num of subdirs */
  strcpy(s1,drive);
  strcat(s1,dir);
  if (*(s1+strlen(s1)-1) == '\\')   /* remove trailing '\' if present */
    *(s1+strlen(s1)-1) = '\0';
  sprintf(s3,"  Dir: %s  SubDirs: %d",s1,dirs);
  if (redir)
    printf("%s",s3);
  else {
    textattr(S_BLUE);
    cprintf("%s",s3);
  }

  if (redir)                            /* date, time from operating system */
    printf("   %s\r\n",systime());
  else {
    textattr(S_MAGENTA);
    if (wherex() < 60) cprintf("   %s",systime());
    textattr(savtextattr);
    cprintf("\r\n ");
  }

  comma_num((long)(files-dirs),s1);      /* files and total size of files */
  comma_num(filebytes,s2);
  sprintf(s3,"Files: %s  Size: %s",s1,s2);
  if (redir)
    printf(s3);
  else {
    textattr(S_YELLOW);
    cprintf(s3);
  }

  getdfree(*(path)-64,&d);               /* disk size and bytes available */
  if (d.df_sclus == 0xffff) d.df_sclus = 0;
  n = (long)d.df_avail * (long)d.df_bsec * (long)d.df_sclus;
  comma_num(n,s1);
  n = (long)d.df_total * (long)d.df_bsec * (long)d.df_sclus;
  comma_num(n,s2);
  if (redir)
    printf("  Free: %s  Total: %s\n",s1,s2);
  else {
    textattr(S_GREEN);
    cprintf("  Free: %s",s1);
    textattr(S_RED);
    cprintf("  Total: %s",s2);
    textattr(S_BLUE);
    if (wherex() < 69) cprintf("  /h Help");
  }
} /* summary_info */
/*--------*/
/* fnexpand() converts and returns fname to a fully qualified file name by
   adding the default drive or the default drive/default directory if not
   included in fname. */

char *fnexpand(char *fname)
{
  char drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE], ext[MAXEXT];
  int flags;

  flags = fnsplit(fname,drive,dir,file,ext);
  if ((!(flags & DRIVE)) && (flags & DIRECTORY)) {  /* drv missing, dir pres */
    getcwd(fname,MAXPATH);
    *(fname+2) = '\0';
    strcat(fname,dir);
    strcat(fname,file);
    strcat(fname,ext);
  }
  if ((!(flags & DRIVE)) && (!(flags & DIRECTORY))) { /* drv & dir missing */
    getcwd(fname,MAXPATH);
    if (strlen(fname) > 3) strcat(fname,"\\"); /* add "\" if not in root dir */
    strcat(fname,file);
    strcat(fname,ext);
  }
  return fname;
} /* fnexpand */
/*--------*/
/* systime() returns the operating system date and time as a 16 character fixed
   length string " MM-DD-YY HH:MMa".  If month or hour is < 10, a space
   character is added to the string.  If day or minute is < 10 a '0' character
   is added to the string. */

char *systime(void)
{
  static char *s = " MM-DD-YY HH:MMa";  /* global variable with local scope */
  struct date d;
  struct time t;
  char ampm;
  int n;

  getdate(&d);
  gettime(&t);
  d.da_year -= 1900;                         /* year (80-199) */
  if (d.da_year > 99) d.da_year -= 100;      /* 100 to 199 = 00 to 99 */
  if (t.ti_hour >= 12) ampm = 'p';
    else ampm = 'a';
  if (t.ti_hour == 0) t.ti_hour = 12;
  if (t.ti_hour > 12) t.ti_hour -= 12;
  sprintf(s," %2d-%02d-%02d %2d:%02d%c",d.da_mon, d.da_day, d.da_year,
                                                    t.ti_hour, t.ti_min, ampm);
  return s;
} /* systime */
/*--------*/
/* packedrevsystime() returns the packed date and time, in format used by
   findfirst() and findnext(), for the current system date less daydecr days.*/

void packedrevsystime(int daydecr, int *date, int *time)
{
  struct date d;
  struct time t;

  getdate(&d);
  gettime(&t);
  d.da_day -= daydecr;
  if (d.da_day < 1) {
    d.da_mon--;
    if (d.da_mon < 1) {
      d.da_mon = 12;
      if (d.da_year > 1980) d.da_year--;
    }
    switch (d.da_mon) {
      case 2:
        if ((d.da_year%4 == 0 && d.da_year%100 != 0) || d.da_year%400 == 0)
          d.da_day += 29;     /* leap year */
        else
          d.da_day += 28;     /* non leap year */
        break;
      case 4: case 6: case 9: case 11:
        d.da_day += 30;
        break;
      default:    /* month 1, 3, 5, 7, 8, 10, 12 */
        d.da_day += 31;
        break;
    }
  }
  *date = (d.da_year-1980)*512 + d.da_mon*32 + d.da_day;
  *time = t.ti_hour*2048 + t.ti_min*32 + t.ti_sec/2;
} /* packedrevsystime */
/*--------*/
void showfiledata1(int n)
{
  printf(" %s%s%s%s",fp[n]->name, fp[n]->ext, filesize(fp[n]->size),
                                          filetime(fp[n]->date,fp[n]->time));
} /* showfiledata1 */
/*--------*/
void showfiledata2(int n)
{
  switch(fp[n]->attr) {
    case 'O': textattr(S_YELLOW);  break;
    case 'E': textattr(S_GREEN);   break;
    case 'D': textattr(S_BLUE);    break;
    case 'H': textattr(S_RED);     break;
    case 'T': textattr(S_NEWFILE); break;
  }
  cprintf(" %s%s%s%s",fp[n]->name, fp[n]->ext, filesize(fp[n]->size),
                                        filetime(fp[n]->date,fp[n]->time));
} /* showfiledata2 */
/*--------*/
/* filetime() returns date and time (as returned by functions findfirst() and
   findnext()) as a 16 character fixed length string " MM-DD-YY HH:MMa".  If
   month or hour is < 10, a space character is added to the string.  If day
   or minute is < 10 a '0' character is added to the string. */

char *filetime(int date, int time)
{
  int n;
  static char *s = " MM-DD-YY HH:MMa";  /* global variable with local scope */

  n = date & 0x001f;                  /* day (1-31) */
  *(s+4) = n/10 + 48;
  *(s+5) = n%10 + 48;
  date >>= 5;
  n = date & 0x000f;                  /* month (1-12) */
  if (n < 10) {
    *(s+1) = ' ';
    *(s+2) = n + 48;
  }
  else {
    *(s+1) = n/10 + 48;
    *(s+2) = n%10 + 48;
  }
  date = ((date >> 4) & 0x007f) + 80;  /* year (80-199) */
  if (date > 99) date -= 100;          /* 2000 to 2099 = 00 to 99 */
  *(s+7) = date/10 + 48;
  *(s+8) = date%10 + 48;
  time >>= 5;
  n = time & 0x003f;                   /* minute (0-59) */
  *(s+13) = n/10 + 48;
  *(s+14) = n%10 + 48;
  time = (time >> 6) & 0x001f;         /* hour (0-23) */
  if (time >= 12) *(s+15) = 'p';
    else *(s+15) = 'a';
  if (time == 0) time = 12;
  if (time > 12) time -= 12;
  if (time < 10) {
    *(s+10) = ' ';
    *(s+11) = time + 48;
  }
  else {
    *(s+10) = time/10 + 48;
    *(s+11) = time%10 + 48;
  }
  return s;
} /* filetime */
/*--------*/
/* filesize() return file size n as a string or <Dir> string if n = -1. */

char *filesize(long n)
{
  static char s[12];

  if (n == MAXLONGINT)  strcpy(s," <Dir>   ");
    else sprintf(s,"%9ld",n);
  return s;
} /* filesize */
/*--------*/
/* output_redirected() returns 1 if standard output (display, file handle 1)
   is redirected to a file variable, output device or null device at the DOS
   prompt using the ">" operator. */

int output_redirected(void)
{
  int dx;

  dx = ioctl(1,0); /* file handle 1, DOS Service 44h/0 - Get Device Info */
  if ((dx & 0x82) == 0x82)
    return 0;      /* output has not been redirected */
  else
    return 1;      /* output has been redirected */
} /* output_redirected */
/*--------*/
/* srch_file() searches file f, starting at address ofst (0 based), for data
   argument arg, with length arglen, by reading data into global buffer sbuf.
   The offset (0 based) into file f is returned.  If no match is found, -1 is
   returned.  File f must be a valid, open file.  */

long srch_file(FILE *f, long ofst, unsigned char *arg, int arglen)
{
  #define SBYTES 200
  char sbuf[SBYTES+50];
  unsigned char *p;
  int byte_ct, j, buflen;

  buflen = SBYTES+arglen-1;  /* allow for argument crossing into next buffer */
  do {
    fseek(f,ofst,SEEK_SET);
    byte_ct = fread((char *)sbuf,1,buflen,f);
    j = 0;
    p = sbuf;
    while (j < byte_ct) {
      if (memcmp((unsigned char *)arg,(unsigned char *)p,arglen) != 0) {
        p++;
        j++;
      }
      else     /* match found */
        return j + ofst;
    } /* while j < byte_ct */
    ofst += (long)SBYTES;
  } while (byte_ct > 0);
  return -1L;   /* match not found */
} /* srch_file */
/*--------*/
/* comma_num() returns number n as a string result with commas inserted every
   3 numbers.  Example:  comma_num(123456,buf) returns "123,456" in buf. */

char *comma_num(long n, char *result)
{
  int commas, i;
  char *srcp, *destp;

  sprintf(result,"%ld",n);
  i = strlen(result);

  if (i > 3) {
    commas = (i - 1) / 3;
    srcp = result + i;
    destp = srcp + commas;
    for (i = 3; commas; --i) {
      *destp-- = *srcp--;
      if (!i) {
        i = 3;
        *destp-- = ',';
        --commas;
      }
    }
  }
  return result;
} /* comma_num */
/*--------*/
/* strndel() deletes n characters from string s starting at position p
   (0 based). */

void strndel(char *s, int p, int n)
{
  for (; (*(s+p) = *(s+p+n)) != '\0'; s++)
    *(s+p+n) = '\0';                /* delete and zero fill */
} /* strndel */
/*--------*/

