/*
**	Copyright  1994 Novell, Inc.  All rights reserved.
**
**	Permission is granted to the recipient of this work ("you") to use,
**	reproduce and distribute Novell's original publication of the work free
**	of charge provided that you reproduce the work in its entirety and
**	include all Novell copyright notices as they originally appear.
**
**	Novell grants you permission to modify and distribute copies of this
**	work including any portion of this work (a "modified work") provided
**	that you include prominent notification of such modification along with
**	the date of modification on a modified work; distribute or publish a
**	modified work to third parties under the same conditions and granting
**	the same rights as are extended to you by Novell under this under
**	permission notice; and provided that you include a copy of Novell's
**	original publication of the work along with any copy of a modified
**	work.
**
**	NOVELL MAKES NO WARRANTY, REPRESENTATION OR PROMISE THAT THIS WORK OR A
**	MODIFIED WORK WILL SATISFY YOUR REQUIREMENTS OR THAT THIS WORK OR A
**	MODIFIED WORK IS WITHOUT DEFECT OR ERROR.  NOVELL DISCLAIMS AND
**	EXCLUDES ANY AND ALL IMPLIED WARRANTIES OF MERCHANTABILITY, TITLE OR
**	FITNESS FOR A PARTICULAR PURPOSE.
**
**	IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW WILL NOVELL OR ANY OTHER
**	PARTY BE LIABLE FOR DAMAGES INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
**	CONSEQUENTIAL, INDIRECT OR PUNATIVE DAMAGES ARISING OUT OF THE USE OF
**	OR INABLITITY TO USE THE WORK OR A MODIFIED WORK.
**
**	DSVADDR.CPP - February 1995
**
**	Definition of the DSVPostalAddress and DSVEmailAddress classes.
**
**	Author: John Buckle, Asia Pacific Support Centre, Novell Australia
**	==================================================================
**	9 Jan 1995           First release              John Buckle
*/

# include "dsdefs.h"
# include "dsvaddr.h"

# include <string.h>
# include <stdio.h>
# include <ctype.h>

/*
** DSVPostalAddress::DSVPostalAddress()
**
**	Initialise the address pointers to null.
*/

DSVPostalAddress::DSVPostalAddress()
{
	for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++)
		Address[i] = 0 ;
}

/*
** NWDSCCODE DSVPostalAddress::assign(NWSYNTAX_ID syntaxID, void * value)
**
**	Initialise the object using the NDS buffer data.
*/

NWDSCCODE DSVPostalAddress::assign(NWSYNTAX_ID syntaxID, void * value)
{
	if (dsValueTypeArray[(WORD)syntaxID] != DSCPP_POSTAL_ADDRESS)
		return DSCPP_WRONG_TYPE ;

	release() ;

	NWPSTR * fields = (NWPSTR *) value ;

	Syntax = syntaxID ;

	for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++)
		Address[i] = strnew(fields[i]) ;

	return 0 ;
}

/*
** NWDSCCODE DSVPostalAddress::assign(DSVPostalAddress * value)
**
**	Initialise the object using another DSVPostalAddress.
*/

NWDSCCODE DSVPostalAddress::assign(DSVPostalAddress * value)
{
	if (this == value) return 0 ;

	release() ;

	Syntax = value->Syntax ;
	for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++)
		Address[i] = strnew(value->Address[i]) ;

	return 0 ;
}

/*
** NWDSCCODE DSVPostalAddress::release(int = DSCPP_DEF_INDEX)
**
**	Release memory used by the object and reset the Syntax field.
*/

NWDSCCODE DSVPostalAddress::release(int)
{
	if (Syntax)
		for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++){
			delete [] Address[i] ;
			Address[i] = 0 ;
			}
	Syntax = 0 ;
	return 0 ;
}

/*
** NWDSCCODE DSVPostalAddress::compare(DSValue * value)
**
**	Compare two DSVPostalAddress objects. Return TRUE(1) if they are
**	equivalent, FALSE(0) if they are not equivalent and negative for
**	an error.
*/

NWDSCCODE DSVPostalAddress::compare(DSValue * value)
{
	if (checkTypeAndSyntax(value,DSCPP_POSTAL_ADDRESS))
		return checkTypeAndSyntax(value,DSCPP_POSTAL_ADDRESS) ;

	NWPSTR * address = ((DSVPostalAddress *) value)->Address ;

	for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++)
		if (stricmp(Address[i],address[i])) return 0 ;

	return 1 ;
}

/*
** NWPSTR DSVPostalAddress::text(int index = DSCPP_DEF_INDEX)
**
**	Return the address fields
*/

NWPSTR DSVPostalAddress::text(int index)
{
	if (index != DSCPP_DEF_INDEX){
		return (index >= 0 && index < NUM_POSTAL_ADDRESS_ELEMENTS) ? Address[index] : "" ;
		}

	char *	pointer = DSValue::Buffer ;
	WORD	count   = 0 ;

	for (int i = 0 ; i < NUM_POSTAL_ADDRESS_ELEMENTS ; i++){
		WORD length = Address[i] ? strlen(Address[i]) : 0 ;
		if (count + length + 3 >= MAX_DN_BYTES) break ;
		count += length + 2 ;
		* pointer++ = '[' ;
		if (length) strcpy(pointer,Address[i]) ; pointer += length ;
		* pointer++ = ']' ;
		}
	* pointer = 0 ;

	return DSValue::Buffer ;
}

/*
** NWDSCCODE DSVEmailAddress::assign(NWSYNTAX_ID syntaxID, void * value)
**
**	Initialise the object using the NDS buffer data.
*/

NWDSCCODE DSVEmailAddress::assign(NWSYNTAX_ID syntaxID, void * value)
{
	if (dsValueTypeArray[(WORD)syntaxID] != DSCPP_EMAIL_ADDRESS)
		return DSCPP_WRONG_TYPE ;

	release() ;

	EMail_Address_T * address = (EMail_Address_T *) value ;

	Syntax  = syntaxID ;
	Type    = address->type ;
	Address = strnew(address->address) ;

	return 0 ;
}

/*
** NWDSCCODE DSVEmailAddress::assign(DSVEmailAddress * value)
**
**	Initialise the object using another DSVEmailAddress.
*/

NWDSCCODE DSVEmailAddress::assign(DSVEmailAddress * value)
{
	if (this == value) return 0 ;

	release() ;

	Syntax  = value->Syntax ;
	Type    = value->Type ;
	Address = strnew(value->Address) ;

	return 0 ;
}

/*
** NWDSCCODE DSVEmailAddress::release(int = DSCPP_DEF_INDEX)
**
**	Release memory used by the object and reset the Syntax field.
*/

NWDSCCODE DSVEmailAddress::release(int)
{
	if (Syntax){
		delete [] Address ;
		Address = 0 ;
		}

	Syntax = 0 ;
	return 0 ;
}

/*
** NWDSCCODE DSVEmailAddress::compare(DSValue * value)
**
**	Compare two DSVEmailAddress objects. Return TRUE(1) if they are
**	equivalent, FALSE(0) if they are not equivalent and negative for
**	an error.
*/

NWDSCCODE DSVEmailAddress::compare(DSValue * value)
{
	if (checkTypeAndSyntax(value,DSCPP_EMAIL_ADDRESS))
		return checkTypeAndSyntax(value,DSCPP_EMAIL_ADDRESS) ;

	DSVEmailAddress * address = (DSVEmailAddress *) value ;

	return stricmp(Address,address->Address) == 0 && Type == address->Type ;
}

/*
** NWPSTR DSVEmailAddress::text(int index = DSCPP_DEF_INDEX)
**
**	Return the address and type fields
*/

NWPSTR DSVEmailAddress::text(int index)
{
	switch (index){
	    case 0:
		sprintf(DSValue::Buffer,"%s",Address) ;
		break ;
	    case 1:
		sprintf(DSValue::Buffer,"%ld",Type) ;
		break ;
	    default:
		sprintf(DSValue::Buffer,"[%s] %ld",Address,Type) ;
		break ;
	    }
	return DSValue::Buffer ;
}

/*
** NWDSCCODE DSVNetAddress::assign(NWSYNTAX_ID syntaxID, void * value)
**
**	Initialise the object using the NDS buffer data.
*/

NWDSCCODE DSVNetAddress::assign(NWSYNTAX_ID syntaxID, void * value)
{
	if (dsValueTypeArray[(WORD)syntaxID] != DSCPP_NET_ADDRESS)
		return DSCPP_WRONG_TYPE ;

	release() ;

	Net_Address_T * address = (Net_Address_T *) value ;

	Syntax  = syntaxID ;
	Type	= address->addressType ;
	Length	= address->addressLength ;
	Address = new BYTE[(int)Length] ;
	memcpy(Address,address->address,(int)Length) ;

	return 0 ;
}

/*
** NWDSCCODE DSVNetAddress::assign(DSVNetAddress * value)
**
**	Initialise the object using another DSVNetAddress.
*/

NWDSCCODE DSVNetAddress::assign(DSVNetAddress * value)
{
	if (this == value) return 0 ;

	release() ;

	Syntax  = value->Syntax ;
	Type    = value->Type ;
	Length  = value->Length ;
	Address = new BYTE[(int)Length] ;
	memcpy(Address,value->Address,(int)Length) ;

	return 0 ;
}

/*
** NWDSCCODE DSVNetAddress::release(int = DSCPP_DEF_INDEX)
**
**	Release memory used by the object and reset the Syntax field.
*/

NWDSCCODE DSVNetAddress::release(int)
{
	if (Syntax){
		delete [] Address ;
		Address = 0 ;
		}

	Syntax = 0 ;
	return 0 ;
}

/*
** NWDSCCODE DSVNetAddress::compare(DSValue * value)
**
**	Compare two DSVNetAddress objects. Return TRUE(1) if they are
**	equivalent, FALSE(0) if they are not equivalent and negative for
**	an error.
*/

NWDSCCODE DSVNetAddress::compare(DSValue * value)
{
	if (checkTypeAndSyntax(value,DSCPP_NET_ADDRESS))
		return checkTypeAndSyntax(value,DSCPP_NET_ADDRESS) ;

	DSVNetAddress * address = (DSVNetAddress *) value ;

	return memcmp(Address,address->Address,(int)Length) == 0 &&
		Type == address->Type ;
}

/*
** NWPSTR DSVNetAddress::text(int index = DSCPP_DEF_INDEX)
**
**	Return the address, length and type fields
*/

NWPSTR DSVNetAddress::text(int index)
{
	if (index == 1){
		sprintf(DSValue::Buffer,"%ld",Type) ;
		return DSValue::Buffer ;
		}

	char * cp = DSValue::Buffer ;

	if (index == DSCPP_DEF_INDEX) * cp++ = '[' ;

	for (int i = 0 ; i < Length ; i++){
		sprintf(cp,"%02x",Address[i]) ;
		cp += 2 ;
		}
	if (index == DSCPP_DEF_INDEX) sprintf(cp,"] %ld",Type) ;

	return DSValue::Buffer ;
}

/*
** int DSVNetAddress::ParseAddress(char * address)
**
**	Convert an ascii hexstring into binary.
*/

int DSVNetAddress::parseAddress(char * address)
{
	Address = new BYTE[strlen(address)/2 + 1] ;

	for (Length = 0 ; * address ; Length ++ ){
		char value = 0 ;
		if (! isxdigit(* address)) address++ ;
		for (int j = 0; j < 2 ; j++ ){
			value <<= 4 ;
			char ch = toupper(*address++) ;
			if (ch >= '0' && ch <= '9')
				value += ch - '0' ;
			else if (ch >= 'A' && ch <= 'F')
				value += ch - 'A' + 10 ;
			else return 1 ;
			}
		Address[(int)Length] = value ;
		}
	return 1 ;
}

/*
** DSValue & dsChangeAddress(DSValue & value, NWPSTR data, NWSYNTAX_ID)
**
**	Modify the Type field of a DSVPostalAddress, DSVEmailAddress or
**	DSVNetAddress object. This function is executed when the Address()
**	stream manipulator is called.
*/

DSValue & dsChangeAddress(DSValue & value, NWPSTR data, NWSYNTAX_ID)
{
char * fields[NUM_POSTAL_ADDRESS_ELEMENTS+1] ;

	if (data == 0) return value ;

	fields[0] = data ;
	for (int count = 1, index = 1 ; index <= NUM_POSTAL_ADDRESS_ELEMENTS ; index++){
		while (* data && * data != '\n') data++ ;
		if (* data == '\n') count ++, * data++ = 0 ;
		fields[index] = data ;
		}

	switch (value.type()){
	    case DSCPP_EMAIL_ADDRESS:{
		DSVEmailAddress * email = (DSVEmailAddress *) & value ;
		email->release() ;
		email->Address = DSValue::strnew(fields[0]) ;
		email->Syntax  = SYN_EMAIL_ADDRESS ;
		}break ;
	    case DSCPP_NET_ADDRESS:{
		DSVNetAddress * net = (DSVNetAddress *) & value ;
		net->release() ;
		net->Syntax = SYN_NET_ADDRESS ;
		net->parseAddress(fields[0]) ;
		}break ;
	    case DSCPP_POSTAL_ADDRESS:{
		int start = 0 ;
		DSVPostalAddress * postal = (DSVPostalAddress *) & value ;
		if (count == 1){
			for (start = 0 ; start < NUM_POSTAL_ADDRESS_ELEMENTS ; start ++)
				if (postal->Address[start] == 0) break ;
			}
		else {
			start = 0 ;
			postal->release() ;
			}
		for (int i = 0 ; start < NUM_POSTAL_ADDRESS_ELEMENTS ; i ++)
			postal->Address[start++] = DSValue::strnew(fields[i]) ;

		postal->Syntax = SYN_PO_ADDRESS ;
		}break ;
	    }

	return value ;
}

/*
** DSManipulator<NWPSTR> Address(NWPSTR data)
**
**	Create an instance of a manipulator object passing the address of
**	dsChangeAddress() function so that the Address field of a
**	DSVPostalAddress, DSVEmailAddress or DSVNetAddress object can be
**	modified.
*/

DSManipulator<NWPSTR> Address(NWPSTR data)
{
	return DSManipulator<NWPSTR>(dsChangeAddress,data) ;
}

/*
** DSValue & dsChangeType(DSValue & value, DWORD data, NWSYNTAX_ID)
**
**	Modify the Type field of a DSVEmailAddress or DSVNetAddress object.
**	This function is executed when the Type() stream manipulator is called.
*/

DSValue & dsChangeType(DSValue & value, DWORD data, NWSYNTAX_ID)
{
	switch (value.type()){
	    case DSCPP_EMAIL_ADDRESS:{
		DSVEmailAddress * email = (DSVEmailAddress *) & value ;
		email->Type = data ;
		}break ;
	    case DSCPP_NET_ADDRESS:{
		DSVNetAddress * net = (DSVNetAddress *) & value ;
		net->Type = data ;
		}break ;
	    }

	return value ;
}

/*
** DSManipulator<DWORD> Type(DWORD data)
**
**	Create an instance of a manipulator object passing the address of
**	dsChangeType() function so that the Type field of a
**	DSVEmailAddress or DSVNetAddress object can be modified.
*/

DSManipulator<DWORD> Type(DWORD data)
{
	return DSManipulator<DWORD>(dsChangeType,data) ;
}


