/*
**	Copyright  1994 Novell, Inc.  All rights reserved.
**
**	Permission is granted to the recipient of this work ("you") to use,
**	reproduce and distribute Novell's original publication of the work free
**	of charge provided that you reproduce the work in its entirety and
**	include all Novell copyright notices as they originally appear.
**
**	Novell grants you permission to modify and distribute copies of this
**	work including any portion of this work (a "modified work") provided
**	that you include prominent notification of such modification along with
**	the date of modification on a modified work; distribute or publish a
**	modified work to third parties under the same conditions and granting
**	the same rights as are extended to you by Novell under this under
**	permission notice; and provided that you include a copy of Novell's
**	original publication of the work along with any copy of a modified
**	work.
**
**	NOVELL MAKES NO WARRANTY, REPRESENTATION OR PROMISE THAT THIS WORK OR A
**	MODIFIED WORK WILL SATISFY YOUR REQUIREMENTS OR THAT THIS WORK OR A
**	MODIFIED WORK IS WITHOUT DEFECT OR ERROR.  NOVELL DISCLAIMS AND
**	EXCLUDES ANY AND ALL IMPLIED WARRANTIES OF MERCHANTABILITY, TITLE OR
**	FITNESS FOR A PARTICULAR PURPOSE.
**
**	IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW WILL NOVELL OR ANY OTHER
**	PARTY BE LIABLE FOR DAMAGES INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
**	CONSEQUENTIAL, INDIRECT OR PUNATIVE DAMAGES ARISING OUT OF THE USE OF
**	OR INABLITITY TO USE THE WORK OR A MODIFIED WORK.
**
**	DSBUFFER.CPP - October 1994
**
**	Initialisation, control and release of directory buffers.
**
**	Author: John Buckle, Asia Pacific Support Centre, Novell Australia
**	==================================================================
**	9 Jan 1995           First release              John Buckle
*/

# include "dsdefs.h"
# include "dsbuffer.h"
# include "dsiterat.h"

/*
** DSBuffer::DSBuffer(NWPSTR context    = 0 ,
**		      DWORD  flags      = DS_CONTEXT_FLAGS,
**		      WORD   inputSize  = DEFAULT_MESSAGE_LEN,
**		      WORD   outputSize = DEFAULT_MESSAGE_LEN,
**		      WORD   filterSize = DEFAULT_MESSAGE_LEN)
**
**	Allocates memory for the three NDS buffers contained inside the
**	object.
*/

DSBuffer::DSBuffer(NWPSTR context, DWORD flags,
			WORD inputSize, WORD outputSize, WORD filterSize) :
		DSContext(context,flags),
		DSInput  (inputSize),
		DSOutput (outputSize),
		DSFilter (filterSize)
{
}

/*
** DSBuffer::~DSBuffer()
**
**	Release the memory used by the buffers.
*/

DSBuffer::~DSBuffer()
{
}

/*
** NWDSCCODE DSBuffer::CloseIteration(
**			NWDS_ITERATION iterHandle,
**			NWDS_OPERATION operation)
**
**	Recover any memory used by an iteration command. The command
**	DSContext::CloseIteration() is not used since this modifies the
**	Status member field.
*/

NWDSCCODE DSBuffer::CloseIteration(NWDS_ITERATION iterHandle, NWDS_OPERATION operation)
{
	return (iterHandle == -1) ? 0 : NWDSCloseIteration(dContext,iterHandle,operation) ;
}

/*
** NWDSCCODE DSBuffer::InitFilter(NWDS_OPERATION function = DSV_SEARCH_FILTER)
**
**	Initialise the filter buffer and create an expression cursor.
*/

NWDSCCODE DSBuffer::InitFilter(NWDS_OPERATION function)
{
	return DSFilter::InitBuf(function) || AllocFilter() ;
}

/*
** NWDSCCODE DSBuffer::ExitFilter()
**
**	Add an END token to the expression and record the expression in the
**	filter buffer.
*/

NWDSCCODE DSBuffer::ExitFilter()
{
	return AddFilterToken(FTOK_END) || PutFilter() ;
}

/*
** void DSBuffer::InitLoop(DSIteration * iteration)
**
**	Prepare for iteration.
*/

void DSBuffer::InitLoop(DSIteration * iteration)
{
	iteration->InitLoop() ;
	DSInput ::InitLoop(iteration) ;
	DSFilter::InitLoop(iteration) ;
	DSOutput::InitLoop(iteration) ;
}

/*
** void DSBuffer::ExitLoop(DSIteration * iteration)
**
**	Terminate iteration.
*/

void DSBuffer::ExitLoop(DSIteration * iteration)
{
	DSOutput::ExitLoop(iteration) ;
	DSFilter::ExitLoop(iteration) ;
	DSInput ::ExitLoop(iteration) ;
	iteration->ExitLoop(Status) ;
}

/*
** NWDSCCODE DSBuffer::DoesObjectExist(NWPSTR object)
**
**	Determine if the specified object exists.
*/

NWDSCCODE DSBuffer::DoesObjectExist(NWPSTR object)
{
NWDS_ITERATION	iterations = -1 ;

	return Status = NWDSRead(dContext,object,0,1,0,& iterations,Output) ;
}

/*
** NWDSCCODE DSBuffer::GetObjectHostServerAddress(
**				NWPSTR		object,
**				NWPSTR		server,
**				DSIteration *	iteration)
**
**	List the net addresses of an NDS object. This function only works for
**	objects with a Host Resource Name attribute (such as Volume objects).
**	NWDSGetObjectHostServerAddress iteration is managed by using a
**	DSIteration object. Developers should derive their own classes from
**	DSIteration and override the class's functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		SetAttrName(AttrName,SYN_NET_ADDRESS)
**		For each address
**			SetAttrValue(SYN_NET_ADDRESS,value)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::GetObjectHostServerAddress(
				NWPSTR		object,
				NWPSTR		server,
				DSIteration *	iteration)
{
	InitLoop(iteration) ;

	Status = NWDSGetObjectHostServerAddress(dContext,object,server,Output) ;

	if (Status == 0) ListAttributes(iteration) ;

	ExitLoop(iteration) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::GetServerAddresses(
**				NWCONN_HANDLE connHandle
**				DSIteration *	iteration)
**
**	List the net addresses of a server object. NWDSGetServerAddresses
**	iteration is managed by using a DSIteration object. Developers should
**	derive their own classes from DSIteration and override the class's
**	functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each address
**			SetAttrValue(SYN_NET_ADDRESS,value)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::GetServerAddresses(
				NWCONN_HANDLE	connHandle,
				DSIteration *	iteration)
{
NWCOUNT	count = 0 ;

	InitLoop(iteration) ;

	Status = NWDSGetServerAddresses(dContext,connHandle,& count,Output) ;

	if (Status == 0) ListAttributes(iteration) ;

	ExitLoop(iteration) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::List(NWPSTR object, DSIteration * iteration)
**
**	List the subordinate objects in the NDS tree using NWDSList().
**	NWDSList iteration is managed by using a DSIteration object. Developers
**	should derive their own classes from DSIteration and override the
**	class's functions to receive the results from NWDSList.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each object
**			InitObject(ObjectName,ObjectInfo)
**			ExitObject(ObjectName)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::List(NWPSTR object, DSIteration * iteration)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = NWDSList(dContext, 	// Directory context
				object,	      	// Base of listing
			      & iterHandle,   	// Iteration handle
				Output) ;     	// Object return buffer

		if (Status || ListObjects(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_LIST) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::ListByClassAndName(
**			NWPSTR		object,
**			NWPSTR		className,
**			NWPSTR		subordinate,
**			DSIteration *	iteration)
**			TimeStamp_T *	timeStamp = DSCPP_NO_TIME_FILTER,
**			WORD		onlyContainers = 0)
**
**	List the subordinate objects of the appropriate class and name.
**	NWDSListByClassAndName iteration is managed by using a DSIteration
**	object. Developers should derive their own classes from DSIteration and
**	override the class's functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each object
**			InitObject(ObjectName,ObjectInfo)
**			ExitObject(ObjectName)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::ListByClassAndName(
				NWPSTR		object,
				NWPSTR		className,
				NWPSTR		subordinate,
				DSIteration *	iteration,
				TimeStamp_T *	timeStamp,
				WORD		onlyContainers)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = (timeStamp == DSCPP_NO_TIME_FILTER)
			? NWDSListByClassAndName(
				dContext,object,className,subordinate,& iterHandle,Output)
			: NWDSExtSyncList(
				dContext,object,className,subordinate,& iterHandle,timeStamp,onlyContainers,Output) ;


		if (Status || ListObjects(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_LIST) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::ListContainers(NWPSTR object, DSIteration * iteration)
**
**	List the subordinate container objects in the NDS tree.
**	NWDSListContainers iteration is managed by using a DSIteration
**	object. Developers should derive their own classes from DSIteration and
**	override the class's functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each object
**			InitObject(ObjectName,ObjectInfo)
**			ExitObject(ObjectName)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::ListContainers(NWPSTR object,DSIteration * iteration)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = NWDSListContainers(
				dContext, 	// Directory context
				object,	      	// Base of listing
			      & iterHandle,   	// Iteration handle
				Output) ;     	// Object return buffer

		if (Status || ListObjects(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_LIST) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::ListAttrsEffectiveRights(
**				NWPSTR		object,
**				NWPSTR		subject,
**				NWFLAGS		allAttrs,
**				DSIteration *	iteration)
**
**	List the effective rights the subject has over the attributes of
**	the object. NWDSListAttrsEffectiveRights iteration is managed by using
**	a DSIteration object. Developers should derive their own classes from
**	DSIteration and override the class's functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each attribute
**			SetAttrName(AttrName,SYN_INTEGER)
**			SetAttrValue(SYN_INTEGER,Rights)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::ListAttrsEffectiveRights(
				NWPSTR		object,
				NWPSTR		subject,
				NWFLAGS		allAttrs,
				DSIteration *	iteration)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = NWDSListAttrsEffectiveRights(
				dContext, 	// Directory context
				object,	      	// Destination of rights
				subject,	// Source of rights
				allAttrs,     	// Return requested attr only
				Input,	      	// Requested attributes
			      & iterHandle,   	// Iteration handle
				Output) ;     	// Object return buffer

		if (Status || ListAttributes(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_READ) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::Read(
**			NWPSTR		object,
**			NWDS_TYPE	infoType,
**			NWFLAGS		allAttrs,
**			DSIteration *	iteration,
**			TimeStamp_T *	timeStamp = DSCPP_NO_TIME_FILTER)
**
**	Execute a read of the NDS tree using NWDSRead(). NWDSRead iteration
**	is managed by using a DSIteration object. Developers should derive
**	their own classes from DSIteration and override the class's
**	functions to receive the results from NWDSRead.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each attribute
**			SetAttrName(AttrName,Syntax)
**			For each value in AttrName
**				SetAttrValue(Syntax,Data)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::Read(
			NWPSTR		object,
			NWDS_TYPE	infoType,
			NWFLAGS		allAttrs,
			DSIteration *	iteration,
			TimeStamp_T *	timeStamp)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = (timeStamp == DSCPP_NO_TIME_FILTER)
			? NWDSRead(
				dContext,object,infoType,allAttrs,Input,& iterHandle,Output)
			: NWDSExtSyncRead(
				dContext,object,infoType,allAttrs,Input,& iterHandle,timeStamp,Output) ;

		if (Status || ListAttributes(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_READ) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::ReadReferences(
**			NWPSTR		server,
**			NWPSTR		object,
**			NWDS_TYPE	infoType,
**			NWFLAGS		allAttrs,
**			DSIteration *	iteration,
**			DWORD		timeStamp)
**
**	Execute a read of the NDS tree using NWDSReadReferences().
**	NWDSRead iteration is managed by using a DSIteration object.
**	Developers should derive their own classes from DSIteration and
**	override the class's functions to receive the results.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each attribute
**			SetAttrName(AttrName,Syntax)
**			For each value in AttrName
**				SetAttrValue(Syntax,Data)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::ReadReferences(
			NWPSTR		server,
			NWPSTR		object,
			NWDS_TYPE	infoType,
			NWFLAGS		allAttrs,
			DSIteration *	iteration,
			DWORD		timeStamp)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = NWDSReadReferences(
				dContext,server,object,infoType,allAttrs,
				Input,timeStamp,& iterHandle,Output) ;

		if (Status || ListAttributes(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_READ) ;

	return Status ;
}

/*
** NWDSCCODE DSBuffer::Search(
**			NWPSTR			base,
**			NWDS_SEARCH_SCOPE	scope,
**			NWFLAGS			searchAliases,
**			NWDS_TYPE		infoType,
**			NWFLAGS			allAttrs,
**			DSIteration *		iteration,
**			TimeStamp_T *		timeStamp = DS_NO_TIME_FILTER)
**
**	Execute a search of the NDS tree using NWDSSearch(). NWDSSearch
**	iteration is managed by using a DSIteration object. Developers should
**	derive their own classes from DSIteration and override the class's
**	functions to receive the results from NWDSSearch.
**
**	DSIteration Protocol:
**		InitLoop()
**		For each object
**			InitObject(ObjectName,ObjectInfo)
**			For each attribute
**				SetAttrName(AttrName,Syntax)
**				For each value in AttrName
**					SetAttrValue(Syntax,Data)
**			ExitObject(ObjectName)
**		ExitLoop(Status)
*/

NWDSCCODE DSBuffer::Search(
			NWPSTR			base,
			NWFLAGS			scope,
			NWFLAGS			searchAliases,
			NWDS_TYPE		infoType,
			NWFLAGS			allAttrs,
			DSIteration *		iteration,
			TimeStamp_T *		timeStamp)
{
NWDS_ITERATION	iterHandle = -1 ;

	InitLoop(iteration) ;

	do {
		Status = (timeStamp == DSCPP_NO_TIME_FILTER)
			? NWDSSearch(
				dContext,base,scope,searchAliases,Filter,
				infoType,allAttrs,Input,& iterHandle,0,0,Output)
			: NWDSExtSyncSearch(
				dContext,base,scope,searchAliases,Filter,timeStamp,
				infoType,allAttrs,Input,& iterHandle,0,0,Output) ;

		if (Status || ListObjects(iteration)) break ;

	} while (iterHandle != -1) ;

	ExitLoop(iteration) ;

	if (Status > 0) CloseIteration(iterHandle,DSV_SEARCH) ;

	return Status ;
}


