/*
        cstr.cpp -- C string class wrapper
        (C) Copyright 1996  John Webster Small
        All rights reserved
*/

#include "cstr.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#define CSTR_GET_CHUNK_SIZE  30

const char cstr::WHITE_SPACE[] = " \t\n\r\f";

char * char2str(char c)
{
    static char char2strBuf[2];
    char2strBuf[0] = c;
    char2strBuf[1] = '\0';
    return char2strBuf;
}

cstr&
cstr:: _ncpy(const char * S, size_t n, size_t extra)
{
    _ok = 1;
    if (S)  {
        if (S == _S)  {
            if (n < _len)
                _S[_len=n] = '\0';
            return *this;
        }
        if (_S)
            _S[_len=0] = '\0';
        return _ncat(S,n,extra);
    }
    else if (extra)  {
        if (_limit && extra == _limit - 1)  {
            assert(_S);
            _S[_len=0] = '\0';
        }
        else  {
            delete[] _S;
            if (extra < SIZE_T_MAX)
                extra++;
            _S = new char[extra];
            if (_S)  {
                _S[_len=0] = '\0';
                _limit = extra;
            }
            else  {
                _len = _limit = 0;
                _ok = 0;
            }
        }
    }
    else if (_S)
        _S[_len=0] = '\0';
    return *this;
}

cstr&
cstr:: _ncat(const char * S, size_t n, size_t extra)
{
    _ok = 1;
    size_t len_ = (S? strlen(S) : 0);
    if (n > len_)
        n = len_;
    len_ = _len + n; //len now result string len!
    assert(len_ >= _len && len_ >= n); //too long!
    size_t limit_ = len_ + 1 + extra;
    if (len_ >= limit_)  {
        limit_ = SIZE_T_MAX; // trunc any excess extra!
        assert(len_ < limit_);//result string too long!
    }
    if (limit_ > _limit)  {
        char * newS = ((limit_ > 1 || S)?
            new char[limit_] : 0);
            // if S == "" then cstr == ""
        if (newS)  {
            if (_S)  {
                assert(strlen(_S) == _len);
                strcpy(newS,_S);
                delete[] _S;
            }
            else
                newS[_len=0] ='\0';
            _S = newS;
            _limit = limit_;
        }
        else  {
            n = 0; //error: nothing concatenated!
            _ok = 0;
        }
    }
    if (_S)  {
        if (n)  {  // if S == 0 then n == 0
            memcpy(&_S[_len],S,n);
            _len += n;
        }
        _S[_len] = '\0';
    }
    return *this;
}

int cstr:: _ncmpi
    (const char * cs, const char * ct, size_t n)
{
    while (n--)  {
        int signum = int(::tolower(*cs++))
            - int(::tolower(*ct++));
        if (signum)
            return signum;
    }
    return 0;
}

cstr& cstr:: _charFill(char c, size_t nfill, size_t extra)
{
    size_t limit_ = nfill + 1 + extra;
    if (nfill >= limit_)  {
        limit_ = SIZE_T_MAX; // trunc any excess extra!
        assert(nfill < limit_);//result string too long!
    }
    _ok = 1;
    if (limit_ > _limit)
        setLimit(limit_);  // sets _ok
    if (_ok)  {
        assert(_S);
        for (size_t i = 0; i < nfill; i++)
            _S[i] = c;
        _S[_len=nfill] = '\0';
    }
    return *this;
}

cstr:: cstr(size_t extra)
    : _S(0), _len(0), _limit(0), _ok(1)
{ (void) _ncat(0,0,extra); }

cstr:: cstr(const cstr& s, size_t extra)
    : _S(0), _len(0), _limit(0), _ok(1)
{ (void) _ncat(s,SIZE_T_MAX,extra); }

cstr:: cstr(const char * S, size_t extra)
    : _S(0), _len(0), _limit(0), _ok(1)
{ (void) _ncat(S,SIZE_T_MAX,extra); }

cstr:: cstr(char c, size_t nfill, size_t extra)
    : _S(0), _len(0), _limit(0), _ok(1)
{
    (void) _charFill(c,nfill,extra);
}

cstr:: cstr(size_t maxLen, const char * format, ...)
    : _S(0), _len(0), _limit(0), _ok(1)
{
    va_list args;
    va_start(args,format);
    (void) cat_vprintf(maxLen,format,args);
    va_end(args);
}

cstr:: ~cstr()  { delete[] _S; }

cstr& cstr:: wordWrap(size_t atColumn)
{
    size_t nl, nlCol, i = 0;

    while (i < _len)  {
      nl = idx_str("\n",i);
      if (nl == CSTR_NOTFOUND)
          nl = _len;
      while (i + atColumn < nl) {  // wrap line
        nlCol = atColumn;
        if (!strchr(WHITE_SPACE,_S[i+nlCol])) {
            while (nlCol &&
              !strchr(WHITE_SPACE,_S[i+nlCol]))
              nlCol--;
            if (!nlCol)  {
                nlCol = atColumn;
                while ((nlCol + i < _len) &&
                  !strchr(WHITE_SPACE,_S[i+nlCol]))
                  nlCol++;
            }
        }
        if (i + nlCol < _len)
           ins(i+nlCol,"\n");
        i += nlCol;
        i++;
      }
      i += nl;
      i++;
    }
    return *this;
}

cstr& cstr:: cpy_printf
    (size_t maxLen, const char * format, ...)
{
    va_list args;
    va_start(args,format);
    (void) clr().cat_vprintf(maxLen,format,args);
    va_end(args);
    return *this;
}

cstr& cstr:: cat_vprintf (size_t maxLen,
     const char * format, va_list arglist)
{
    assert(maxLen < SIZE_T_MAX && format);
    cstr buf(maxLen);
    if (buf.ok())  {
        if (EOF == ::vsprintf
            ((char *)buf.str(),format,arglist))
            _ok = 0;
        else  {
            buf.setLen();
            cat(buf);
        }
 
    }
    else
        _ok = 0;
    return *this;
}

cstr& cstr:: cat_printf
    (size_t maxLen, const char * format, ...)
{
    va_list args;
    va_start(args,format);
    (void) cat_vprintf(maxLen,format,args);
    va_end(args);
    return *this;
}

cstr& cstr:: cat_itoa(int value, int radix)
{
    if (value < 0 && radix == 10)  {
        value = -value;
        cat("-");
    }
    return cat_ultoa((unsigned long)value,radix);
}

cstr& cstr:: cat_ltoa(long value, int radix)
{
    if (value < 0 && radix == 10)  {
        value = -value;
        cat("-");
    }
    return cat_ultoa((unsigned long)value,radix);
}

cstr& cstr:: cat_ultoa(unsigned long value, int radix)
{
    int digit, wrap = 0;
    unsigned long divisor = 1;
    size_t extra = 1;
    assert((radix >= 2) && (radix <= 36));
    while (value / divisor)  {
        if (divisor > (ULONG_MAX / radix))  {
            wrap = 1;
            break;
        }
        divisor *= radix;
        extra++;
    }
    if ((divisor > 1) && !wrap) {
        divisor /= radix;
        extra--;
    }
    _ncat(0,0,extra);
    while (1)  {
        assert(divisor);
        digit = (int) (value / divisor);
        assert(digit < radix);
        if (digit <= 9)
            cat(char2str('0' + digit));
        else
            cat(char2str('A' + digit - 10));
        value %= divisor;
        if (divisor == 1)
            break;
        else
            divisor /= radix;
    };
    return *this;
}

double cstr:: atof()
{
    if (_len > 0)  {
        assert(_S);
        return  ::strtod(_S,(char**)0);
    }
    return 0;
}

long cstr:: strtol(int base)
{
    if (_len > 0)  {
        assert(_S);
        return ::strtol(_S,(char**)0,base);
    }
    return 0;
}

double cstr:: strtod(cstr& suffix)
{
    if (_len > 0)  {
        assert(_S);
        char * endp;
        double answer = ::strtod(_S,&endp);
        suffix.cpy(endp);
        return answer;
    }
    return 0;
}

long cstr:: strtol(int base, cstr& suffix)
{
    if (_len > 0)  {
        assert(_S);
        char * endp;
        long answer = ::strtol(_S,&endp,base);
        suffix.cpy(endp);
        return answer;
    }
    return 0;
}

unsigned long cstr:: strtoul(int base, cstr& suffix)
{
    if (_len > 0)  {
        assert(_S);
        char * endp;
        unsigned long answer = ::strtoul(_S,&endp,base);
        suffix.cpy(endp);
        return answer;
    }
    return 0;
}

cstr& cstr:: nins
  (size_t pos, const char * S, size_t n, size_t extra)
{
    _ok = 1;
    if (!S)
        return *this;
    // allow idx() calls for pos!
    if (pos == CSTR_NOTFOUND)  {
        _ok = 0;
        return *this;
    }
    if (pos >= _len)
        return _ncat(S,n,extra);
    size_t len_ = strlen(S);
    if (!len_)   // if S == "" then cstr at least ""
        return *this;
    assert(_S);  // i.e. _len > pos >= 0
    // we can now
    // assert(_S && _len && S && len_ && pos < _len);
    if (n > len_)
        n = len_;
    if (n > vacancy())  {
        len_ = _len + n; //len_ now result string len!
        assert(len_ >= _len && len_ >= n); //too long!
        size_t limit_ = len_ + 1 + extra;
        if (len_ >= limit_)  {
            limit_ = SIZE_T_MAX;
              // trunc any excess extra!
            assert(len_ < limit_);
              // result string too long!
        }
        char * newS = new char[limit_];
        if (!newS)  {
            _ok = 0;
            return *this;  // no room for insertion!
        }
        strcpy(newS,_S);
        delete[] _S;
        _S = newS;
        _limit = limit_;
    }
    // _S now has the room for the insertion!
    memmove(&_S[pos+n],&_S[pos],_len-pos);
    memcpy(&_S[pos],S,n);
    _S[_len += n] = '\0';
    return *this;
}

cstr& cstr:: ncpy2
  (size_t pos, const char * S, size_t n, size_t extra)
{
    _ok = 1;
    if (!S)
        return *this;
    // allow idx() calls for pos!
    if (pos == CSTR_NOTFOUND)  {
        _ok = 0;
        return *this;
    }
    if (pos >= _len)
        return _ncat(S,n,extra);
    size_t len_ = strlen(S);
    if (!len_)   // if S == "" then cstr at least ""
        return *this;
    assert(_S);  // i.e. _len > pos >= 0
    // we can now
    // assert(_S && _len && S && len_ && pos < _len);
    if (n > len_)
        n = len_;
    size_t tail = pos + n;
    assert(tail >= pos && tail >= n); // too long!
    if (tail >= _limit)  {
        size_t limit_ = tail + 1 + extra;
        if (tail >= limit_)  {
            limit_ = SIZE_T_MAX;
              // trunc any excess extra!
            assert(tail < limit_);
              // result string too long!
        }
        char * newS = new char[limit_];
        if (!newS)  {
            _ok = 0;
            return *this;  // no room for copy!
        }
        strcpy(newS,_S);
        delete[] _S;
        _S = newS;
        _limit = limit_;
    }
    // _S now has the room for the cpy!
    memcpy(&_S[pos],S,n);
    if (tail > _len)
        _S[_len = pos + n] = '\0';
    return *this;
}

cstr& cstr:: rmv(size_t pos, size_t n)
{
    if (pos >= _len)
        return *this;
    assert(_S); // i.e. _len > pos >= 0
    size_t stop = pos + n;
    if (stop >= _len || stop < pos || stop < n)
        _S[_len=pos] = '\0';
    else  {
        strcpy(&_S[pos],&_S[pos+n]);
        _len -= n;
    }
    return *this;
}

cstr& cstr:: substr(cstr& dst, size_t pos,
    size_t n, size_t extra)
{
    if (pos < _len)  {
        assert(_S);
        return dst.ncpy(&_S[pos],n,extra);
    }
    return dst.clr();
}

cstr& cstr:: replace
  (size_t pos, size_t n, const char * S, size_t extra)
{
    assert(pos + n >= pos && pos + n >= n);
    if (pos + n <= _len)
        return rmv(pos,n).nins(pos,S,SIZE_T_MAX,extra);
    return *this;
}

cstr& cstr:: replace
  (size_t pos, size_t n, char c, size_t extra)
{
    assert(pos + n >= pos && pos + n >= n);
    if (pos + n <= _len)
        return rmv(pos,n).nins(pos,c,n,extra);
    return *this;
}

cstr& cstr:: replace(const char * P, const char * S,
    int caseIgnore, size_t extra)
{
    size_t pos = idx_str(P,0,caseIgnore);
    if (pos != CSTR_NOTFOUND)
        return replace(pos,strlen(P),(S?S:""),extra);
    return *this;
}

cstr& cstr:: replaceAll(const char * P, const char * S,
    int caseIgnore, size_t extra)
{
    size_t pos = idx_str(P,0,caseIgnore);
    size_t p = (P? strlen(P) : 0);
    size_t s = (S? strlen(S) : 0);
    while (pos != CSTR_NOTFOUND)
        if (pos + p >= pos && pos + p >= p
            && pos + p <= _len)  {
            (void) replace(pos,p,S,extra);
            pos = idx_str(P,pos+s,caseIgnore);
        }
        else
            break;
    return *this;
}

cstr& cstr:: trim(format f, const char * strip)
{
    assert(f != center && strip && strlen(strip));
    if (f == left || f == both)  {
        if (_len)  {
            assert(_S);
            (void) rmv(0,strspn(_S,strip));
        }
    }
    if (f == right || f == both)  {
        assert(_S || !_len);
        for (size_t i = _len; i--; )
            if (strcspn(&_S[i],strip))  {
                trunc(i+1);
                break;
            }
    }
    return *this;
}

cstr& cstr:: pad
    (size_t fieldWidth, format f, char fillch)
{
  assert(fieldWidth && ((f == left) || (f == right)));
  setLimit(fieldWidth+1);
  if (f == left)
      return nins(0,fillch,fieldWidth-len());
  return nins(len(),fillch,fieldWidth-len());
}

cstr& cstr::justify
    (size_t fieldWidth, format f, char fillch)
{
    assert(fieldWidth);
    setLimit(fieldWidth+1);
    size_t n = fieldWidth-len();
    if (n)
      switch (f)  {
        case right: (void) prefix(fillch,n); break;
        case center:
            (void) prefix(fillch,n/2);
            (void) ncat(fillch,fieldWidth-len());
            break;
        case left: (void) ncat(fillch,n); break;
        case both: /* n/a */ break;
      }
    return *this;
}

size_t cstr::idx(const char * S) const
{
    if (_S &&
       ((unsigned long)S) >= ((unsigned long)_S))  {
        unsigned long pos =
            ((unsigned long)S) - ((unsigned long)_S);
        if (pos < ((unsigned long)_len))
            return size_t(pos);
    }
    return CSTR_NOTFOUND;
}

size_t cstr::idx_chr(char c) const
{
    assert(!_len || _S);
    for (register size_t i = 0; i < _len; i++)
        if (_S[i] == c)
            return i;
    return CSTR_NOTFOUND;
}

size_t cstr::idx_rchr(char c) const
{
    assert(!_len || _S);
    register size_t i = _len;
    while (i--)
        if (_S[i] == c)
            return i;
    return CSTR_NOTFOUND;
}

size_t cstr::idx_brk(const char * S) const
{
    if (S)  {
        assert(!_len || _S);
        for (register size_t i = 0; i < _len; i++)
            if (strchr(S,_S[i]))
                return i;
    }
    return CSTR_NOTFOUND;
}

size_t cstr::idx_str
  (const char * S, size_t start, int caseIgnore) const
{
    if (S && start < _len)  {
        register size_t i = start;
        assert(!_len || _S);
        size_t len_ = strlen(S);
        if (i + len_ >= i && i + len_ >= len_)
            if (caseIgnore)  {
                for (; i + len_ <= _len; i++)
                    if (!_ncmpi(&_S[i],S,len_))
                        return i;
            }
            else
                for (; i + len_ <= _len; i++)
                    if (!strncmp(&_S[i],S,len_))
                        return i;
    }
    return CSTR_NOTFOUND;
}

cstr& cstr::tolower()
{
    if (_S)
        for (size_t i = 0; i < _len; i++)
            _S[i] = char(::tolower(_S[i]));
    return *this;
}

cstr& cstr::toupper()
{
    if (_S)
        for (size_t i = 0; i < _len; i++)
            _S[i] = char(::toupper(_S[i]));
    return *this;
}

cstr& cstr::toproper()
{
    tolower();
    for (char * S = _S;
        S && *(S += strspn(S,WHITE_SPACE));
        S += strcspn(S,WHITE_SPACE))
        *S = ::toupper(*S); 
    return *this;
}


cstr& cstr::setLimit(size_t limit_)
{
    _ok = 1;
    if (limit_)  {
        if (limit_ != _limit)  {
            char * newS = new char[limit_];
            if (newS)  {
                size_t len_ = ((limit_ >= _len + 1)?
                    _len : (limit_ - 1));
                if (len_)
                    memcpy(newS,_S,len_);
                delete[] _S;
                _S = newS;
                _S[_len=len_] = '\0';
                _limit = limit_;
            }
            else
                _ok = 0;
        }
    }
    else if (_S)  {
        delete[] _S;
        _S = 0;
        _limit = _len = 0;
    }
    return *this;
}

cstr& cstr:: atLeastLimit(size_t minLimit)
{
    if (minLimit > _limit)
        return setLimit(minLimit);
    return *this;
}

cstr& cstr::trunc(size_t len_)
{
    if (len_)  {
        if (len_ <= _len)
            setLimit(len_+1);
    }
    else
        setLimit(0);
    return *this;
}

cstr& cstr::setLen(size_t bufLen)
{
    if (_S)
        if (bufLen == SIZE_T_MAX)
            _len = strlen(_S);
        else
            if (bufLen < _limit)
                _S[_len=bufLen] = '\0';
    return *this;
}

int cstr::cmp(const char * S, int caseIgnore) const
{
    if (_S)
        if (S)
            if (caseIgnore)
                return _ncmpi(_S,S,_len+1);
            else
                return strcmp(_S,S);
        else
            return 1;
    else if (S)
        return -1;
    return 0;
}

int cstr::ncmp
    (const char * S, size_t n, int caseIgnore) const
{
    if (!n)
        return 0;
    if (_S)
        if (S)
            if (caseIgnore)
                return _ncmpi(_S,S,n);
            else
                return strncmp(_S,S,n);
        else
            return 1;
    else if (S)
        return -1;
    return 0;
}

ostream& cstr::storeOn(ostream& os) const
{
    if ((os << _len << '\n'))  {
        char * S = _S;
        for (unsigned chunks = _len / INT_MAX;
            chunks--; S += INT_MAX)
            if (!os.write(S,INT_MAX))
                return os;
        int lastLength = _len % INT_MAX;
        if (lastLength)
            (void) os.write(S,lastLength);
    }
    return os;
}

istream& cstr::readFrom(istream& is)
{
    size_t len_;
    if ((is >> len_))  {
        (void) is.get(); // skip '\n'
        if (len_)  {
            if (len_ >= _limit)  {
                delete[] _S;
                _S = new char[len_+1];
                if (_S)
                    _limit = len_ + 1;
                else
                    _len = _limit = 0;
            }
            if (_S)  {
                _S[_len = len_] = '\0';
                char * S = _S;
                for (unsigned chunks = len_ / INT_MAX;
                    chunks; chunks--)
                {
                    if (!is.read(S,INT_MAX))  {
                        *S = '\0';
                        _len = strlen(_S);
                        return is;
                    }
                    S += INT_MAX;
                }
                int lastLength = len_ % INT_MAX;
                if (lastLength)
                    if (!is.read(S,lastLength))  {
                        *S = '\0';
                        _len = strlen(_S);
                    }
            }
        }
    }
    return is;
}

istream& cstr::ignore(istream& is)
{
    size_t len_;
    if ((is >> len_))  {
        (void) is.get(); // skip '\n'
        for (unsigned chunks = len_ / INT_MAX;
            chunks; chunks--)
            (void) is.ignore(INT_MAX);
        int lastLength = len_ % INT_MAX;
        if (lastLength)
            (void) is.ignore(lastLength);
    }
    return is;
}

istream& cstr::get(istream& is, char delimiter)
{
    static char buf[CSTR_GET_CHUNK_SIZE];
    clr();
    do  {
        if (is.get(buf,sizeof(buf),delimiter))
           (void) _ncat(buf,size_t(is.gcount()),0);
    } while (is.good() && is.gcount());
    return is;
}

istream& cstr::getline(istream& is, char delimiter)
{
    (void) get(is,delimiter);
    (void) is.get();
    return is;
}


//#define CSTR_DEMO
#ifdef CSTR_DEMO

#include <iomanip.h>
#include <fstream.h>

main()
{
    cstr s("Now is the time for all",10);
    s.cat(" programmers to take a string break!");
    cout << (const char *)s << endl;
    s.toproper().printOn(cout) << endl;
    s.tolower();
    ofstream os("cstr.tmp");
    if (os)  {
        os << s;
        os.close();
        ifstream is("cstr.tmp");
        if (is)  {
            cstr t(80);
            is >> t;
            cout << (const char *)t << endl;
            (void) is.seekg(0);
            (void) t.getline(is);
            cout << (const char *)t << endl;
            (void) t.get(is);
            cout << (const char *)t << endl;
            is.close();
        }
    }
    cstr p("for all programmers ");
    size_t idx = s.idx(strstr(s,p));
    s.rmv(idx,p.len());
    cout << (const char *) s << endl;
    p.cpy("for everyone ");
    s.ins(idx,p);
    cout << (const char *) s << endl;
    s.ncpy2(11,'*',4);
    cout << (const char *) s << endl;
    s("Hello World!"); p("hello World!");
    cout << "cmp(" << s.str() << ","
         << p.str() << ") = "
         << s.cmp(p) << endl;
    cout << "Ignoring case we have: " << endl;
    cout << "cmp(" << s.str() << ","
         << p.str() << ") = "
         << s.cmp(p,1) << endl;
    s.replace("World","Goodbye");
    cout << s.str() << endl;
    return 0;
}

#endif
