/*
        cstr.h -- C string class wrapper
        (C) Copyright 1996  John Webster Small
        All rights reserved
        Demo: CC -DCSTR_DEMO cstr
*/

#ifndef cstr_h
#define cstr_h

/* LINTLIBRARY */

#ifndef assert_h
  #include <assert.h>
  #ifndef assert_h
    #define assert_h
  #endif
#endif

#ifndef string_h
  #include <string.h>
  #ifndef string_h
    #define string_h
  #endif
#endif

#ifndef stddef_h
  #include <stddef.h>
  #ifndef stddef_h
    #define stddef_h
  #endif
#endif

#ifndef limits_h
  #include <limits.h>
  #ifndef limits_h
    #define limits_h
  #endif
#endif

#ifndef stdarg_h
  #include <stdarg.h>
  #ifndef stdarg_h
    #define stdarg_h
  #endif
#endif

#ifndef iostream_h
  #include <iostream.h>
  #ifndef iostream_h
    #define iostream_h
  #endif
#endif

// HEAP_LOSS is a fudge in case the heap can't
// allocate the maximum size of size_t.
#ifndef HEAP_LOSS
  #define HEAP_LOSS  0
#endif

#ifndef SIZE_T_MAX
  #define SIZE_T_MAX \
    size_t(size_t(ULONG_MAX) - HEAP_LOSS)
#endif

#define CSTR_NOTFOUND SIZE_T_MAX

// Enable cstr to handle char's
extern char * char2str(char c);

class cstr
{
    char * _S;
    size_t _len, _limit;
    int _ok; // last operation ok?: cstr(chfill),
      // _ncpy(), _ncat(), nins(), ncpy2(), setLimit()
      // cstr(maxLen,format,...), vprintf(), printf()
    cstr&
      _ncpy(const char * S, size_t n, size_t extra);
    cstr&
      _ncat(const char * S, size_t n, size_t extra);
    static int _ncmpi
      (const char * cs, const char * ct, size_t n);
    cstr& _charFill
      (char c, size_t nfill, size_t extra = 0);
  public:
    enum format { left, center, right, both };
    static const char WHITE_SPACE[];
    cstr(size_t extra = 0);
    cstr(const cstr& s, size_t extra = 0);
    cstr(const char * S, size_t extra = 0);
    cstr(char c, size_t nfill, size_t extra = 0);
    cstr(size_t maxLen, const char * format, ...);
    ~cstr();
    cstr& wordWrap(size_t atColumn);
    int ok() { return _ok; }
    operator const char *() const { return _S; }
    const char * str() const { return _S; }
    char& operator[](size_t n)
        { assert(_S && n < _len);
          return _S[n]; }
    cstr& cpy(const char * S, size_t extra = 0)
        { return _ncpy(S,SIZE_T_MAX,extra); }
    cstr& operator()(size_t extra)
        { return _ncat(0,0,extra); }
    cstr& operator()(const char * S)
        { return cpy(S,0); }
    cstr& operator()(const char * S, size_t extra)
        { return cpy(S,extra); }
    cstr& operator=(const char * S)
        { return cpy(S); }
    cstr& operator=(const cstr& s)
        { return cpy(s); }
    const char * operator()
        (char c, size_t nfill, size_t extra)
        { return _charFill(c,nfill,extra).str(); }
    cstr& ncpy
        (const char * S, size_t n, size_t extra = 0)
        { return _ncpy(S,n,extra); }
    cstr& ncpy(char c, size_t nfill = 1,
        size_t extra = 0)
        { return _charFill(c,nfill,extra); }
    cstr& cpy_vprintf(size_t maxLen,
        const char * format, va_list arglist)
        { return clr().cat_vprintf(maxLen,format,arglist); }
    cstr& cpy_printf
        (size_t maxLen, const char * format, ...);
    cstr& cat(const char * S, size_t extra = 0)
        { return _ncat(S,SIZE_T_MAX,extra); }
    cstr& operator+=(const char * S)
        { return cat(S); }
    cstr& ncat
        (const char * S, size_t n, size_t extra = 0)
        { return _ncat(S,n,extra); }
    cstr& ncat(char c, size_t nfill = 1,
        size_t extra = 0)
        { return ncat(cstr(c,nfill,0),nfill,extra); }
    cstr& cat_vprintf(size_t maxLen,
        const char * format, va_list arglist);
    cstr& cat_printf
        (size_t maxLen, const char * format, ...);
    cstr& cat_itoa(int value, int radix);
    cstr& cat_ltoa(long value, int radix);
    cstr& cat_ultoa(unsigned long value, int radix);
    double atof();
    long strtol(int base);
    int atoi() { return (int) strtol(10); }
    long atol() { return strtol(10); }
    double strtod(cstr& suffix);
    long strtol(int base, cstr& suffix);
    unsigned long strtoul(int base, cstr& suffix);
    cstr& nins(size_t pos, const char * S,
        size_t n, size_t extra = 0);
    cstr& nins(size_t pos, char c,
        size_t nfill = 1, size_t extra = 0)
        { return nins(pos,cstr(c,nfill,0),nfill,extra); }
    cstr& ins(size_t pos, const char * S,
        size_t extra = 0)
        { return nins(pos,S,SIZE_T_MAX,extra); }
    cstr& prefix(const char * S, size_t extra = 0)
        { return nins(0,S,SIZE_T_MAX,extra); }
    cstr& prefix
        (char c, size_t nfill = 1, size_t extra = 0)
        { return prefix(cstr(c,nfill,0),extra); }
    cstr& ncpy2(size_t pos, const char * S,
        size_t n, size_t extra = 0);
    cstr& ncpy2(size_t pos, char c,
        size_t nfill = 1, size_t extra = 0)
        { return ncpy2(pos,cstr(c,nfill,0),nfill,extra);}
    cstr& cpy2(size_t pos, const char * S,
        size_t extra = 0)
        { return ncpy2(pos,S,SIZE_T_MAX,extra); }
    cstr& rmv(size_t pos, size_t n = SIZE_T_MAX);
    cstr& substr(cstr& dst, size_t pos,
        size_t n = SIZE_T_MAX, size_t extra = 0);
    cstr& extract(cstr& dst, size_t pos,
        size_t n = SIZE_T_MAX, size_t extra = 0)
        { (void) substr(dst,pos,n,extra);
          return rmv(pos,n); }
    cstr& replace(size_t pos, size_t n,
        const char * S = 0, size_t extra = 0);
    cstr& replace(size_t pos, size_t n, char c,
        size_t extra = 0);
    cstr& replace(const char * P, const char * S = 0,
        int caseIgnore = 0, size_t extra = 0);
    cstr& replaceAll
        (const char * P, const char * S = 0,
        int caseIgnore = 0, size_t extra = 0);
    cstr& trim(format f = both,
        const char * strip = WHITE_SPACE);
    cstr& pad(size_t fieldWidth, format f = left,
        char fillch = ' ');
    cstr& justify(size_t fieldWidth, format f = left,
        char fillch = ' ');
    size_t idx(const char * S) const;
    size_t idx_chr(char c) const;
    size_t idx_rchr(char c) const;
    size_t idx_brk(const char * S) const;
    size_t idx_str(const char * S,
        size_t start = 0, int caseIgnore = 0) const;
    cstr& tolower();
    cstr& toupper();
    cstr& toproper();
    size_t len() const { return _len; }
    size_t vacancy() const
        { return (_limit? _limit - _len - 1 : 0); }
    size_t limit() const { return _limit; }
    cstr& setLimit(size_t limit_);
    cstr& atLeastLimit(size_t minLimit);
    cstr& pack() { return setLimit(_len? _len+1 : 0); }
    cstr& unpack() { return cat(""); }
    cstr& trunc(size_t len_ = 0);
    cstr& clr()
        { if (_S) _S[_len=0] = '\0'; return *this; }
    cstr& setLen(size_t bufLen = SIZE_T_MAX);
    int cmp(const char * S, int caseIgnore = 0) const;
    int ncmp(const char * S, size_t n,
        int caseIgnore = 0) const;
    int operator==(const char * S) const
        { return !cmp(S); }
    int operator!=(const char * S) const
        { return cmp(S); }
    int operator< (const char * S) const
        { return cmp(S) <  0; }
    int operator<=(const char * S) const
        { return cmp(S) <= 0; }
    int operator> (const char * S) const
        { return cmp(S) >  0; }
    int operator>=(const char * S) const
        { return cmp(S) >= 0; }
    int operator==(const cstr& cs) const
        { return !cmp(cs); }
    int operator!=(const cstr& cs) const
        { return cmp(cs); }
    int operator< (const cstr& cs) const
        { return cmp(cs) <  0; }
    int operator<=(const cstr& cs) const
        { return cmp(cs) <= 0; }
    int operator> (const cstr& cs) const
        { return cmp(cs) >  0; }
    int operator>=(const cstr& cs) const
        { return cmp(cs) >= 0; }
    ostream& printOn(ostream& os) const
        { return os << str(); }
    ostream& storeOn(ostream& os) const;
    istream& readFrom(istream& is);
    static istream& ignore(istream& is);
    istream& get(istream& is, char delimiter = '\n');
    istream& getline
        (istream& is, char delimiter = '\n');
    cstr(istream& is) : _S(0), _len(0), _limit(0)
        { (void) readFrom(is); }
};

inline ostream& operator<<(ostream& os, const cstr& cs)
    { return cs.storeOn(os); }
inline istream& operator>>(istream& is, cstr& cs)
    { return cs.readFrom(is); }

#endif
