// File    : HEAPVIEW.CPP
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Wed 08/02/95 17:54:19
// Note    : Copyright 1993-95, Eric Woodruff, All rights reserved
// Compiler: Borland C++ 3.1 to 4.xx
//
// This file contains the Heap Viewer class.  This heap viewer will report the
// proper amount of memory regardless of whether the application is running
// in real mode, 16 bit protected mode, or 32 bit protected mode.
//
// To enable free EMS/XMS memory reporting, add the REPORT_EMS_XMS definition
// to the project or make file.
//

#if defined(__FLAT__)
#include <windows.h>
#endif

#include <alloc.h>
#include <stdio.h>
#include <string.h>

#define Uses_TRect
#define Uses_TView
#include <tv.h>

#if !defined(__DPMI16__) && !defined(__DPMI32__)

#if defined(REPORT_EMS_XMS) && !defined(SHAREWARE)
#include <tvirtmem.h>       // Report free EMS/XMS memory too.
#endif

#endif

#include <heapview.h>

//
// Heap Viewer functions
//
THeapView::THeapView(TRect &r) : TView(r)
{
    oldMem = 0L;
    update();
}

void THeapView::draw()
{
    TDrawBuffer b;
    char c = getColor(2);

    b.moveChar(0, ' ', c, size.x);
    b.moveStr(0, heapStr, c);
    writeLine(0, 0, size.x, 1, b);
}

void THeapView::update()
{
#if !defined(__DPMI16__) && !defined(__DPMI32__)
    // In real mode, it will operate as it always has done.
    long total = farcoreleft();
    struct farheapinfo heap;
#else
    #if !defined(__FLAT__)
        // In 16 bit protected mode, a call to farcoreleft() is all
        // that is required.
        long total = farcoreleft();
    #else
        // Even though you free() and/or delete memory in 32 bit protected
        // mode, a call to farcoreleft() will not reflect a change that
        // increases the amount of free memory, thus it always appears that
        // your application has memory leaks when it really doesn't.
        // The same is true of the code below, but it does reflect a more
        // accurate amount when the GlobalAlloc()-type functions are used.
        _MEMORYSTATUS MemStat;
        MemStat.dwLength = sizeof(_MEMORYSTATUS);   // *Must* set length!
        GlobalMemoryStatus(&MemStat);

        // Total free = Available physical memory + Available paging file space
        long total = MemStat.dwAvailPhys + MemStat.dwAvailPageFile;
    #endif
#endif

    switch(heapcheck())
    {
        case _HEAPEMPTY:
            strcpy(heapStr, "No heap!");
            total = -1;
            break;

        case _HEAPCORRUPT:
            strcpy(heapStr, "Heap corrupt!");
            total = -2;
            break;

        case _HEAPOK:
#if !defined(__DPMI16__) && !defined(__DPMI32__)
            // Count up the free blocks scattered about prior to the highest
            // allocated block in real mode.
            heap.ptr = NULL;
            while(farheapwalk(&heap) != _HEAPEND)
                if(!heap.in_use)
                    total += heap.size;

    #if defined(REPORT_EMS_XMS) && !defined(SHAREWARE)
            long freeEMS, freeXMS;

            TVirtualMemory::queryFree(&freeEMS, &freeXMS);

            // Report all values in K-bytes.
            sprintf(heapStr, "E:%5ld X:%5ld C:%3ld", freeEMS / 1024L,
                freeXMS / 1024L, total / 1024L);

            // Make oldMem change value if necessary.  The value isn't
            // important, just as long as it forces a redraw.
            total += freeEMS;
            total += freeXMS;
    #else
            sprintf(heapStr, "Free Memory: %8ld", total);
    #endif
#else
            sprintf(heapStr, "Free Memory: %8ld", total);
#endif
            break;
    }

    if(total != oldMem)
    {
        oldMem = total;
        drawView();
    }
}
