/*
   CPPTask - A Multitasking Kernel For C++

   Version 1.0 08-12-91

   Ported by Rich Smith from:

   Public Domain Software written by
      Thomas Wagner
      Patschkauer Weg 31
      D-1000 Berlin 33
      West Germany

   TSKMSG.CPP - Message handling routines.

   Subroutines:
        mailbox::mailbox
        mailbox::~mailbox
        mailbox::send_mail
        mailbox::wait_mail
        mailbox::c_wait_mail
        mailbox::check_mailbox

*/

#include <stdio.h>

#include "task.hpp"
#include "tsklocal.hpp"


/*
   mailbox - initialises mailbox.
*/

mailbox::mailbox ()
{
   waiting = NULL;
   mail_first = mail_last = NULL;
}


/*
   ~mailbox - kills all processes waiting for mail
*/

mailbox::~mailbox ()
{
   CRITICAL;

   C_ENTER;
   tsk_kill_queue (&waiting);
   mail_first = mail_last = NULL;
   C_LEAVE;

}


/*
   wait_mail - Wait until mail arrives. If there is mail in the box on
               entry, the first mail block is assigned to the caller,
               and the task continues to run.
*/

farptr far mailbox::wait_mail (dword timeout)
{
   msgptr msg;
   CRITICAL;

   C_ENTER;
   if ((msg = mail_first) != NULL)
      {
      if ((mail_first = msg->next) == NULL)
         mail_last = NULL;
      C_LEAVE;
      return msg;
      }

   tsk_wait (&waiting, timeout);
   return tsk_current->get_retptr();
}

/*
   c_wait_mail - If there is mail in the box on entry, the first mail 
                 block is assigned to the caller, else an error is returned.
*/

farptr far mailbox::c_wait_mail (void)
{
   msgptr msg;
   CRITICAL;

   C_ENTER;
   if ((msg = mail_first) != NULL)
      if ((mail_first = msg->next) == NULL)
         mail_last = NULL;
   C_LEAVE;
   return msg;
}


/*
   send_mail - Send a mail block to a mailbox. If there are tasks waiting
               for mail, the first waiting task is assigned the block and
               is made eligible.
*/

void far mailbox::send_mail (farptr msg)
{
   tcbptr curr;
   CRITICAL;

   C_ENTER;
   if ((curr = waiting) == NULL)
      {
      if (mail_first == NULL)
         mail_first = (msgptr)msg;
      else
         mail_last->next = (msgptr)msg;
      ((msgptr)msg)->next = NULL;
      mail_last = (msgptr)msg;
      C_LEAVE;
      return;
      }
   waiting = curr->tsk_runable ();
   curr->set_retptr(msg);
   C_LEAVE;
}


/*
   check_mailbox - returns TRUE if there is mail in the box.
*/

int far mailbox::check_mailbox (void)
{
   return mail_first != NULL;
}


