/*****************************************************************/
//	This program is an example of using combuf. In this example
//	I am not using the full stream potentials because of the
//	program specs.
//	Specs: San Jose State University has an on-campus digital phone
//	service that requires a seperate "box" that connects you to a
//	modem pool. To access, you perform the following (at 9600-7E1):
//	<in>  ENTER - DIRECTORY(D), NETWORK(N), OUTSIDE(O), OR TERMINAL TYPE
//		  CHANGE(T) >
//		* This is the main prompt *
//	<out> T<CR>
//	<in>  ENTER TERMINAL TYPE >
//	<out> 90<CR>
//		* This sets up a system where the user communicates to the pool
//		  at 9600 baud, but the pool communicates to the modem at 2400.
//	<in>  ENTER - ... CHANGE(T) >
//	<out> O<CR>
//		* Request an outside line through the modem pool *
//	<in>  ENTER AUTHORIZATION CODE >
//	<out> idStr + <CR>
//		* The authorization code allows the campus to charge you for the call.
//	<in>  ENTER DESTINATION NUMBER >
//	<out> phoneStr + <CR>
//		* This is CompuServe's phone number *
//	<in>  TERMINAL TYPE CHANGE IN EFFECT
//		  MODEM CALL INITIATED
//		  CONNECT etc
//
//	Now the program is communicating through the modem. Unfortunately, it
//	is doing so at EVEN parity, 7 bits, 1 stopbits. For CIM to transmit
//	files correctly, it must talk in 8N1. So the last step required is
//	to place the modem in command mode (send "+++" after 1 second of no
//	transmission), wait for the OK response, change to 8N1, send ATO\r
//	(modem adjusts to the sending format), and wait for the OK.
/********************************************************************/

#include "Comstr.h"
#include <Constream.h>
#include <Strstream.h>
#include "Stopwatch.h"
#include <String.h>

char pbuf[1024];				// Buffer for combuf.
StopWatch sw;
char *idStr = "xxxxxxxxx";		// My authorization code.
char *phoneStr = "9885366";		// Phone nr for CIS.

// The "box" (and most modems) requires an inter-character delay.
// sendSlow() provides between 55 and 110 mSec of delay between chars.

void sendSlow( ostream& buf, char *str )
	{
	while( *str )
		{
		sw.start();
		while( sw.elapsed() < 2 );
		buf.put( *str++ );
		}
	}

// Function to wait for the '>' character (indicates a prompt).

int waitForPrompt( ostream& win, istream& buf )
	{
	int ch;

	while( (ch = buf.get()) != EOF )
		{
		win << (char) ch;
		if( ch == '>' )
			return 1;
		}

	return 0;
	}

// Function to wait for a particular string.

int waitForString( ostream& win, istream& buf, char *str )
	{
	char *ptr = str;
	int ch;

	while( (ch = buf.get()) != EOF )
		{
		win << (char) ch;
		if( ch == *ptr )
			{
			ptr++;
			if( *ptr == 0 )
				return 1;
			}
		else
			ptr = str;
		}
	return 0;
	}

// Function to wait for a connect string. With the modems in the pool, that
// is "CONNECT RELIABLE" or "CONNECT 2400".

int waitForConnectString( ostream& win, istream& coms )
	{
	char buf[20];
	int i;
	int ch;

	i = 0;
	while( (ch = coms.get()) != EOF )
		{
		win << (char) (ch & 0x7F);
		buf[i++] = (ch & 0x7F);
		if( strnicmp(buf, "CONNECT 2400", i) == 0 )
			{
			if( i == 12 )
				return 1;
			continue;
			}
		else if( strnicmp(buf, "CONNECT RELIABLE", i) == 0 )
			{
			if( i == 16 )
				return 1;
			continue;
			}
		else
			i = 0;
		}
	return 0;
	}

// Function to put the modem in command mode, change to 8N1, and
// go back online.

int changeModem( ostream& win, iostream& coms )
	{
	combuf *b;

	sw.start();
	while( sw.elapsed() < 19 );
	sendSlow( coms, "+++" );
	if( waitForString(win, coms, "OK\r\n") == 0)
		return 0;
	b = (combuf *) coms.rdbuf();
	b->portServer()->setBits( 8 );
	b->portServer()->setParity( NONE );
	sendSlow( coms, "ATO\r" );
	return waitForString(win, coms, "CONNECT");
	}

int main( int argc, char **argv )
	{
	int result;
	int oldPIC = inportb(0x21);
	// I'm running this program from the DOS shell in CIM. I want the
	// PIC to return to the same value it had when I shelled, so I
	// save it in oldPIC.

	constream win;							// console stream.
	StdPort com1(0);						// Uses COM1.
	combuf mybuf( &com1, pbuf, 512, 512 );	// the combuf.
	iostream stream( &mybuf );				// iostream attached to combuf.

	// Clear the screen.
	win.window(1,1,80,25);
	win.clrscr();

	// Usage: TEMPCIS ON | OFF
	// ON performs the notes at the beginning of the file.
	// OFF disables the port (unasserts DTR).
	if( argc != 2 || ( (result = stricmp(argv[1], "OFF")) != 0 &&
			stricmp(argv[1],"ON") != 0 ) )
		{
		win << "ERROR! Improper commandline." << endl;
		win << "Usage: TEMPCIS ON | OFF" << endl;
		return 0;
		}

	// result = 0 if OFF argument.
	if( result == 0 )
		{
		mybuf.open();
		mybuf.close();
		win << "Port is turned off" << endl;
		return 0;
		}

	// set port to 9600-7E1.
	mybuf.portServer()->setFormat( CommFormat(9600L, 7, EVEN, 1) );

	// And timeout to 5 seconds.
	mybuf.setTimeout( 500L );

	// Open the buffer for communications.
	if( mybuf.open() == 0 )
		{
		win << "Failure opening port" << endl;
		return 0;
		}
	// Assert DTR.
	mybuf.portServer()->assertLine(DTR);

	// "Box" automatically responds to DTR with the main prompt.
	result = waitForPrompt( win, stream );

	if( result != 0 )
		{
		// Request new terminal setting.
		sendSlow( stream, "T\r" );

		// And wait for terminal type prompt.
		result = waitForPrompt( win, stream );
		}

	if( result != 0 )
		{
		// Set terminal type 90.
		sendSlow( stream, "90\r" );

		// And wait for main prompt.
		result = waitForPrompt( win, stream );
		}

	if( result != 0 )
		{
		// Request outside line.
		sendSlow( stream, "O\r" );

		// And wait for authorization code prompt.
		result = waitForPrompt( win, stream );
		}

	if( result != 0 )
		{
		// Send authorization code + CR.
		sendSlow( stream, idStr );
		sendSlow( stream, "\r" );

		// And wait for the phone number prompt.
		result = waitForPrompt( win, stream );
		}

	if( result != 0 )
		{
		// Send CIS phone number + CR.
		sendSlow( stream, phoneStr );
		sendSlow( stream, "\r" );

		// And wait for prompt ending in INITIATED.
		mybuf.setTimeout( 1000L );	// use timeout of 10 seconds.
		result = waitForString( win, stream, "INITIATED" );
		}

	if( result != 0 )
		{
		// use timeout of 15 seconds.
		mybuf.setTimeout( 1500L );

		// and get the connect string.
		result = waitForConnectString( win, stream );
		}

	if( result != 0 )
		{
		// use timeout of 5 seconds.
		mybuf.setTimeout( 500L );

		// And change the modem to 9600-8N1.
		win << endl << "(Ensuring modem is 96008N1)" << endl;
		result = changeModem( win, stream );
		}

	win << "\nResult = " << result << endl;

	if( result != 0 )
		// Leave DTR asserted when we close the port.
		com1.setDtrOnClose(1);

	// Close the port.
	mybuf.close();

	// Restore the original PIC value.
	outportb(0x21,oldPIC);
	return 0;
	}
