/*
        cl.h -- Container Lite 2.0
        (C) Copyright 1996  John Webster Small
        All rights reserved
*/

#ifndef cl_h
#define cl_h

/* LINTLIBRARY */

#ifndef limits_h
  #include <limits.h>
  #ifndef limits_h
    #define limits_h
  #endif
#endif

#ifndef iostream_h
  #include <iostream.h>
  #ifndef iostream_h
    #define iostream_h
  #endif
#endif

#ifndef iomanip_h
  #include <iomanip.h>
  #ifndef iomanip_h
    #define iomanip_h
  #endif
#endif

#ifndef string_h
  #include <string.h>
  #ifndef string_h
    #define string_h
  #endif
#endif

#ifndef except_h
  #include <except.h>
  #ifndef except_h
    #define except_h
  #endif
#endif

#ifndef assert_h
  #include <assert.h>
  #ifndef assert_h
    #define assert_h
  #endif
#endif

#ifndef typeinfo_h
  #include <typeinfo.h>
  #ifndef typeinfo_h
    #define typeinfo_h
  #endif
#endif

/*  stream delimiters/manipulators  */

inline ostream& endm(ostream& os)
{ return os << '\n'; }
inline istream& nextm(istream& is)
{ (void) is.get(); return is; }

/*  special functions for strings  */

extern char * CL_strdup(const char * D);
    // CL_strdup() uses new instead of malloc!
extern ostream&  CL_puts(ostream& os, const char * D);
extern istream&  CL_gets(istream& is, char *& D);

/*  cl compare pointer types  */

#define CLblank
#define CLcmP(ID,ITEM)  \
        int (*ID)(const ITEM * D1, const ITEM * D2)
#define CLkeyCmP(ID,ITEM) \
        int (*ID)(const void * K, const ITEM * D)
typedef CLcmP(voidCmP,void);
typedef CLkeyCmP(voidKeyCmP,void);
#define CLcmPcast(ID,ITEM)  \
        (CLcmP(CLblank,ITEM)) ID
#define CLkeyCmPcast(ID,ITEM) \
        (CLkeyCmP(CLblank,ITEM)) ID
#define CLKCC(ITEM) (CLkeyCmP(CLblank,ITEM))
#define CLKCCSM(ITEM,CMP) (CLKCC(ITEM) ITEM :: CMP)
#define CLcmP0(ITEM)  CLcmPcast(0,ITEM)
#define CLkeyCmP0(ITEM) CLkeyCmPcast(0,ITEM)
#define voidCmP0 ((voidCmP)0)
#define voidKeyCmP0 ((voidKeyCmP)0)
#define CmPfnc cmP() const

#define CL_HONOR_SORTED 1

/*  cl constants  */

#ifndef CL_MAXNODES
#define CL_MAXNODES     ((unsigned) \
                        (UINT_MAX/sizeof(void *)))
#endif

#ifndef CL_LIMIT
#define CL_LIMIT        20U
#endif

#ifndef CL_DELTA
#define CL_DELTA        10U
#endif

#define CL_NOTFOUND     CL_MAXNODES
#define CL_SORTED       0x0001U
#define CL_BIND_ONLY    0x0000U
#define CL_ASG          0x0002U
#define CL_READ         0x0004U
#define CL_READ_DEL     0x0008U
#define CL_NEW          0x0010U
#define CL_NEWED        0x0020U
#define CL_DEL          0x0040U
#define CL_SAVE         0x0080U
#define CL_ALL_FLAGS    0x00FFU
#define CL_ANDS         (CL_ASG|CL_NEW|CL_DEL|CL_SAVE)

class cl
{

  private:

        unsigned lowLimit, lowThreshold, first;
        void **  linkS;
        unsigned _limit,    _delta,   _nodes;
        unsigned _maxNodes, _flags;
        voidCmP  _cmP;

        void init(unsigned flags_);
        cl&     operator=(const cl& b)  // not allowed!
              { assign(b); return *this; }
        void * deleteLeak(unsigned n, void * D,
              void * (cl::*method)(unsigned, void *));
        void * deleteLeak(void * D,
              void * (cl::*method)(void *));
        void * cloneD(const void * D)
            { return ((D && _flags & CL_NEW)?
                 newD(D) : 0); }

  protected:

        unsigned _curNode;

        void    assign(const cl& b, int deep = 1);
        virtual voidCmP cmPD() const
                        { return 0; }
        virtual void * assignD
                        (void *, const void *) const
                        { return (void *)0; }
        virtual void * newD(const void *) const
                        { return (void *)0; }
        virtual void   deleteD(void * D) const throw();
        virtual int    attachD(void *) const
                        { return 1; }
        virtual void   detachD(void *) const throw();
        virtual int    putD(ostream&, void *) const
                        { return 0; }
        virtual int    printD(ostream&, void *) const
                        { return 0; }
        virtual void * getD(istream&) const
                        { return (void *)0; }
        ostream&       _storeOn(ostream& os);
        virtual ostream& v_storeOn(ostream& os)
                        { return _storeOn(os); }
        ostream&       _printOn(ostream& os);
        virtual ostream& v_printOn(ostream& os)
                        { return _printOn(os); }
        istream&       _readFrom(istream& is);
        virtual istream& v_readFrom(istream& is)
                        { return _readFrom(is); }

public:

        class NoCmP
           // No compare function set exception!
        {
          public:
           NoCmP() {}
           NoCmP(const NoCmP& c) {}
        };
        static NoCmP NoCmPException;

        cl     (unsigned flags_ = CL_BIND_ONLY)
                { init(flags_); }
        cl     (const cl& b) { assign(b); }
        int    load(const char * filename, int io_mode
                 = int(ios::in)|int(ios::binary));
        int    save(const char * filename, int io_mode
                 = int(ios::out)|int(ios::binary));
        ostream& storeOn(ostream& os)
                { return (flags(CL_SAVE)?
                    v_storeOn(os):os); }
        ostream& printOn(ostream& os)
                { return v_printOn(os); }
        istream& readFrom(istream& is)
                { return v_readFrom(is); }
        cl (const char * filename, int io_mode
                  = int(ios::in)|int(ios::binary))
                { (void) load(filename,io_mode); }
        cl (istream& is) { (void) readFrom(is); }
        int     cmp(const cl& b) const
                    throw(NoCmP);
        virtual ~cl() { allClr(); }

/*  Housekeeping Primitives  */

        unsigned limit() const { return _limit; }
        void     setLimit(unsigned newLimit);
        void     pack()  { setLimit(_nodes); }
        unsigned delta() const { return _delta; }
        void     setDelta(unsigned newDelta
                        = CL_DELTA);
        unsigned nodes() const { return _nodes; }
        unsigned maxNodes() const { return _maxNodes; }
        void     setMaxNodes(unsigned newMaxNodes
                        = CL_MAXNODES);
        unsigned vacancy() const
                        { return _maxNodes - _nodes; }
        unsigned vacancyNonElastic() const
                        { return _limit - _nodes; }
        unsigned flags(unsigned flags_ = CL_ALL_FLAGS)
                        const
                    { return (_flags & flags_); }
        void     setFlags(unsigned flags_)
                    { _flags |= flags_; }
        void     resetFlags(unsigned flags_)
                    { _flags &= ~flags_; }

/*  Elastic Array Primitives  */

        void *   atIns(unsigned n, void * D);
        void *   atInsDL(unsigned n, void * D)
                     { return deleteLeak
                         (n,D,&cl::atIns); }
        void *   atInsNew(unsigned n, const void * D)
                     { return atInsDL(n,cloneD(D)); }
        void *   atRmv(unsigned n);
        void     allRmv();
        int      atDel(unsigned n);
        void *   atDelAsg(unsigned n, void * D)
                     { return (atGetAsg(n,D)?
                       (atDel(n), D) : 0); }
        int      allDel();
        int      atClr(unsigned n)
                     { return (atDel(n)?
                       1 : (atRmv(n) != 0)); }
        void     allClr();
        void *   atPut(unsigned n, void * D);
        void *   atPutDL(unsigned n, void * D)
                     { return deleteLeak
                         (n,D,&cl::atPut); }
        void *   atPutNew(unsigned n, const void * D);
        void *   atPutAsg(unsigned n, const void * D);
        void *   atGet(unsigned n) const;
        void *   operator[](unsigned n) const
                        { return atGet(n); }
        void *   atGetAsg(unsigned n, void * D);
        void *   atXchg(unsigned n, void * D);
        unsigned indexOf(const void * D) const;

/*  Stack - Deque - Queue Primitives  */

        void *   push(void * D)
                        { return atIns(0U,D); }
        void *   pushDL(void * D)
                     { return atInsDL(0U,D); }
        void *   pushNew(const void * D)
                        { return atInsNew(0U,D); }
        void *   pop()  { return atRmv(0U); }
        int      popDel()  { return atDel(0U); }
        void *   popDelAsg(void * D)
                        { return atDelAsg(0U,D); }
        void *   top() const { return atGet(0U); }
        void *   topAsg(void * D)
                        { return atGetAsg(0U,D); }
        void *   insQ(void * D)
                        { return atIns(_nodes,D); }
        void *   insQDL(void * D)
                        { return atInsDL(_nodes,D); }
        void *   insQNew(const void * D)
                        { return atInsNew(_nodes,D); }
        void *   unQ()  { return atRmv(_nodes-1); }
        int      unQDel()  { return atDel(_nodes-1); }
        void *   unQDelAsg(void * D)
                        { return atDelAsg(_nodes-1,D);}
        void *   rear() const
                        { return atGet(_nodes-1); }
        void *   rearAsg(void * D)
                        { return atGetAsg(_nodes-1,D);}

/*  List (single and double linked) Primitives  */

        unsigned curNode() const
                 { return ((_curNode < _nodes)?
                   _curNode : CL_NOTFOUND); }
        void *   setCurNode(unsigned n = CL_NOTFOUND);
        void *   insBefore(void * D);
        void *   insBeforeDL(void * D)
                     { return deleteLeak
                         (D,&cl::insBefore); }
        void *   insBeforeNew(const void * D)
                     { return insBeforeDL(cloneD(D)); }
        void *   rmvStayPut();
        int      delStayPut();
        void *   delAsgStayPut(void * D)
                     { return (atGetAsg(_curNode,D)?
                       (delStayPut(), D) : 0); }
        int      clrStayPut();
        void *   insAfter(void * D);
        void *   insAfterDL(void * D)
                     { return deleteLeak
                         (D,&cl::insAfter); }
        void *   insAfterNew(const void * D)
                     { return insAfterDL(cloneD(D)); }
        void *   rmvBackup();
        int      delBackup();
        void *   delAsgBackup(void * D)
                     { return (atGetAsg(_curNode,D)?
                       (delBackup(), D) : 0); }
        int      clrBackup();
        void *   put(void * D)
                        { return atPut(_curNode,D); }
        void *   putDL(void * D)
                        { return atPutDL(_curNode,D); }
        void *   putNew(const void * D)
                        { return atPutNew(_curNode,D); }
        void *   putAsg(const void * D)
                        { return atPutAsg(_curNode,D); }
        void *   get() const { return atGet(_curNode); }
        void *   getAsg(void * D)
                        { return atGetAsg(_curNode,D); }
        void *   next();
        void *   operator++()  { return next(); }
        void *   operator++(int) { void * D = get();
                     (void) next(); return D; }
        void *   nextAsg(void * D);
        void *   prev();
        void *   operator--()  { return prev(); }
        void *   operator--(int) { void * D = get();
                     (void) prev(); return D; }
        void *   prevAsg(void * D);

/* Priority Q, Set, Bag, Dictionary, Sort Primitives */

        unsigned sorted() const
                    { return (_flags & CL_SORTED); }
        void     unSort()  { _flags &= ~CL_SORTED; }
        void     setCmP(voidCmP cmP_ = voidCmP0,
                     int resetSortFlag = 1);
        voidCmP  cmP() const { return _cmP; }
        int      sort(voidCmP cmP_ = voidCmP0);
        void *   insSort(void * D);
        void *   insSortDL(void * D)
                     { return deleteLeak
                         (D,&cl::insSort); }
        void *   insSortNew(const void * D)
                     { return insSortDL(cloneD(D)); }
        void *   insUnique(void * D);
        void *   insUniqueDL(void * D)
                     { return deleteLeak
                         (D,&cl::insUnique); }
        void *   insUniqueNew(const void * D);
        cl&      asSet();
        void *   findFirst(const void * K,
                        unsigned& idx,
                        voidKeyCmP kcmP = 0,
                        int honorSortFlag = 1) const;
        void *   findNext(const void * K,
                        unsigned& idx,
                        voidKeyCmP kcmP = 0,
                        int honorSortFlag = 1) const;
        void *   findFirst(const void * K,
                        voidKeyCmP kcmP = 0,
                        int honorSortFlag = 1)
                        { return findFirst(K,_curNode,
                          kcmP,honorSortFlag); }
        void *   findNext(const void * K,
                        voidKeyCmP kcmP = 0,
                        int honorSortFlag = 1)
                        { return findNext(K,_curNode,
                          kcmP,honorSortFlag); }

};      /*  class cl  */

inline ostream& operator<<(ostream& os, cl& b)
{ return b.storeOn(os); }

inline istream& operator>>(istream& is, cl& b)
{ return b.readFrom(is); }

template <class ITEM>
class CL : public cl
{

protected:

        virtual  void   deleteD(void * D)
                        const throw();

public:

/*  Constructors and destructor  */

        CL (unsigned flags = CL_ANDS)
           : cl(flags) {}
        CL (const CL<ITEM >& b) : cl(b) {}
        CL (const char * filename,
           int io_mode = int(ios::in)|int(ios::binary))
                : cl(filename,io_mode) {}
        CL (istream& is) : cl(is) {}
        CL<ITEM >& operator=
                (const CL<ITEM >& b)
                { assign(b); return *this; }
        int     cmp(const CL<ITEM >& b) const
                  throw(NoCmP);
        virtual ~CL() { allClr(); }

/*  Elastic Array Primitives  */

        ITEM *   atIns(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atIns(n,(void *)D); }
        ITEM *   atInsDL(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atInsDL(n,(void *)D); }
        ITEM *   atInsNew(unsigned n, const ITEM * D)
                        { return (ITEM *)cl::
                        atInsNew(n,(const void *)D); }
        ITEM *   atRmv(unsigned n)
                        { return (ITEM *)cl::
                        atRmv(n); }
        ITEM *   atDelAsg(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atDelAsg(n,(void *)D); }
        ITEM *   atPut(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atPut(n,(void *)D); }
        ITEM *   atPutDL(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atPutDL(n,(void *)D); }
        ITEM *   atPutNew(unsigned n, const ITEM * D)
                        { return (ITEM *)cl::
                        atPutNew(n,(const void *)D); }
        ITEM *   atPutAsg(unsigned n, const ITEM * D)
                        { return (ITEM *)cl::
                        atPutAsg(n,(const void *)D); }
        ITEM *   atGet(unsigned n) const
                        { return (ITEM *)cl::
                        atGet(n); }
        ITEM *   operator[](unsigned n) const
                        { return atGet(n); }
        ITEM *   atGetAsg(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atGetAsg(n,(void *) D); }
        ITEM *   atXchg(unsigned n, ITEM * D)
                        { return (ITEM *)cl::
                        atXchg(n,(void *) D); }
        unsigned indexOf(const ITEM * D) const
                        { return cl::
                        indexOf((const void *)D); }

/*  Stack - Deque - Queue Primitives  */

        ITEM *   push(ITEM * D)
                        { return (ITEM *)cl::
                        push((void *) D); }
        ITEM *   pushDL(ITEM * D)
                        { return (ITEM *)cl::
                        pushDL((void *) D); }
        ITEM *   pushNew(const ITEM * D)
                        { return (ITEM *)cl::
                        pushNew((const void *)D); }
        ITEM *   pop()
                        { return (ITEM *)cl::
                        pop(); }
        ITEM *   popDelAsg(ITEM * D)
                        { return (ITEM *)cl::
                        popDelAsg((void *)D); }
        ITEM *   top() const
                        { return (ITEM *)cl::
                                top(); }
        ITEM *   topAsg(ITEM * D)
                        { return (ITEM *)cl::
                        topAsg((void *)D); }
        ITEM *   insQ(ITEM * D)
                        { return (ITEM *)cl::
                        insQ((void *)D); }
        ITEM *   insQDL(ITEM * D)
                        { return (ITEM *)cl::
                        insQDL((void *)D); }
        ITEM *   insQNew(const ITEM * D)
                        { return (ITEM *)cl::
                        insQNew((const void *)D); }
        ITEM *   unQ()  { return (ITEM *)cl::unQ(); }
        ITEM *   unQDelAsg(ITEM * D)
                        { return (ITEM *)cl::
                        unQDelAsg((void *)D); }
        ITEM *   rear() const
                        { return (ITEM *)cl::
                                rear(); }
        ITEM *   rearAsg(ITEM * D)
                        { return (ITEM *)cl::
                        rearAsg((void *)D); }

/*  List (single and double linked) Primitives  */

        ITEM *   setCurNode(unsigned n = CL_NOTFOUND)
                        { return (ITEM *)cl::
                        setCurNode(n); }
        ITEM *   insBefore(ITEM * D)
                        { return (ITEM *)cl::
                        insBefore((void *)D); }
        ITEM *   insBeforeDL(ITEM * D)
                        { return (ITEM *)cl::
                        insBeforeDL((void *)D); }
        ITEM *   insBeforeNew(const ITEM * D)
                        { return (ITEM *)cl::
                        insBeforeNew((const void *)D); }
        ITEM *   rmvStayPut()
                        { return (ITEM *)cl::
                        rmvStayPut(); }
        ITEM *   delAsgStayPut(ITEM * D)
                        { return (ITEM *)cl::
                        delAsgStayPut((void *)D); }
        ITEM *   insAfter(ITEM * D)
                     { return (ITEM *)cl::
                       insAfter((void *)D); }
        ITEM *   insAfterDL(ITEM * D)
                        { return (ITEM *)cl::
                        insAfterDL((void *)D); }
        ITEM *   insAfterNew(const ITEM * D)
                     { return (ITEM *)cl::
                       insAfterNew((const void *)D); }
        ITEM *   rmvBackup()
                     { return (ITEM *)cl::
                       rmvBackup(); }
        ITEM *   delAsgBackup(ITEM * D)
                     { return (ITEM *)cl::
                       delAsgBackup((void *)D); }
        ITEM *   put(ITEM * D)
                        { return (ITEM *)cl::
                        put((void *)D); }
        ITEM *   putDL(ITEM * D)
                        { return (ITEM *)cl::
                        putDL((void *)D); }
        ITEM *   putNew(const ITEM * D)
                        { return (ITEM *)cl::
                        putNew((const void *)D); }
        ITEM *   putAsg(const ITEM * D)
                        { return (ITEM *)cl::
                        putAsg((const void *)D); }
        ITEM *   get() const
                        { return (ITEM *)cl::
                        get(); }
        ITEM *   getAsg(ITEM * D)
                        { return (ITEM *)cl::
                        getAsg((void *)D); }
        ITEM *   next()
                        { return (ITEM *)cl::
                        next(); }
        ITEM *   operator++()
                        { return next(); }
        ITEM *   operator++(int) { ITEM * D = get();
                     (void) next(); return D; }
        ITEM *   nextAsg(ITEM * D)
                        { return (ITEM *)cl::
                        nextAsg((void *)D); }
        ITEM *   prev()
                        { return (ITEM *)cl::
                        prev(); }
        ITEM *   operator--()
                        { return prev(); }
        ITEM *   operator--(int) { ITEM * D = get();
                     (void) prev(); return D; }
        ITEM *   prevAsg(ITEM * D)
                        { return (ITEM *)cl::
                        prevAsg((void *)D); }

/* Priority Q, Set, Bag, Dictionary, Sort Primitives */

        void     setCmP(CLcmP(cmP_,ITEM)
                     = CLcmP0(ITEM),
                     int resetSortFlag = 1)
                     { cl::setCmP
                      ((voidCmP)cmP_,resetSortFlag); }
        CLcmP(CmPfnc,ITEM) { return CLcmPcast
                        (cl::cmP(),ITEM); }
        int      sort(CLcmP(cmP_,ITEM)
                        = CLcmP0(ITEM))
                        { return cl::sort
                        ((voidCmP)cmP_); }
        ITEM *   insSort(ITEM * D)
                        { return (ITEM *)cl::
                        insSort((void *)D); }
        ITEM *   insSortDL(ITEM * D)
                        { return (ITEM *)cl::
                        insSortDL((void *)D); }
        ITEM *   insSortNew(const ITEM * D)
                        { return (ITEM *)cl::
                        insSortNew((const void *)D); }
        ITEM *   insUnique(ITEM * D)
                        { return (ITEM *)cl::
                        insUnique((void *)D); }
        ITEM *   insUniqueDL(ITEM * D)
                        { return (ITEM *)cl::
                        insUniqueDL((void *)D); }
        ITEM *   insUniqueNew(const ITEM * D)
                        { return (ITEM *)cl::
                        insUniqueNew((const void *)D);}
        CL<ITEM >& asSet()
                        { (void) cl::asSet();
                          return*this; }
        ITEM *   findFirst(const void * K,
                        unsigned& idx,
                        CLkeyCmP(kcmP,ITEM)
                          = CLkeyCmP0(ITEM),
                        int honorSortFlag = 1) const
                        { return (ITEM *) cl::
                        findFirst(K,idx,
                        voidKeyCmP(kcmP),
                        honorSortFlag); }
        ITEM *   findNext(const void * K,
                        unsigned& idx,
                        CLkeyCmP(kcmP,ITEM)
                          = CLkeyCmP0(ITEM),
                        int honorSortFlag = 1) const
                        { return (ITEM *) cl::
                        findNext(K,idx,
                        voidKeyCmP(kcmP),
                        honorSortFlag); }
        ITEM *   findFirst(const void * K,
                        CLkeyCmP(kcmP,ITEM)
                          = CLkeyCmP0(ITEM),
                        int honorSortFlag = 1)
                        { return findFirst(K,_curNode,
                          kcmP,honorSortFlag); }
        ITEM *   findNext(const void * K,
                        CLkeyCmP(kcmP,ITEM)
                          = CLkeyCmP0(ITEM),
                        int honorSortFlag = 1)
                        { return findNext(K,_curNode,
                          kcmP,honorSortFlag); }

};      /*  class CL  */

template <class ITEM>
int CL<ITEM >::cmp(const CL<ITEM >& b)
const throw(cl::NoCmP)
{ return cl::cmp(b); }

template <class ITEM>
void CL<ITEM >:: deleteD(void * D) const throw()
{ delete (ITEM *)D; }


template <class ITEM, class TRAITS>
class CL_ : public CL<ITEM >
{

private:

        static int CL_traits_cmp
            (const ITEM * I1, const ITEM * I2)
            { return TRAITS().cmp(*I1,*I2); }

protected:

        virtual  voidCmP cmPD() const
                        { return ((voidCmP)
                        CL_traits_cmp); }
        virtual  void * assignD(void * D,
                        const void * S) const
                        { return TRAITS().
                        assign(*(ITEM *)D,
                        *(const ITEM *)S,
                        flags()); }
        virtual  void * newD(const void * D) const
                        { return  TRAITS().
                        newCopy(*(const ITEM *)D); }
        virtual  void   deleteD(void * D)
                        const throw();
        virtual  int    attachD(void * D) const
                        { return TRAITS().
                        attach(*(ITEM *)D,this); }
        virtual  void   detachD(void * D)
                        const throw();
        virtual  int    putD(ostream& os, void * D)
                        const
                        { return TRAITS().
                        put(os,*(ITEM *)D); }
        virtual  int    printD(ostream& os, void * D)
                        const { return TRAITS().
                          print(os,*(ITEM *)D); }
        virtual  void * getD(istream& is) const
                        { return TRAITS().
                        get(is); }
        ostream&       _printOn(ostream& os)
                        { return CL_<ITEM,TRAITS >::
                          _printOn(os
                          << typeid(CL_<ITEM,TRAITS >)
                           .name()); }
        virtual ostream& v_printOn(ostream& os)
                        { return _printOn(os); }

public:

/*  Constructors and destructor  */

        CL_ (unsigned flags = CL_ANDS)
           : CL<ITEM >(flags) {}
        CL_ (const CL_<ITEM,TRAITS >& b)
             : CL<ITEM >(b) {}
        CL_ (const char * filename,
           int io_mode = int(ios::in)|int(ios::binary))
                : CL<ITEM >(filename,io_mode) {}
        CL_ (istream& is) : CL<ITEM >(is) {}
        CL_<ITEM,TRAITS >& operator=
                (const CL_<ITEM,TRAITS >& b)
                { assign(b); return *this; }
        int     cmp(const CL_<ITEM,TRAITS >& b) const
                   throw(NoCmP);
        virtual ~CL_() { allClr(); }

        CL_<ITEM,TRAITS >& asSet()
                        { (void) cl::asSet();
                          return*this; }
};      /*  class CL_  */

template <class ITEM, class TRAITS>
int CL_<ITEM,TRAITS >::cmp
(const CL_<ITEM,TRAITS >& b) const throw(cl::NoCmP)
{ return CL<ITEM >::cmp(b); }

template <class ITEM, class TRAITS>
void CL_<ITEM,TRAITS >::
deleteD(void * D) const throw()
{ TRAITS()._delete((ITEM *)D); }

template <class ITEM, class TRAITS>
void CL_<ITEM,TRAITS >::
detachD(void * D) const throw()
{ TRAITS().detach(*(ITEM *)D,this); }

/*  traits templates  */

#define CL_template(CL_NAME,TRAIT_NAME) \
template <class ITEM> \
class CL_NAME \
  : public CL_<ITEM,TRAIT_NAME<ITEM > > \
{ \
public: \
  CL_NAME (unsigned flags = CL_ANDS) \
    : CL_<ITEM,TRAIT_NAME<ITEM > >(flags) {} \
  CL_NAME (const CL_NAME<ITEM >& b) \
   : CL_<ITEM,TRAIT_NAME<ITEM > >(b) {} \
  CL_NAME (const char * filename, \
     int io_mode = int(ios::in)|int(ios::binary)) \
   : CL_<ITEM,TRAIT_NAME<ITEM > > \
     (filename,io_mode) {} \
  CL_NAME (istream& is) \
   : CL_<ITEM,TRAIT_NAME<ITEM > >(is) {} \
  CL_NAME<ITEM >& operator= \
   (const CL_NAME<ITEM >& b) \
     { assign(b); return *this; } \
  int     cmp(const CL_NAME<ITEM >& b) const \
         throw(NoCmP); \
  virtual ~CL_NAME() { allClr(); } \
  CL_NAME<ITEM >& asSet() \
    { (void) CL_<ITEM,TRAIT_NAME<ITEM > > \
       ::asSet(); return*this; } \
}; \
template <class ITEM> \
int CL_NAME<ITEM >::cmp \
(const CL_NAME<ITEM >& b) const \
throw(cl::NoCmP) \
{ return CL_<ITEM,TRAIT_NAME<ITEM > >::cmp(b); }

template <class ITEM>
struct CL_nativeTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    void _delete(ITEM * I) throw();
    int attach(ITEM& i, const cl * C)
        { return 1; }
    void detach(ITEM&, const cl *) throw();
    int cmp(const ITEM& i1, const ITEM& i2)
        { return ((i1 > i2)? 1 :
           ((i1 == i2)? 0 : -1)); }
    int put(ostream& os, ITEM& i)
        { return ((os << i << endm)? 1 : 0); }
    int print(ostream& os, ITEM& i)
        { return ((os << i)? 1 : 0); }
    void * get(istream& is)
        {
          ITEM * I = new ITEM;
          if (I) if (!(is >> *I >> nextm))
              { delete I; I = 0; }
          return I;
        }
};

template <class ITEM>
void CL_nativeTraits<ITEM >:: _delete(ITEM * I) throw()
{ delete I; }

template <class ITEM>
void CL_nativeTraits<ITEM >::
  detach(ITEM&, const cl *) throw()
{ return; }

CL_template(CL_native,CL_nativeTraits)

template <class ITEM>
struct CL_defaultTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    void _delete(ITEM * I) throw();
    int attach(ITEM&, const cl *)
        { return 1; }
    void detach(ITEM&, const cl *) throw();
    int cmp(const ITEM&, const ITEM&)
        { assert(0); return 0; }
    int put(ostream&, ITEM&)
        { return 0; }
    int print(ostream&, ITEM&)
        { return 0; }
    void * get(istream&)
        { return 0; }
};

template <class ITEM>
void CL_defaultTraits<ITEM >:: _delete(ITEM * I) throw()
{ delete I; }

template <class ITEM>
void CL_defaultTraits<ITEM >::
  detach(ITEM&, const cl *) throw()
{ return; }

CL_template(CL_default,CL_defaultTraits)

template <class ITEM>
struct CL_wellEndowedTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    void _delete(ITEM * I) throw();
    int attach(ITEM& i, const cl * C)
        { return 1; }
    void detach(ITEM&, const cl *) throw();
    int cmp(const ITEM& i1, const ITEM& i2)
        { return i1.cmp(i2); }
    int put(ostream& os, ITEM& i)
        { return ((os << i << endm)? 1 : 0); }
    int print(ostream&, ITEM&)
        { return 0; }
    void * get(istream& is)
        {
          ITEM * I = new ITEM;
          if (I) if (!(is >> *I >> nextm))
              { delete I; I = 0; }
          return I;
        }
};

template <class ITEM>
void CL_wellEndowedTraits<ITEM >:: _delete(ITEM * I) throw()
{ delete I; }

template <class ITEM>
void CL_wellEndowedTraits<ITEM >::
  detach(ITEM&, const cl *) throw()
{ return; }

CL_template(CL_wellEndowed,CL_wellEndowedTraits)

struct CL_charTraits
{
    void * assign(char&, const char&, unsigned)
        {  return 0; }
    void * newCopy(const char& i)
        { return CL_strdup(&i); }
    void _delete(char * I) throw();
    int attach(char&, const cl *)
        { return 1; }
    void detach(char&, const cl *) throw();
    int cmp(const char& i1, const char& i2)
        { return strcmp(&i1,&i2); }
    int put(ostream& os, char& i)
        { return (CL_puts(os,&i)? 1 : 0); }
    int print(ostream& os, char& i)
        { return ((os << &i)?1:0); }
    char * get(istream& is)
        { char * I = 0; (void) CL_gets(is,I);
          return I; }
};

typedef CL_<char,CL_charTraits> CL_char;

template <class ITEM>
struct CL_cmpTraits
{
  void * assign(ITEM&, const ITEM&, unsigned)
    { return 0; }
  void * newCopy(const ITEM &)
    { return 0; }
  void _delete(ITEM * I) throw();
  int attach(ITEM&, const cl *)
    { return 1; }
  void detach(ITEM&, const cl *) throw();
  int cmp(const ITEM& i1, const ITEM& i2)
    { return i1.cmp(i2); }
  int put(ostream&, ITEM&)
    { return 0; }
  int print(ostream&, ITEM&)
    { return 0; }
  void * get(istream&)
    { return 0; }
};

template <class ITEM>
void CL_cmpTraits<ITEM >::
  _delete(ITEM * I) throw()
    { delete I; }

template <class ITEM>
void CL_cmpTraits<ITEM >::
  detach(ITEM&, const cl *) throw()
    { return; }

CL_template(CL_cmp,CL_cmpTraits)

template <class ITEM>
struct CL_streamTraits
{
  void * assign(ITEM&, const ITEM&, unsigned)
    { return 0; }
  void * newCopy(const ITEM &)
    { return 0; }
  void _delete(ITEM * I) throw();
  int attach(ITEM&, const cl *)
    { return 1; }
  void detach(ITEM&, const cl *) throw();
  int cmp(const ITEM&, const ITEM&)
    { assert(0); return 0; }
  int put(ostream& os, ITEM& i)
      { return ((os << i << endm)? 1 : 0); }
  int print(ostream&, ITEM&)
      { return 0; }
  void * get(istream& is)
      {
        ITEM * I = new ITEM;
        if (I) if (!(is >> *I >> nextm))
           { delete I; I = 0; }
        return I;
      }
};

template <class ITEM>
void CL_streamTraits<ITEM >::
  _delete(ITEM * I) throw()
    { delete I; }

template <class ITEM>
void CL_streamTraits<ITEM >::
detach(ITEM&, const cl *) throw()
    { return; }

CL_template(CL_stream,CL_streamTraits)

template <class ITEM>
struct CL_cmpStreamTraits
{
  void * assign(ITEM&, const ITEM&, unsigned)
    { return 0; }
  void * newCopy(const ITEM &)
    { return 0; }
  void _delete(ITEM * I) throw();
  int attach(ITEM&, const cl *)
    { return 1; }
  void detach(ITEM&, const cl *) throw();
  int cmp(const ITEM& i1, const ITEM& i2)
    { return i1.cmp(i2); }
  int put(ostream& os, ITEM& i)
      { return ((os << i << endm)? 1 : 0); }
  int print(ostream&, ITEM&)
      { return 0; }
  void * get(istream& is)
      {
        ITEM * I = new ITEM;
        if (I) if (!(is >> *I >> nextm))
           { delete I; I = 0; }
        return I;
      }
};

template <class ITEM>
void CL_cmpStreamTraits<ITEM >::
  _delete(ITEM * I) throw()
    { delete I; }

template <class ITEM>
void CL_cmpStreamTraits<ITEM >::
detach(ITEM&, const cl *) throw()
    { return; }

CL_template(CL_cmpStream,CL_cmpStreamTraits)


template <class ITEM>
struct CL_allTraits
{
    void * assign(ITEM& i1, const ITEM& i2,
         unsigned flags)
        { i1 = i2; return &i1; }
    void * newCopy(const ITEM& i)
        { return new ITEM(i); }
    void _delete(ITEM * I) throw();
    int attach(ITEM& i, const cl * C)
        { return 1; }
    void detach(ITEM&, const cl *) throw();
    int cmp(const ITEM& i1, const ITEM& i2)
        { return i1.cmp(i2); }
    int put(ostream& os, ITEM& i)
        { return ((os << i << endm)? 1 : 0); }
    int print(ostream&, ITEM& i)
        { return (i.printOn(os)? 1 : 0); }
    void * get(istream& is)
        {
          ITEM * I = new ITEM;
          if (I) if (!(is >> *I >> nextm))
              { delete I; I = 0; }
          return I;
        }
};

template <class ITEM>
void CL_allTraits<ITEM >:: _delete(ITEM * I) throw()
{ delete I; }

template <class ITEM>
void CL_allTraits<ITEM >::
  detach(ITEM&, const cl *) throw()
{ return; }

CL_template(CL_all,CL_allTraits)

/* iterators */

#define CL_forEach(b) for ((void)(b).setCurNode(0); \
    (b).get(); (void)(b).next())

// Use only via CL_do() macro!
// Requires ITEM to be selectable.
template <class ITEM>
class CL_do_iter
{
    CL<ITEM >& _b;
    unsigned _idx;
  public:
    CL_do_iter(CL<ITEM >& b) : _b(b), _idx(0) {}
    unsigned indexOf() { return _idx; }
    void next() { ++_idx; }
    operator ITEM * () { return b[_idx]; }
    ITEM& operator*() { return *b[_idx]; }
    ITEM * operator->() { return b[_idx]; }
};
#define CL_do(b,ITEM,D) \
    for(CL_do_iter<ITEM> (D)(b); (D); (D).next())

// Requires:  int ITEM::cmp(const KEY&) const;
template <class ITEM, class KEY>
class CL_find
{
    CL<ITEM >& _b;
    static int cmpKey2Item(const KEY * K, const ITEM * D)
        { return - D->cmp(*K); }
  public:
    CL_find(CL<ITEM >& b) : _b(b) {}
    operator ITEM * () { return _b.get(); }
    ITEM * operator->() { return _b.get(); }
    ITEM * first(const KEY * K, int honorSortFlag = 1)
      { return _b.findFirst
      ((const void *)K,CLKCC(ITEM)cmpKey2Item,
      honorSortFlag); }
    CL_find(CL<ITEM >& b, const KEY * K,
       int honorSortFlag = 1) : _b(b)
       { (void) first(K,honorSortFlag); }
    ITEM * operator[](const KEY * K)
      { return first(K); }
    ITEM * next(const KEY * K, int honorSortFlag = 1)
      { return _b.findNext
      ((const void *)K,CLKCC(ITEM)cmpKey2Item,
      honorSortFlag); }
};

#endif  /*  cl_h  */
