/*
        cl.cpp -- Container Lite 2.0
        (C) Copyright 1996  John Webster Small
        All rights reserved
*/

#include "cl.h"
#include <fstream.h>
#ifndef assert_h
  #include <assert.h>
#endif

char * CL_strdup(const char * D)
{
    if (D)  {
        size_t len = strlen(D);
        char * newD = new char[len+1];
        if (newD)
            return strcpy(newD,D);
    }
    return 0;
}

ostream& CL_puts(ostream& os, const char * D)
{
    size_t len = (D? strlen(D) : 0);
    if ((os << len << endm))
        if (len)
            if (len > INT_MAX)
            {
                (void) os.write(D,INT_MAX);
                (void) os.write(&D[INT_MAX],
                    (int)(len-INT_MAX));
            }
            else
                (void) os.write(D,(int)len);
    return os;
}

istream& CL_gets(istream& is, char *& D)
{
    if (D)  {
        delete[] D;
        D = 0;
    }
    size_t len;
    if ((is >> len >> nextm))
        if (len)  {
          D = new char[len+1];
          if (D)  {
            size_t count = 0;
            try  {
              while (is && len)
                if (len > INT_MAX)  {
                  (void) is.read(&D[count],INT_MAX);
                  count += INT_MAX;
                  len -= INT_MAX;
                }
                else  {
                  (void) is.read(&D[count],len);
                  count += len;
                  len = 0;
                }
            } catch (...)  {
              delete[] D;
              D = 0;
              throw;
            }
            if (is)
              D[len] = '\0';
            else  {
              delete[] D;
              D = 0;
            }
          }
        }
    return is;
}

void cl:: init(unsigned flags_)
{
    _curNode = first = _nodes = 0U;
    _cmP = voidCmP0;
/*
        The following relationships are maintained
        during operation of a cl:
        1 <= delta <= lowLimit <= limit <= maxNodes
                <= CL_MAXNODES
        lowThreshold = lowLimit - delta;
*/
    assert(unsigned(CL_DELTA));
    assert(unsigned(CL_DELTA) <= unsigned(CL_LIMIT));
    assert(unsigned(CL_LIMIT) <= unsigned(CL_MAXNODES));
    linkS = (void **)0;
    _limit = unsigned(CL_LIMIT);
    _delta = unsigned(CL_DELTA);
    _maxNodes = unsigned(CL_MAXNODES);
    lowLimit = unsigned(CL_LIMIT);
    lowThreshold =
        unsigned(CL_LIMIT) - unsigned(CL_DELTA);
    _flags = (flags_ | CL_SORTED);
}

void * cl:: deleteLeak(unsigned n, void * D,
    void * (cl::*method)(unsigned, void *))
{
    if (D)  {
        void * ok;
        try { ok = (this->*method)(n,D); }
        catch (...) { if (D) deleteD(D); throw; }
        if (!ok) deleteD(D);
        return ok;
    }
    return 0;
}

void * cl:: deleteLeak(void * D,
    void * (cl::*method)(void *))
{
    if (D)  {
        void * ok;
        try { ok = (this->*method)(D); }
        catch (...) { if (D) deleteD(D); throw; }
        if (!ok) deleteD(D);
        return ok;
    }
    return 0;
}

void cl:: assign(const cl& b, int deep)
{
        allClr();
        setDelta(1);
        setLimit(1);
        setMaxNodes(1);
        setMaxNodes(b.maxNodes());
        setLimit(b.limit());
        setDelta(b.delta());
        resetFlags(CL_ALL_FLAGS);
        setFlags(b.flags());
        unsigned i;
        if (deep)  {
            setFlags(CL_NEW | CL_DEL);
            for (i = 0U; i < b.nodes(); i++)
                if (!insQNew(b.atGet(i)))
                        break;
        }
        else  {
            resetFlags(CL_DEL);
            for (i = 0U; i < b.nodes(); i++)
                if (!insQ(b.atGet(i)))
                        break;
        }
        setCmP(b.cmP());
        setFlags(b.flags());  // CL_SORTED?
        (void) setCurNode(b.curNode());
}

#pragma argsused
void cl:: deleteD(void * D) const throw()
    { assert(0); }

void cl:: detachD(void *) const throw() {}

ostream& cl:: _storeOn(ostream& os)
{
        if (!(os << _maxNodes << endm << _limit << endm
                << _delta << endm << _nodes << endm
                << _curNode << endm << _flags << endm))
                return os;
        for (unsigned i = 0U; i < _nodes; i++)
                if (!putD(os,atGet(i)))  {
                        os.clear(ios::failbit);
                        break;
                }
        return os;
}

ostream& cl:: _printOn(ostream& os)
{
        os << '(';
        unsigned i = 1U;
        if (_nodes)
            if (printD(os,atGet(0)))
                do {
                  os << ", ";
                  if (!printD(os,atGet(i)))
                      break;
                } while (++i < _nodes);
        if (i >= _nodes)
            os << ')';
        else
            os.clear(ios::failbit);
        return os;
}

istream& cl:: _readFrom(istream& is)
{
        unsigned maxNodes_, limit_, delta_, nodes_;
        unsigned curNode_, flags_;
        if (!(is >> maxNodes_ >> nextm >> limit_ >> nextm
                >> delta_ >> nextm >> nodes_ >> nextm
                >> curNode_ >> nextm >> flags_ >> nextm))
                return is;
        allClr();
        flags_ |= CL_DEL;
        //reloaded nodes are dynamic!
        setFlags(flags_);
        // In case getD() or insQ() throws exception
        // below; curNode setting is lost then!
        setMaxNodes(maxNodes_);
        setLimit(limit_);
        setDelta(delta_);
        void * cD, * ok;
        for (unsigned i = 0U; i < nodes_; i++)  {
                cD = getD(is);
                if (!cD) break;
                try  {
                  ok = insQ(cD);
                } catch (...) { deleteD(cD); throw; }
                if (!ok)  {
                        deleteD(cD);
                        break;
                }
        }
        setFlags(flags_);
          // CL_SORTED may be raised!
        (void) setCurNode(curNode_);
        if (nodes_ != _nodes)
            is.clear(ios::failbit);
        return is;
}

int cl:: load(const char * filename, int io_mode)
{
    int ok = 0;
    if (filename)  {
        ifstream is(filename,io_mode);
        if (is)  {
            ok = (readFrom(is)?1:0);
            is.close();
        }
    }
    return ok;
}

int cl:: save(const char * filename, int io_mode)
{
    int ok = 0;
    if (flags(CL_SAVE) && filename)  {
        ofstream os(filename,io_mode);
        if (os)  {
            ok = (storeOn(os)?1:0);
            os.close();
        }
    }
    return ok;
}

cl::NoCmP cl:: NoCmPException;

int cl:: cmp(const cl& b) const throw(cl::NoCmP)
{
    voidCmP cmP_ = (_cmP? _cmP : cmPD());
    if (!cmP_)
        throw NoCmPException;
    int signum;
    for (unsigned i = 0;
        i < nodes() && i < b.nodes();
        i++)
    {
        signum = (*cmP_)(atGet(i),b.atGet(i));
        if (signum)
           return signum;
    }
    if (nodes() > b.nodes())
        return 1;
    if (nodes() < b.nodes())
        return -1;
    return 0;
}

void cl:: setLimit(unsigned newLimit)
{
        void ** newLinkS;
        unsigned headNodes;
        if (newLimit < _nodes)
                newLimit = _nodes;
        else if (newLimit > _maxNodes)
                newLimit = _maxNodes;
        if (newLimit < _delta)
                newLimit = _delta;
        if (!newLimit || newLimit == _limit)
                return;
        if (!linkS)  {
                _limit = newLimit;
                return;
        }
        newLinkS = new void *[newLimit];
        if (!newLinkS)
                return;
        if ((headNodes = _limit - first) > _nodes)
                headNodes = _nodes;
        if (headNodes)  // move leading nodes
                memcpy(newLinkS,&linkS[first],
                        sizeof(linkS[0U])*headNodes);
        /* copy wrap around trailing nodes */
        if (headNodes < _nodes)
                memcpy(&newLinkS[headNodes],linkS,
                        sizeof(linkS[0U])*
                        (_nodes-headNodes));
        if (newLimit > _limit)
                if (newLimit - _delta > _limit)
                        lowLimit = newLimit - _delta;
                else
                        lowLimit = _limit;
        else
                if (newLimit - _delta > _delta)
                        lowLimit = newLimit - _delta;
                else
                        lowLimit = _delta;
        lowThreshold = lowLimit - _delta;
        delete[] linkS;
        linkS = newLinkS;
        _limit = newLimit;
        first = 0U;
        return;
}

void cl:: setDelta(unsigned newDelta)
{
        if (newDelta && (newDelta <= lowLimit))
            _delta = newDelta;
}

void cl:: setMaxNodes(unsigned newMaxNodes)
{
        if (newMaxNodes < _limit)
            if (newMaxNodes > _nodes)
                setLimit(newMaxNodes);
        if (newMaxNodes >= _limit)
            if (newMaxNodes > CL_MAXNODES)
                _maxNodes = CL_MAXNODES;
            else
                _maxNodes = newMaxNodes;
}

void * cl:: atIns(unsigned n, void * D)
{
    void ** newLinkS;
    unsigned newLimit, i;
    int ok;
    if (!D)
        return (void *)0;
    if (_nodes == _limit)  { // no vacancy - expand
        if (_limit == _maxNodes)
            return (void *)0;
        newLimit = ((_maxNodes - _delta > _limit)?
            (_limit + _delta) : _maxNodes);
        newLinkS = new void *[newLimit];
        if (!newLinkS)  // expansion unavailable
            return (void *)0;
        try {
          ok = attachD(D);
        } catch (...) { delete[] newLinkS; throw; }
        if (!ok)  {// data refuses binding
            delete[] newLinkS;
            return (void *)0;
        }
        newLinkS[((n <= _nodes)? n : (n = _nodes))] = D;
        // calc headNodes
        if (n <= (i = _limit - first))  {
            // insert in head section
            if (n)  {// postfix or split head
                memcpy(newLinkS,&linkS[first],
                    sizeof(linkS[0U])*n);
                if (n < i) // it's a split
                    memcpy(&newLinkS[n+1],
                        &linkS[first+n],
                        sizeof(linkS[0U])*(i-n));
            }
            else    // prefix head
                memcpy(&newLinkS[1],&linkS[first],
                    sizeof(linkS[0U])*i);
            if (i < _nodes)  // copy tail
                memcpy(&newLinkS[1+i],linkS,
                    sizeof(linkS[0U])
                    *(_nodes-i));
        }
        else  {    // insert in tail section
            // copy head first
            memcpy(newLinkS,&linkS[first],
                sizeof(linkS[0U])*i);
            if (i < _nodes)  { // copy tail
                memcpy(&newLinkS[i],linkS,
                    sizeof(linkS[0U])*(n-i));
                if (n < _nodes) // it's a tail split
                    memcpy(&newLinkS[n+1],
                        &linkS[n-i],
                        sizeof(linkS[0U])
                        *(_nodes-n));
            }
        }
        // Compute next smaller linkS size
        // and threshold for shrinking.
        lowLimit = _limit;
        lowThreshold = lowLimit - _delta;
        // swap new for old
        delete[] linkS;
        linkS = newLinkS;
        _limit = newLimit;
        first = 0U;
    }
    else  {  // vacancy
        if (!linkS)
            linkS = new void *[_limit];
        if (!linkS)
            return (void *)0;
        try  {
          if (!attachD(D))  {
              if (!_nodes)  {
                  delete[] linkS;
                  linkS = (void **)0;
              }
              return (void *)0;
          }
        } catch(...) {
            if (!_nodes)  {
                delete[] linkS;
                linkS = (void **)0;
            }
            throw;
        }
        if (!n)  // push
            linkS[first? --first
                : (first = _limit - 1)] = D;
        else if (n >= _nodes) // insQ
            linkS[(first+(n=_nodes))%_limit] = D;
        else  { // insert interior
            i = (first + n) % _limit;
            if (i < first || !first)
                // move rear rightward
                memmove(&linkS[i+1],&linkS[i],
                    sizeof(linkS[0U])
                    * (_nodes-n));
            else  { // move front leftward
                memmove(&linkS[first-1],
                    &linkS[first],
                    sizeof(linkS[0U])
                    *(n));
                first--;
                i--;
            }
            linkS[i] = D;
        }
    }
    _nodes++;
    if (n <= _curNode)
        _curNode++;
    _flags &= ~CL_SORTED;
    return D;
}

void * cl:: atRmv(unsigned n)
{
    void ** newLinkS = (void **)0;
    unsigned newLimit, i;
    void * D;
    if (!linkS || n >= _nodes)
        return (void *)0;
    D = linkS[i=((first+n)%_limit)];
    if (_nodes <= lowThreshold) {
        // get ready to contract
        newLimit = lowLimit;
        newLinkS = new void *[newLimit];
        // Compute next smaller linkS
        // size and threshold for
        // shrinking.
        if (newLinkS)  {
            if (_nodes < _delta)
                lowThreshold = 0;
            else
                lowThreshold -= _delta;
            lowLimit = lowThreshold + _delta;
        }
    }
    if (newLinkS)  {  // remove and contract
        // calc head nodes
        if ((i = _limit - first) > _nodes)
                i = _nodes;
        if (n < i)  {
            // rmv from head section
            if (n)  { // rmv from head
                memcpy(newLinkS,&linkS[first],
                    sizeof(linkS[0U])*n);
                if (n < i - 1)
                        memcpy(&newLinkS[n],
                            &linkS[first+n+1],
                            sizeof(linkS[0U])*(i-n-1));
            }
            else if (i > 1)   // pop head
                memcpy(newLinkS,&linkS[first+1],
                    sizeof(linkS[0U])*(i-1));
            if (i < _nodes)  // copy tail
                memcpy(&newLinkS[i-1],linkS,
                    sizeof(linkS[0U])
                    *(_nodes-i));
        }
        else  { // i <= n < nodes
            // rmv from tail section
            // copy head first
            memcpy(newLinkS,&linkS[first],
                sizeof(linkS[0U])*i);
            if (i == n)  { // pop tail
                if (n < _nodes - 1)
                    // copy trailing tail
                    memcpy(&newLinkS[i],&linkS[1],
                        sizeof(linkS[0U])*(_nodes-n-1));
            }
            else  {  // copy up to n in tail
                memcpy(&newLinkS[i],linkS,
                    sizeof(linkS[0U])*(n-i));
                // skip n'th node
                if (n < _nodes - 1)
                    // copy anything trailing n'th node
                    memcpy(&newLinkS[n],&linkS[n-i+1],
                        sizeof(linkS[0U])*(_nodes-n-1));
            }
        }
        // swap new for old
        delete[] linkS;
        linkS = newLinkS;
        _limit = newLimit;
        first = 0U;
    }
    else  { // simply remove
        if (!n)  {  // pop
            if (++first >= _limit)
                first = 0U;
        }
        else if (n != _nodes-1)  // del interior
            if (i < first) // move tail leftward
                memmove(&linkS[i],&linkS[i+1],
                    sizeof(linkS[0U])
                    *(_nodes-n-1));
            else {  // move head rightward
                memmove(&linkS[first+1],
                    &linkS[first],
                    sizeof(linkS[0U])*n);
                first++;
            }
    }
    if (n < _curNode)
        _curNode--;
    else if (n == _curNode)
        _curNode = _nodes-1;
    if (--_nodes == 0U)  {
        _flags |= CL_SORTED;
        _flags &= ~CL_NEWED;
        delete[] linkS;
        linkS = (void **)0;
    }
    detachD(D);
    return D;
}

void cl:: allRmv()
{
    if (linkS)  {
        unsigned i, j;
        for (i = first, j = _nodes;
            i < _limit && j; i++, j--)
            detachD(linkS[i]);
        for (i = 0; j; i++, j--)
            detachD(linkS[i]);
        delete[] linkS;
    }
    init(_flags);
    _maxNodes = _maxNodes;
    _limit = _delta;
}

int  cl:: atDel(unsigned n)
{
        void * D;
        if (_flags & CL_DEL)
                if ((D = atRmv(n)) != (void *)0)  {
                        deleteD(D);
                        return 1;
                }
        return 0;
}

int cl:: allDel()
{
    if (_flags & CL_DEL)  {
        if (linkS)  {
            unsigned i = first, j = _nodes;
            for (i = first, j = _nodes;
                i < _limit && j; i++, j--)  {
                detachD(linkS[i]);
                deleteD(linkS[i]);
            }
            for (i = 0; j; i++, j--)  {
                detachD(linkS[i]);
                deleteD(linkS[i]);
            }
            delete[] linkS;
        }
        init(_flags);
        _maxNodes = _maxNodes;
        _limit = _delta;
        return 1;
    }
    return 0;
}

void cl:: allClr()
{
        if (_flags & CL_DEL)
                (void) allDel();
        else
                allRmv();
}

void * cl:: atPut(unsigned n, void * D)
{
        if (linkS && D && (n < _nodes))  {
                void *& N = linkS[(first+n)%_limit];
                if (D != N) if (attachD(D))  {
                    _flags &= ~CL_SORTED;
                    detachD(N);
                    if (_flags & CL_DEL)
                        deleteD(N);
                    return (N=D);
                }
        }
        return (void *)0;
}

void * cl:: atPutNew(unsigned n, const void * D)
{
        void * oldD, * cD;
        if (!D || !(_flags & CL_NEW))
                return (void *)0;
        if ((oldD = atGet(n)) == (void *)0)
                return (void *)0;
        if (oldD == D)
                return (void *)0;
        cD = newD(D);
        if (cD)  {
                void * ok;
                try {
                  ok = atPut(n,cD);
                } catch(...) { deleteD(cD); throw; }
                if (ok)  {
                        _flags |= CL_NEWED;
                        return cD;
                }
                else
                        deleteD(cD);
        }
        return (void *)0;
}

void * cl:: atPutAsg(unsigned n, const void * D)
{
        void * oldD;
        if (D && _flags & CL_ASG)
                if ((oldD = atGet(n)) != (void *)0)
                        if (D == oldD)
                                return oldD;
                        else  {
                                _flags &= ~(CL_READ
                                  | CL_READ_DEL);
                                if (assignD(oldD,D))
                                  return oldD;
                        }
        return (void *)0;
}

void * cl:: atGet(unsigned n) const
{
        return ((linkS && (n < _nodes))?
                linkS[(first+n)%_limit] : (void *)0);
}

void * cl:: atGetAsg(unsigned n, void * D)
{
        void * N;
        if (D && _flags & CL_ASG)
                if ((N = atGet(n)) != (void *)0)
                        if (D == N)
                                return D;
                        else  {
                          _flags &= ~CL_READ_DEL;
                          _flags |=  CL_READ;
                          if (assignD(D,N)) return D;
                        }
        return (void *)0;
}

void * cl:: atXchg(unsigned n, void * D)
{
        if (linkS && D && (n < _nodes))
                if (attachD(D))  {
                        _flags &= ~CL_SORTED;
                        void *& N = linkS[(first+n)
                                %_limit];
                        void * oldD = N;
                        N = D;
                        detachD(oldD);
                        return oldD;
                }
        return (void *)0;
}

unsigned cl:: indexOf(const void * D) const
{
        unsigned i;
        if (linkS && D)
                for (i = 0U; i < _nodes; i++)
                        if (D == linkS[(first+i)
                                %_limit])
                                return i;
        return CL_NOTFOUND;
}

void *  cl:: setCurNode(unsigned n)
{
    if (n > _nodes)
        _curNode = _nodes;
    else
        _curNode = n;
    if ((_curNode < _nodes) && linkS)
        return linkS[(first+_curNode)%_limit];
    return (void *) 0;
}

void * cl:: insBefore(void * D)
{ 
    unsigned n = _curNode;
    if (n > _nodes)
        n = _nodes;
    D =  atIns(n,D);
    _curNode = n;
    return D;
}

void * cl:: rmvStayPut()
{ 
    unsigned n = _curNode;
    void * D =  atRmv(n);
    _curNode = n;
    return D;
}

int cl:: delStayPut()
{
    unsigned n = _curNode;
    int ok = atDel(n);
    _curNode = n;
    return ok;
}

int cl:: clrStayPut()
{
    if (_flags & CL_DEL)
        return delStayPut();
    return (rmvStayPut() != 0);
}

void * cl:: insAfter(void * D)
{
        if (atIns(_curNode+1,D))  {
                if (++_curNode >= _nodes)
                        _curNode = _nodes - 1;
                return D;
        }
        return (void *)0;
}

void * cl:: rmvBackup()
{
        void * D;
        unsigned n;
        if ((D = atRmv(n=_curNode)) != (void *)0)
                if (n)
                        _curNode = n - 1;
                else
                        _curNode = _nodes;
        return D;
}

int  cl:: delBackup()
{
        void * D;
        if (_flags & CL_DEL)
                if ((D = rmvBackup()) != (void *)0)  {
                        deleteD(D);
                        return 1;
                }
        return 0;
}

int cl:: clrBackup()
{
    if (_flags & CL_DEL)
        return delBackup();
    return (rmvBackup() != 0);
}

void * cl:: next()
{
        if (linkS)  {
                if (_curNode >= _nodes)
                        _curNode = 0U;
                else
                        _curNode++;
                if (_curNode < _nodes)
                        return linkS[(first+_curNode)
                                %_limit];
        }
        return (void *)0;
}

void * cl:: nextAsg(void * D)
{
    if (_curNode >= _nodes)
        _curNode = 0;  // wrap
    else
        _curNode++;
    if (atGetAsg(_curNode,D))
        return D;
    // if _curNode is a valid node, backup
    if (!_curNode)
        _curNode = _nodes;
    else if (_curNode < _nodes)
        _curNode--;
    return 0;
}

void * cl:: prev()
{
        if (linkS)  {
                if (_curNode)  {
                        if (_curNode > _nodes)
                                _curNode = _nodes;
                        _curNode--;
                }
                else
                        _curNode = _nodes;
                if (_curNode < _nodes)
                        return linkS[(first+_curNode)
                                %_limit];
        }
        return (void *)0;
}

void *  cl:: prevAsg(void * D)
{
    if (_curNode)
    {
        _curNode--;
        if (atGetAsg(_curNode,D))
            return D;
        _curNode++;
    }
    else
        _curNode = _nodes;
    return 0;
}

void cl:: setCmP(voidCmP cmP_, int resetSortFlag)
{
    _cmP = (cmP_? cmP_ : cmPD());
    if (resetSortFlag && _nodes)
        _flags &= ~CL_SORTED;
}

int   cl:: sort(voidCmP cmP_)
{
        unsigned low, mid, high;
        unsigned d;
        void * D;
/*
        The current node is always reset to undefined
        regardless of the outcome of sort.
*/
        _curNode = _nodes;
        if (!_cmP)
                _cmP = cmPD();
        if (_flags & CL_SORTED)  {
                if (_cmP == cmP_ || !cmP_)
                        return 1;
        }
        if (cmP_)  {
                _cmP = cmP_;
                _flags &= ~CL_SORTED;
        }
        else if (!_cmP)
                return 0;
        if (!_nodes || !linkS)
                return (int) (_flags |= CL_SORTED);
        if (first)  {
                /* form contiguous block at front */
                d = (first + _nodes) % _limit;
                if (d > first)
                        memmove(linkS,&linkS[first],
                            sizeof(linkS[0U])*_nodes);
                else if (d < first)
                        memmove(&linkS[d],
                                &linkS[first],
                                sizeof(linkS[0U])
                                *(_limit-first));
                /* else array is full/contiguous */
                first = 0U;
        }
        for (high = d = 1; d < _nodes; high = ++d)  {
                low = 0U;
                D = linkS[d];
                while (low < high)  {
                        mid = ((high + low) >> 1);
                        if ((*_cmP)(D,
                                linkS[mid]) <= 0)
                                high = mid;
                        else
                                low = mid + 1;
                }
                if (high < d)  {
                        memmove(&linkS[high+1],
                                &linkS[high],
                                sizeof(linkS[0U])
                                *(d-high));
                        linkS[high] = D;
                }
        }
        return (int) (_flags |= CL_SORTED);
}

void * cl:: insSort(void * D)
{
        unsigned low, high;
        void * ok;
/*
        The current node is left undefined if
        anything fails, otherwise it is set to the
        newly inserted node.  If the compare function
        throws an exception, the current node is set
        to the node choked on.
*/
        _curNode = _nodes;
        if (!_cmP) if ((_cmP = cmPD())
                == voidCmP0) return (void *)0;
        if (!D || _nodes >= _maxNodes)
                return (void *)0;
        if (!(_flags & CL_SORTED))
                if (!sort())
                        return (void *)0;
        low = 0U;
        high = _nodes;  // linkS == 0 -> nodes == 0!
        while (low < high)  {
                _curNode = ((high + low) >> 1);
                if ((*_cmP)(D,
                  linkS[(first+_curNode)%_limit]) <= 0)
                  high = _curNode;
                else
                  low = _curNode + 1;
        }
        if ((ok = atIns(high,D)) == (void *)0)
                _curNode = _nodes;
        else
                _curNode = high;
        /*  atIns() resets sorted to zero  */
        _flags |= CL_SORTED;
        return ok;
}

void * cl:: insUnique(void * D)
{
        if (!findFirst(D))
            if (sorted())
                return insSort(D);
            else
                return insQ(D);
        return (void *)0;
}

void * cl:: insUniqueNew(const void * D)
{
        if (!findFirst(D))
            if (sorted())
                return insSortNew(D);
            else
                return insQNew(D);
        return (void *)0;
}

cl& cl:: asSet()
{
        cl tmp;
        if (cmP())
            tmp.setCmP(cmP());
        else
            tmp.setCmP(cmPD());
        if (tmp.cmP())  {
            void * D;
            for (D = setCurNode(0); D; D = next())
                tmp.insUnique(D);
            for (D = setCurNode(0); D; D = next())
                if (tmp.indexOf(D) == CL_NOTFOUND)
                    clrBackup();
        }
        return *this;
}

void * cl:: findFirst(const void * K, unsigned& idx,
    voidKeyCmP kcmP, int honorSortFlag) const
{
        unsigned low, high;
/*
        idx is left undefined if anything fails,
        otherwise it is set to the newly found node.
        If the compare function throws an exception,
        the current node is set to the node choked on.
*/
        idx = _nodes;
        if (!kcmP)  {
          honorSortFlag = 1;
          if ((kcmP = voidKeyCmP(_cmP)) == voidKeyCmP0)
            if ((kcmP = voidKeyCmP(cmPD()))
                == voidKeyCmP0) return (void *) 0;
        }
        if (!linkS || !K || !_nodes)
                return (void *)0;
        if (honorSortFlag && (_flags & CL_SORTED))  {
            /*  binary search for first match  */
            low = 0U;
            high = _nodes - 1;
            while (low < high)  {
                idx = ((high + low) >> 1);
                if ((*kcmP)(K,linkS[(first+idx)
                    %_limit]) <= 0)
                    high = idx;
                else
                    low = idx + 1;
            }
            idx = high;
            if ((*kcmP)(K,linkS[(first+high)%_limit]))
                idx = _nodes;
        }
        else  {  /*  linear search!  */
            idx = 0;
            do  {
                if (!(*kcmP)(K,linkS[(first+idx)
                    %_limit]))
                    break;
            } while (++idx < _nodes);
        }
        return ((idx < _nodes)?
            linkS[(first+idx)%_limit] : (void *)0);
}

void * cl:: findNext(const void * K, unsigned& idx,
    voidKeyCmP kcmP, int honorSortFlag) const
{
/*
        For sorted cl's you must first call
        findFirst() to insure consistent results!
*/
        unsigned low, high;
        void * D;
/*
        idx is left undefined if anything fails,
        otherwise it is set to the newly found node.
        If the compare function throws an exception,
        the current node is set to the node choked on.
*/
        if (!kcmP)  {
          honorSortFlag = 1;
          if ((kcmP = voidKeyCmP(_cmP)) == voidKeyCmP0)
            kcmP = voidKeyCmP(cmPD());
        }
        if (!linkS || !K || !kcmP  || !_nodes
                || (++idx == _nodes))  {
                idx = _nodes;
                return (void *)0;
        }
        if (idx > _nodes)
                idx = 0;
        if (!(*kcmP)(K,D=linkS[(first+idx)%_limit]))
                return D;
        if (honorSortFlag && (_flags & CL_SORTED))
            return (void *) 0;
        while (++idx < _nodes)
            if (!(*kcmP)(K,linkS[(first+idx)%_limit]))
                break;
        return ((idx < _nodes)?
            linkS[(first+idx)%_limit] : (void *)0);
}

void CL_charTraits:: _delete(char * I) throw()
{ delete[] I; }

void CL_charTraits:: detach(char&, const cl *) throw()
{}
