//
// The KDBF framework shipped by Borland has a small bug in
// the function BDatabase::forceWrite in BDATABAS.CPP.  The curList is
// preloaded with space for pointers to four cursors.  As each cursor is
// created, the pointer is loaded.  Unfortunately, BDatabase::forceWrite()
// attempts to dereference the pointers even if they are set to NULL.
//
// This file is an updated BDATABAS.CPP with the bug corrected.
//
//  Stephen Birch
//


/**********************************************************************
**
**                         BDATABAS.CPP
**
** Member functions of the BDatabase class.
**
**********************************************************************/

// DBF - (C) Copyright 1994 by Borland International

#include "kdbf.h"
#pragma hdrstop

#ifdef __DLL__
   #define EXPORT _export
#else
   #define EXPORT
#endif

//
// prototypes for utility functions
//
Retcode EXPORT GetOptionalParams(pCHAR szDriver, UINT16 *iFields,
                                 pFLDDesc pfldDesc, pCHAR szData);
Retcode EXPORT addDatabase(engdef *, BDatabase *);   
void EXPORT deleteDatabase(engdef *, BDatabase *);   
Retcode EXPORT addKeymap(engdef *, const char *, int, int, const int *);
Retcode EXPORT getDesc(TABLEHANDLE, int&, FieldDesc far *);

// Constructor for universal database.

BDatabase::BDatabase(BEngine *eng,
                     DBIOpenMode openMode,
                     DBIShareMode shareMode)
{
    dbdef *dob;

    try
    {
        dob = new dbdef;            // See INTSTRCT.H file.
    }
    catch (xalloc)
    {
        isOpen = FALSE;
        strcpy(Alias, "");
        strcpy(Driver, "");
        strcpy(Password, "");
        lastError = DBIERR_NOMEMORY;
        return;
    }
    dob->handleCnt = 0;
    dob->curList = 0;
    dob->engH = eng;
    dbobj = (void *)dob;

    if (!eng->isOpen)
    {
        isOpen = FALSE;
        lastError = PXERR_ENGINENOTOPEN;
        return;                        
    }

    lastError = DbiOpenDatabase(NULL, NULL, openMode, shareMode,
                                NULL, 0, NULL, NULL, &hDb);
    if (lastError != DBIERR_NONE)
    {
        isOpen = FALSE;
        return;
    }
    isOpen = TRUE;
    isLocal = TRUE;
    strcpy(Alias, "");
    strcpy(Driver, szPARADOX);
    strcpy(Password, "");

    lastError = DbiGetNetUserName(userName);
    if (lastError != DBIERR_NONE)
    {
        DbiCloseDatabase(&hDb);
        return;
    }

    lastError = addDatabase((engdef *)eng->engobj, this);
    if (lastError != DBIERR_NONE)
    {
        DbiCloseDatabase(&hDb);
        return;
    }
}

BDatabase::BDatabase(BEngine *eng,
                     const char *BDDriver,
                     const char *BDAlias,
                     const char *BDPassword,
                     DBIOpenMode openMode,
                     DBIShareMode shareMode)
{
    dbdef *dob = 0;
    hDb = 0;

    try
    {
        dob = new dbdef;            // See INTSTRCT.H file.
        dob->handleCnt = 0;
        dob->curList = 0;
        dob->engH = eng;
        dbobj = (void *)dob;

        if (!eng->isOpen)
        {
            lastError = PXERR_ENGINENOTOPEN;
            throw lastError;
        }

        // If standard database
        if ((!strcmp(BDDriver, szPARADOX)) ||
            (!strcmp(BDDriver, szDBASE))   ||
            (!strcmp(BDDriver, szASCII)))
        {
            lastError = DbiOpenDatabase(NULL, NULL, openMode, shareMode,
                                        NULL, 0, NULL, NULL, &hDb);
            isLocal = TRUE;
        }
        else
        {
            lastError = DbiOpenDatabase((pCHAR)BDAlias, (pCHAR)BDDriver,
                                        openMode, shareMode,
                                        (pCHAR)BDPassword, 0, NULL, NULL, &hDb);
            isLocal = FALSE;
        }
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }
        isOpen = TRUE;

        strcpy(Alias, BDAlias);

        if (BDDriver)
        {
            strcpy(Driver, BDDriver);
        }

        if (BDPassword)
        {
            strcpy(Password, BDPassword);
        }

        lastError = DbiGetNetUserName(userName);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }

        lastError = addDatabase((engdef *)eng->engobj, this);
    }
    catch(xalloc)
    {
        if (hDb)
        {
            DbiCloseDatabase(&hDb);
        }

        isOpen = FALSE;
        strcpy(Alias, "");
        strcpy(Driver, "");
        strcpy(Password, "");
        lastError = DBIERR_NOMEMORY;
    }
    catch(Retcode)
    {
        if (hDb)
        {
            DbiCloseDatabase(&hDb);
        }

        strcpy(Alias, "");
        strcpy(Driver, "");
        strcpy(Password, "");
        isOpen = FALSE;
    }
}


// Destructor; if the database is open, close it first.

BDatabase::~BDatabase()
{
    if (isOpen)
    {
        close();
    }

    dbdef *dob  = (dbdef *)dbobj;
    if (dob)
    {
        // Reset pointers in any dependent cursor objects.
        for (int i=0; i < dob->handleCnt; i++)
        {
            if (dob->curList[i])
            {
                ((curdef *)dob->curList[i]->curobj)->dbH = 0;
            }
        }
        if (dob->handleCnt)
        {
            delete [] dob->curList;
        }
        if (dob->engH)       // If engine is operating,
        {
           deleteDatabase((engdef *)        // remove from Engine's
                          dob->engH->engobj,this);  // database list.
        }
        delete dob;
    }
}

// Open the universal database

Retcode BDatabase::open(DBIOpenMode openMode,
                        DBIShareMode shareMode)
{
    if (isOpen)
    {
        return (lastError = PXERR_DBALREADYOPEN);
    }

    dbdef *dob  = (dbdef *)dbobj;
    if (!dob->engH || ! dob->engH->isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenDatabase(NULL, NULL, openMode, shareMode, NULL, 0,
                                NULL, NULL, &hDb);

    if (lastError != DBIERR_NONE)
    {
        isOpen = FALSE;
        return lastError;
    }

    isOpen = TRUE;
    isLocal = TRUE;
    strcpy(Alias, "");
    strcpy(Driver, szPARADOX);
    strcpy(Password, "");
    lastError = DbiGetNetUserName(userName);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    return (lastError = DBIERR_NONE);
}

// Open a database

Retcode BDatabase::open(const char *BDDriver,
                        const char *BDAlias,
                        const char *BDPassword,
                        DBIOpenMode openMode,
                        DBIShareMode shareMode)
{
    if (isOpen)
    {
        return (lastError = PXERR_DBALREADYOPEN);
    }

    dbdef *dob  = (dbdef *)dbobj;
    if (!dob->engH || ! dob->engH->isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    // If standard database
        if ((!strcmp(BDDriver, szPARADOX))  ||
            (!strcmp(BDDriver, szDBASE))    ||
            (!strcmp(BDDriver, szASCII)))
    {
        lastError = DbiOpenDatabase((pCHAR)BDAlias, (pCHAR)BDDriver,
                                    openMode, shareMode, NULL, 0,
                                    NULL, NULL, &hDb);
        isLocal = TRUE;
    }
    else
    {
        lastError = DbiOpenDatabase((pCHAR)BDAlias, (pCHAR)BDDriver,
                                    openMode, shareMode,
                                    (pCHAR)BDPassword, 0, NULL, NULL, &hDb);
        isLocal = FALSE;
    }
    if (lastError != DBIERR_NONE)
    {
        isOpen = FALSE;
        return lastError;
    }

    isOpen = TRUE;
    strcpy(Alias, BDAlias);
    strcpy(Driver, BDDriver);
    strcpy(Password, BDPassword);
    lastError = DbiGetNetUserName(userName);
    if (lastError != DBIERR_NONE)
    {
        throw lastError;
    }

    return (lastError = DBIERR_NONE);
}

//  Close the database; close all cursors associated with the database.

Retcode BDatabase::close(void)
{
    int i;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    dbdef *dob = (dbdef *)dbobj;

    for (i=0; i < dob->handleCnt; i++)
    {
        if (dob->curList[i] && dob->curList[i]->isOpen)
        {
            if ((lastError = dob->curList[i]->close()) != 0)   // Close cursor.
            {
                return lastError;
            }
        }
    }

    lastError = DbiCloseDatabase(&hDb);
    if (lastError == DBIERR_NONE)
    {
        isOpen = FALSE;
    }

    if (TableList.isOpen)
    {
        TableList.close();
    }

    return lastError;
}

// See if a table exists in the database.
BOOL BDatabase::tableExists(const char *tableName)
{
    BOOL        exists = FALSE;
    char        *tblName;
    TBLFullDesc tblDesc;
    CHAR        remoteName[DBIMAXNAMELEN+1];

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return exists;
    }

    if (isLocal)
    {
        tblName = _fstrtok((pCHAR)tableName, ".");
    }
    else
    {
        if ((!strcmp(Driver, "ORACLE")) ||
            (!strcmp(Driver, "SYBASE")))
        {
            strcpy(remoteName, userName);
            strcat(remoteName, ".");
            strcat(remoteName, tableName);
        }
        else
        {
            strcpy(remoteName, tableName);
        }
        tblName = (pCHAR)tableName;
    }

    if (tblName == NULL)
    {
        lastError = DBIERR_INVALIDTABLENAME;
        return FALSE;
    }

    resetDBTableList();

    lastError = TableList.gotoTop();
    if (lastError != DBIERR_NONE)
    {
        return FALSE;
    }

    while (TableList.getNextRecord(tblDesc) == DBIERR_NONE)
    {
        if (!stricmp(tblDesc.tblBase.szName, tblName))
        {
            // Check if the types match
            if (strcmp(tblDesc.tblBase.szType, Driver) && isLocal)
            {
                // keep searching it the table is of the wrong type
                continue;
            }
            lastError = DBIERR_NONE;
            return TRUE;
        }
    }

    lastError = DBIERR_NOSUCHTABLE;

    return exists;
}

// Create a table based on its descriptor.
Retcode BDatabase::createTable(const char *tableName,
                               int numFields, const FieldDesc *fldDescs,
                               ValidityDesc *valDesc, RefDesc *refDesc
                               )
{
    CRTblDesc   crTblDesc;
    pFLDDesc    pfldDesc    = 0;
    pFLDDesc    pfldDescOpt = 0;
    pCHAR       Data        = 0;       
    UINT16      iFields     = 0;
    UINT16      iOffset     = 0;// Offset in the default values buffer
    CHAR        Temp[DBIMAXSCFLDLEN];         // Temporary Buffer
    int         i;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    try
    {
        pfldDesc = new FLDDesc[numFields];

        memset(&crTblDesc, 0, sizeof(CRTblDesc)); // Clear the buffer.

        for (i=0; i < numFields; i++)
        {
            pfldDesc[i].iFldNum     = fldDescs[i].fldNum;
            strcpy(pfldDesc[i].szName, fldDescs[i].fldName);
            pfldDesc[i].iFldType    = fldDescs[i].fldType;
            pfldDesc[i].iSubType    = fldDescs[i].fldSubtype;
            pfldDesc[i].iUnits1     = fldDescs[i].fldLen;

            if (!strcmp(Driver, szDBASE))
            {
                // dBASE float field needs a size - dBASE doesn't
                if ((fldDescs[i].fldType == fldDouble) &&
                    (fldDescs[i].fldLen == 0))
                {
                    pfldDesc[i].iUnits1     = 20;
                }
                if ((fldDescs[i].fldType == fldBcd))
                {
                    pfldDesc[i].iUnits1     = 20;
                    pfldDesc[i].iFldType    = fldDouble;
                } 
                if ((fldDescs[i].fldType == fldTime))
                {
                    pfldDesc[i].iUnits1     = 20;
                    pfldDesc[i].iFldType    = fldChar;
                } 
                if ((fldDescs[i].fldType == fldTimeStamp))
                {
                    pfldDesc[i].iUnits1     = 40;
                    pfldDesc[i].iFldType    = fldChar;
                } 
            }

            pfldDesc[i].iUnits2     = 0;
            pfldDesc[i].iOffset     = 0;
            if (fldDescs[i].fldLen != 0)
            {
                pfldDesc[i].iLen        = pfldDesc[i].iUnits1 + 1;
            }
            else
            {
                pfldDesc[i].iLen        = 0;
            }
            pfldDesc[i].iNullOffset = 0;
            pfldDesc[i].efldvVchk   = fldvNOCHECKS;
            pfldDesc[i].efldrRights = fldrUNKNOWN;
        }

        // Set the name and the type of the table
        strcpy(crTblDesc.szTblName, tableName) ; // name of the table
        strcpy(crTblDesc.szTblType, Driver) ; // Type of table

        // Set the field information for the table
        crTblDesc.iFldCount     = numFields ;   // number of fields
        crTblDesc.pfldDesc      = pfldDesc ;      // Field descriptor

        if (!strcmp(Driver, szPARADOX))
        {
            if (valDesc != NULL)
            {
                crTblDesc.iValChkCount  = valDesc->valChkCount;
                crTblDesc.pecrValChkOp  = valDesc->cropType;
                crTblDesc.pvchkDesc     = valDesc->vchkDesc;
            }
            if (refDesc != NULL)
            {
                crTblDesc.iRintCount    = refDesc->rintCount;
                crTblDesc.pecrRintOp    = refDesc->cropType;
                crTblDesc.printDesc     = refDesc->rintDesc;
            }
        }
        
        // Allocate enough space for the maximum number of fields
        pfldDescOpt = new FLDDesc[DBIMAXSCFIELDS];
        Data        = new CHAR[DBIMAXSCRECSIZE];

        // Initialize the pfldDesc variable to all zero's
        memset((void*)pfldDescOpt, 0, sizeof(FLDDesc) * DBIMAXSCFIELDS);
        memset((void*)Data, 0, sizeof(char) * DBIMAXSCRECSIZE);

        // Get the default values for the optional parameters
        // Optional parameters are only valid for local tables
        if ((!strcmp(Driver, szPARADOX)) ||
            (!strcmp(Driver, szDBASE)))
        {
            lastError = GetOptionalParams(Driver, &iFields, pfldDescOpt, Data);
            if (lastError != DBIERR_NONE)
            {
                throw lastError;
            }
        }

        dbdef   *dob  = (dbdef *)dbobj;
        BEnv    env;

        (dob->engH)->getDefaults(env);

        if (!strcmp(Driver, szPARADOX))
        {
            for (i = 0; i < iFields; i++)
            {
                if (!strcmp(pfldDescOpt[i].szName, "LEVEL"))
                {
                    strcpy(&Data[iOffset], itoa(env.tabCrtMode, Temp, 10));
                }
                // Set the maximum size of the table
                if (!strcmp(pfldDescOpt[i].szName, "BLOCK SIZE"))
                {
                    strcpy(&Data[iOffset], itoa(env.tblMaxSize, Temp, 10));
                }
                // Increment to point to the next field within the
                // array of default values.
                iOffset += pfldDescOpt[i].iLen;
            }
        }

        iOffset = 0;

        if (!strcmp(Driver, szDBASE))
        {
            for (i = 0; i < iFields; i++)
            {
                // Set the level of the table
                if (!strcmp(pfldDescOpt[i].szName, "LEVEL"))
                {
                    strcpy(&Data[iOffset], itoa(env.tabCrtMode, Temp, 10));
                }
                // Increment to point to the next field within the
                // array of default values.
                iOffset += pfldDescOpt[i].iLen;
            }
        }

        if ((!strcmp(Driver, szPARADOX)) ||
            (!strcmp(Driver, szDBASE)))
        {
            crTblDesc.iOptParams    = iFields;
            crTblDesc.pfldOptParams = pfldDescOpt;
            crTblDesc.pOptData      = (pBYTE)Data;
        }

        // Create the table using information supplied in the Table
        // Descrpitor above
        lastError = DbiCreateTable(hDb, TRUE, &crTblDesc);
        if (lastError)
        {
            throw lastError;
        }

        resetDBTableList();

        delete pfldDesc;
        pfldDesc = 0;                                              
        delete pfldDescOpt;
        pfldDescOpt = 0;
        delete Data;
        Data = 0;
    }
    catch(xalloc)
    {
        if (pfldDesc)
        {
            delete pfldDesc;
        }
        if (pfldDescOpt)
        {
            delete pfldDescOpt;
        }
        if (Data)
        {
            delete Data;
        }
    }
    catch (Retcode)
    {
        if (pfldDesc)
        {
            delete pfldDesc;
        }
        if (pfldDescOpt)
        {
            delete pfldDescOpt;
        }
        if (Data)
        {
            delete Data;
        }
        return lastError;
    }
    return lastError;
}

// Copy one table family to another.

Retcode BDatabase::copyTable(const char *srcTable, const char *destTable,
                             const char *destType)
{
    BOOL    bOverWrite;
    char    destTableType[DBIMAXNAMELEN + 1];

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    bOverWrite = FALSE;

    if (destType == NULL)
    {
        strcpy(destTableType, Driver);
    }
    else
    {
        strcpy(destTableType, destType);
    }
    
    lastError = DbiCopyTable(hDb, bOverWrite, (pCHAR)srcTable, destTableType,
                             (pCHAR)destTable);

    if (lastError == DBIERR_NONE)
    {
        resetDBTableList();
    }

    return lastError;
}

// Append records of a table to the destination table.
Retcode BDatabase::appendTable(const char *srcTable,
                               const char *destTable,
                               const char *destType,
                               const BDatabase &db)
{
    BATTblDesc  srcTblDesc;
    BATTblDesc  destTblDesc;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    memset((pVOID)&srcTblDesc, 0, sizeof(BATTblDesc));
    memset((pVOID)&srcTblDesc, 0, sizeof(BATTblDesc));

    // Source table values
    //   Database
    srcTblDesc.hDb = hDb;
    //   tablename
    strcpy(srcTblDesc.szTblName, srcTable);
    //   Table Type
    strcpy(srcTblDesc.szTblType, Driver);

    // Destinarion table
    //   Database
    if (&db == NULL)
    {
        destTblDesc.hDb = hDb;
    }
    else
    {
        destTblDesc.hDb = db.hDb;
    }

    //   Table Name
    strcpy(destTblDesc.szTblName, destTable);

    //   Table Type
    if (destType == NULL)
    {
        strcpy(destTblDesc.szTblType, Driver);
    }
    else
    {
        strcpy(destTblDesc.szTblType, destType);
    }

    lastError = DbiBatchMove(&srcTblDesc, NULL, &destTblDesc, NULL, batAPPEND,
                             NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                             NULL, NULL, NULL, NULL, TRUE, TRUE,
                             NULL, TRUE);

    return lastError;
}

// Renames a table and its family members.

Retcode BDatabase::renameTable(const char *oldName, const char *newName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiRenameTable(hDb, (pCHAR)oldName, Driver, (pCHAR)newName);

    if (lastError == DBIERR_NONE)
    {
        resetDBTableList();
    }

    return lastError;
}

// Delete a table and its family.

Retcode BDatabase::deleteTable(const char *tableName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiDeleteTable(hDb, (pCHAR)tableName, Driver);

    if (lastError == DBIERR_NONE)
    {
        resetDBTableList();
    }

    return lastError;
}

// Encrypt a table based on a supplied password.

Retcode BDatabase::encryptTable(const char *tableName,
          const char *password)
{
    CRTblDesc   crTblDesc;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (!isLocal)
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }

    // Clear the buffer
    memset(&crTblDesc, 0, sizeof(CRTblDesc));

    // name of the table
    strcpy(crTblDesc.szTblName, tableName);
    strcpy(crTblDesc.szTblType, Driver);
    // Master password suplied for the table
    crTblDesc.bProtected = TRUE;
    // Password for the table
    strcpy(crTblDesc.szPassword, password);

    lastError = DbiDoRestructure(hDb, 1, &crTblDesc, NULL, NULL, NULL, FALSE);
    
    return lastError;
}

// Decrypt a table using a password.

Retcode BDatabase::decryptTable(const char *tableName)
{
    CRTblDesc   crTblDesc;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (!isLocal)
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }
    
    // Clear the buffer
    memset(&crTblDesc, 0, sizeof(CRTblDesc));

    // name of the table
    strcpy(crTblDesc.szTblName, tableName);
    strcpy(crTblDesc.szTblType, Driver) ; // Type of table

    lastError = DbiDoRestructure(hDb, 1, &crTblDesc, NULL, NULL, NULL, FALSE);

    return lastError;
}

// Empty a table. (Delete all its records).

Retcode BDatabase::emptyTable(const char *tableName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiEmptyTable(hDb, NULL,(pCHAR)tableName, Driver);

    return lastError;
}

// Upgrade a table from Paradox 3.5 to Paradox 4.0 format.

Retcode BDatabase::upgradeTable(const char *tableName)
{
    CRTblDesc   crTblDesc;
    pFLDDesc    pfldDescOpt = 0;
    pCHAR       Data        = 0;
    UINT16      iFields     = 0;
    int         i;
    CHAR        Temp[DBIMAXSCFLDLEN];         // Temporary Buffer
    UINT16      iOffset     = 0;// Offset in the default values buffer

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (!isLocal)
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }

    try
    {
        memset(&crTblDesc, 0, sizeof(CRTblDesc)); // Clear the buffer.

        // Set the name and the type of the table
        strcpy(crTblDesc.szTblName, tableName) ; // name of the table
        strcpy(crTblDesc.szTblType, Driver) ; // Type of table

        // Allocate enough space for the maximum number of fields
        pfldDescOpt = new FLDDesc[DBIMAXSCFIELDS];
        Data        = new CHAR[DBIMAXSCRECSIZE];

        // Initialize the pfldDesc variable to all zero's
        memset((void*)pfldDescOpt, 0, sizeof(FLDDesc) * DBIMAXSCFIELDS);
        memset((void*)Data, 0, sizeof(CHAR) * DBIMAXSCRECSIZE);

        // Get the default values for the optional parameters
        lastError = GetOptionalParams(Driver, &iFields, pfldDescOpt, Data);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }

        if (!strcmp(Driver, szPARADOX))
        {
            for (i = 0; i < iFields; i++)
            {
                if (!strcmp(pfldDescOpt[i].szName, "LEVEL"))
                {
                    strcpy(&Data[iOffset], itoa(px50Fmt, Temp, 10));
                }
                // Increment to point to the next field within the
                // array of default values.
                iOffset += pfldDescOpt[i].iLen;
            }
        }

        iOffset = 0;

        if (!strcmp(Driver, szDBASE))
        {
            for (i = 0; i < iFields; i++)
            {
                // Set the level of the table
                if (!strcmp(pfldDescOpt[i].szName, "LEVEL"))
                {
                    strcpy(&Data[iOffset], itoa(px50Fmt, Temp, 10));
                }
                // Increment to point to the next field within the
                // array of default values.
                iOffset += pfldDescOpt[i].iLen;
            }
        }

        crTblDesc.iOptParams    = iFields;
        crTblDesc.pfldOptParams = pfldDescOpt;
        crTblDesc.pOptData      = (pBYTE)Data;
        
        lastError = DbiDoRestructure(hDb, 1, &crTblDesc, (pCHAR)tableName,
                                     NULL, NULL, FALSE);

        delete pfldDescOpt;
        pfldDescOpt = 0;
        delete Data;
        Data = 0;
    }
    catch(xalloc)
    {
        if (pfldDescOpt)
        {
            delete pfldDescOpt;
        }
        if (Data)
        {
            delete Data;
        }
    }
    catch (Retcode)
    {
        if (pfldDescOpt)
        {
            delete pfldDescOpt;
        }
        if (Data)
        {
            delete Data;
        }
        return lastError;
    }
    
    return lastError;
}

Retcode BDatabase::restructureTable(CRTblDesc &tableDesc,
                                    const char *tableName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (!isLocal)
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }
    
    lastError = DbiDoRestructure(hDb, 1, &tableDesc, (pCHAR)tableName,
                                 NULL, NULL, FALSE);

    if (lastError == DBIERR_NONE)
    {
        resetDBTableList();
    }

    return lastError;
}

// Determine if a table is protected.

BOOL BDatabase::isProtected(const char *tableName)
{
    BOOL        protect = FALSE;
    char        *tblName;
    TBLFullDesc  tblDesc;

    // For local Tables
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return protect;
    }

    // Different methodology required for local and remote tables
    if (isLocal)
    {
        tblName = _fstrtok((pCHAR)tableName, ".");
        if (tblName == NULL)
        {
            lastError = DBIERR_INVALIDTABLENAME;
            return FALSE;
        }

        resetDBTableList();

        lastError = TableList.gotoTop();
        if (lastError != DBIERR_NONE)
        {
            return FALSE;
        }

        while (TableList.getNextRecord(tblDesc) == DBIERR_NONE)
        {
            if (!stricmp(tblDesc.tblBase.szName, tblName))
            {
                // Make certain file type match - needed in case a dBASE and
                // Paradox table of the same name exist in the same directory.
                if (stricmp(tblDesc.tblBase.szType, Driver))
                {
                    // Keep searching if the type doesn't match
                    continue;
                }
                if (tblDesc.tblExt.bProtected == TRUE)
                {
                    // Table is protected
                    lastError = DBIERR_NONE;
                    return TRUE;
                }
                else
                {
                    lastError = DBIERR_NONE;
                    return FALSE;
                }
            }
        }

        lastError = DBIERR_NOSUCHTABLE;
    }
    else
    {
        TABLEHANDLE tabH = 0;

        lastError = DbiOpenTable(hDb, (pCHAR)tableName, Driver,
                                 NULL, NULL, NULL, dbiREADONLY, dbiOPENSHARED,
                                 xltNONE, FALSE, NULL, &tabH);
        if (lastError == DBIERR_NONE)
        {
            DbiCloseCursor(&tabH);
            protect = FALSE;
            return protect;
        }
        else if (lastError == DBIERR_INVALIDPASSWORD)
        {
            lastError = DBIERR_NONE;
            protect = TRUE;
            return protect;
        }
    }

    return protect;
}

// Return the number of fields (columns) in a table.

int BDatabase::getFieldCount(const char *tableName)
{
    int         count = 0;
    char        *tblName;
    TBLFullDesc  tblDesc;

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return count;
    }

    // Different methodology required for local and remote tables
    if (isLocal)
    {
        tblName = _fstrtok((pCHAR)tableName, ".");

        if (tblName == NULL)
        {
            lastError = DBIERR_INVALIDTABLENAME;
            return 0;
        }

        resetDBTableList();

        lastError = TableList.gotoTop();
        if (lastError != DBIERR_NONE)
        {
            return FALSE;
        }

        while (TableList.getNextRecord(tblDesc) == DBIERR_NONE)
        {
            if (!stricmp(tblDesc.tblBase.szName, tblName))
            {
                // For local tables, check if the types match
                if ((!strcmp(tblDesc.tblBase.szType, szPARADOX)) ||
                    (!strcmp(tblDesc.tblBase.szType, szDBASE)))
                {
                    if (stricmp(tblDesc.tblBase.szType, Driver))
                    {
                        // Keep searching if the type doesn't match
                        continue;
                    }
                }
                // Table is protected
                lastError = DBIERR_NONE;
                return tblDesc.tblExt.iFields;
            }
        }

        lastError = DBIERR_NOSUCHTABLE;
    }
    else
    {
        TABLEHANDLE tabH = 0;
        CURProps    curProps;

        lastError = DbiOpenTable(hDb, (pCHAR)tableName, Driver,
                                 NULL, NULL, NULL, dbiREADONLY, dbiOPENSHARED,
                                 xltNONE, FALSE, NULL, &tabH);
        if (lastError != DBIERR_NONE)
        {
           return 0;
        }

        lastError = DbiGetCursorProps(tabH, &curProps);
        if (lastError != DBIERR_NONE)
        {
           return 0;
        }

        count = curProps.iFields;

        DbiCloseCursor(&tabH);
    }

    return count;
}


// Given a table, return the number of fields in it and an array
// of its field descriptors (FieldDesc structures). The caller is
// responsible for deleting the free store memory allocated for
// the field descriptors (using delete [] desc, where
// desc is pointer to FieldDesc object).

Retcode BDatabase::getDescVector(char *tableName,
                                 int& numFields,
                                 FieldDesc *&fldDescs)
{
    char            *tblName;
    TBLFullDesc     tblDesc;
    TABLEHANDLE     tabH = 0;

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        throw lastError;
    }

    try
    {
        // Need to handle local tables differently from remote tables
        if (isLocal)
        {
            tblName = _fstrtok((pCHAR)tableName, ".");

            if (tblName == NULL)
            {
                lastError = DBIERR_INVALIDTABLENAME;
                throw lastError;
            }

            resetDBTableList();

            lastError = TableList.gotoTop();
            if (lastError != DBIERR_NONE)
            {
                return FALSE;
            }

            while (TableList.getNextRecord(tblDesc) == DBIERR_NONE)
            {
                if (!stricmp(tblDesc.tblBase.szName, tblName))
                {
                    // For local tables, check if the types match
                    if ((!strcmp(tblDesc.tblBase.szType, szPARADOX)) ||
                        (!strcmp(tblDesc.tblBase.szType, szDBASE)))
                    {
                        if (stricmp(tblDesc.tblBase.szType, Driver))
                        {
                            // Keep searching if the type doesn't match
                            continue;
                        }
                    }

                    numFields = tblDesc.tblExt.iFields;

                    lastError = DBIERR_NONE;

                    fldDescs = new FieldDesc[numFields];  // Allocate
                                                          // descriptor array.

                    lastError = getDesc(tableName, numFields,
                                        (FieldDesc far *)fldDescs);
                    if (lastError)
                    {
                        throw lastError;
                    }

                    return lastError;
                }
            }

            lastError = DBIERR_NOSUCHTABLE;
            throw lastError;
        }
        else
        {
            CURProps    curProps;

            lastError = DbiOpenTable(hDb, (pCHAR)tableName, Driver,
                                     NULL, NULL, NULL, dbiREADONLY,
                                     dbiOPENSHARED, xltNONE, FALSE, NULL,
                                     &tabH);
            if (lastError != DBIERR_NONE)
            {
               throw lastError;
            }

            lastError = DbiGetCursorProps(tabH, &curProps);
            if (lastError != DBIERR_NONE)
            {
               throw lastError;
            }

            DbiCloseCursor(&tabH);
            tabH = 0;
            
            numFields = curProps.iFields;

            fldDescs = new FieldDesc[numFields];  // Allocate
                                                  // descriptor array.
            lastError = getDesc(tableName, numFields,
                                (FieldDesc far *)fldDescs);
            if (lastError)
            {
                throw lastError;
            }

            return lastError;
       }
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        if (tabH)
        {
            DbiCloseCursor(&tabH);
        }
        if (fldDescs)
        {
            delete fldDescs;
        }
        return lastError;
    }
    catch(Retcode)
    {
        if (tabH)
        {
            DbiCloseCursor(&tabH);
        }
        if (fldDescs)
        {
            delete fldDescs;
        }
        return lastError;
    }
}

// Returns information about the indexes in a table

Retcode BDatabase::getIndexVector(char *tableName,
                                  int& numIndexes,
                                  IDXDesc *&idxDescs)
{
    char            *tblName;
    TBLFullDesc     tblDesc;

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        throw lastError;
    }

    try
    {
        // Need to handle local tables differently from remote tables
        if (isLocal)
        {
            tblName = _fstrtok((pCHAR)tableName, ".");

            if (tblName == NULL)
            {
                lastError = DBIERR_INVALIDTABLENAME;
                throw lastError;
            }

            resetDBTableList();

            lastError = TableList.gotoTop();
            if (lastError != DBIERR_NONE)
            {
                return FALSE;
            }

            while (TableList.getNextRecord(tblDesc) == DBIERR_NONE)
            {
                if (!stricmp(tblDesc.tblBase.szName, tblName))
                {
                    if (stricmp(tblDesc.tblBase.szType, Driver))
                    {
                        // Keep searching if the type doesn't match
                        continue;
                    }

                    numIndexes = tblDesc.tblExt.iIndexes;

                    lastError = DBIERR_NONE;

                    idxDescs = new IDXDesc[numIndexes]; // Allocate descriptor
                                                        //array.

                    getIndexDesc(tableName, numIndexes,
                                 (IDXDesc far *)idxDescs);
                                 
                    return lastError;
                }
            }

            lastError = DBIERR_NOSUCHTABLE;
            throw lastError;
        }
        else
        {
            TABLEHANDLE tabH = 0;
            CURProps    curProps;

            lastError = DbiOpenTable(hDb, (pCHAR)tableName, Driver,
                                     NULL, NULL, NULL, dbiREADONLY,
                                     dbiOPENSHARED, xltNONE, FALSE, NULL,
                                     &tabH);
            if (lastError != DBIERR_NONE)
            {
               throw lastError;
            }

            lastError = DbiGetCursorProps(tabH, &curProps);
            if (lastError != DBIERR_NONE)
            {
               throw lastError;
            }

            DbiCloseCursor(&tabH);
            tabH = 0;

            numIndexes = curProps.iIndexes;

            idxDescs = new IDXDesc[numIndexes];  // Allocate
                                                 // descriptor array.

            getIndexDesc(tableName, numIndexes,
                         (IDXDesc far *)idxDescs);
            return lastError;
        }
   }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        if (idxDescs)
        {
            delete idxDescs;
        }
        return lastError;
    }
    catch(Retcode)
    {
        if (idxDescs)
        {
            delete idxDescs;
        }
        return lastError;
    }
}

// Given an ordered list of fields, establish a virtual field
// as a composite of all the fields.

Retcode BDatabase::defineCompoundKey (const char *tableName,
                                      int numFields,
                                      const FIELDNUMBER *fldArray,
                                      const char *indexName,
                                      BOOL caseSen,
                                      FIELDNUMBER& indexHandle)
{
    IDXDesc idxDesc;
    int i;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    // Composite indexes referenced by an index ID are only supported for
    // Paradox tables
    if (strcmp(Driver, szPARADOX))
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }

    memset(&idxDesc, 0, sizeof(IDXDesc));

    strcpy(idxDesc.szName, indexName);
    idxDesc.bMaintained         = TRUE;
    idxDesc.iFldsInKey          = numFields;

    if (numFields > DBIMAXFLDSINKEY)
    {
        return (lastError = DBIERR_OUTOFRANGE);
    }
    for (i = 0; i < numFields; i++)
    {
        idxDesc.aiKeyFld[i]     = fldArray[i];
    }
    idxDesc.bCaseInsensitive    = !caseSen;

    lastError = getIndexId(tableName, fldArray, indexHandle);

    // Check if the index exists
    if (lastError != DBIERR_NOSUCHINDEX)
    {
        return lastError;
    }                           

    lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                            Driver, &idxDesc, NULL);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    // Set the indexHandle to the ID of the Index
    lastError = getIndexId(tableName, indexName, indexHandle);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    return lastError;
}

// Given an ordered list of fields, created a composite index

Retcode BDatabase::defineCompoundKey (const char *tableName,
                                      int numFields,
                                      const FIELDNUMBER *fldArray,
                                      const char *indexName,
                                      BOOL caseSen)
{
    IDXDesc     idxDesc;
    INT16       numFlds;
    FieldDesc   *fldDescs = NULL;

    int i;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    memset(&idxDesc, 0, sizeof(IDXDesc));

    strcpy(idxDesc.szName, indexName);

    idxDesc.bMaintained         = TRUE;
    if (!strcmp(Driver, szDBASE))
    {
        idxDesc.iFldsInKey          = 1;
        idxDesc.bExpIdx             = TRUE;
        strcpy(idxDesc.szName, tableName);
        strcat(idxDesc.szName, ".MDX");
        strcpy(idxDesc.szTagName, indexName);

        lastError = getDescVector((pCHAR)tableName, numFlds, fldDescs);
        if (lastError != DBIERR_NONE)
        {
            return lastError;
        }

        strcpy(idxDesc.szKeyExp, fldDescs[(fldArray[0] - 1)].fldName);

        for (int i = 1; i < numFields; i++)
        {
            strncat(idxDesc.szKeyExp, " + ", DBIMAXKEYEXPLEN);
            idxDesc.szKeyExp[DBIMAXKEYEXPLEN] = 0;
            strncat(idxDesc.szKeyExp, fldDescs[(fldArray[i] - 1)].fldName,
                    DBIMAXKEYEXPLEN);
            idxDesc.szKeyExp[DBIMAXKEYEXPLEN] = 0;
        }
    }
    else
    {
        idxDesc.iFldsInKey          = numFields;

        for (i = 0; i < numFields; i++)
        {
            idxDesc.aiKeyFld[i]     = fldArray[i];
        }
    }

    if (numFields > DBIMAXFLDSINKEY)
    {
        return (lastError = DBIERR_OUTOFRANGE);
    }
    
    idxDesc.bCaseInsensitive    = !caseSen;

    lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                            Driver, &idxDesc, NULL);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    return lastError;
}

// Create a secondary index on the table.

Retcode BDatabase::createSIndex (const char *tableName,
          FIELDNUMBER fieldH, PXKeyCrtMode keyMode)
{
    IDXDesc     idxDesc;
    FieldDesc   *fldDescs = 0;
    INT16       numFields;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    try
    {
        memset(&idxDesc, 0, sizeof(IDXDesc));

        lastError = getDescVector((pCHAR)tableName, numFields, fldDescs);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }

        if (fieldH > numFields)
        {
            lastError = DBIERR_OUTOFRANGE;
            throw lastError;
        }

        if (!strcmp(Driver, szPARADOX))
        {
            strcpy(idxDesc.szName, fldDescs[fieldH-1].fldName);
            // If a maintained index
            if (keyMode != pxSecondary)
            {
                idxDesc.bMaintained     = TRUE;
            }
            idxDesc.iIndexId = fieldH;
        }
        else if (!strcmp(Driver, szDBASE))
        {
            if (keyMode != pxSecondary)
            {
                strcpy(idxDesc.szName, tableName);
                strcat(idxDesc.szName, ".MDX");
                strcpy(idxDesc.szTagName, fldDescs[fieldH-1].fldName);
            }
            else
            {
                strncpy(idxDesc.szName, tableName, 3);
                idxDesc.szName[3] = 0;
                strncat(idxDesc.szName, fldDescs[fieldH-1].fldName, 8);
                // Make certain to NULL terminate
                idxDesc.szName[8] = 0;
                strcat(idxDesc.szName, ".NDX");
                idxDesc.bMaintained = FALSE;
            }
        }
        else
        {
            strcpy(idxDesc.szName, tableName);
            strncat(idxDesc.szName, "_", DBIMAXNAMELEN);
            strncat(idxDesc.szName, fldDescs[fieldH-1].fldName, DBIMAXNAMELEN);
            idxDesc.szName[DBIMAXNAMELEN] = 0;
            idxDesc.bMaintained     = TRUE;
        }

        delete fldDescs;
        fldDescs = 0;

        idxDesc.iFldsInKey          = 1;
        idxDesc.aiKeyFld[0]         = fieldH;

        lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                                Driver, &idxDesc, NULL);

        throw lastError;
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        if (fldDescs)
        {
            delete fldDescs;
        }
        return lastError;
    }
    catch(Retcode)
    {
        if (fldDescs)
        {
            delete fldDescs;
        }        
        return lastError;
    }
}

// Create a primary key by using the first numFields fields as the key.

Retcode BDatabase::createPIndex(const char *tableName, int numFields)
{
    IDXDesc     idxDesc;
    FieldDesc   *fldDescs       = 0;
    DBIKEYEXP   expression      = "";
    int         i;
    INT16       numberOfFields;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (numFields < 1)
    {
        return (lastError = DBIERR_NOTSUPPORTED);
    }

    try
    {
        if (!strcmp(Driver, szPARADOX))
        {
            memset(&idxDesc, 0, sizeof(IDXDesc));

            idxDesc.bPrimary            = TRUE;
            idxDesc.bUnique             = TRUE;
            idxDesc.bMaintained         = TRUE;
            idxDesc.iFldsInKey          = numFields;

            for (i = 0; i < numFields; i++)
            {
                idxDesc.aiKeyFld[i]     = i+1;
            }

            lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                                    Driver, &idxDesc, NULL);

            throw lastError;
        }
        else if (!strcmp(Driver, szDBASE))
        {
            memset(&idxDesc, 0, sizeof(IDXDesc));

            lastError = getDescVector((pCHAR)tableName, numberOfFields, fldDescs);
            if (lastError != DBIERR_NONE)
            {
                throw lastError;
            }

            if (numFields > numberOfFields)
            {
                lastError = DBIERR_OUTOFRANGE;
                throw lastError;
            }

            strcat(expression, fldDescs[0].fldName);
            for (int i = 1; i < numFields; i++)
            {
                strcat(expression, " + ");
                strcat(expression, fldDescs[i].fldName);
            }

            strcpy(idxDesc.szTagName, "Primary");
            strcpy(idxDesc.szKeyExp, expression);
            idxDesc.bUnique             = TRUE;
            idxDesc.bMaintained         = TRUE;
            idxDesc.bExpIdx             = TRUE;
            idxDesc.iFldsInKey          = 1;

            delete fldDescs;
            fldDescs = 0;
            lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                                    Driver, &idxDesc, NULL);

            return lastError;
        }
        else if (!strcmp(Driver, szASCII))
        {
            lastError = DBIERR_NOTSUPPORTED;
            return lastError;
        }
        else
        {
            // See what happens on the server...
            memset(&idxDesc, 0, sizeof(IDXDesc));

            idxDesc.bPrimary            = FALSE;
            idxDesc.bUnique             = TRUE;
            idxDesc.bMaintained         = TRUE;
            idxDesc.iFldsInKey          = numFields;

            strcpy(idxDesc.szName, tableName);
            strncat(idxDesc.szName, "_", DBIMAXNAMELEN);
            strncat(idxDesc.szName, "Primary", DBIMAXNAMELEN);
            idxDesc.szName[DBIMAXNAMELEN] = 0;

            for (i = 0; i < numFields; i++)
            {
                idxDesc.aiKeyFld[i]     = i+1;
            }

            lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName,
                                    Driver, &idxDesc, NULL);

            throw lastError;
        }
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;

        if (fldDescs)
        {
            delete fldDescs;
        }

        return lastError;
    }
    catch(Retcode)
    {
        if (fldDescs)
        {
            delete fldDescs;
        }
        return lastError;
    }
}

Retcode BDatabase::createIndex(const char *tableName, IDXDesc &indexDesc)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiAddIndex(hDb, NULL, (pCHAR)tableName, NULL, &indexDesc,
                            NULL);

    return lastError;
}

// Delete an index on the table; the value of indexID for the
// primary key is zero.

Retcode BDatabase::dropIndex(const char *tableName, FIELDNUMBER indexID)
{
    char        indexName[DBIMAXNAMELEN + 1];
    char        indexTagName[DBIMAXNAMELEN + 1];
    pIDXDesc    pidxDescs = NULL;
    FieldDesc   *fldDescs = NULL;
    INT16       numFields;
    INT16       numIndexes;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    try
    {
        if (!strcmp(Driver, szPARADOX))
        {
            lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver, NULL,
                                       NULL, indexID);
        }
        else if (!strcmp(Driver, szDBASE))
        {
            if (indexID == 0)
            {
                strcpy(indexName, tableName);
                strcpy(indexTagName, "Primary");
                lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                                           (pCHAR)indexName, (pCHAR)indexTagName, 0);
            }
            else 
            {
                if (!tableExists(tableName))
                {
                    return lastError;
                }

                lastError = getDescVector((pCHAR)tableName, numFields,
                                          fldDescs);
                if (lastError != DBIERR_NONE)
                {
                    throw lastError;
                }

                if (indexID > numFields)
                {
                    lastError = DBIERR_OUTOFRANGE;
                    throw lastError;
                }

                lastError = getIndexVector((pCHAR)tableName, numIndexes,
                                           pidxDescs);
                if (lastError != DBIERR_NONE)
                {
                    throw lastError;
                }

                strncpy(indexName, tableName, 3);
                indexName[3] = 0;
                strncat(indexName, fldDescs[indexID-1].fldName, 8);
                indexName[8] = 0;
                strcat(indexName, ".NDX");
                
                // Check if the index exist as an .NDX index
                if (!access(indexName, 00))
                {
                    lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName,
                                               Driver, (pCHAR)indexName, NULL,
                                               0);
                }
                else
                {
                    // Check if it is part of the production index
                    int i;
                    
                    for (i=0; i < numIndexes;i++)
                    {
                        if (!strcmp(pidxDescs[i].szTagName,
                                    fldDescs[indexID-1].fldName))
                        {
                            break;
                        }
                    }

                    // No Index Exists
                    if (i == numIndexes)
                    {
                        lastError = DBIERR_NOSUCHINDEX;
                        throw lastError;
                    }
                    
                    lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName,
                                               Driver,
                                               (pCHAR)pidxDescs[i].szName,
                                               (pCHAR)pidxDescs[i].szTagName, 0);
                }
            }
        }
        else 
        {
            if (indexID == 0)
            {
                strcpy(indexName, tableName);
                strncat(indexName, "_", DBIMAXNAMELEN);
                indexName[DBIMAXNAMELEN] = 0;
                strncat(indexName, "Primary", DBIMAXNAMELEN);
                indexName[DBIMAXNAMELEN] = 0;

                lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                                           (pCHAR)indexName, NULL, 0);
            }
            else
            {
                lastError = getDescVector((pCHAR)tableName, numFields,
                                          fldDescs);
                if (lastError != DBIERR_NONE)
                {
                    throw lastError;
                }

                if (indexID > numFields)
                {
                    lastError = DBIERR_OUTOFRANGE;
                    throw lastError;
                }

                strcpy(indexName, tableName);
                strncat(indexName, "_", DBIMAXNAMELEN);
                indexName[DBIMAXNAMELEN] = 0;
                strncat(indexName, fldDescs[indexID-1].fldName, DBIMAXNAMELEN);
                indexName[DBIMAXNAMELEN] = 0;

                lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                                           (pCHAR)indexName, NULL, 0);
            }
        }
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;

        if (fldDescs)
        {
            delete fldDescs;
        }

        if (pidxDescs)
        {
            delete pidxDescs;
        }

        return lastError;
    }
    catch(Retcode one)
    {
        if (fldDescs)
        {
            delete fldDescs;
        }

        if (pidxDescs)
        {
            delete pidxDescs;
        }

        return one;
    }

    return lastError;
}

Retcode BDatabase::dropIndex(const char *tableName, const char *indexName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    if (!strcmp(Driver, szDBASE))
    {
        CHAR    idxName[DBIMAXNAMELEN + 1];

        strcpy(idxName, tableName);
        strncat(idxName, ".MDX", DBIMAXNAMELEN);
        idxName[DBIMAXNAMELEN] = 0;
        
        lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                                   (pCHAR)idxName, (pCHAR)indexName, 0);
    }
    else
    {
        lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                                   (pCHAR)indexName, NULL, 0);
    }
                               
    return lastError;
}

Retcode BDatabase::dropIndex(const char *tableName, const char *indexName,
                             const char *indexTagName)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiDeleteIndex(hDb, NULL, (pCHAR)tableName, Driver,
                               (pCHAR)indexName, (pCHAR)indexTagName, 0);

    return lastError;
}

// Get the number of primary key fields for the table.

int BDatabase::getNumPFields(const char *tableName)
{
    TABLEHANDLE tabH        = 0;
    pIDXDesc    pidxDesc    = 0;
    char        *fldName    = 0;
    INT16       nFields     = 0;
    INT16       i;
    INT16       numIndexes  = 0;


    if (!isOpen)                                   
    {
        lastError = PXERR_DBNOTOPEN;
        return 0;
    }

    try
    {
        lastError = getIndexVector((pCHAR)tableName, numIndexes, pidxDesc);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }

        if (!strcmp(Driver, szPARADOX))
        {
            for (i = 0; i < numIndexes; i++)
            {
                if (pidxDesc[i].iIndexId == 0)
                {
                    nFields = pidxDesc[i].iFldsInKey;
                    break;
                }
            }
            
            if (i == numIndexes)
            {
                lastError = DBIERR_NOSUCHINDEX;
                throw lastError;
            }

            delete pidxDesc;
            pidxDesc = 0;
            
            return nFields;
        }
        else if (!strcmp(Driver, szDBASE))
        {
            // Search for the "Primary" index.
            for (i = 0; i < numIndexes; i++)
            {
                if (!strcmp(pidxDesc[i].szTagName, "Primary"))
                {
                    break;
                }
            }

            if (i == numIndexes)
            {
                lastError = DBIERR_NOSUCHINDEX;
                throw lastError;
            }

            // Count the number of fields in the expression
            fldName = strtok(pidxDesc[i].szKeyExp, "+");
            nFields++;
            while(fldName != NULL)
            {
                fldName = strtok(NULL, "+");
                if(fldName != NULL)
                {
                    nFields++;
                }
            }

            delete pidxDesc;
            pidxDesc = 0;

            return nFields;
        }
        else
        {
            char indexName[DBIMAXNAMELEN + 1];

            strcpy(indexName, tableName);
            strncat(indexName, "_", DBIMAXNAMELEN);
            indexName[DBIMAXNAMELEN] = 0;
            strncat(indexName, "Primary", DBIMAXNAMELEN);
            indexName[DBIMAXNAMELEN] = 0;

            // Search for the "Primary" index.
            for (i = 0; i < numIndexes; i++)
            {
                if (!strcmp(pidxDesc[i].szTagName, indexName))
                {
                    nFields = pidxDesc[i].iFldsInKey;
                    break;
                }
            }

            if (i == numIndexes)
            {
                lastError = DBIERR_NOSUCHINDEX;
                throw lastError;
            }

            delete pidxDesc;
            pidxDesc = 0;

            return nFields;
        }    
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;

        if (pidxDesc)
        {
            delete pidxDesc;
        }

        if (tabH)
        {
            DbiCloseCursor(&tabH);
        }
        return 0;
    }
    catch(int i)
    {
        if (tabH)
        {
            DbiCloseCursor(&tabH);
        }
        if (pidxDesc)
        {
            delete pidxDesc;
        }
        return i;
    }
}

// Return information on a composite secondary index.

#pragma argsused
Retcode BDatabase::getSKeyInfo(const char *indexName,
                               char *fldName,
                               int& numFields,
                               BOOL& caseSen,
                               FIELDNUMBER *fldArray,
                               FIELDNUMBER& indexID)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DBIERR_NOTSUPPORTED;

    return lastError;
}

// Given a single field secondary index file name, return
// information about the field.

#pragma argsused
Retcode BDatabase::getSKeyInfo(const char *indexName,
                               char *fldName,
                               BOOL& caseSen,
                               FIELDNUMBER& indexID)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DBIERR_NOTSUPPORTED;
    return lastError;
}

// Acquire a file lock on the network.

#pragma argsused
Retcode BDatabase::lockNetFile(const char *fileName, PXLockMode lockMode)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiAcqPersistTableLock(hDb, (pCHAR)fileName, Driver);

    return lastError;
}

// Release a previously acquired lock on a file.

#pragma argsused
Retcode BDatabase::unlockNetFile(const char *fileName, PXLockMode lockMode)
{
    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiRelPersistTableLock(hDb, (pCHAR)fileName, Driver);

    return lastError;
}

// Write the changed record buffers associated with this database to disk.


Retcode BDatabase::forceWrite(void)
{
    int i;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    for (i = 0; i < ((dbdef *)dbobj)->handleCnt; i++)
    {
        // For each table in the database
        dbdef * pdb;
        pdb = (dbdef *)dbobj;
        if (pdb->curList[i] != NULL) {                // Bug fix here (SGB)
            DbiSaveChanges(pdb->curList[i]->tabH);
        }
    }

    return lastError;
}

//  Return the name of the user causing a network locking error.

char *BDatabase::getNetErrUser(void)
{
    dbdef *dob = (dbdef *)dbobj;
    if (! isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return ("");
    }

    lastError = DbiGetErrorContext(ecUSERNAME, dob->errUser);

    return (dob->errUser);
}

Retcode BDatabase::query(const char *qryString, BCursor &answer,
                         DBIQryLang qryLang)
{
    TABLEHANDLE tabH = 0;

    if (!isOpen)
    {
        return (lastError = PXERR_DBNOTOPEN);
    }

    lastError = DbiQExecDirect(hDb, qryLang, (pCHAR)qryString, &tabH);
    if (lastError)
    {
        return lastError;
    }

    lastError = answer.attach(this, tabH);

    DbiCloseCursor(&tabH);

    return lastError;
}

const char *BDatabase::getDriver(void)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return NULL;
    }

    return Driver;
}

Retcode BDatabase::beginTransaction(hDBIXact &tranHandle, eXILType tranType)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiBeginTran(hDb, tranType, &tranHandle);

    return lastError;
}

Retcode BDatabase::endTransaction(eXEnd tranType)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiEndTran(hDb, NULL, tranType);

    return lastError;
}

Retcode BDatabase::getTransactionInfo(XInfo &tranInfo)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiGetTranInfo(hDb, NULL, &tranInfo);

    return lastError;
}

Retcode BDatabase::setProp(UINT32 prop, UINT32 value)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiSetProp((hDBIObj)hDb, prop, value);

    return lastError;
}

Retcode BDatabase::getProp(UINT32 prop, pVOID value, UINT16 maxLen,
                           UINT16 &retLen)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiGetProp((hDBIObj)hDb, prop, value, maxLen, &retLen);

    return lastError;
}

Retcode BDatabase::setDirectory(const char *Directory)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiSetDirectory(hDb, (pCHAR)Directory);

    if (lastError == DBIERR_NONE)
    {
        resetDBTableList();
        if (lastError != DBIERR_NONE)
        {
            return lastError;
        }
    }

    return lastError;
}

// Get the current working directory

Retcode BDatabase::getDirectory(char *Directory)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiGetDirectory(hDb, FALSE, (pCHAR)Directory);

    return lastError;
}

// Get the default directory

Retcode BDatabase::getDefaultDirectory(char *Directory)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    lastError = DbiGetDirectory(hDb, TRUE, (pCHAR)Directory);

    return lastError;
}

Retcode BDatabase::getIndexId(const char *tableName, const char *indexName,
                              UINT16 &indexID)
{
    TABLEHANDLE tabH = 0;
    IDXDesc     idxDesc;

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    if (strcmp(Driver, szPARADOX))
    {
        lastError = DBIERR_NOTSUPPORTED;
        return lastError;
    }

    lastError = DbiOpenIndexList(hDb, (pCHAR)tableName, Driver, &tabH);
    if (lastError != DBIERR_NONE)
    {
        throw lastError;
    }

    while (DbiGetNextRecord(tabH, dbiNOLOCK, (pBYTE)&idxDesc, NULL)
           == DBIERR_NONE)
    {
        if (!strcmp(idxDesc.szName, indexName))
        {
            indexID = idxDesc.iIndexId;
            break;            
        }
    }

    if (tabH)
    {
        DbiCloseCursor(&tabH);
    }

    return lastError;
}

Retcode BDatabase::getIndexId(const char *tableName,
                              const FIELDNUMBER *fldArray, UINT16 &indexID)
{
    TABLEHANDLE tabH = 0;
    IDXDesc     idxDesc;

    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return lastError;
    }

    if (strcmp(Driver, szPARADOX))
    {
        lastError = DBIERR_NOTSUPPORTED;
        return lastError;
    }

    lastError = DbiOpenIndexList(hDb, (pCHAR)tableName, Driver, &tabH);
    if (lastError != DBIERR_NONE)
    {
        throw lastError;
    }

    while (DbiGetNextRecord(tabH, dbiNOLOCK, (pBYTE)&idxDesc, NULL)
           == DBIERR_NONE)
    {
        for (int i = 0; i < idxDesc.iFldsInKey; i++)
        {
            if (idxDesc.aiKeyFld[i] != fldArray[i])
            {
                lastError = DBIERR_NOSUCHINDEX;
                indexID = 0;
                break;
            }
            lastError = DBIERR_NONE;
            indexID = idxDesc.iIndexId;
            break;
        }
    }

    if (tabH)
    {
        DbiCloseCursor(&tabH);
    }

    return lastError;
}

DATABASEHANDLE BDatabase::getHandle(void)
{
    if (!isOpen)
    {
        lastError = PXERR_DBNOTOPEN;
        return 0;
    }

    lastError = DBIERR_NONE;

    return hDb;
}

Retcode BDatabase::getDesc(const char *tableName, int& numFields,
                           FieldDesc far *fldDescs)
{
    FLDDesc     fldDesc;
    UINT16      fldNum;
    TABLEHANDLE tabH    = 0;

    lastError = DbiOpenFieldList(hDb, (pCHAR)tableName, Driver, FALSE,
                                 &tabH);
    if (lastError != DBIERR_NONE)
    {
        throw lastError;
    }

    for (fldNum = 0; fldNum < numFields; fldNum++)
    {
        lastError = DbiGetNextRecord(tabH, dbiNOLOCK,
                                     (pBYTE)&fldDesc, NULL);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }

        fldDescs[fldNum].fldNum      = fldDesc.iFldNum;
        strcpy(fldDescs[fldNum].fldName, fldDesc.szName);
        fldDescs[fldNum].fldType     =
                                  (PXFieldType)fldDesc.iFldType;
        fldDescs[fldNum].fldSubtype  =
                               (PXFieldSubtype)fldDesc.iSubType;
        fldDescs[fldNum].fldLen      = fldDesc.iLen;
    }

    if (tabH)
    {
        DbiCloseCursor(&tabH);
    }

    return lastError;
}

Retcode BDatabase::getIndexDesc(const char *tableName, int& numIndexes,
                                IDXDesc far *idxDescs)
{
    UINT16      idxNum;
    TABLEHANDLE tabH    = 0;

    lastError = DbiOpenIndexList(hDb, (pCHAR)tableName, Driver, &tabH);
    if (lastError != DBIERR_NONE)
    {
        throw lastError;
    }

    for (idxNum = 0; idxNum < numIndexes; idxNum++)
    {
        lastError = DbiGetNextRecord(tabH, dbiNOLOCK,
                                     (pBYTE)&(idxDescs[idxNum]), NULL);
        if (lastError != DBIERR_NONE)
        {
            throw lastError;
        }
    }

    if (tabH)
    {
        DbiCloseCursor(&tabH);
    }
    
    return lastError;
}

Retcode BDatabase::resetDBTableList()
{
    if (TableList.isOpen)
    {
        TableList.close();
        if (lastError != DBIERR_NONE)
        {
            return lastError;
        }
    }

    return openTableList(TableList, "*.*", TRUE);

}

Retcode BDatabase::openTableList(BListCursor<TBLBaseDesc> &tableList,
                                 const char *wildCard,
                                 BOOL system)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenTableList(hDb, FALSE, system, (pCHAR)wildCard,
                                 &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = tableList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openTableList(BListCursor<TBLFullDesc> &tableList,
                                 const char *wildCard,
                                 BOOL system)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenTableList(hDb, TRUE, system, (pCHAR)wildCard,
                                 &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = tableList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openFileList(BListCursor<FILEDesc> &fileList,
                                const char *wildCard)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenFileList(hDb, (pCHAR)wildCard, &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = fileList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openIndexList(BListCursor<IDXDesc> &indexList,
                                 const char *tableName,
                                 const char *tableType)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenIndexList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                 &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = indexList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openFieldList(BListCursor<FLDDesc> &fieldList,
                                 const char *tableName,
                                 const char *tableType, BOOL physicalTypes)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenFieldList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                 physicalTypes, &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = fieldList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openFamilyList(BListCursor<FMLDesc> &familyList,
                                  const char *tableName,
                                  const char *tableType)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenFamilyList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                  &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = familyList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openVchkList(BListCursor<VCHKDesc> &vchkList,
                                const char *tableName,
                                const char *tableType)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenVchkList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = vchkList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openRintList(BListCursor<RINTDesc> &rintList,
                                const char *tableName,
                                const char *tableType)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenRintList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = rintList.attach(tabH);

    return lastError;
}

Retcode BDatabase::openSecurityList(BListCursor<SECDesc> &securityList,
                                    const char *tableName,
                                    const char *tableType)
{
    TABLEHANDLE tabH;
    
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenSecurityList(hDb, (pCHAR)tableName, (pCHAR)tableType,
                                    &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = securityList.attach(tabH);

    return lastError;
}

// Redefine pure virtuals from the BDbObject class.

char * BDatabase::nameOf() const
{
    return ("BDatabase");
}

void BDatabase::printOn( ostream& os)
{
    os << nameOf() << " { DB Name = " << Alias << " Driver = " << Driver
       << "Open Status = " << (int) isOpen << " }\n";
}

// The following two routines are used to maintain a list of databases for
// the Paradox Engine. The Database Framework does not assume the presense
// of any Class library (not even Borland's container library Classlib).
// Consequently, no List class is used.

Retcode addDatabase(engdef *eng, BDatabase *db)
{
    // Add this database object to the Database vector in the Paradox Engine.

    int i;
    BDatabase **newvec;
    for (i = 0; i < eng->handleCnt; i++)
    {
        if (!eng->dbList[i])  // Determine an empty slot.
        {
            eng->dbList[i] = db;
            return DBIERR_NONE;
        }
    }
    try
    {
        // Four more handles.
        newvec = new BDatabase *[eng->handleCnt+4];
    }
    catch(xalloc)
    {
        return DBIERR_NOMEMORY;
    }
    
    for (i = 0; i < eng->handleCnt; i++)
    {
        newvec[i] = eng->dbList[i];
    }
    newvec[i++] = db;
    newvec[i] = newvec[i+1] = newvec[i+2] = 0;
    if (eng->handleCnt)
    {
        delete [] eng->dbList;                             // Delete old vector.
    }
    eng->handleCnt += 4;
    eng->dbList = newvec;

    return DBIERR_NONE;
}

void deleteDatabase(engdef *eng, BDatabase *db)
{
  // Invoke when a Database object's destructor is called.

    for (int i = 0; i < eng->handleCnt; i++)
    {
        if (eng->dbList[i] == db)
        {
            eng->dbList[i] = 0;
            return;
        }
    }
}

// Add key map defined by an application to the Paradox Engine
// object; the key map results in a compound index's index
// handle that is valid only within this instance of the Engine.
// Paradox Engine semantics require the key map definition for
// every session. This information is required for some of the
// search functions provided in BCursor class.

Retcode addKeymap(engdef *eng, const char *tableName, int numFields,
          int indexHandle, const int *fldArray)
{
    int i,j;
    keyMapdef *newmap = 0;

    // See if an entry already exists.

    try
    {
        for (i = 0; i < eng->compIdxCnt; i++)
        {
            if (eng->keymap[i].indexId == indexHandle &&
                eng->keymap[i].keyCnt  == numFields   &&
                ! strcmp(eng->keymap[i].tableName, tableName))
            {
                break;
            }
        }
        if (i == eng->compIdxCnt)              // Add a new entry.
        {
            newmap = new keyMapdef[eng->compIdxCnt+1];
            for (j = 0; j < eng->compIdxCnt; j++)
            {
                newmap[j] = eng->keymap[j];
            }
            if (eng->compIdxCnt)             // Free old structure if
            {
                delete [] eng->keymap;          // this is not the first.
                eng->keymap = 0;
            }
            eng->keymap = newmap;
            eng->compIdxCnt += 1;
        }
        else
        {
            // Delete allocated memory in old structure for the index.

            delete eng->keymap[i].tableName;
            eng->keymap[i].tableName = 0;
            delete [] eng->keymap[i].fieldArray;
            eng->keymap[i].fieldArray = 0;
        }

        // Add information at index position i. The memory allocated
        // here will be freed when the engine is closed.

        eng->keymap[i].indexId = indexHandle;
        eng->keymap[i].keyCnt  = numFields;
        // Need to know that value is 0 on failure
        eng->keymap[i].tableName = 0;
        eng->keymap[i].tableName  = new char[strlen(tableName)+1];
        strcpy(eng->keymap[i].tableName,tableName);
        // Need to know that value is 0 on failure
        eng->keymap[i].fieldArray = 0;
        eng->keymap[i].fieldArray = new int[numFields];
        for (j = 0; j < numFields; j++)
        {
            eng->keymap[i].fieldArray[j] = fldArray[j];
        }

        return DBIERR_NONE;
    }
    catch(xalloc)
    {
        if (newmap)
        {
            delete newmap;
        }

        if (eng->keymap[i].tableName)
        {
            delete eng->keymap[i].tableName;
        }

        if (eng->keymap[i].fieldArray)
        {
            delete eng->keymap[i].fieldArray;
        }

        return DBIERR_NOMEMORY;
    }
}

Retcode getDesc(TABLEHANDLE tabH, int& numFields, FieldDesc far *fldDescs)
{
    pFLDDesc    pfldDesc = 0;
    Retcode     rslt;
    int         i;

    try
    {
        pfldDesc = new FLDDesc[numFields];

        rslt = DbiGetFieldDescs(tabH, pfldDesc);
        if (rslt != DBIERR_NONE)
        {
            throw rslt;
        }

        for (i=0; i < numFields; i++)
        {
            fldDescs[i].fldNum      = pfldDesc[i].iFldNum;
            strcpy(fldDescs[i].fldName, pfldDesc[i].szName);
            fldDescs[i].fldType     = (PXFieldType)pfldDesc[i].iFldType;
            fldDescs[i].fldSubtype  = (PXFieldSubtype)pfldDesc[i].iSubType;
            fldDescs[i].fldLen      = pfldDesc[i].iLen;
        }

        delete pfldDesc;
        pfldDesc = 0;

        return DBIERR_NONE;
    }
    catch(xalloc)
    {
        if (pfldDesc)
        {
            delete pfldDesc;
        }
        return DBIERR_NOMEMORY;
    }
    catch(Retcode)
    {
        return rslt;
    }
}

//=====================================================================
//  Code:   GetOptionalParams(pCHAR szDriver, UINT16 *iFields,
//                            pFLDDesc pfldDesc, pCHAR szData);
//
//  Input:  szDriver    - Name of the driver
//          iFields     - Used to return the number of fields
//          pfldDesc    - Returns the descriptors of the fields.
//                          Memory for this variable must be allocated by
//                          the calling calling.
//          szData      - Contains the values of the fields. Memory for
//                          this variable must be allocated by the calling
//                          function.
//
//  Return: DBIResult - success of the function
//
//  Description:
//          This function is used to return the available optional
//          parameters when creating a table of the given type. This
//          information includes the number of optional parameters
//          (iFields), an array of field descriptors describing those
//          parameters, and the default values for the parameters.
//=====================================================================
Retcode                          
GetOptionalParams (pCHAR szDriver, UINT16 *iFields,
                   pFLDDesc pfldDesc, pCHAR szData)
{
    DBIResult   rslt;       // Return value from IDAPI functions
    TABLEHANDLE tabH = 0;   // Handle to the Cursor
    pCHAR       szNode = 0; // String which contains the name of the node
    pCFGDesc    CfgDesc = 0;// Configuration Descriptor
    CHAR        szCfgPath[DBIMAXPATHLEN + 1]; // Maximum length of the path
                            // in the configuration file.
    UINT16      iOffset = 0;

    try
    {
        // Set up the option to get from the Configuration File
        strcpy(szCfgPath, "\\DRIVERS\\");
        strcat(szCfgPath, szDriver);
        strcat(szCfgPath, "\\TABLE CREATE\\");
    
        // Open the Configuration file - returns the configuration options
        // for the current level in a schema table referenced by tabH.
        rslt = DbiOpenCfgInfoList(NULL, dbiREADONLY, cfgPersistent,
                                  szCfgPath, &tabH);
        if (rslt != DBIERR_NONE)
        {
            throw rslt;
        }

        // Allocate resources
        szNode = new CHAR[256];
        CfgDesc = new CFGDesc;

        *iFields = 0;
        // Process all nodes
        //   Get the next record in the table - contains the next option
        //   of the given level in the tree.
        while (DbiGetNextRecord(tabH, dbiNOLOCK, (pBYTE)CfgDesc, NULL)
               == DBIERR_NONE)
        {
            pfldDesc[*iFields].iFldNum = *iFields + 1;
            pfldDesc[*iFields].iFldType = CfgDesc->iDataType;
            pfldDesc[*iFields].iUnits1 = DBIMAXSCFLDLEN - 1;
            pfldDesc[*iFields].iLen = DBIMAXSCFLDLEN;
            strcpy(pfldDesc[*iFields].szName, CfgDesc->szNodeName);
            if (!strcmp(szDriver, szDBASE))
            {
                pfldDesc[*iFields].iOffset = DBIMAXSCFLDLEN * (*iFields);
            }

            // Display the remaining information
            //    Display the Description of the node
            sprintf(szNode, "%s", CfgDesc->szValue);
            strcpy(&szData[iOffset], szNode);
            iOffset += pfldDesc[*iFields].iLen;
            (*iFields)++;
        }

        // Clean up
        delete szNode;
        szNode = 0;
        delete CfgDesc;
        CfgDesc = 0;
        
        rslt = DbiCloseCursor(&tabH);
        if (rslt != DBIERR_NONE)
        {
            throw rslt;
        }
        tabH = 0;
        
        return rslt;
    }
    catch(xalloc)
    {
        if (szNode)
        {
            delete szNode;
        }

        if (CfgDesc)
        {
            delete CfgDesc;
        }

        return rslt;
    }
    catch(Retcode)
    {
        if (szNode)
        {
            delete szNode;
        }

        if (CfgDesc)
        {
            delete CfgDesc;
        }

        return rslt;
    }
}

#pragma
static void makeDBInst()
{
    BListCursor<TBLBaseDesc> bListBaseTableDesc;
    BListCursor<TBLFullDesc> bListExtendedTableDesc;
    BListCursor<FILEDesc> bListFileDesc;
    BListCursor<IDXDesc> bListIndexDesc;
    BListCursor<FLDDesc> bListFieldDesc;
    BListCursor<FMLDesc> bListFamilyDesc;
    BListCursor<VCHKDesc> bListVchkDesc;
    BListCursor<RINTDesc> bListRintDesc;
    BListCursor<SECDesc> bListSecurityDesc;
}

