******************************************************************************
*
* FUNCTION:    But_Last - Returns All But Last Token From String Of Delimited Tokens
*
* DESCRIPTION:  Given a string of delimited Tokens, and the delimiting
* character sequence, this function will remove the Last Token from the
* string, returning the Rest of the Tokens of the string.  The delimiting
* character sequence separating the Last Token from the rest of the string
* is also removed.  If "SkipNullToken" is true, then all consecutive
* occurrences of the delimiting character sequence preceeding the last token
* are removed.
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Token_Str    I  C A line of text containing delimited Token data.
* Delimiter    I  C The Token delimiting character sequence.
* SkipNullTokenI  B True if there must be data within a Token for it to be
*                   counted as a Token.  A "Null" Token would be defined by
*                   two or more consecutive occurences of the delimiting 
*                   character sequence within "Token_Str".  
*                   This is an optional parameter defaulting to false.
*
* MODIFICATION HISTORY
* --------------------
* 03/26/89  PAT WEISSER -- Created.
*
******************************************************************************
FUNCTION But_Last
PARAMETERS Token_Str, Delimiter, SkipNullToken
PRIVATE Token_Str_Pos, Start_Of_Token, Str_Buff, Last_Delim_Pos, Last_Char_Pos

SkipNullToken = IIF( TYPE( "SkipNullToken" ) != "L", .F., SkipNullToken )

IF( SkipNullToken )
  * Ignore possible block of consecutive delimiters following last Token:

  Token_Str_Pos = LEN( Token_Str ) - LEN( Delimiter ) + 1

  DO WHILE( SUBSTR( Token_Str, Token_Str_Pos, LEN( Delimiter ) ) == Delimiter )
    Token_Str_Pos = Token_Str_Pos - LEN( Delimiter )

    IF( Token_Str_Pos <= 0 )
      RETURN( "" )
    ENDIF

  ENDDO

  * "Token_Str_Pos" will be pointing to the last "LEN( Delimiter )" characters
  * of the last token of "Token_Str".  Need to advance "Token_Str_Pos" to the
  * last character of the last token of "Token_Str":

  Token_Str_Pos = Token_Str_Pos + LEN( Delimiter ) - 1

ELSE
  Token_Str_Pos = LEN( Token_Str )
ENDIF


* "Token_Str_Pos" should now be pointing to the last character of the last
* token of "Token_Str".

Str_Buff = SUBSTR( Token_Str, 1, Token_Str_Pos )

Last_Delim_Pos = RAT( Delimiter, Str_Buff )

IF( Last_Delim_Pos == 0 )
  * The "Last" token is also the "First" and only token.
  RETURN( "" )
ENDIF

IF( SkipNullToken )
  * Must not include any trailing delimiter character sequences:
  DO WHILE( SUBSTR( Str_Buff, Last_Delim_Pos, LEN( Delimiter ) ) == Delimiter )
    Last_Delim_Pos = Last_Delim_Pos - LEN( Delimiter )

    IF( Last_Delim_Pos <= 0 )
      * Then the "Last" token was the only token.  It was just preceeded by
      * one or more consecutive occurrences of the delimiting character sequence.
      RETURN( "" )
    ENDIF
  ENDDO

  Last_Char_Pos = Last_Delim_Pos + LEN( Delimiter ) - 1
ELSE
  Last_Char_Pos = Last_Delim_Pos - 1
ENDIF

RETURN( SUBSTR( Str_Buff, 1, Last_Char_Pos ) )





******************************************************************************
*
* FUNCTION:    Last_Token - Returns Last Token From String Of Delimited Tokens
*
* DESCRIPTION:  Given a string of delimited Tokens, the delimiting character
* sequence, and whether or not to ignore null tokens, this function will find 
* and return the last Token in the string.  If the list is empty, a null 
* string will be returned.
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Token_Str    I  C A line of text containing delimited Token data.
* Delimiter    I  C The Token delimiting character sequence.
* SkipNullTokenI  B True if there must be data within a Token for it to be
*                   counted as a Token.  A "Null" Token would be defined by
*                   two or more consecutive occurences of the delimiting 
*                   character sequence within "Token_Str".  
*                   This is an optional parameter defaulting to false.
*
* MODIFICATION HISTORY
* --------------------
* 03/26/89  PAT WEISSER -- Created.
*
******************************************************************************
FUNCTION Last_Token
PARAMETERS Token_Str, Delimiter, SkipNullToken
PRIVATE Token_Str_Pos, Start_Of_Token, Str_Buff

IF( !( Delimiter $ Token_Str ) )
  * Delimiter character sequence not in Token_Str at all, 1 or less Tokens exist:
  RETURN( Token_Str )
ENDIF

SkipNullToken = IIF( TYPE( "SkipNullToken" ) != "L", .F., SkipNullToken )

IF( SkipNullToken )
  * Ignore possible block of consecutive delimiters following last Token:

  Token_Str_Pos = LEN( Token_Str ) - LEN( Delimiter ) + 1

  DO WHILE( SUBSTR( Token_Str, Token_Str_Pos, LEN( Delimiter ) ) == Delimiter )
    Token_Str_Pos = Token_Str_Pos - LEN( Delimiter )

    IF( Token_Str_Pos <= 0 )
      RETURN( "" )
    ENDIF

  ENDDO

  * "Token_Str_Pos" will be pointing to the last "LEN( Delimiter )" characters
  * of the last token of "Token_Str".  Need to advance "Token_Str_Pos" to the
  * last character of the last token of "Token_Str":

  Token_Str_Pos = Token_Str_Pos + LEN( Delimiter ) - 1

ELSE
  Token_Str_Pos = LEN( Token_Str )
ENDIF


* "Token_Str_Pos" should be now be pointing to the last character of the last
* token of "Token_Str".

Str_Buff = SUBSTR( Token_Str, 1, Token_Str_Pos )

Start_Of_Token = RAT( Delimiter, Str_Buff ) + LEN( Delimiter )

IF( Start_Of_Token == LEN( Delimiter ) )
  * The "Last" token is also the "First".
  RETURN( Str_Buff )
ENDIF

RETURN( SUBSTR( Str_Buff, Start_Of_Token ) )




******************************************************************************
*
* FUNCTION:    But_First - Returns All But First Token From String Of Delimited Tokens
*
* DESCRIPTION:  Given a string of delimited Tokens, the delimiting character
* sequence, and whether or not to ignore null tokens, this function will 
* remove the first Token from the string, returning the Rest of the Tokens of 
* the string.  The delimiting character sequence separating the first Token 
* from the rest of the string is NOT returned.
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Token_Str    I  C A line of text containing delimited Token data.
* Delimiter    I  C The Token delimiting character sequence.
* SkipNullTokenI  B True if there must be data within a Token for it to be
*                   counted as a Token.  A "Null" Token would be defined by
*                   two or more consecutive occurences of the delimiting 
*                   character sequence within "Token_Str".  Also, the first Token
*                   would be considered empty if the first occurrence of the
*                   delimiting character sequence was not preceeded by any
*                   characters at all.
*                   This is an optional parameter defaulting to false.
*
* EXTERNAL LIBRARY REFERENCES
* ---------------------------
* Get_Token()
*
* MODIFICATION HISTORY
* --------------------
* 12/31/88  PAT WEISSER -- Created.
* 03/26/89  PAT WEISSER -- Incorportated use of "Get_Token()" function.
*
******************************************************************************
FUNCTION But_First
PARAMETERS Token_Str, Delimiter, SkipNullToken
PRIVATE Start_Of_Token2

IF( !( Delimiter $ Token_Str ) )
  * Delimiter character sequence not in Token_Str at all, 1 or less Tokens exist:
  RETURN( "" )
ENDIF

SkipNullToken = IIF( TYPE( "SkipNullToken" ) != "L", .F., SkipNullToken )

Start_Of_Token2 = 0
Get_Token( Token_Str, 2, Delimiter, SkipNullToken, @Start_Of_Token2 )

IF( Start_Of_Token2 == 0 )
  RETURN( "" )
ELSE
  RETURN( SUBSTR( Token_Str, Start_Of_Token2 ) )
ENDIF




******************************************************************************
*
* FUNCTION   :  Get_Token -- Returns The Nth Delimited Token From "Token_Str"
*
* DESCRIPTION:  Given a character string of delimited Tokens, the number of 
* the desired Token, the delimiting character sequence, and whether or not to 
* ignore null tokens, this function will extract the Nth Token from the string
* and return it.  A null string will be returned if the end of the token 
* string is encountered before the desired token is encountered.
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Token_Str    I  C Character string containing delimited Tokens.
* Sought_Token I  N Number of desired Token.
* Delimiter    I  C Token delimiting character sequence.
* SkipNullTokenI  B True if there must be data within a token for it to be
*                   counted as a token.  A "Null" token would be defined by
*                   two or more consecutive occurences of the delimiting 
*                   character sequence within "Token_Str".  Also, the first token
*                   would be considered empty if the first occurrence of the
*                   delimiting character sequence was not preceeded by any
*                   characters at all.
*                   This is an optional parameter defaulting to false.
*@Tok_Start_PosO  N This is an optional return parameter.  The starting character
*                   position in "Token_Str" of the sought token is returned.
*                   If the sought token is not found, then zero is returned.
*
* MODIFICATION HISTORY
* --------------------
* 03/01/89  PAT WEISSER    Created.
* 03/26/89  PAT WEISSER -- Added the "SkipNullToken" parameter, and the
*           "Tok_Start_Pos" parameter.
* 01/04/90  PAT WEISSER -- Modified to allow for the case of first token in
*           "Token_Str" being null.
*
******************************************************************************
FUNCTION Get_Token
PARAMETERS Token_Str, Sought_Token, Delimiter, SkipNullToken, Tok_Start_Pos
PRIVATE Num_Tokens, Token_Str_Pos, Next_Token_Len

* Token_Str_Pos = 1
SkipNullToken = IIF( TYPE( "SkipNullToken" ) != "L", .F., SkipNullToken )

IF( TYPE( "Tok_Start_Pos" ) != "N" )
  PRIVATE Tok_Start_Pos
ENDIF
Tok_Start_Pos = 0

IF( !SkipNullToken .AND. ( SUBSTR( Token_Str, 1, LEN( Delimiter ) ) == Delimiter ) )
  * Then the first field is empty, but we need to count it:
  Num_Tokens = 1
  Token_Str_Pos = LEN( Delimiter ) + 1

ELSE
  Num_Tokens = 0
  Token_Str_Pos = 1

ENDIF


IF( EMPTY( Token_Str ) )

  RETURN( "" )

ENDIF


IF( !( Delimiter $ Token_Str ) )
  * Then the string argument contains no Delimiter character sequences -- 
  * we are dealing with 1 Token.

  IF( Sought_Token = 1 )
    * Then Return the token string itself:
    Tok_Start_Pos = 1
    RETURN( Token_Str )

  ENDIF

  RETURN( "" )

ENDIF


* If this point is reached, the "Token_Str" argument contains at least one
* Delimiting character sequence:

DO WHILE( Token_Str_Pos <= LEN( Token_Str ) )

  IF( SkipNullToken )
    * Ignore possible block of consecutive delimiters preceeding Current Token,
    * but take care to update "Tok_Start_Pos":

    DO WHILE( SUBSTR( Token_Str, Token_Str_Pos, LEN( Delimiter ) ) == Delimiter )
      Token_Str_Pos = Token_Str_Pos + LEN( Delimiter )
      Tok_Start_Pos = Token_Str_Pos

      IF( Token_Str_Pos > LEN( Token_Str ) )
        Tok_Start_Pos = 0
        RETURN( "" )
      ENDIF

    ENDDO

  ENDIF

  Next_Token_Len = AT( Delimiter, SUBSTR( Token_Str, Token_Str_Pos ) ) - 1
  Num_Tokens = Num_Tokens + 1

  IF( Num_Tokens = Sought_Token )
    * We've found the Token to return, must extract it from "Token_Str":

    Tok_Start_Pos = Token_Str_Pos

    DO CASE

      CASE( Next_Token_Len == -1 )
        * Then the token sought is the last in the string of tokens.
  
        RETURN( SUBSTR( Token_Str, Token_Str_Pos ) )


      CASE( Next_Token_Len == 0 )
        * The sought token is null (an EMPTY string).

        RETURN( "" )


      OTHERWISE
        * Extract token from its place in the string:

        RETURN( SUBSTR( Token_Str, Token_Str_Pos, Next_Token_Len ) )


    ENDCASE

  ENDIF


  IF( Next_Token_Len == -1 )
    * No further occurences of the delimiting character sequence were found.
    EXIT
  ENDIF

  Token_Str_Pos = Token_Str_Pos + Next_Token_Len + LEN( Delimiter )

ENDDO

Tok_Start_Pos = 0

RETURN( "" )





******************************************************************************
*
* FUNCTION   :  Token_Count -- Counts Number Of Delimited Tokens In A String
*
* DESCRIPTION:  Given a string of delimited Tokens, the delimiting character
* sequence, and whether or not to ignore null tokens, this function will 
* return the number of delimited Tokens in the character string.
*
* PARAMETERS  I/O T Purpose
* ----------  --- - -------
* Token_Str    I  C Character string to count the number of "Tokens" in.
* Delimiter    I  C The Token delimiting character sequence.
* SkipNullTokenI  B True if there must be data within a Token for it to be
*                   counted as a Token.  A "Null" Token would be defined by
*                   two or more consecutive occurences of the delimiting 
*                   character sequence within "Token_Str".  A "Null" Token is
*                   also defined by the first or last character(s) of "Token_Str"
*                   being equal to the Delimiting character sequence.
*                   This is an optional parameter defaulting to false.
*
* EXTERNAL LIBRARY REFERENCES
* ---------------------------
* Get_Token(), But_First()
*
* MODIFICATION HISTORY
* --------------------
* 08/09/88  PAT WEISSER    Created.
* 01/04/89  PAT WEISSER -- Added the "SkipNullToken" parameter.
*
******************************************************************************
FUNCTION Token_Count
PARAMETER Token_Str, Delimiter, SkipNullToken
PRIVATE Num_Tokens

SkipNullToken = IIF( TYPE( "SkipNullToken" ) != "L", .F., SkipNullToken )

Num_Tokens = 0

IF( SkipNullToken )

  IF( EMPTY( Token_Str ) )
    * Ignore the single null token, return a count of 0:
    RETURN( 0 )

  ENDIF

  IF( !EMPTY( Get_Token( Token_Str, 1, Delimiter, SkipNullToken ) ) )
    * We have at least one non-null token.  Since the "But_First()" function
    * loop will count all but the first non-null token, we must increment the
    * token counter once here:

    Num_Tokens = 1

  ENDIF

ELSE

  IF( !( Delimiter $ Token_Str ) )
    * There is a single (possibly null) token to return the count for:

    RETURN( 1 )

  ENDIF


  * Since the "But_First()" function
  * loop will count all but the first non-null token, we must increment the
  * token counter once here:

  Num_Tokens = 1


  IF( SUBSTR( Token_Str, -( LEN( Delimiter ) ) ) == Delimiter )
    * Then the last token field is null (empty).  Since the terminal condition
    * for the counting loop is based on the "But_First()" function returning a
    * null string, it would not detect this last field.  We must therefore
    * increment the Token Counter here:

    Num_Tokens = Num_Tokens + 1

  ENDIF


ENDIF


Work_Str = Token_Str  && Make local copy in case user passed by reference.


DO WHILE( .T. )

  Work_Str = But_First( Work_Str, Delimiter, SkipNullToken )

  IF( EMPTY( Work_Str ) )
    EXIT
  ENDIF

  Num_Tokens = Num_Tokens + 1

ENDDO

RETURN( Num_Tokens )


