#include <clipdefs.h>
#include <filesys.h>
#include <extend.h>
#include <set.h>
#include <errsys.h>
#include <error.ch>

#define ESC 			"\x1B"
#define RLE_MASK 		0xC0
#define LENGTH_MASK	0x3F

#define E_PCX_ARG			1
#define E_PCX_MEMORY		2
#define E_PCX_FILE		3
#define E_PCX_BADPCX		4
#define E_PCX_NOTMONO   5

struct {
	char manufacturer;
	char version;
	char encoding;
	unsigned char bitsPerPixel;
	int x0, y0, x1, y1;
	int xDPI, yDPI;
	struct {
		char red;
		char green;
		char blue;
	} pal16[16];
	char reserved;
	char planes;
	int bytesPerLine;
	int paletteInfo;
	int xScreenSize, yScreenSize;
	char filler[54];
} *PCXHeader = 0;

static BYTE *decodeBuffer = NULL;
static BYTE *printBuffer = NULL;
static int fileHandle = 0;

static BYTE BitMask[8] =
	{ 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };


static int FileOpen(char *fname, int mode)
{
	ERROR e;
	BOOL retry = FALSE;
	int handle;
	memset(&e, 0, sizeof(ERROR));
	e.tries = 1;
	e.severity = 2;
	e.flags = EF_CANRETRY | EF_CANDEFAULT;
	e.subSystem = "PCX";
	e.genCode = EG_OPEN;
	e.subCode = 1;
	do {
		handle = _topen(fname, mode);
		if (handle != -1)
			retry = FALSE;
		else
			if (_eError(&e) == 1) {
				retry = TRUE;
				e.tries++;
			}
			else
				retry = FALSE;
	} while (retry);
	return handle;
}


static int OpenPCX(char *fname)
{
	fileHandle = FileOpen(fname, FS_READ);
	if (fileHandle == -1) {
		return 0;
	}
	if (_tread(fileHandle, PCXHeader, sizeof *PCXHeader) != sizeof *PCXHeader) {
		_tclose(fileHandle);
		_eErrorp(EG_READ, "PCX", E_PCX_FILE, EF_CANDEFAULT);
		return 0;
	}
	if (PCXHeader->manufacturer != 0x0A) {
		_tclose(fileHandle);
		_eErrorp(EG_CORRUPTION, "PCX", E_PCX_BADPCX, EF_CANDEFAULT);
		return 0;
	}
	if (PCXHeader->bitsPerPixel != 1) {
		_tclose(fileHandle);
		_eErrorp(EG_UNSUPPORTED, "PCX", E_PCX_NOTMONO, EF_CANDEFAULT);
		return 0;
	}
	return 1;
}


static void ClosePCX()
{
	_tclose(fileHandle);
}


static BYTE ReadByte()
{
	BYTE b;
	_tread(fileHandle, &b, 1);
	return b;
}


static void DecodeLine()
{
	BYTE b, *bufPtr = decodeBuffer;
	short n = PCXHeader->bytesPerLine;
	while (n > 0)
	{
		b = ReadByte();
		if ((b & RLE_MASK) == RLE_MASK) {
			short len = b & LENGTH_MASK;
			b = ReadByte();
			if ( (n -= len) < 0 ) {
				_eErrorp(EG_CORRUPTION, "PCX", E_PCX_BADPCX, EF_CANDEFAULT);
				return;
			}
			memset(bufPtr, b, len);
			bufPtr += len;
		} else {
			n--;
			*bufPtr++ = b;
		}
	}
}


static int PrinterInit()
{
	int handle;

	if (_Set.printer.isOpen)
		handle = _Set.printer.handle;
	else
		handle = 4;

	// set the printer to raw mode
	if ( _tisdevice(handle) )
		_tdevraw(handle, 1);

	// set line spacing to 8/60ths
	_Set.PrintOut( ESC "A\x8", 3 );
	return 1;
}


static void PrinterEnd()
{
	int handle;
	// set standard 6lpi spacing
	_Set.PrintOut( ESC "2\r\n", 4 );

	// restore "cooked" mode
	if (_Set.printer.isOpen)
		handle = _Set.printer.handle;
	else
		handle = 4;
	if ( _tisdevice(handle) )
		_tdevraw(handle, 0);
}


static void PrintGraphicBuffer(BYTE *buffer, short len)
{
	WORD columns = len / 3;
	_Set.PrintOut(ESC "*\x27", 3);
	_Set.PrintOut((char *)&columns, 2);
	_Set.PrintOut(buffer, len);
	_Set.PrintOut("\r\n", 2);
}


void PrintImage(void)
{
	short width, height, printBytes;
	short x, y;

	width = PCXHeader->x1 - PCXHeader->x0 + 1;
	height = PCXHeader->y1 - PCXHeader->y0 + 1;
	printBytes = width * 3;
	decodeBuffer = _xalloc(PCXHeader->bytesPerLine);
	printBuffer = _xalloc(printBytes);

	if (!decodeBuffer || !printBuffer) {
		if (decodeBuffer) _xfree(decodeBuffer);
		if (printBuffer) _xfree(printBuffer);
		_eErrorp(EG_MEM, "PCX", E_PCX_MEMORY, EF_CANDEFAULT);
		return;
	}

	memset(printBuffer, 0, printBytes);

	for (y = 0; y < height; y++)
	{
		DecodeLine();

		for (x = 0; x < width; x++)
			if ( !(decodeBuffer[x / 8] & BitMask[x & 7]) )
				printBuffer[(y % 24) / 8 + x * 3] |= BitMask[y & 7];

		if ((y % 24) == 23) {
			PrintGraphicBuffer(printBuffer, printBytes);
			memset(printBuffer, 0, printBytes);
		}
	}

	if ((y % 24) != 0)
		PrintGraphicBuffer(printBuffer, printBytes);

	_xfree(decodeBuffer);
	_xfree(printBuffer);
}


void pascal PRINTPCX(void)
{
	char *fname;

	if (_pcount != 1) {
		_eArg("PCX", E_PCX_ARG, EF_CANDEFAULT);
		return;
	}

	if ( !PrinterInit() ) return;

	fname = _parc(1);
	if (!fname) {
		_eArg("PCX", E_PCX_ARG, EF_CANDEFAULT);
		return;
	}

	PCXHeader = _xalloc(sizeof *PCXHeader);
	if (!PCXHeader) {
		_eErrorp(EG_MEM, "PCX", E_PCX_MEMORY, EF_CANDEFAULT);
		return;
	}

	if (!OpenPCX(fname)) return;

	PrintImage();

	ClosePCX();
	PrinterEnd();
}























