/***
*	Errorsys.prg
*	Standard Clipper 5.0 error handler
*	Copyright (c) 1990 Nantucket Corp.  All rights reserved.
*
*	Compile:  /m/n/w
*/

#include "error.ch"
#include "fileio.ch"

// put messages to STDERR
#command ? <list,...>   =>  ?? Chr(13) + Chr(10) ; ?? <list>
#command ?? <list,...>  =>  OutErr(<list>)


// used below
#define NTRIM(n)		( LTrim(Str(n)) )



/***
*	ErrorSys()
*
*	Note:  automatically executes at startup
*/

proc ErrorSys()
	ErrorBlock( {|e| DefError(e)} )
return

/***
*	DefError()
*/
static func DefError(e)
local i, scratch, PrePrint, PreDevice, PreConsole, eMessage:="", cMessage
local handle, OpenError, lkey, eTitle
	PrePrint:=set(_SET_PRINTER)
	set print off
	PreDevice:=set(_SET_DEVICE)
	set device to screen
	PreConsole:=set(_SET_CONSOLE)
	set console on

	// by default, division by zero yields zero
	if ( e:genCode == EG_ZERODIV )
		set(_SET_PRINTER,PrePrint)
		set(_SET_DEVICE,PreDevice)
		set(_SET_CONSOLE,PreConsole)
        return (0)
	end
	// for network open error, set NETERR() and subsystem default
	if ( e:genCode == EG_OPEN .and. e:osCode == 32 .and. e:canDefault )
		NetErr(.t.)
		set(_SET_PRINTER,PrePrint)
		set(_SET_DEVICE,PreDevice)
		set(_SET_CONSOLE,PreConsole)
        return (.f.)                                    // NOTE
	end
	// for lock error during APPEND BLANK, set NETERR() and subsystem default
	if ( e:genCode == EG_APPENDLOCK .and. e:canDefault )
		NetErr(.t.)
		set(_SET_PRINTER,PrePrint)
		set(_SET_DEVICE,PreDevice)
		set(_SET_CONSOLE,PreConsole)
        return (.f.)                                    // NOTE
	end

	// If printer is offline wait until it is put back online
	if (e:gencode == EG_PRINT .and. !isprinter())
        ?? chr(7)
		scratch:=xMessage(10,10,"Printer is offline, correct or press [esc] to abort...","w/r,w+/r",.t.)
		while !isprinter()
			inkey()
			lkey:=lastkey()
			if(lkey == 27)
				break
			endif
		end
		set(_SET_PRINTER,PrePrint)
		set(_SET_DEVICE,PreDevice)
		set(_SET_CONSOLE,PreConsole)
        win_rest(scratch)
		return(.t.)
	endif

	if (e:oscode == 19)
		?? chr(7)
		eTitle:=EEMessage(e:oscode)
		eMessage:="The disk being used is write protected.  Disable the write protection "+;
		"to continue or press [esc] to abort."
		if (lastkey() == 27)
		set(_SET_PRINTER,PrePrint)
		set(_SET_DEVICE,PreDevice)
		set(_SET_CONSOLE,PreConsole)
            break
		else
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            return(.t.)
		endif
	endif

	if (e:oscode == 21)
		?? chr(7)
		eTitle:=EEMessage(e:oscode)
		eMessage:="The drive being used is not ready close the floppy door to continue "+;
		"or press [esc] to abort."
		if (lastkey() == 27)
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            break
		else
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            return(.t.)
		endif
	endif

	if (e:oscode == 25)
		?? chr(7)
		eTitle:=EEMessage(e:oscode)
		eMessage:="The operating system has reported a seek error.  This generally means "+;
		"there is a bad block on the disk.  Use a disk fixing utility such as "+;
		"The Norton Disk Doctor to fix it.  Select OK to retry or [esc] to abort."
		if (lastkey() == 27 )
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            break
		else
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            return(.t.)
		endif
	endif

	if (e:oscode == 27)
		?? chr(7)
		eTitle:=EEMessage(e:oscode)
		eMessage:="The operating system has reported a 'sector not found' error.  This generally means "+;
		"there is a physical problem with the disk.  Use a disk fixing utility such as "+;
		"The Norton Disk Doctor to fix it.  Select OK to retry or [esc] to abort."
		if (lastkey() == 27 )
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            break
		else
			set(_SET_PRINTER,PrePrint)
			set(_SET_DEVICE,PreDevice)
			set(_SET_CONSOLE,PreConsole)
            return(.t.)
		endif
	endif

	do case
		case (e:gencode == EG_CREATE)
			eMessage:="The file "+e:filename+" could not be created for the "+;
			"following reason: "+EEMessage(e:oscode)+".  This file is required for "+;
			"further processing.  Please correct to continue.  "
		case (e:gencode == EG_OPEN)
			eMessage:="The file "+e:filename+" could not be opened for the "+;
			"following reason: "+EEMessage(e:oscode)+".  This file is required for "+;
			"further processing.  Please correct to continue.  "
		case (e:gencode == EG_WRITE)
			eMessage:="Unable to write to "+e:filename+" for the following reason: "+;
			".  Please correct to continue.  "
		case (e:gencode == EG_READ)
			eMessage:="Unable to read "+e:filename+" for the following reason: "+;
			". Please correct to continue.  "
	endcase

	do case
		case (e:oscode == 4)
			eTitle:=EEMessage(e:oscode)
			eMessage:=eMessage+"The operating system has run out of file handles.  This can "+;
			"usually be fixed by enlarging the 'Files=' variable in the 'Config.sys' file which "+;
			"is located in the root directory of the boot drive.  If this file doesn't exitst "+;
			"one should be created.  "
		case (e:oscode == 8)
			eTitle:=EEMessage(e:oscode)
			eMessage:=eMessage+"The computer does not have enough memory to continue.  This "+;
			"error should not occur on a computer with 640k of base memory.  If it does report this "+;
			"error to the software author, otherwise, increase the base memory to continue. "
		otherwise
			eTitle:="Software Error"
			eMessage:=eMessage+"An error has occurred within the software.  A description of "+;
			"the problem has been saved.  "
    endcase
	eMessage:=eMessage+"If you are unable to fix this problem you may "+;
	"contact EMSoft for technical support.  The correct address and phone number "+;
	"should be in your online documentation which is accessible by pressing [F1]."

	? chr(7)
	Message(10,10,,,eTitle,eMessage,"w/r,w+/r",.t.)
	if file("program.err")
		handle:=fopen("program.err",66)
		fseek(handle,0,2)
	else
		handle:=fcreate("program.err",0)
	endif
	OpenError:=ferror()
	if (OpenError == 0)
		fwrite(handle,dtoc(date())+"   "+time()+chr(13)+chr(10))

		fwrite(handle,if( e:severity > ES_WARNING, "Error: ", "Warning: " ))
		if(e:oscode > 0)
			fwrite(handle,EEMessage(e:oscode)+chr(13)+chr(10))
		endif
        // add subsystem name if available
		if ( ValType(e:subsystem) == "C" )
			fwrite(handle,e:subsystem())
		else
			fwrite(handle,"???")
		end

		// add subsystem's error code if available
		if ( ValType(e:subCode) == "N" )
			fwrite(handle,("/" + NTRIM(e:subCode)))
		else
			fwrite(handle,"/???")
		end

		// add error description if available
		if ( ValType(e:description) == "C" )
			fwrite(handle,("  " + e:description))
		end

		// add either filename or operation
		if ( !Empty(e:filename) )
			fwrite(handle,(": " + e:filename))
		elseif ( !Empty(e:operation) )
			fwrite(handle,(": " + e:operation))
		end
		fwrite(handle,chr(13)+chr(10))
		i := 2
		while ( !Empty(ProcName(i)) )
			 fwrite(handle,"Called from "+ Trim(ProcName(i)) + ;
				"(" + NTRIM(ProcLine(i)) + ")  "+chr(13)+chr(10))
			i++
		end
		if !empty(alias())
			fwrite(handle,"Database:"+chr(13)+chr(10))
			fwrite(handle,space(5)+alias()+" =>")
		endif
		if !empty(recno())
			fwrite(handle,"RecNo(")
			fwrite(handle,alltrim(str(recno())))
			fwrite(handle,")"+chr(13)+chr(10))
		endif
		i:=1
		if !empty(indexkey(i))
			fwrite(handle,"Indeces:"+chr(13)+chr(10))
			while !empty(indexkey(i))
				fwrite(handle,space(5)+indexkey(i))
				if(indexkey(i) == indexkey(0))
					fwrite(handle," <=(Master)")
				endif
				fwrite(handle,chr(13)+chr(10))
				i++
			end
		endif

		fwrite(handle,"----------------------------------------------"+chr(13)+chr(10))
        fclose(handle)
    else
		cMessage := if( e:severity > ES_WARNING, "Error ", "Warning " )

		if(e:oscode > 0)
			cMessage+="( "+EEMessage(e:oscode)+") "
		endif
        // add subsystem name if available
		if ( ValType(e:subsystem) == "C" )
			cMessage += e:subsystem()
		else
			cMessage += "???"
		end

		// add subsystem's error code if available
		if ( ValType(e:subCode) == "N" )
			cMessage += ("/" + NTRIM(e:subCode))
		else
			cMessage += "/???"
		end

		// add error description if available
		if ( ValType(e:description) == "C" )
			cMessage += ("  " + e:description)
		end

		// add either filename or operation
		if ( !Empty(e:filename) )
			cMessage += (": " + e:filename)

		elseif ( !Empty(e:operation) )
			cMessage += (": " + e:operation)
		end
		? cMessage
		i := 2
		while ( !Empty(ProcName(i)) )
			? "Called from "+ Trim(ProcName(i)) + ;
				"(" + NTRIM(ProcLine(i)) + ")  "
			i++
		end
		if !empty(alias())
			? "Database:"
			?? space(5)+alias()+"=>"
		endif
		if !empty(recno())
			? recno()
		endif
		i:=1
		if !empty(indexkey(i))
			? "Indeces:"
			while !empty(indexkey(i))
				?? space(5)+indexkey(i)
				if(indexkey(i) == indexkey(0))
					? " <=(Master)"
				else
					?
				endif
				i++
			end
		endif
	endif
	set(_SET_PRINTER,PrePrint)
	set(_SET_DEVICE,PreDevice)
	set(_SET_CONSOLE,PreConsole)
    break
return (.f.)

function EEMessage(_Code)
Local ErrorMsg[88]
if(_Code == 0)
	return("No Error")
endif
ErrorMsg[1]:="Invalid function"
ErrorMsg[2]:="File not found"
ErrorMsg[3]:="Path not found"
ErrorMsg[4]:="No handles available"
ErrorMsg[5]:="Access Denied"
ErrorMsg[6]:="Invalid handle"
ErrorMsg[7]:="Memory control blocks destroyed"
ErrorMsg[8]:="Insufficient Memory"
ErrorMsg[9]:="Invalid memory block address"
ErrorMsg[10]:="Invalid environment"
ErrorMsg[11]:="Invalid format"
ErrorMsg[12]:="Invalid access code"
ErrorMsg[13]:="Invalid data"
ErrorMsg[14]:="Reserved"
ErrorMsg[15]:="Invalid drive"
ErrorMsg[16]:="Attempt to remove current directory"
ErrorMsg[17]:="Not the same device"
ErrorMsg[18]:="No more files"
ErrorMsg[19]:="Disk write-protected"
ErrorMsg[20]:="Unknown unit"
ErrorMsg[21]:="Drive not ready"
ErrorMsg[22]:="Unknown command"
ErrorMsg[23]:="CRC error"
ErrorMsg[24]:="Bad request structure length"
ErrorMsg[25]:="Seek error"
ErrorMsg[26]:="Unknown media type"
ErrorMsg[27]:="Sector not found"
ErrorMsg[28]:="Out of paper"
ErrorMsg[29]:="Write fault"
ErrorMsg[30]:="Read fault"
ErrorMsg[31]:="General failure"
ErrorMsg[32]:="Sharing violation"
ErrorMsg[33]:="Lock violation"
ErrorMsg[34]:="Invalid disk change"
ErrorMsg[35]:="FCB unavailable"
ErrorMsg[36]:="Sharing buffer overflow"
ErrorMsg[37]:="Code page mismatch"
ErrorMsg[38]:="Error handling EOF"
ErrorMsg[39]:="Handle disk full"
ErrorMsg[40]:="Reserved"
ErrorMsg[41]:="Reserved"
ErrorMsg[42]:="Reserved"
ErrorMsg[43]:="Reserved"
ErrorMsg[44]:="Reserved"
ErrorMsg[45]:="Reserved"
ErrorMsg[46]:="Reserved"
ErrorMsg[47]:="Reserved"
ErrorMsg[48]:="Reserved"
ErrorMsg[49]:="Reserved"
ErrorMsg[50]:="Network request not supported"
ErrorMsg[51]:="Remote computer not listening"
ErrorMsg[52]:="Duplicate name on network"
ErrorMsg[53]:="Network name not found"
ErrorMsg[54]:="Network busy"
ErrorMsg[55]:="Network device no longer exists"
ErrorMsg[56]:="Net BIOS command limit exceeded"
ErrorMsg[57]:="Network adapter error"
ErrorMsg[58]:="Incorrect network response"
ErrorMsg[59]:="Unexpected network error"
ErrorMsg[60]:="Incompatible remote adapter"
ErrorMsg[61]:="Print queue full"
ErrorMsg[62]:="Not enough space for print file"
ErrorMsg[63]:="Print file deleted"
ErrorMsg[64]:="Network name deleted"
ErrorMsg[65]:="Access denied"
ErrorMsg[66]:="Network device type incorrect"
ErrorMsg[67]:="Network name not found"
ErrorMsg[68]:="Network name limit exceeded"
ErrorMsg[69]:="Net BIOS session limit exceeded"
ErrorMsg[70]:="Temporarily paused"
ErrorMsg[71]:="Netword request not accepted"
ErrorMsg[72]:="Print or disk redirection is paused"
ErrorMsg[73]:="Reserved"
ErrorMsg[74]:="Reserved"
ErrorMsg[75]:="Reserved"
ErrorMsg[76]:="Reserved"
ErrorMsg[77]:="Reserved"
ErrorMsg[78]:="Reserved"
ErrorMsg[79]:="Reserved"
ErrorMsg[80]:="File already exists"
ErrorMsg[81]:="Reserved"
ErrorMsg[82]:="Cannot made directory entry"
ErrorMsg[83]:="Fail on INT 24"
ErrorMsg[84]:="Too many redirections"
ErrorMsg[85]:="Duplicate redirection"
ErrorMsg[86]:="Invalid password"
ErrorMsg[87]:="Invalid parameter"
ErrorMsg[88]:="Network data fault"
return(ErrorMsg[_Code])
