/*

Program Name  npeditor.prg

Purpose       Editor for Funck(y) II's notepad()

Author        Stephen L. Woolstenhulme    CIS 73060,1302
              3805 Indigo Drive
              Plano, Texas 75075

Notice        Copyright (c) 1992
              Stephen L. Woolstenhulme
              All Rights Reserved

Function 1    cText := npEditor( cText, nTop, nLeft, nBottom, nRight, ;
                                 nAttr, nSize, nLines, nLineLen, nTab )
        
Parameters    Name, Default Value, & Description
              ---------------------------------------------------------
              cFile     File to edit -- fileman() Text file to edit.
              nTop      3                         Top row of edit box.
              nLeft     6                         Left column of box.
              nBottom   maxrow()-4                Bottom row of box.
              nRight    maxcol()-5                Right column of box.
              nAttr     23                        Funck(y) color.
              nSize     16000                     Maximum buffer size.
              nLines    nSize/20                  Maximum buffer lines.
              nLineLen  nRight - ( nLeft + 4 )    Line length to wrap.
              nTab      4                         Tab expansion spaces.

*/

#include "inkey.ch"
#include "FUNCky.ch"
#define K_ALT_BS 270

#xcommand ifNil <xVar> is <xVal> => ;
                <xVar> := iif( <xVar> == Nil, <xVal>, <xVar> )

FUNCTION npEditor( cFile, nTop, nLeft, nBottom, nRight, ;
                   nAttr, nSize, nLines, nLineLen, nTab )

   local np, cText, cScrn := savescreen( 0, 0, maxrow(), maxcol() )
   local nRow := row(), nCol := col(), nChoice 
   local cDelim := chr(9) + chr(10) + chr(12) + chr(13) + chr(32) + ;
                   chr(38) + '~`$%^@.,-_!?/\|;:(){}[]<>#+=*"0123456789'
   
   ifNil nTop     is 3
   ifNil nLeft    is 6
   ifNil nBottom  is maxrow()-4
   ifNil nRight   is maxcol()-5
   ifNil nAttr    is 23
   ifNil nSize    is 16000
   ifNil nLines   is 800
   ifNil nLineLen is maxcol()-15
   ifNil nTab     is 4
   
   nTop     := iif( ! valtype( ntop )     == 'N', val( nTop ),     nTop )
   nleft    := iif( ! valtype( nleft )    == 'N', val( nleft ),    nleft )
   nbottom  := iif( ! valtype( nbottom )  == 'N', val( nbottom ),  nbottom )
   nright   := iif( ! valtype( nright )   == 'N', val( nright ),   nright )
   nattr    := iif( ! valtype( nattr )    == 'N', val( nattr ),    nattr )
   nsize    := iif( ! valtype( nsize )    == 'N', val( nsize ),    nsize )
   nlines   := iif( ! valtype( nlines )   == 'N', val( nlines ),   nlines )
   nlinelen := iif( ! valtype( nlinelen ) == 'N', val( nlinelen ), nlinelen )
   ntab     := iif( ! valtype( ntab )     == 'N', val( ntab ),     ntab )

   np := npCreate( nTop + 1, nLeft + 2, nBottom - 1, nRight - 2, ;
                   nAttr, nSize, nLines, nLineLen, nTab )
   
   setcolor( 'w+/b' )
   cls
   
   if ! cFile == Nil
      if file( cFile )
         cText := memoread( cFile )
      else
         cText := ''                    // started with new file name.
      endif
   endif

   if cText == Nil
        cFile := fileman( 5, 5, MAXROW() - 1, "W+/B, B/W ,,, W/N" )
        cText := iif( empty( cFile ), '', memoread( cFile ) )
   endif

   npColor( np, nAttr )
   npBlockColor( np, 79 )
   cls( 23, chr( 177 ) )
   settitle( " Editing Notepad ", 1, 31 )
   
   box( npWinTop( np )-1,               ;
        npWinLeft( np )-2,              ;
        npWinBottom( np )+1,            ;
        npWinRight( np )+2, '', 23 )          
   @ npWinBottom( np ) + 1, npWinLeft( np ) say 'F1:Help' 
   @ npWinBottom( np ) + 1, npWinLeft( np ) + 10 say 'F2:Spell' 
   @ npWinBottom( np ) + 1, npWinRight( np ) - 19 say  'F10:Exit' 
   @ npWinBottom( np ) + 1, npWinRight( np ) - 8 say 'Esc:Abort' 
   
   npStore( np, NP_WRAP, cText)
   
   do while .t.
       NotePad( np, cDelim )
       nChoice := 2

       if lastkey() == K_ESC
           nChoice := alert( 'You pressed Escape!', ;
                           { ' Abort ', ' Save ', ' Continue ' } )
       endif

       if ! nChoice == 3
           exit
       endif
   enddo

   if nChoice == 2                 // write the edited file.
        set printer to ( cFile )
        set print on
        set console off
        ?? npRetrieve( np, NP_VERBATIM )
        set console on
        set print off
        set printer to
   endif

   restscreen( 0, 0, maxrow(), maxcol(), cScrn )
   setPos( nRow - 1, nCol )
return npSave( np )

FUNCTION NotePad( np, cDelim )
   LOCAL nKey, i, nMargin, nLines
   LOCAL InsertMode := FALSE
   LOCAL cLine, cScrn, cBookMark
   
   if cDelim == Nil
       cDelim := chr(9) + chr(10) + chr(12) + chr(13) + chr(32) + ;
                 chr(38) + '~`$%^@.,-_!?/\|;:(){}[]<>#+=*"0123456789'
   endif

   csrtype( 0 )
   
   DO WHILE ( TRUE )
       npDisplay( np )
       csrput( npScrRow( np ), npScrCol( np ))
       nKey := inkey(0)
    
       DO CASE
       
       CASE( nKey EQ K_F1 )
             showHelp()

       CASE( nKey EQ K_F2 )
             npSpell( np, cDelim )

       CASE( nKey EQ ENTER .or. nKey EQ K_TAB )
             npInsert( np, nKey )
               
       CASE( nKey EQ BACKSPACE )
             npBackSpace( np )
               
       CASE( nKey EQ K_DEL )
             npDelete( np )
               
       CASE( nKey EQ K_CTRL_BS .or. nKey EQ K_CTRL_T )
             npDeleteWord( np )
               
       CASE( nKey EQ K_ALT_BS )
             npDeleteToken( np, cDelim )
       
       CASE( nKey EQ K_CTRL_Y .or. nKey EQ K_ALT_K )
             npBeginMark( np )
             npCsrEnd( np )
             npDeleteBlock( np )

       CASE( nKey == K_ALT_L )
             npDeleteLine( np )

       CASE( nKey >= 376 .AND. nKey <= 384 )  // K_ALT_1 to K_ALT_9
             npSetBookMark( np, nKey - 375 )
             npMarkToken( np, cDelim )        // quick light show
             npDisplay( np )
             nap( 4 )
             npUnMark( np )
             npDisplay( np )

       CASE( nKey == K_ALT_J )                // Jump to bookmark 1 to 9
             cBookMark := ''
             
             for i := 1 to 9
                if npGetBookMark( np, i ) > 0
                    cBookMark += transform( i, ' 9 ' )
                endif
             next 
             
             if ! empty( cBookMark )
                 cScrn := savescreen( maxrow() - 3, 0, maxrow(), maxcol() )
                 @ maxrow() - 2, 0 say padc( ' Select a bookmark ', maxcol() + 1 ) color 'n/w'
                 @ maxrow() - 1, 0 say padc( cBookMark, maxcol() + 1 ) color 'n/w'
                 @ maxrow() - 3, 0 to maxrow(), maxcol() 
                 inkey(0)
                 restscreen( maxrow() - 3, 0, maxrow(), maxcol(), cScrn )
                 
                 if chr( lastkey() ) $ cBookMark
                      npGoBookMark(np, val( chr( lastkey() ) ) )  
                 endif
             endif

       CASE( nKey EQ INS )
             InsertMode := (! InsertMode)
             csrtype( IIF(InsertMode, 1, 0) )
               
       CASE( nKey EQ LEFTARROW )
             npCsrLeft(np, 1)
               
       CASE( nKey EQ RIGHTARROW )
             npCsrRight(np, 1)
               
       CASE( nKey EQ UPARROW )
             npCsrUp(np, 1)
               
       CASE( nKey EQ DOWNARROW )
             npCsrDown(np, 1)
               
       CASE( nKey EQ PGUP )
             npPageUp( np )
               
       CASE( nKey EQ PGDN )
             npPageDown( np )
               
       CASE( nKey EQ K_END )
             npCsrEnd( np )
               
       CASE( nKey EQ K_HOME )
             npCsrHome( np )
               
       CASE( nKey EQ K_CTRL_RIGHT )
             npWordRight( np )
               
       CASE( nKey EQ K_CTRL_LEFT )
             npWordLeft( np )
               
       CASE( nKey EQ K_CTRL_HOME )
             npCsrTop( np )
               
       CASE( nKey EQ K_CTRL_END )
             npCsrBottom( np )
               
       CASE( nKey EQ K_CTRL_PGUP )
             npHome( np )
               
       CASE( nKey EQ K_CTRL_PGDN )
             npEnd( np )
               
       CASE( nKey EQ K_F10 .or. nKey EQ K_ESC )
             npSave( np )
             return NIL
       
       CASE( nKey EQ K_CTRL_RET )  // Hard Page Break
             npInsert( np, 12 )
             npInsert( np, 13 )
       
       CASE( nKey EQ K_ALT_P )     // Print
             if prnstatus() == 0
                nMargin := max( ( 79 - npLineLength( np, 0 ) ) / 2, 0 )
                npSetBookMark( np, 10 )
                npHome( np )
                nLines := npLineCount( np )
                
                for i := 1 to nLines
                   npMarkLine( np )
                   npDisplay( np )
                   cLine := space( nMargin ) + ;
                            strtran( npGetLine( np ), ;
                                     chr( K_TAB ),    ;
                                     space( npTabSize( np ) ) )
                   lPrintLine( cLine )
                   npUnMark( np )
                   npCsrDown( np )
                next

                npGoBookMark( np, 10 )
             else
                cScrn := savescreen( maxrow(), 0, maxrow(), maxcol() )
                @ maxrow(), 0 say padc( 'Printer not ready!  Press any key ... ', maxcol() + 1 )
                tone( 200, 1 )
                inkey(0)
                restscreen( maxrow(), 0, maxrow(), maxcol(), cScrn )
             endif   

       CASE( nKey EQ K_ALT_R )     // Right Justify
             npJustify( np )
       
       OTHERWISE
             IF( InsertMode .or. ;
                 chr( npGetChar( np ) ) $ chr( K_TAB ) + chr( K_ENTER ) )
                 npInsert( np, nKey )
             ELSE
                 npOverWrite( np, nKey )
             ENDIF
       ENDCASE
   ENDDO

return NIL

static function showHelp
    local cText, cScrn := savescreen( 0, 0, maxrow(), maxcol() ), hp
    local nTop := iif( row() > maxrow() / 2, 2, ( maxrow() / 2 + 1 ) )
    local cTitleBuffer := '', nTitlePos := 0, nTitleAttr := 0

    if ! file( 'nphelp.txt' )
       @ maxrow(), 0 say padc( 'Cannot locate NPHELP.TXT!  Press any key ...', maxcol() + 1 ) 
       inkey(0)
    else
      cText := memoread( 'nphelp.txt' )
      hp := npCreate( nTop, 5, nTop + ( maxrow() / 2 ) - 4, maxcol() - 5, ;
                      atoAttr( 'w+/bg' ), 4000, 200, 66, 4 )
      npColor( hp, atoAttr( 'w+/bg' ) )
      gettitle( @cTitleBuffer, @nTitlePos, @nTitleAttr )
      settitle( " Help Text Notepad ", 1, atoAttr( 'w+/rb' ) )
      box( npWinTop( hp )-1,               ;
           npWinLeft( hp )-2,              ;
           npWinBottom( hp )+1,            ;
           npWinRight( hp )+2, '', atoAttr( 'w+/bg' ) )          
      npStore( hp, NP_WRAP, cText)
      NotePad( hp )
      settitle( cTitleBuffer, nTitlePos, nTitleAttr )
      restscreen( 0, 0, maxrow(), maxcol(), cScrn )
    endif

return Nil
