/*
 * File......: CAPFLAGS.PRG
 * Author....: Kevin Maher/Steve Tyrakowski
 * CIS ID....: 73766,1224
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Kevin Maher and Steve Tyrakowski
 * and is placed in the public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"



/*  $DOC$
 *  $FUNCNAME$
 *     CAPTURE
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Start Capture and set Capture Flags
 *  $SYNTAX$
 *
 *     CAPTURE  [PRN | LPT1 | LPT2 | LPT3 | L <nDevice>]
 *              [JOBRELEASE | NOJOBRELEASE]
 *              [FORMFEED abbr:FF | NOFORMFEED abbr:NFF]
 *              [NOTABS abbr:NT | TABS abbr:T <nTabs>]
 *              [SERVERPRINTER <nPrinter>]
 *              [COPIES abbr:C <nCopies>]
 *              [TIMEOUT abbr:TI <nSec>]
 *              [AUTOENDCAP or FLUSHDEVICE abbr:A]
 *              [NOAUTOENDCAP or NOFLUSHDEVICE abbr:NA]
 *              [MAXLINES <nLines>]
 *              [MAXCHARS <nChars>]
 *              [FORM abbr:F <cnForm>]
 *              [NOBANNER abbr:NB | BANNER abbr:B <(cBanner)>]
 *              [NAM <(cName)>]
 *              [QUEUE abbr:Q <(cPQ)>]
 *              [CREATE abbr:CR <(cPath)>]
 *
 *
 *  $ARGUMENTS$
 *
 *     LPT1 | LPT2 | LPT3 | PRN | L <nDevice> which device
 *     to send output to.  If you specify PRN or nothing, CAPTURE
 *     will capture the default device.
 *
 *     JOBRELEASE The print job is released for printing if the
 *     capture is interrupted by a loss of connection time to the
 *     server.
 *
 *     NOJOBRELEASE The print job is not released for printing if the
 *     capture is interrupted by a loss of connection time to the
 *     server.
 *
 *     FORMFEED abbr:FF The print service will print a formfeed after
 *     the print job is finished.
 *
 *     NOFORMFEED abbr:NFF The print service suppresses automatic
 *     form feed after the print job is printed.
 *
 *     NOTABS abbr:NT  This will make sure tab characters get sent
 *     to the printer unchanged.  This is ignored if TABS is specified.
 *
 *     TABS abbr:T <nTabs> Set tab size from 1 to 18.  Default - 8
 *
 *     SERVERPRINTER <nPrinter> printer number that services files
 *     captured to the device.
 *
 *     COPIES abbr:C <nCopies> number of copies to print of the
 *     captured file.  Default - 1
 *
 *     TIMEOUT abbr:TI <nSec> starts counting down every time an
 *     application executes a print command.  When the timeout
 *     expires, the server flushes the capture file and queues
 *     it to a printer.  If an application executes a second print
 *     command before the first timeout expires, the timeout starts
 *     over from the original value.
 *     Default - 0 seconds, Range 0 - 3640 seconds.
 *
 *     AUTOENDCAP abbr:A The server will flush the capture file
 *     when the application ends the capture of the device.
 *     FLUSHDEVICE can also be used.
 *
 *     NOAUTOENDCAP abbr:NA The server will not flush the capture
 *     file when the application ends the capture of the device.
 *     NOFLUSHDEVICE can also be used.
 *
 *     MAXLINES <nLines> maximum lines per page.  Default 66
 *
 *     MAXCHARS <nChars> maximum characters per line.  Default 132
 *
 *     FORM abbr:F <cnForm> type of form that a user put in the printer
 *     to print files captured to the device.  If the current form
 *     is different from the one specified, the file server console
 *     displays a message to put in correct form.  The form can be
 *     specified as a numeric or a character expresion.
 *
 *     NOBANNER abbr:NB No banner page will be printed.  This is
 *     ignored if BANNER is specified.
 *
 *     BANNER abbr:B <(cBanner)> a 12 byte string that appears on the
 *     bottom half of the banner page.  All letters are upper
 *     case.  If the string is 12 null characters, the file name
 *     appears on the banner page.  The banner can be specified as
 *     a literal or a character expression enclosed in parentheses.
 *
 *     NAM <(cName)> a 12 byte string that appears on the upper part
 *     of the banner page. The name can be specified as a literal
 *     or a character expression enclosed in parentheses.
 *
 *     QUEUE abbr:Q <(cPQ)> makes a call the fn_printq function, to
 *     set capture print queue.  Print queue can be specified as a
 *     literal or a character expresion enclosed in parentheses.
 *
 *     CREATE abbr:CR <(cPath)> specify the volume, directory and
 *     filename of where to put the capture file.  This only works
 *     for the default device.  The path can be specified as a literal
 *     or a character expression enclosed in parentheses.
 *
 *  $RETURNS$
 *     N/A
 *
 *  $DESCRIPTION$
 *
 *  This command calls the functions to set the capture flags and then
 *  start the capture.  The NOTIFY, NONOTIFY, SERVER, JOB CONFIG, KEEP,
 *  and SHOW options of the dos command version are not supported by
 *  this command.  Also you cannot specify parameters with an "=" or "/",
 *  they must have a space between the option and the parameters.
 *  For example:
 *                   CAPTURE /q=testq
 *  Would change to:
 *                   CAPTURE q testq
 *
 *  $EXAMPLES$
 *
 *  #include "netto.ch"
 *  FUNCTION Main()
 *     //capture default device with no formfeed, no tabs and no banner
 *     CAPTURE NFF NT NB
 *     . . . print some stuff . . .
 *     ENDCAP PRN
 *
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_sDefCpF() fn_sSPCapF() ENDCAP
 *  $END$
 */



/*  $DOC$
 *  $FUNCNAME$
 *     ENDCAP
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     End Capture
 *  $SYNTAX$
 *
 *     ENDCAP [PRN | LPT1 | LPT2 | LPT3 | <nDevice> ]
 *
 *  $ARGUMENTS$
 *
 *     <nDevice> LPT device, 1 - LPT1, 2 - LPT2, 3 - LPT3
 *
 *  $RETURNS$
 *
 *
 *  $DESCRIPTION$
 *
 *  This user defined command ends the capture.
 *
 *
 *  $EXAMPLES$
 *
 *  #include "netto.ch"
 *  FUNCTION Main()
 *     CAPTURE PRN NOFORMFEED NOTABS NOBANNER
 *     . . . print some stuff . . .
 *     ENDCAP PRN
 *
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_sDefCpF() fn_sSPCapF() CAPTURE
 *  $END$
 */


/*  $DOC$
 *  $FUNCNAME$
 *     SAVE/RESTORE CAPTURE
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Saves/Restores current capture settings
 *  $SYNTAX$
 *
 *     SAVE CAPTURE [ PRN | LPT1 | LPT2 | LPT3] TO memvar
 *
 *     RESTORE CAPTURE [ PRN | LPT1 | LPT2 | LPT3] FROM memvar
 *
 *  $ARGUMENTS$
 *
 *     memvar = variable name
 *
 *  $RETURNS$
 *
 *
 *  $DESCRIPTION$
 *
 *  This command saves the current capture settings to a memvar or restores
 *  them from a memvar.  Note that the restore does not ensure that the
 *  capture gets turned back on, it only resets the capture flags.
 *
 *  $EXAMPLES$
 *
 *  #include "netto.ch"
 *  FUNCTION Main()
 *  LOCAL aFlags
 *     SAVE CAPTURE PRN TO aFlags                // save the current settings
 *     CAPTURE PRN NOFORMFEED NOTABS NOBANNER    // start the capture
 *     . . . print some stuff . . .              // go do your print stuff
 *     ENDCAP PRN                                // end the capture
 *     RESTORE CAPTURE PRN FROM aFlags           // restore flags
 *     IF aFlags[CF_LPTCAPTUREFLAG]              // check if capture was on
 *        CAPTURE PRN                            // if so, reset it
 *     ENDIF
 *
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_sDefCpF() fn_sSPCapF() CAPTURE
 *  $END$
 */

/*  $DOC$
 *  $FUNCNAME$
 *     fn_gDefCpF()
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Gets Print Job Flags for the default LPT
 *  $SYNTAX$
 *
 *     fn_gDefCpF() -> aCapFlags
 *
 *  $ARGUMENTS$
 *
 *     None
 *
 *  $RETURNS$
 *
 *	  <aCapFlags> is an array filled with the printer job flags for the
 *     default LPT.
 *
 *
 *  $DESCRIPTION$
 *
 *  This function returns the print job flags for the default LPT. The
 *  array contains twenty-six (26), or CF_FLAGCOUNT elements as
 *  described below and defined in netto.ch. Also CF_PRINTFLAGS is a 4
 *  dimensional array, or CF_PRINTFLAGCOUNT.
 *
 *      Element           Type Description
 *  --------------------------------------------------------
 *
 *  1 CF_STATUSFLAG         N  Status Flag    always set to 0
 *  2 CF_PRINTFLAGS         A  Printer Flags  Below
 *
 *   1 CF_PRINTRELEASE      L  The print job is released for printing
 *                             if the capture is interrupted by a loss
 *                             of connection time to the server
 *   2 CF_SUPPRESSFORMFEED  L  The print service suppresses automatic form
 *                             feed after the print job is printed
 *   3 CF_INTERPRETCTRLSEQ  L  Tabs converted to spaces
 *   4 CF_PRINTBANNER       L  Banners printed before document
 *
 *  3 CF_TABSIZE            N  Tab Size
 *  4 CF_SERVERPRINTER      N  Server Printer
 *  5 CF_NUMBERCOPIES       N  Number of Copies
 *  6 CF_FORMTYPE           N  Form Type
 *  7 CF_RESERVEDFLAG1         Reserved
 *  8 CF_BANNERTEXT         C  14 byte string indicating the name appearing
 *                             on the bottom half of a banner page.
 *                             All letters are uppercase and if it is 14 null
 *                             characters the capture file name apperars on
 *                             banner page.
 *  9 CF_RESERVEDFLAG2         Reserved
 * 10 CF_LOCALLPTDEVICE     N  Local LPT Device 1-Lpt1, 2-Lpt2, 3-Lpt3
 * 11 CF_FLUSHCAPTIMEOUT    N  Flush Capture Timeout (in seconds)
 * 12 CF_FLUSHCAPONDEVCLOSE L  Flush on Device Close
 * 13 CF_MAXLINES           N  Max Lines on Page
 * 14 CF_MAXCHARS           N  Max Characters per Line
 * 15 CF_FORMNAME           C  Form Name
 * 16 CF_LPTCAPTUREFLAG     L  LPT Capture Flag
 * 17 CF_FILECAPTUREFLAG    L  LPT File Flag
 * 18 CF_TIMINGOUTFLAG      L  Timing Out Flag
 * 19 CF_PRTRSETBUFFERADD   N  Printer Setup Buffer Address
 * 20 CF_PRTRESETBUFFADD    N  Printer Reset Buffer Address
 * 21 CF_CONIDQUEPRTJOB     N  Connection ID of Server
 * 22 CF_CAPTUREINPROG      L  Capture in Progress Flag
 * 23 CF_PRINTQUEUEFLAG     L  Print Queue Flag
 * 24 CF_PRINTJOBVALID      L  Print Job Valid Flag
 * 25 CF_PRINTQUEUEID       N  Print Queue ID
 * 26 CF_PRINTJOBNUMBER     N  Print Job Number
 *
 *  $EXAMPLES$
 *
 *    #include "netto.ch"
 *    function main()
 *
 *        local   aCapFlags:={}
 *        aCapFlags := fn_gDefCpF()
 *        ? "Current timeout setting in seconds:"
 *        ?? aCapflags[CF_FLUSHCAPTIMEOUT]
 *
 *    return NIL
 *
 *  $SEEALSO$
 *  fn_sDefCpF() fn_gSPCapF fn_sSPCapF
 *  $END$
 */

#ifdef FT_TEST

  FUNCTION main()
    CAPTURE PRN
    CAPTURE LPT2
    CAPTURE LPT3
  RETURN (NIL)

#endif



FUNCTION fn_gDefCpF()
RETURN (GetFlags())


/*  $DOC$
 *  $FUNCNAME$
 *     fn_gSPCapF()
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Get Specified Capture Flags
 *  $SYNTAX$
 *
 *     fn_gSPCapF(<nDevice>) -> <aCapFlags>
 *
 *  $ARGUMENTS$
 *
 *     <nDevice> LPT device, 1 - LPT1, 2 - LPT2, 3 - LPT3
 *
 *  $RETURNS$
 *
 *	  <aCapFlags> is an array filled with the printer job flags for the
 *     specified LPT. The Array is defined in netto.ch
 *
 *  $DESCRIPTION$
 *
 *  This function returns an array of capture flags for the specified
 *  LPT device.  The array is described in fn_gDefCpF in more detail
 *  and defined in netto.ch.
 *
 *
 *  $EXAMPLES$
 *
 *  #include "netto.ch"
 *  FUNCTION Main()
 *  LOCAL aCapflags
 *     aCapflags := fn_sLPTCap(1)        // get current flags for lpt1
 *     aCapflags[CF_NUMBERCOPIES] := 5   // change copies entry
 *     fn_sSPCapF(aCapflags,1)           // set the flags
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_sDefCpF() fn_sSPCapF()
 *  $END$
 */

FUNCTION fn_gSPCapF(nDevice)
RETURN (GetFlags(nDevice))

/*  $DOC$
 *  $FUNCNAME$
 *     fn_sDefCPF()
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Set Default Capture Flags
 *  $SYNTAX$
 *
 *     fn_sDefCPF(<aFlags>) -> NIL
 *
 *  $ARGUMENTS$
 *
 *     <aFlags> the structure for aflags is in netto.ch
 *
 *  $RETURNS$
 *
 *	  <aCapFlags> is an array filled with the printer job flags for the
 *     specified LPT. The Array is defined in netto.ch
 *
 *  $DESCRIPTION$
 *
 *  This function sets capture flags for the default LPT.  First you must
 *  get the default capture flags useing the fn_gDefCpF function.  It will
 *  return an array which is described in fn_gDefCpF in more detail and
 *  defined in netto.ch.  Then make any changes to the flags and pass the
 *  array to fn_sDefCPF(aFlags).  Only changes up to CF_FORMNAME or element
 *  15 in the array are allowed.
 *
 *  $EXAMPLES$
 *  #include "netto.ch"
 *
 *  FUNCTION Main
 *  LOCAL aFlags := Array(CF_FLAGCOUNT)
 *    aFlags[CF_PRINTFLAGS] := Array(CF_PRINTFLAGCOUNT)
 *    aFlags[CF_NUMBERCOPIES] := 2
 *    fn_sDefCpF(aFlags)
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_gSPCapF() fn_sSPCapF()
 *  $END$
 */

FUNCTION fn_sDefCpF(aFlags)
RETURN (SetFlags(aFlags))


/*  $DOC$
 *  $FUNCNAME$
 *     fn_sSPCapF()
 *  $CATEGORY$
 *     Print
 *  $ONELINER$
 *     Set Specific Capture Flags
 *  $SYNTAX$
 *
 *     fn_sSPCapF(<aFlags>,<nDevice>) -> NIL
 *
 *  $ARGUMENTS$
 *
 *     <aFlags> the structure for aflags is in netto.ch
 *
 *     <nDevice> LPT device, 1 - LPT1, 2 - LPT2, 3 - LPT3
 *
 *  $RETURNS$
 *
 *     NIL
 *
 *  $DESCRIPTION$
 *
 *  This function sets capture flags for a specific device.  First you must
 *  get the default capture flags useing the fn_gSPCapF() function.  It will
 *  return an array which is described in fn_gDefCpF in more detail and
 *  defined in netto.ch.  Then make any changes to the flags and pass the
 *  array to fn_sSPCapF(aFlags,1).  Only changes up to CF_FORMNAME or element
 *  15 in the array are allowed.
 *
 *  $EXAMPLES$
 *
 *  #include "netto.ch"
 *  FUNCTION Main
 *  LOCAL aFlags := Array(CF_FLAGCOUNT)
 *    aFlags[CF_PRINTFLAGS]   := Array(CF_PRINTFLAGCOUNT)
 *    aFlags[CF_NUMBERCOPIES] := 2
 *    fn_sSPCapF(aFlags, 3)
 *  RETURN (NIL)
 *
 *  $SEEALSO$
 *   fn_gDefCpF() fn_sDefCpF() fn_gSPCapF()
 *  $END$
 */

FUNCTION fn_sSPCapF(aFlags, nDevice)
RETURN (SetFlags(aFlags, nDevice))



STATIC FUNCTION GetFlags(nDevice)
LOCAL cPrintFlags
LOCAL cRecv := SetGetFlags(nDevice)
LOCAL aFlags := Array(CF_FLAGCOUNT)
  aFlags[CF_PRINTFLAGS        ] := Array(CF_PRINTFLAGCOUNT)


  aFlags[CF_STATUSFLAG        ] := BYTE2I(substr(cRecv, 1, 1))

  cPrintFlags := substr(cRecv, 2, 1)
  aFlags[CF_PRINTFLAGS, CF_PRINTRELEASE     ] := FT_IsBit(cPrintFlags, 2)
  aFlags[CF_PRINTFLAGS, CF_SUPPRESSFORMFEED ] := FT_IsBit(cPrintFlags, 3)
  aFlags[CF_PRINTFLAGS, CF_INTERPRETCTRLSEQ ] := FT_IsBit(cPrintFlags, 6)
  aFlags[CF_PRINTFLAGS, CF_PRINTBANNER      ] := FT_IsBit(cPrintFlags, 7)

  aFlags[CF_TABSIZE           ] := BYTE2I(substr(cRecv, 3, 1))
  aFlags[CF_SERVERPRINTER     ] := BYTE2I(substr(cRecv, 4, 1))
  aFlags[CF_NUMBERCOPIES      ] := BYTE2I(substr(cRecv, 5, 1))
  aFlags[CF_FORMTYPE          ] := BYTE2I(substr(cRecv, 6, 1))
  aFlags[CF_RESERVEDFLAG1     ] := BYTE2I(substr(cRecv, 7, 1))
  aFlags[CF_BANNERTEXT        ] :=        substr(cRecv, 8,13)
  aFlags[CF_RESERVEDFLAG2     ] := BYTE2I(substr(cRecv,21, 1))
  // add one to it to match up with lpt1,lpt2,lpt3
  aFlags[CF_LOCALLPTDEVICE    ] := BYTE2I(substr(cRecv,22, 1)) + 1
  // Convert timeout from 18ths of second to seconds for array.
  aFlags[CF_FLUSHCAPTIMEOUT   ] := Bin2I(substr(cRecv,23, 2)) / 18
  aFlags[CF_FLUSHCAPONDEVCLOSE] := BYTE2I(substr(cRecv,25, 1)) == 0
  aFlags[CF_MAXLINES          ] := HILO2W(substr(cRecv,26, 2))
  aFlags[CF_MAXCHARS          ] := HILO2W(substr(cRecv,28, 2))
  aFlags[CF_FORMNAME          ] :=        substr(cRecv,30,13)
  aFlags[CF_LPTCAPTUREFLAG    ] := BYTE2I(substr(cRecv,43, 1)) == 255
  aFlags[CF_FILECAPTUREFLAG   ] := BYTE2I(substr(cRecv,44, 1)) == 255
  aFlags[CF_TIMINGOUTFLAG     ] := BYTE2I(substr(cRecv,45, 1)) == 255
  aFlags[CF_PRTRSETBUFFERADD  ] := HILO2L(substr(cRecv,46, 4))
  aFlags[CF_PRTRESETBUFFADD   ] := HILO2L(substr(cRecv,50, 4))
  aFlags[CF_CONIDQUEPRTJOB    ] := BYTE2I(substr(cRecv,54, 1))
  aFlags[CF_CAPTUREINPROG     ] := BYTE2I(substr(cRecv,55, 1)) == 255
  aFlags[CF_PRINTQUEUEFLAG    ] := BYTE2I(substr(cRecv,56, 1)) == 255
  aFlags[CF_PRINTJOBVALID     ] := BYTE2I(substr(cRecv,57, 1)) == 255
  aFlags[CF_PRINTQUEUEID      ] := HILO2L(substr(cRecv,58, 4))
  aFlags[CF_PRINTJOBNUMBER    ] := HILO2W(substr(cRecv,62, 2))

RETURN (aFlags)

STATIC FUNCTION SetPrintBits(aFlags, cPrintFlags, nPos, nBit)
  IF aFlags[CF_PRINTFLAGS, nPos     ] <> NIL
    IF aFlags[CF_PRINTFLAGS, nPos]
      cPrintFlags := FT_BitSet(cPrintFlags, nBit)
    ELSE
      cPrintFlags := FT_BitClr(cPrintFlags, nBit)
    ENDIF
  ENDIF
RETURN (NIL)

STATIC FUNCTION SetFlags(aFlags, nDevice)
LOCAL cFlags := SetGetFlags(nDevice)
LOCAL cPrintFlags

  cPrintFlags := Substr(cFlags, 2, 1)
  SetPrintBits(aFlags, @cPrintFlags, CF_PRINTRELEASE    , 2)
  SetPrintBits(aFlags, @cPrintFlags, CF_SUPPRESSFORMFEED, 3)
  SetPrintBits(aFlags, @cPrintFlags, CF_INTERPRETCTRLSEQ, 6)
  SetPrintBits(aFlags, @cPrintFlags, CF_PRINTBANNER     , 7)

  FlagStuff(@cFlags, cPrintFlags ,2,1)

  FlagStuff(@cFlags, aFlags[CF_TABSIZE           ] ,3,1, "I")
  FlagStuff(@cFlags, aFlags[CF_SERVERPRINTER     ] ,4,1, "I")
  FlagStuff(@cFlags, aFlags[CF_NUMBERCOPIES      ] ,5,1, "I")
  FlagStuff(@cFlags, aFlags[CF_FORMTYPE          ] ,6,1, "I")
  FlagStuff(@cFlags, aFlags[CF_RESERVEDFLAG1     ] ,7,1, "I")
  FlagStuff(@cFlags, aFlags[CF_BANNERTEXT        ] ,8,13)
  FlagStuff(@cFlags, aFlags[CF_RESERVEDFLAG2     ] ,21,1, "I")

  //FlagStuff(@cFlags, Iif(aFlags[CF_LOCALLPTDEVICE] == NIL,NIL,;
  //            aFlags[CF_LOCALLPTDEVICE    ] - 1) ,22,1, "I")
  //


  // Capture timeout array element is in seconds, and needs to be
  // converted to 18ths of a second for Netware call

  // Bug Fix 02/04/93 by Kevin Maher
  // was
  //FlagStuff(@cFlags, ft_iswap(18 * aFlags[CF_FLUSHCAPTIMEOUT]) ,23,2, "W")
  // change to this because it's supposed to allow a nil value
  // which means no to set a new value
  FlagStuff(@cFlags, Iif(aFlags[CF_FLUSHCAPTIMEOUT]==NIL,NIL;
                    ,ft_iswap(18 * aFlags[CF_FLUSHCAPTIMEOUT])) ,23,2, "W")


  FlagStuff(@cFlags, Iif(aFlags[CF_FLUSHCAPONDEVCLOSE]==NIL,NIL;
                      ,Iif(aFlags[CF_FLUSHCAPONDEVCLOSE],0,1))   ,25,1, "I")
  FlagStuff(@cFlags, aFlags[CF_MAXLINES          ] ,26,2, "W")
  FlagStuff(@cFlags, aFlags[CF_MAXCHARS          ] ,28,2, "W")
  FlagStuff(@cFlags, aFlags[CF_FORMNAME          ] ,30,13)

  SetGetFlags(nDevice, Left(cFlags,42))

RETURN (NIL)

STATIC FUNCTION FlagStuff(cFlags, xStuff, nStart, nLen, cConvert)
  IF xStuff <> NIL
    IF cConvert <> NIL
      xStuff := Iif(cConvert == "I", I2BYTE(xStuff), W2HILO(xStuff))
    ENDIF
    cFlags := Stuff(cFlags,nStart,nLen,Pad(xStuff,nLen," "))
  ENDIF
RETURN (NIL)

STATIC FUNCTION SetGetFlags(nDevice,cFlags)
LOCAL aRegs[INT86_MAX_REGS]
LOCAL nAl

  IF nDevice == NIL
    IF cFlags == NIL
      nAl := 0
      cFlags := Space(63)
    ELSE
      nAl := 1
    ENDIF
  ELSE

    IF cFlags == NIL
      nAl := 2
      cFlags := Space(63)
    ELSE
      nAl := 3
    ENDIF

    // so lpt1, lpt2, lpt3 match, nDevice is actually 1 less
    aRegs[ DX ] := makeHI( (nDevice - 1) )

  ENDIF

  aRegs[ AX ] := makeHI( 184 ) + nAl
  aRegs[ BX ] := REG_ES
  aRegs[ CX ] := LEN(cFlags)
  aRegs[ ES ] := cFlags

  if ft_int86( INT21, aRegs )
      _fnSetErr( UNSIGNED( lowbyte( aRegs[ AX ] ) ) )
  else
      _fnSetErr( EINT86 )
  endif

RETURN (aRegs[ ES ])

