/*
 * File......: XQJOB.PRG
 * Author....: Glenn Scott
 * CIS ID....: 71620,1521
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Glenn Scott and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "netto.ch"


/* -----------------------------------------------------------------
   _fnJs2Pkt( <aJob> ) -> cPkt
   -----------------------------------------------------------------

   Converts a queue job structure to a valid packet suitable for
   sending to a NetWare API call like CreateQueueJobAndFile().
   Does some special translations as follows:

     -  Converts all the logical flags like AutoStart and Restart
        to a one byte "flags" variable by setting the various bits
     -  Converts date/time pairs to 6 byte NetWare date/time strings

*/

/* ------------------------------------------------------------------ */

function _fnJs2pkt( aJob )
   local cPkt, cFlags := chr(0)

   /* This could probably use some optimizing: */

   cFlags := iif( aJob[ QJ_AUTOSTART ],    ft_bitset( cFlags, 3 ), cFlags )
   cFlags := iif( aJob[ QJ_RESTART   ],    ft_bitset( cFlags, 4 ), cFlags )
   cFlags := iif( aJob[ QJ_ENTRYOPEN ],    ft_bitset( cFlags, 5 ), cFlags )
   cFlags := iif( aJob[ QJ_USERHOLD  ],    ft_bitset( cFlags, 6 ), cFlags )
   cFlags := iif( aJob[ QJ_OPERATORHOLD ], ft_bitset( cFlags, 7 ), cFlags )

   cPkt :=    I2BYTE(    aJob[ QJ_CLIENT_STATION     ]   ) +   ;
              I2BYTE(    aJob[ QJ_CLIENT_TASK        ]   ) +   ;
              L2HILO(    aJob[ QJ_CLIENT_ID          ]   ) +   ;
              L2HILO(    aJob[ QJ_TARGET_SERVER      ]   ) +   ;
           _fnDt2Str(    aJob[ QJ_TARGET_EXEC_DATE   ],        ;
                         aJob[ QJ_TARGET_EXEC_TIME   ]         ;
                    )                                      +   ;
           _fnDt2Str(    aJob[ QJ_JOB_ENTRY_DATE     ],        ;
                         aJob[ QJ_JOB_ENTRY_TIME     ]         ;
                    )                                      +   ;
              W2HILO(    aJob[ QJ_JOB_NUMBER         ]   ) +   ;
              W2HILO(    aJob[ QJ_JOB_TYPE           ]   ) +   ;
              I2BYTE(    aJob[ QJ_JOB_POSITION       ]   ) +   ;
              cFlags                                       +   ;
              padr(      aJob[ QJ_JOB_FILE_NAME      ],14) +   ;
                         aJob[ QJ_JOB_FILE_HANDLE    ]     +   ;
              I2BYTE(    aJob[ QJ_SERVER_STATION     ]   ) +   ;
              I2BYTE(    aJob[ QJ_SERVER_TASK        ]   ) +   ;
              L2HILO(    aJob[ QJ_SERVER_ID          ]   ) +   ;
                padr(    aJob[ QJ_JOB_DESCRIPTION    ],50) +   ;
                         aJob[ QJ_CLIENT_RECORD_AREA ]

   return cPkt


/* -----------------------------------------------------------------
   _fnPkt2Js( <cPkt> ) -> aJob
   -----------------------------------------------------------------

   Converts a queue job entry structure from it's native NetWare
   format (a long character string) to the NetWare toolkit
   standard JOBSTRUCT array.

   Does some special translations as follows:

     -  Converts the one-byte "flags" variable to a series of
        five logicals
     -  Converts 6 byte NetWare date/time strings to date/time pairs

*/

/* ------------------------------------------------------------------ */

function _fnpkt2js( cPkt )
   local aJob, aDtEx, aDtEn, cFlags

   aDtEx  := _fnStr2Dt( subs( cPkt, 11, 6 ) )
   aDtEn  := _fnStr2Dt( subs( cPkt, 17, 6 ) )
   cFlags := subs( cPkt, 28, 1 )

   aJob := _fnqjob( BYTE2I( subs( cPkt,  1,  1 ) ) , ;
                    BYTE2I( subs( cPkt,  2,  1 ) ) , ;
                    HILO2L( subs( cPkt,  3,  4 ) ) , ;
                    HILO2L( subs( cPkt,  7,  4 ) ) , ;
                    aDtEx[1]                       , ;
                    aDtEx[2]                       , ;
                    aDtEn[1]                       , ;
                    aDtEn[2]                       , ;
                    HILO2W( subs( cPkt, 23,  2 ) ) , ;
                    HILO2W( subs( cPkt, 25,  2 ) ) , ;
                    BYTE2I( subs( cPkt, 27,  1 ) ) , ;
                    subs( cPkt, 29, 14 )           , ;
                    subs( cPkt, 43,  6 )           , ;
                    BYTE2I( subs( cPkt, 49,  1 ) ) , ;
                    BYTE2I( subs( cPkt, 50,  1 ) ) , ;
                    HILO2L( subs( cPkt, 51,  4 ) ) , ;
                    subs( cPkt, 55, 50 )           , ;
                    subs( cPkt, 105    )             ;
                  )

   aJob[ QJ_AUTOSTART    ] := ft_isbit( cFlags, 3 )
   aJob[ QJ_RESTART      ] := ft_isbit( cFlags, 4 )
   aJob[ QJ_ENTRYOPEN    ] := ft_isbit( cFlags, 5 )
   aJob[ QJ_USERHOLD     ] := ft_isbit( cFlags, 6 )
   aJob[ QJ_OPERATORHOLD ] := ft_isbit( cFlags, 7 )

   return aJob

/* ------------------------------------------------------------------ */

/*  $DOC$
 *  $FUNCNAME$
 *     CREATE JOBSTRUCT
 *  $CATEGORY$
 *     Queue
 *  $ONELINER$
 *     Create a QMS Job Structure (array) the easy way
 *  $SYNTAX$
 *
 *     CREATE JOBSTRUCT <aJob>               ;
 *       [ QUEUE SERVER          <nTarg>   ] ;
 *       [ STARTDATE             <dXdate>  ] ;
 *       [ STARTTIME             <cXtime>  ] ;
 *       [ JOB TYPE              <nType>   ] ;
 *       [ AUTOSTART                       ] ;
 *       [ RESTART                         ] ;
 *       [ USERHOLD                        ] ;
 *       [ OPERATORHOLD                    ] ;
 *       [ JOB DESCRIPTION       <cDesc>   ] ;
 *       [ CLIENT RECORD AREA    <cCliRec> ]
 *
 *
 *  $ARGUMENTS$
 *
 *       QUEUE SERVER         <nTarg>
 *
 *         Queue Server Binder ObjectID, or -1 if any server.
 *         Defaults to -1 (any server).
 *
 *       STARTDATE            <dXdate>
 *
 *         Date you want job serviced.  Defaults to ASAP.
 *
 *       STARTTIME            <cXtime>
 *
 *         Time you want job serviced.  Defaults to ASAP.
 *
 *       JOB TYPE              <nType>
 *
 *         Job type.  Up to queue servers to determine valid job types.
 *         Defaults to 0.
 *
 *       AUTOSTART
 *
 *         If specified, the job will start upon a break to the job server
 *         connection, even if you have not explicitly started the job.  If
 *         AUTOSTART is omitted and there is a connection break before you
 *         have release the job to the queue, the job will be removed from
 *         the queue.  Defaults to no AUTOSTART.
 *
 *       RESTART
 *
 *         If specified and the queue server for some reason aborts the job,
 *         the job will remain in the queue.  Otherwise the job is removed
 *         from the queue.  Defaults to no RESTART.
 *
 *       USERHOLD
 *
 *         If specified the job will continue the advance in the queue but
 *         will not be serviced until the user releases this flag.  Defaults
 *         to no USERHOLD
 *
 *       OPERATORHOLD
 *
 *         Same as for USERHOLD, however only queue operators may clear or
 *         set this flag.  Defaults to no OPERATORHOLD
 *
 *       JOB DESCRIPTION       <cDesc>
 *
 *         Text job description.  Max len=50 char.
 *
 *       CLIENT RECORD AREA    <cCliRec>
 *
 *         Free area for queue servers to define.  You may use this area
 *         to send information to custom queue servers that you devise.
 *         NetWare Print Servers require a specially formatted client area;
 *         use CREATE PRINTJOB to make it.  Max Len=152 char.
 *
 *  $RETURNS$
 *
 *      <aJob>, an array that should be considered a "queue job entry
 *      structure."  You will pass this to other QMS API calls that
 *      require such a structure.
 *
 *      The elements of the array are referenced with the QJ_ constants
 *      in netto.CH
 *
 *  $DESCRIPTION$
 *
 *      There are several steps involved when using the Novell QMS system,
 *      a very important one of which is setting up a proper job
 *      structure.
 *
 *      This command helps you create the front end of a queue job entry
 *      structure.  Use it in conjuction with CREATE PRINTJOB to create
 *      jobs for a Novell Print Server.
 *
 *      For people who are writing client-server applications, you will
 *      want to define your own Client Record Area.  See a Novell API
 *      reference manual for a complete description of the queue job
 *      entry structure.
 *
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *      "CREATE PRINTJOB"
 *  $INCLUDE$
 *
 *  $END$
 */

function  _fnqjob( nClient,  nTask,   nCliID, nTarg,  dXdate, cXtime,     ;
                   dEdate,   cEtime,  nNum,   nType,  nPos,   lAutoStart, ;
                   lReStart, lOpen,   lUHold, lOHold, cFile,  cHandle,    ;
                   nSerSta,  nSerTsk, nSerID, cDesc,  cCliRec )

  /*
   *      Note that _fnQjob() doesn't validate what the user sends in;
   *      i.e,. he could really screw it up with invalid types.
   *      Caveat emptor.
   */

  default nClient    to 0,;
          nTask      to 0,;
          nCliID     to 0,;
          nTarg      to -1,;
          dXdate     to ctod("  /  /  "),;   // is this getting converted?
          cXtime     to "99:99:99",;
          dEdate     to ctod("  /  /  "),;
          cEtime     to "99:99:99",;
          nNum       to 0,;
          nType      to 0,;
          nPos       to 0,;
          lAutostart to .f.,;              // What should these defaults be?
          lReStart   to .f.,;
          lOpen      to .f.,;
          lUHold     to .f.,;
          lOHold     to .f.,;
          cFile      to REPLICATE( CHR(0), 14 ),;
          cHandle    to REPLICATE( CHR(0), 6 ),;
          nSerSta    to 0,;
          nSerTsk    to 0,;
          nSerID     to 0,;
          cDesc      to REPLICATE( CHR(0),  50 ),;
          cCliRec    to REPLICATE( CHR(0), 152 )

  cDesc    := padr( subs( cDesc,   1,  49 ),  50, CHR(0) )
  cCliRec  := padr( subs( cCliRec, 1, 151 ), 152, CHR(0) )


  return { nClient,  nTask,   nCliID, nTarg,  dXdate, cXtime, ;
           dEdate,   cEtime,  nNum,   nType,  nPos,   lAutoStart, ;
           lReStart, lOpen,   lUHold, lOHold, cFile,  cHandle,    ;
           nSerSta,  nSerTsk, nSerID, cDesc,  cCliRec ;
         }

/* ------------------------------------------------------------------ */
