/****************************************************************************
  FILENAME
    MSELECT.PRG

  DESCRIPTION
    Multiple select picklist and support routines for FiveWin v1.9.1

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    24 Jan 96

  FUNCTIONS
    MultiSelect()

  PROCEDURES
    CopySelected()
    DeleteSelected()
    SelectAll()
    LbxPopUp()

  REQUIRES
    FIVEWIN.CH
    COMMON.CH

  COMMENTS
    You've heard the old saying, "Necessity is the mother of invention"?
    Well, it was out of necessity that I wrote this routine. If you see
    anything that could be improved, have questions or suggestions,
    please forward them to my email address given above. I usually lurk
    about in the comp.lang.clipper newsgroup on the 'net.

****************************************************************************/

#include 'FIVEWIN.CH'
#include 'COMMON.CH'


/****************************************************************************
  FUNCTION
    MultiSelect()

  DESCRIPTION
    Multiple select picklist routine

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    24 Jan 96

  ARGUMENTS
    <aItems>        List of items for the 'Available' list
    [<aPreSelects>] List of items for pre-populating the 'Selected' list
    [<cTitle>]      Optional dialog title string
    [<oFont>]       Font to use for dialog and controls
    [<oWin>]        Window or dialog handle

  VARIABLES
    <aSelected>     List of items the user has selected
    <oDlgMSelect>   Multiple select dialog box
    <oBtnAddOne>    'Add' button
    <oBtnAddAll>    'Add all' button
    <oBtnDelOne>    'Remove' button
    <oBtnDelAll>    'Remove all' button
    <oBtnSave>      'Save' button
    <oBtnDefault>   'Default' button
    <oBtnCancel>    'Cancel' button
    <oLbxAvailable> Listbox containing 'available' items
    <oLbxSelected>  Listbox containing 'selected' items
    <cAvailable>    Dummy variable for <oLbxAvailable>
    <cSelected>     Dummy variable for <oLbxSelected>

  RETURNS
    An array containing the items in the 'selected' listbox. If user
    clicks on the 'Cancel' button, an empty array is returned.

  COMMENTS

****************************************************************************/
FUNCTION MultiSelect(aItems,aPreSelects,cTitle,oFont,oWin)
  local aSelected     := nil
  local oDlgMSelect   := nil
  local oBtnAddOne    := nil
  local oBtnAddAll    := nil
  local oBtnDelOne    := nil
  local oBtnDelAll    := nil
  local oBtnSave      := nil
  local oBtnDefault   := nil
  local oBtnCancel    := nil
  local oLbxAvailable := nil
  local oLbxSelected  := nil
  local cAvailable    := ''
  local cSelected     := ''


  // assign values to missing optional arguments
  default aPreSelects := {}                         ,;
          cTitle      := 'Multiple select picklist' ,;
          oFont       := nil                        ,;
          oWin        := GetWndDefault()  // GetWndFrame()

  /********************************************************************
  * If <aPreSelects> is not empty, populate <aSelected> with list of  *
  * preselected items, otherwise <aSelected> starts off as an empty   *
  * array.                                                            *
  ********************************************************************/
  aSelected := if(empty(aPreSelects),{},aPreSelects)

  // if argument <oFont> was not passed, or it's not an object...
  if oFont == nil .or. !valtype(oFont) == 'O'

    // define it
    define font oFont       ;
      name 'MS Sans Serif'  ;
      size 6,15             ;
      bold

  endif

  /* Notice we're using PIXELS for everything here! */

  // define the dialog box itself
  define dialog oDlgMSelect  ;
    from 0,0 to 283,472      ;
    pixel                    ;
    font oFont               ;
    title cTitle             ;
    of oWin                  ;
    style nOR(DS_MODALFRAME ,;
              WS_POPUP      ,;
              WS_CAPTION)

  // display a title above the listbox containing available items
  @ 7,12 say 'Available'  ;
    pixel                 ;
    font oFont            ;
    of oDlgMSelect

  // display the listbox containing available items
  @ 15,12 listbox oLbxAvailable ;
    var cAvailable              ;
    items aItems                ;
    size 79,103                 ;
    pixel                       ;
    update                      ;
    multiple                    ;
    sort                        ;
    of oDlgMSelect

  /********************************************************************
  * When user clicks right mouse button, display a small popup menu   *
  * containing menu items which emulate the 'Add' and 'Add all'       *
  * buttons on the dialog. Notice that the button OBJECTS are passed! *
  ********************************************************************/
  oLbxAvailable:bRClicked :=                       ;
    {|nRow,nCol| LbxPopUp(nRow,nCol,oLbxAvailable ,;
                          oBtnAddone,oBtnAddAll)}

  /********************************************************************
  * Whenever the 'change' event for the 'Available' listbox fires, we *
  * re-evaluate the bWhen instance variable for the 'Add' button. If  *
  * it is satisfied, the 'Add' button is enabled, otherwise it is     *
  * disabled. Likewise for the 'Add all' button.                      *
  *                                                                   *
  * The 'change' event for the 'Selected' listbox is then triggered.  *
  ********************************************************************/
  oLbxAvailable:bChange :=           ;
    {|| (if(eval(oBtnAddOne:bWhen)  ,;
            oBtnAddOne:Enable()     ,;
            oBtnAddOne:Disable()     ;
           )                        ,;
         if(eval(oBtnAddAll:bWhen)  ,;
            oBtnAddAll:Enable()     ,;
            oBtnAddAll:Disable()     ;
           )                        ,;
         oLbxSelected:Change()       ;
        )                            ;
    }

  /********************************************************************
  * When the <oLbxAvailable> 'gotfocus' event fires, we trigger it's  *
  * 'change' event, then make it the active control.                  *
  ********************************************************************/
  oLbxAvailable:bGotFocus :=       ;
    {|| oLbxAvailable:Change()    ,;
        oLbxAvailable:SetFocus()}

  // display a title above listbox containing selected items
  @ 7,147 say 'Selected'  ;
    pixel                 ;
    font oFont            ;
    of oDlgMSelect

  // display the listbox containing selected items
  @ 15,147 listbox oLbxSelected ;
    var cSelected               ;
    items aSelected             ;
    size 79,103                 ;
    pixel                       ;
    update                      ;
    multiple                    ;
    sort                        ;
    of oDlgMSelect

  /********************************************************************
  * When user clicks the right mouse button, display a small popup    *
  * menu containing selections which emulate the 'Remove' and         *
  * 'Remove all' buttons in the center of the dialog.                 *
  ********************************************************************/
  oLbxSelected:bRClicked :=                        ;
    {|nRow,nCol| LbxPopUp(nRow,nCol,oLbxSelected  ,;
                          oBtnDelOne,oBtnDelAll)}

  /********************************************************************
  * Whenever the 'change' event for the 'Selected' listbox fires, we  *
  * re-evaluate the :bWhen instance variable for the 'Remove' button. *
  * If it is satisfied, the 'Remove' button is enabled, otherwise it  *
  * is disabled. Likewise for the 'Remove all' button. We're using    *
  * the exact same technique here as for the 'change' event for       *
  * <oLbxAvailable>, with one exception...                            *
  *                                                                   *
  * Notice that the 'change' event for the 'Selected' listbox doesn't *
  * fire the 'change' event for the 'Available' listbox. If it did,   *
  * these events would ping-pong back and forth, resulting in an      *
  * eventual 650 stack fault error.                                   *
  ********************************************************************/
  oLbxSelected:bChange :=            ;
    {|| (if(eval(oBtnDelOne:bWhen)  ,;
            oBtnDelOne:Enable()     ,;
            oBtnDelOne:Disable()     ;
           )                        ,;
         if(eval(oBtnDelAll:bWhen)  ,;
            oBtnDelAll:Enable()     ,;
            oBtnDelAll:Disable()     ;
           )                         ;
        )                            ;
    }

  /********************************************************************
  * When the <oLbxSelected> 'gotfocus' event fires, we trigger it's   *
  * 'change' event, see above, then make it the active control.       *
  ********************************************************************/
  oLbxSelected:bGotFocus :=        ;
    {|| oLbxSelected:Change()     ,;
        oLbxSelected:SetFocus()}

  /********************************************************************
  * The 'Add' button is enabled only if there are highlighted items   *
  * in the 'Available' listbox. When the button is clicked, those     *
  * items that are highlighted are copied to the 'Selected' list.     *
  * Then we fire the 'change' event for the other listbox and make    *
  * the 'Available' listbox the active control.                       *
  ********************************************************************/
  @ 28,97 button oBtnAddOne                      ;
    prompt '> Add'                               ;
    size 42,10                                   ;
    font oFont                                   ;
    pixel                                        ;
    of oDlgMSelect                               ;
    when oLbxAvailable:GetSelCount() > 0         ;
    action                                       ;
      (CopySelected(oLbxAvailable,oLbxSelected) ,;
       oLbxAvailable:Change()                   ,;
       oLbxAvailable:SetFocus())                 ;
    message 'Copy highlighted item(s) to "Selected" list'

  /********************************************************************
  * The 'Add all' button selects all items in the 'Available' list,   *
  * then copies them to the 'Selected' list. It is enabled only if    *
  * the 'Available' listbox contains any items.                       *
  ********************************************************************/
  @ 48,97 button oBtnAddAll                      ;
    prompt '>> Add all'                          ;
    size 42,10                                   ;
    font oFont                                   ;
    pixel                                        ;
    of oDlgMSelect                               ;
    when oLbxAvailable:Len() > 0                 ;
    action                                       ;
      (SelectAll(oLbxAvailable)                 ,;
       CopySelected(oLbxAvailable,oLbxSelected) ,;
       oLbxSelected:SetFocus())                  ;
    message 'Copy all available items to "Selected" list'

  /********************************************************************
  * The 'Remove' button is enabled only if there are highlighted      *
  * items in the 'Selected' list. When the button is clicked, those   *
  * highlighted items are deleted from the list. If any items are     *
  * left in the 'Selected' list, it remains the active control.       *
  * Otherwise the 'Available' listbox becomes the active control.     *
  ********************************************************************/
  @ 75,97 button oBtnDelOne               ;
    prompt 'Remove <'                     ;
    size 42,10                            ;
    font oFont                            ;
    pixel                                 ;
    of oDlgMSelect                        ;
    when oLbxSelected:GetSelCount() > 0   ;
    action                                ;
      (DeleteSelected(oLbxSelected)      ,;
       if(oLbxSelected:Len() > 0         ,;
          oLbxSelected:SetFocus()        ,;
          oLbxAvailable:SetFocus()        ;
         )                                ;
      )                                   ;
    message 'Remove highlighted item(s) from "Selected" list'

  /********************************************************************
  * The 'Remove all' button is enabled only if there are items in the *
  * 'Selected' listbox. When the button is clicked, the :Reset()      *
  * method clears out the 'Selected' listbox. The 'Available' listbox *
  * then becomes the active control.                                  *
  ********************************************************************/
  @ 95,97 button oBtnDelAll       ;
    prompt 'Remove all <<'        ;
    size 42,10                    ;
    font oFont                    ;
    pixel                         ;
    of oDlgMSelect                ;
    when oLbxSelected:Len() > 0   ;
    action                        ;
      (oLbxSelected:Reset()      ,;
       oLbxAvailable:SetFocus())  ;
    message 'Remove all items from "Selected" list'

  /********************************************************************
  * The 'Save' button simple closes down the dialog                   *
  ********************************************************************/
  @ 126,32 button oBtnSave    ;
    prompt 'OK'               ;
    size 42,10                ;
    font oFont                ;
    pixel                     ;
    of oDlgMSelect            ;
    action oDlgMSelect:End()  ;
    default                   ;
    message 'Save current selections and exit'

  // if <aPreSelects> isn't empty, add a 'Default' button
  if len(aPreSelects) > 0

    /******************************************************************
    * When user clicks on the 'Default' button, the contents of       *
    * <aPreSelects> is placed back in the 'Selected' listbox,         *
    * replacing whatever was already there. This allows the user to   *
    * set the 'Selected' list back to the state it was when the       *
    * dialog was first activated.                                     *
    ******************************************************************/
    @ 126,97 button oBtnDefault                ;
      prompt '&Default'                        ;
      size 42,10                               ;
      font oFont                               ;
      pixel                                    ;
      of oDlgMSelect                           ;
      action                                   ;
        (oLbxSelections:SetItems(aPreselects) ,;
         oLbxSelections:Refresh())             ;
      message 'Restore default selections'

  endif

  /********************************************************************
  * The 'Cancel' button clears out the 'Selected' listbox via the     *
  * :Reset() method and closes the dialog. In other words, when the   *
  * user cancels, an empty array is returned.                         *
  ********************************************************************/
  @ 126,162 button oBtnCancel  ;
    prompt '&Cancel'           ;
    size 42,10                 ;
    font oFont                 ;
    pixel                      ;
    cancel                     ;
    of oDlgMSelect             ;
    action                     ;
      (oLbxSelected:Reset()   ,;
       oDlgMSelect:End())      ;
    message 'Cancel selections and exit'

  // display the dialog, centered on the screen
  activate dialog oDlgMSelect ;
    centered

return oLbxSelected:aItems()
// EOProc MultiSelect()



/****************************************************************************
  PROCEDURE
    CopySelected()

  DESCRIPTION
    Copies selected items from one listbox to another

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    24 Jan 96

  ARGUMENTS
    <oLbxSource>    Listbox which contains selected items
    <oLbxTarget>    Listbox to place selected items in

  VARIABLES
    <aSource>       List of selected items in <oLbxSource>
    <aSelects>      List of element numbers selected in <oLbxSource>
    <nLen>          Number of elements in <aSource>
    <nAt>           Returned by ascan()
    <I>             For...next loop counter variable

  COMMENTS

****************************************************************************/
STATIC PROCEDURE CopySelected(oLbxSource,oLbxTarget)
  local aSource   := oLbxSource:aSelections()
  local aSelects  := oLbxSource:GetSelItems()
  local nLen      := len(aSource)
  local nAt       := 0
  local I         := 0


  // for each selected item in <oLbxSource>
  for I := 1 to nLen

    // if it's not already in <oLbxTarget>...
    if (nAt := ascan(oLbxTarget:aItems(),aSource[I])) == 0

      // add it
      oLbxTarget:Add(aSource[I])

    endif

  next I

  // deselect highlighted items in <oLbxSource>
  aeval(aSelects,{|ele| oLbxSource:SetSel(ele,FALSE)})

  // refresh target listbox
  oLbxTarget:Refresh()

return
// EOProc CopySelected()



/****************************************************************************
  PROCEDURE
    DeleteSelected()

  DESCRIPTION
    Deletes selected items in a listbox

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    24 Jan 96

  ARGUMENTS
    <oLbxSource>    Listbox from which selected items are to be deleted

  VARIABLES
    <aItems>        List of all items in <oLbxSource>
    <aSource>       List of selected items in <oLbxSource>
    <aTarget>       Temporary work array
    <nAt>           Returned by ascan()
    <I>             For...next loop counter variable

  COMMENTS

****************************************************************************/
STATIC PROCEDURE DeleteSelected(oLbxSource)
  local aItems  := oLbxSource:aItems()
  local aSource := oLbxSource:aSelections()
  local aTarget := {}
  local nAt     := 0
  local I       := 0


  // for each item in <oLbxSource>...
  for I := 1 to len(aItems)

    // if it's not in <aSource>, meaning it's not selected...
    if (nAt := ascan(aSource,aItems[I])) == 0

      // add it to <aTarget>
      aadd(aTarget,aItems[I])

    endif

  next I

  // place new list of selections in <oLbxSource>, refresh it
  oLbxSource:SetItems(aTarget)
  oLbxSource:Refresh()

return
// EOProc DeleteSelected()



/****************************************************************************
  PROCEDURE
    SelectAll()

  DESCRIPTION
    Selects all items in a listbox

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    24 Jan 96

  ARGUMENTS
    <oLbx>      The listbox object in which all items are to be selected

  VARIABLES
    <I>         For...next loop counter variable

  COMMENTS
    There's probably a way to reduce this to a one-liner using
    aeval(), eval(), or a combination thereof, but it's late I'm too
    damned tired to think about it right now! <ZZZZZzzzzzz>

****************************************************************************/
PROCEDURE SelectAll(oLbx)
  local I := 0


  // for each item in <oLbx>...
  for I := 1 to oLbx:Len()

    // select it
    oLbx:SetSel(I,TRUE)

  next I

return
// EOProc SelectAll()



/****************************************************************************
  PROCEDURE
    LbxPopUp()

  DESCRIPTION
    Listbox popup menu for MultiSelect()

  AUTHOR
    Patrick K. Spence - pkspence@conc.tdsnet.com

  DATE
    27 Jan 96

  ARGUMENTS
    <nRow>        Row number when right mouse button was clicked
    <nCol>        Column number when right mouse button was clicked
    <oLbx>        Active listbox object
    <oBtnOne>     Button object which acts on highlighted listbox items
    <oBtnAll>     Button object which acts on all listbox items

  VARIABLES
    <oMnuPopUp>   Popup menu
    <oItemOne>    Menu item which acts on highlighted listbox items
    <oItemAll>    Menu item which acts on all listbox items

  COMMENTS
    The reason the button objects are passed, is because we use their
    captions, 'when' clauses and action blocks to construct the popup
    menu and assign actions to the menu items. This way if changes are
    necessary in a button action block or 'when' clause, it only needs
    to be changed in one place instead of several.

****************************************************************************/
PROCEDURE LbxPopUp(nRow,nCol,oLbx,oBtnOne,oBtnAll)
  local oMnuPopUp := nil
  local oItemOne  := nil
  local oItemAll  := nil


  menu oMnuPopUp popup

    menuitem oItemOne                 ;
      prompt oBtnOne:cCaption()       ;
      action eval(oBtnOne:bAction())  ;
      of oMnuPopUp                    ;
      message oBtnOne:cMsg

    separator

    menuitem oItemAll                 ;
      prompt oBtnAll:cCaption()       ;
      action eval(oBtnAll:bAction())  ;
      of oMnuPopUp                    ;
      message oBtnAll:cMsg

  endmenu

  // enable/disable menu items according to button 'when' clauses
  if(eval(oBtnOne:bWhen()),oItemOne:Enable(),oItemOne:Disable())
  if(eval(oBtnAll:bWhen()),oItemAll:Enable(),oItemAll:Disable())

  // display the popup menu
  activate popup oMnuPopUp  ;
    at nRow,nCol            ;
    of oLbx

return
// EOProc LbxPopUp()



/***************************************************************************
  EOFile MSELECT.PRG
***************************************************************************/
