
//  SECURITY.PRG  //
//                                                           
// AUTHOR : Rick Hellewell                                   
// DATE   : Tue  6/20/95  8:14:26 AM
// NOTICE : Copyright (c) 1991 - 1995 by Rick Hellewell
//             All Rights Reserved                           
// VERSION: Clipper 5.2c
// PURPOSE: User Login, Security checking                    
// Notes  : Released to public domain for use with attribution                                                 
//                                                           
// Program Version :                                         
//  //

/* User Login, Logoff, Maintenance, Security checking

   Uses USERLIST.DBF, with structure of:
   .    Field Name  Type       Width    Dec
   .    USERNAME    Character     12            User Name
   .    FULLNAME    Character     20            User Full Name
   .    PASSCODE    Numeric       10      8     Numeric password
   .    LEVEL       Character      1            Security Level
   .    LOTACCESS   Character      2            Accessable Lots (*=all)
   .    NAMEDATE    Date           8            Date User Name changed
   .    PASSDATE    Date           8            Date Password changed
   .    LEVELDATE   Date           8            Date Security level changed
   .    NAMEBY      Character     12            User Name changed by
   .    PASSBY      Character     12            Password changed by
   .    LEVELBY     Character     12            Security Level changed by

   All passwords are character entry, and then converted to a 
   numeric value with 8 decimal places using a formula based 
   on the ascii value of each password character.  This numeric
   value is stored in the USERLIST.DBF database.  All password
   entries are coverted and compared to the stored numeric value.
   This prevents viewing the database to gain password access, since the
   stored numeric password value cannot be converted back into a text value.

   The security levels could also encoded using a simple character
   substitution scheme.  

   Some Z..() functions are used to display messages, save screens, etc. 
   These will need to be replaced with your similar functions.

   The program keeps the USERLIST.DBF open for short periods of time, and
   never has the file open during data entry. This helps ensure that access
   to the file on a network will not cause problems.
   
   Returns the aUser array, containing user information. If the user presses
   the ESC key, then the aUser array is empty, signalling to the calling 
   program that the user wants to escape/quit the calling program.
   
   The aUser array has this structure:
         aUser[1]  := userlist->username     user name
         aUser[2]  := userlist->passcode     numeric password
         aUser[3]  := userlist->level        security level
         aUser[4]  := userlist->namedate     date name last changed
         aUser[5]  := userlist->passdate     date password last changed
         aUser[6]  := userlist->leveldate    date security level last changed
         aUser[7]  := userlist->nameby       who changed the name
         aUser[8]  := userlist->passby       who changed the password
         aUser[9]  := userlist->levelby      who changed the security level
         aUser[10] := userlist->lotaccess    what lots the user can access

   
   Note to outside users: there are several functions in here that use
   external programs. They are identified; you'll need to substitute 
   similar functions in your program.
*/

#include "inkey.ch"
#include "achoice.ch"
static cTempName
static lHide                           // hide input if true


function SECURITY()
   
   // main entry point for user logon, logoff, changing
   
   local xcolor
   private cUserName := ""
   // test
   // private vers_date := date()
   // private vers_num := "3.05"
   // zsysinit()
   // test
   private aSecMenu := {}
   private aSecure := {}
   
   lHide := .f.
   
   // set up the aUser array if not already public
   if type("auser") <> "A"
      public aUser := {"", "0", "", date(), date(), date(), "", "", "", ""}
   endif
   
   // open the user name database
   dbfopen("USERLIST") // opens the database (EXTERNAL)
   if reccount() = 0   // empty, so create a dummy entry
      aUser := {"", "", "9", "", "", "", "", "", "", ""}
      adduser(.t.)                     // add a master user with .t. flag
   endif
   close USERLIST
      
   // logoff current user
   if len(alltrim(aUser[1])) > 0
      logoff()
   endif
   
   // logon new user
   if ! logon()                        // no entry, so quit
      alerter("Program cancelled...")    // EXTERNAL, similar to ALERT()
      ? ""
      aUser := {"", "", "9", "", "", "", "", "", "", ""}
      return 
   endif
   
   // user maintenance menu
   do while.t.
      // cUserName := aUser[1]
      scrnhead("User Maintenance")        // EXTERNAL, clears screen, puts text at top
      
      /* shows values (test only)
      @11,0
      for i = 1 to len(aUser)
      ? aUser[i]
      next
      */
      
      // no user, force a logon before displaying the menu
      if len(alltrim(aUser[1])) = 0
         // aSecMenu := {}
         logon()
         loop
      endif
      if len(aSecMenu) = 0
         aAdd(aSecMenu, "1. Start Program")
         aadd(aSecMenu, "2. Log On as Another User")
         aadd(aSecMenu, "3. Log Off Current User")
         aadd(aSecMenu, "4. Add a New User")
         aadd(aSecMenu, "5. Change User Access Levels")
         aadd(aSecMenu, "6. Delete a User Name")
         aadd(aSecMenu, "7. Change my Own Password")
         aadd(aSecMenu, "8. User File Reports")
         // set security flag array for use by ACHOICE()
         aSecure := {if(aUser[3] $ "0123456789", .t., .f.), ; // start program
                      if(aUser[3] $ "0123456789", .t., .f.), ; // log on
                      if(aUser[3] $ "0123456789", .t., .f.), ; // logoff
                      if(aUser[3] $ "9", .t., .f.), ;          // add user
                      if(aUser[3] $ "9", .t., .f.), ;          // change access levels
                      if(aUser[3] $ "9", .t., .f.), ;          // delete User
                      if(aUser[3] $ "0123456789", .t., .f.), ; // change own password
                      if(aUser[3] $ "9", .t., .f.)}            // change own password
         
      endif
      xcolor := setcolor()                       
      setcolor(zcolor("w+/b, b/w, b,b,w/b"))    // EXTERNAL, zcolor() converts color to mono
      
      nChoice := achoice(4, 20, 22, 60, aSecMenu, aSecure)
      setcolor(zcolor(xcolor))
      do case
         case nChoice == 1
            exit
         case nChoice == 2
            if logon()
               userlog("Logged on")
            endif
            if aUser[5] < date()-90 
               do while .t.
                  alerter("Time to change your password!")
                  if ! changepass()
                     loop
                  else
                     exit
                  endif
               enddo
            endif
         case nChoice == 3
            if logoff()
               userlog("Logged off")
            endif
         case nChoice == 4
            if adduser(.f.)
               userlog("Added user")
            endif
         case nChoice == 5
            if changeuser()
               userlog("Changed user")
            endif
         case nChoice == 6
            if deluser()
               userlog("Deleted user")
            endif
         case nChoice == 7
            if changepass()
               userlog("Changed password")
            endif
         case nChoice == 8
            userprint()
         otherwise
            exit
      end case
   enddo
   
   if lastkey() = K_ESC
      return aUser := {"", "", "", "", "", "", "", "", "", ""}
   endif
   
return aUser  // return aUser info to calling program

//  //
static function getinput(cPrompt, cAnswer, nPromptLine)
   // get input to a question
   local ln_length, ln_start, xColor
   if pcount() < 3
      nPromptLine := maxrow() / 2
   endif
   ln_length := len(cPrompt) + len(cAnswer) + 2
   ln_start  := (maxcol() / 2) - (ln_length / 2)
   @ nPromptLine, ln_start say cPrompt
   if lHide
      xColor := setcolor()
      setcolor(",N/N")
   endif
   @ nPromptLine, ln_start + len(cPrompt) + 1 get cAnswer pict "@!"
   read
   if lHide
      setcolor(xColor)
   endif
   @ nPromptline,0 clear to nPromptline + 1,maxcol()
   
return cAnswer

//  //
static function getname()
   // get user name
   // scrnhead("User Logon")
   local cUser := space(12)
   do while len(alltrim(cUser)) < 5
      cUser := getinput("Enter User Name", space(12), 8)
      if len(alltrim(cUser)) = 0       // allow no entry
         return ""
      endif
      if len(alltrim(cUser)) < 5
         alerter("The user name must contain at least five characters.")
      endif
   enddo
   
return cUser

//  //
static function getpass()
   // get user password
   local cPassword := ""
   lHide := .t.
   do while len(alltrim(cPassWord)) < 5
      cPassWord := input_hide(" Enter Password", 9, 25, space(8))
      @ 9,0 clear to 9,maxcol()
      if len(alltrim(cPassWord)) < 5
         alerter("Your password must contain at least five characters.")
      endif
   enddo
   lHide := .f.
return cPassWord

//  //
static function getlevel()
   // get user security level
return getinput(" Enter Security Levels", space(10), 10)

//  //
static function getLotAccess()
   // get user lot access
   
return  getinput("   Enter Lots (*=all)", space(2), 11)

//  //
static function PassEncode(cPassword)
   
   // covert password to numeric value with 8 decimal places
   local first := ""       // padded password
   local nPassNum := 0     // encoded password
   local ndecSave := 0     // # of decimals setting
   local a1, a2, a3, a4, a5, a6, a7, a8
   
   cPassword := alltrim(cPassword)     // trim leading/trailing spaces
                                       // fill the rest of the password with "x"
   first := padr(cPassword,10,"x")
   release cPassword              // clear password variable
   nDecSave := set(_SET_DECIMALS) // save current setting
   set decim to 8
   // covert each letter to ascii value
   a1 := asc(substr(first, 1, 1))
   a2 := asc(substr(first, 2, 1))
   a3 := asc(substr(first, 3, 1))
   a4 := asc(substr(first, 4, 1))
   a5 := asc(substr(first, 5, 1))
   a6 := asc(substr(first, 6, 1))
   a7 := asc(substr(first, 7, 1))
   a8 := asc(substr(first, 8, 1))
   release first
   // perform calculation of password by a formula to get a 8 decimal number value
   nPassNum := round((a1 / a3 * a5 / a7 * a2 / a4 * a6 / a8), 8)
   set decimals to nDecSave            // restore old setting
   
return nPassNum

//  //
static function finduser(cTempName, lUpdtUser, lClose)
   // find the user name in the database
   // update the aUser array values if lUpdtUser = .t.
   
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   if pcount() = 1
      lUpdtUser := .t.
      lClose := .t.
   endif
   if select("USERLIST") = 0
      dbfopen("USERLIST")
   endif
   if reccount() = 0
      aUserTemp := {"", "", "9", "", "", "", "", "", "", ""}
      adduser(.t.)
   endif
   // do an exact seek without messing with SET EXACT
   seek padr(cTempName,12)     
   if ! eof()
      aUserTemp[1]  := userlist->username
      aUserTemp[2]  := userlist->passcode
      aUserTemp[3]  := userlist->level
      aUserTemp[4]  := userlist->namedate
      aUserTemp[5]  := userlist->passdate
      aUserTemp[6]  := userlist->leveldate
      aUserTemp[7]  := userlist->nameby
      aUserTemp[8]  := userlist->passby
      aUserTemp[9]  := userlist->levelby
      aUserTemp[10] := userlist->lotaccess
   endif
   if lClose                           // close dbf if flag is true
      close USERLIST
   endif   
   
return aUserTemp                        

//  //
static function logon()
   // log on a user: give them three tries only, then off the system
   
   local nAttempts := 0  // logon attempts
   local nEnterPass := 0 // password value
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}   // temporary user array

   scrnhead("Logon User")
   do while .t.
      if nAttempts > 3                 // 4 strikes and you're out!
         alerter("Strike 4 !;(What game are we playing?);Access denied!")
         ? ""
         quit
      endif
      cTempName := getname()           // get user name
      if len(alltrim(cTempName)) = 0
         return .f.
      endif
      aUserTemp := finduser(cTempName,.t., .t.)      // find user name , close dbf
      if len(alltrim(aUserTemp[1])) = 0   // it's empty, so not found
         nAttempts ++
         alerter("Can't find that user!;(Strike " + alltrim(str(nAttempts))+")")
         loop
      else
         nEnterPass := PassEncode(getpass()) // get encoded password
         aUser := aUserTemp
      endif
      if nEnterPass == aUser[2]        // password ok
         exit
      else                             // invalid password
         nAttempts ++
         alerter("Incorrect password!;(Strike " + alltrim(str(nAttempts))+")")
      endif
   enddo
   aUser := aUserTemp      
   
   // set up current user security levels for menu display by ACHOICE()
   aSecure := {if(aUser[3] $ "0123456789", .t., .f.), ; // start program
                if(aUser[3] $ "0123456789", .t., .f.), ; // log on
                if(aUser[3] $ "0123456789", .t., .f.), ; // logoff
                if(aUser[3] $ "9", .t., .f.), ;          // add user
                if(aUser[3] $ "9", .t., .f.), ;          // change access levels
                if(aUser[3] $ "9", .t., .f.), ;          // delete User
                if(aUser[3] $ "0123456789", .t., .f.), ; // change own password
                if(aUser[3] $ "9", .t., .f.)}
                // change own password
   
   if select("USERLIST") > 0
      close userlist
   endif
return .t. 

//  //
static function logoff()
   // logoff a user
   
   // clear user array
   aUser     := {"", "", "", "", "", "", "", "", "", ""}
   cTempName := ""
   
return .t. // if successful

//  //
static function adduser(lFirstOne)
   // add a user, if lFirstOne .t., add initial user with full access level
   local nUserPass, cAnswer
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   
   // check user's access
   scrnhead("Add User Name")
   if "9" $ aUser[3]
      do while .t.
         cTempName := getname()
         // no entry, exit
         if len(alltrim(cTempName)) = 0
            alerter("No changes made!")
            return .f.
         endif
         if ! lFirstOne
            // check if user already exists
            aUserTemp := finduser(cTempName, .f., .t.)   // then close dbf
            if aUserTemp[1] == cTempName
               alerter("That user name already exists!") 
               loop
            endif
         endif
         nUserPass := PassEncode(getpass())
         if lFirstOne
            aUserTemp[3]  := "0123456789"
            aUserTemp[10] := "*"
            alerter("Master user added with all security levels")
         else
            aUserTemp[3]  := getlevel()
            aUserTemp[10] := getlotaccess()
         endif
         
         cAnswer := zYesNoQ(15, "OK to add " + alltrim(cTempName) + " ?")
         do case
            case cAnswer = 1           // Yes
               exit
            case cAnswer = 2           // No
               loop
            case cAnswer = 3           // Quit
               alerter("User name not added to list.")
               return .f.
         end case
      enddo
      // add the user record
      dbfopen("USERLIST")
      append blank
      if ! reclock("Database record in use;user record not added")
         return .f.
      endif
      replace username with cTempName, passcode with nUserPass, level with aUserTemp[3], ;
          namedate with date(), passdate with date(), leveldate with date(), ;
          nameby with aUser[1], passby with aUser[1], levelby with aUser[1], lotaccess with aUserTemp[10]
      commit    
      unlock
      close userlist
   else
      alerter("Sorry, your security level doesn't allow adding users.")
      return .f.
   endif
return .t.

//  //
static function deluser()
   // delete a user name

   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   local cTempName := ""
   scrnhead("Delete User Name")
   // check user's access
   if "9" $ aUser[3]
      // get user name
      cTempName := getname()
      if len(alltrim(cTempName)) = 0
         alerter("No changes made.")
         return .f.
      endif
      aUserTemp:= finduser(cTempName, .f., .f.) // don't close dbf yet
      if len(aUserTemp[1]) = 0
         alerter("Couldn't find that user name.")
         close userlist
         return .f.
      endif

      // make sure it not current logged on user
      if aUserTemp[1] == aUser[1]
         alerter("Sorry - you can't delete your own user name!")
         return .f.
      endif
      // verify deletion
      if zYesNoQ(15, "OK to delete user " + alltrim(cTempName) + " ?") = 1
         dbfopen("USERLIST")
         if ! reclock("Couldn't access that user record;User not deleted.")
            return .f.
         endif
         delete                        // go for it
         unlock
         close USERLIST
      else
         alerter("No changes made")
      endif
   else
      alerter("Sorry - your security level doesn't allow deleting users.")
   endif
   
return .t.

//  //
static function changeuser()
   // change user level
   local cUserLevel
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   
   scrnhead("Change User Access Level")
   // check user's access
   if "9" $ aUser[3]
      // get user name
      cTempName := getname()
      if len(alltrim(cTempName)) = 0
         alerter("No changes made")
         return .f.
      endif
      aUserTemp := finduser(cTempName, .f., .f.)   // don't close dbf yet
      if len(aUserTemp[1]) = 0
         alerter("Couldn't find that user name.")
         close userlist
         return .f.
      endif
      // make sure it not current logged on user
      if aUserTemp[1] == aUser[1]
         alerter("Sorry, you can't change your own security level!")
         return .f.
      endif
      aUserTemp[3]  := getlevel()
      aUserTemp[10] := getlotaccess()
      dbfopen("USERLIST")
      if ! reclock("Couldn't access that user record;data not changed")
         return .f.
      endif   
      replace level with aUserTemp[3], ;
          leveldate with date(), ;
          levelby with aUser[1], ;
          lotaccess with aUserTemp[10]
      commit    
      unlock
      close userlist
   else
      alerter("Sorry - you're not allowed to change security levels.")
      return .f.
   endif
   
return .t.

//  //
static function changepass()
   // change user password
   local cSecure
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   local cPass1 := 0
   local cPass2 := 0
   local nPassCode := 0
   
   scrnhead("Change Personal Password")
   // user can only change their own password
   do while .t.
      @ 6,0 say padc("Please enter your new password", maxcol())
      cPass1 := getpass()
      if len(alltrim(cPass1)) = 0
         alerter("No changes made")
         return .f.
      endif
      @ 6,0 say padc("Enter the new password again to verify", maxcol())
      cPass2 := getpass()
      if cPass1 <> cPass2
         alerter("Oops! The passwords didn't match!;Please type carefully.")
         if zYesNoQ(15, "Do you want to try again?") = 1
            loop                       // yes, try again
         else
            return .f.                 // no or quit, forget it
         endif
      endif
      nPassCode := PassEncode(cPass1)
      release cPass1, cPass2           // for debug hackers
      if nPassCode == aUser[2]
         alerter("Your new password must be different from the old password.")
         loop
      else
         exit
      endif
   enddo
   // find the user, replace the record
   aUserTemp := finduser(aUser[1], .t., .f.)    // don't close dbf yet
   if aUserTemp[1] = aUser[1]
      aUser[2] := nPassCode
      if ! reclock("Couldn't access the data;password not changed")
         return .f.
      endif 
      alerter("Storing new password", .f.)  
      replace passcode with nPassCode, ;
          passdate with date(), ;
          passby with aUser[1]
      commit    
      unlock
      close userlist
   else
      alerter("Couldn't find that user!", .t.)
      close userlist
      return .f.
   endif
   
return .t.

//  //
static function changelevel()
   // change user security level
   local aUserTemp := {"", "", "", "", "", "", "", "", "", ""}
   local cTempName, cSecure
   
   scrnhead("Change User Security Level")
   if "9" $ aUser[3]
      cTempName := getname()
      if len(alltrim(cTempName)) = 0
         alerter("No changes made.")
         return .f.
      endif
      aUserTemp :=  finduser(cTempName, .f., .f.)  // don't close dbf yet
      if aUserTemp[1] <> cTempName
         alerter("Sorry - couldn't find that user name.")
         close userlist
         return .f.
      endif
      if len(alltrim(aUserTemp[1])) > 0
         cSecure       := getlevel()
         aUserTemp[10] := getlotaccess()
         dbfopen("USERLIST")
         if ! reclock("Couldn't access the data record;security level not changed")
            return .f.
         endif   
         replace level with cSecure, ;
             leveldate with date(), ;
             levelby with aUser[1], lotaccess with aUserTemp[10]
         commit    
         unlock
         close userlist
      endif
      
   else
      alerter("Sorry, you aren't allowed to change security levels")
      return .f.
   endif
   
return .t.

//  //
static function userlog(cAction)
   // logon/logoff/changes to log database
   dbfopen("LOGFILE")
   append blank
   if ! reclock("Couldn't access the data record;log not updated!")
      return .f.
   endif   

   replace logfile->logdate with date(), ;
       logfile->logtime with time(), ;
       logfile->action with aUser[1] + " " + cAction + ": " + cTempName, ;
       logfile->statuscode with 50,  ;
       logfile->errorcode with 0,    ;
       logfile->operator with aUser[1]
   commit    
   unlock
   close logfile
   
return .t.

//  //
static function userprint()
   // print a report from the user logon/logoff database
   
   local pUnderOn := chr(27) + "&d0D" // underline on
   local pUnderOff := chr(27) + "&d@" // underline off
   scrnhead("User List Report")
   zSayMess(10, "Printing User Security Report")
   
   prn_output(1)
   ? padc("USER SECURITY REPORT", 80)
   ? padc(dtoc(date()) + " at " + left(time(), 5), 80)
   ? ""
   ? ""
   dbfopen("USERLIST")
   go top
   ? space(5) + pUnderOn + "USER NAME      LEVEL  LOT     PASSWORD LAST CHANGED " + pUnderOff
   do while .not. eof()
      ? space(5)
      ?? userlist->username
      ?? space(5)
      ?? userlist->level
      ?? space(5)
      ?? userlist->lotaccess
      ?? space(5)
      ?? userlist->passdate
      ?? " by " 
      ?? userlist->passby

//          aUserTemp[4]  := userlist->namedate
//          aUserTemp[5]  := userlist->passdate
//          aUserTemp[6]  := userlist->leveldate
//          aUserTemp[7]  := userlist->nameby
//          aUserTemp[8]  := userlist->passby
//          aUserTemp[9]  := userlist->levelby
      skip
   enddo
   ? ""
   ? ""
   ? space(5) + padc(" End of Report ", 75, "")
   ?? chr(12)
   prn_close()
   
return .f.

//  //
function LotLock(aUser, mLot)
   // checks the current users lot access levels to see if they match
   // returns .t./.f. depending on user's security
   // must be available to all programs, since it is used to limit
   // users to data from their own lot
   
   // you can use a similar function to prevent access to certain
   // functions if the security level is not OK. In this program,
   // the user array includes which lot is assigned to the user.
   // If the lot name doesn't match the lot name of the current record,
   // the user can't edit/delete/add data.
   
   local lOK := .t.
   // make sure mLot parameter passed
   if valtype(mLot) <> "C"
      alerter("Access to LOTLOCK without proper parameters!;Contact the Computer Wizard!")
      return .f.
   endif
   
   // check lot access codes in current user profile
   if ! alltrim(aUser[10]) $ alltrim(mLot)
      lOK := .f.
   endif
   // but allow "all lots" user to access
   if alltrim(aUser[10]) = "*"
      lOK := .t.
   endif
   if ! lOK
      alerter("Sorry, you can't do that for this lot!;Your user is assigned only to Lot " + aUser[10])
   endif
   
return lOK
//  //

//  //
//                            end of file                            //
//  //
