/***
*
*   Lblrun.prg
*   Clipper 5.0 LABEL FORM runtime system
*   Copyright (c) 1990 Nantucket Corp.  All rights reserved
*
*   Modified by Frederic Steppe for U.S.C. [76164,3333]
*
*   Compile:  /m/n/w
*/
 
#include "lbldef.ch"                            // Label array definitions
#include "error.ch"
 
// File-wide static declarations
STATIC aLabelData   := {}                       // Label definition array
STATIC aBandToPrint := {}
STATIC cBlank       := ""
STATIC lOneMoreBand := .T.
STATIC nCurrentCol  := 1                        // The current column in the band
STATIC lConcatEmpty := .T.
STATIC bAskMoreSamples := NIL
 
/***
*  SetLblEmpty( ... ) --> Last value
*  Specify if any empty field in the LBL definition must be skipped
*  at the bottom of the label
*/
FUNCTION SetLblEmpty( lLblEmpty )

   LOCAL LastValue := lConcatEmpty

   IF lLblEmpty!=NIL
      lConcatEmpty := lLblEmpty
   ENDIF
   
   RETURN LastValue


/***
*  SetLblAskSamples( ... ) --> Last code block
*  Specify a code block to execute in order to ask if more samples are
*  to be printed
*/
FUNCTION SetLblAskSamples( bAskSample )

   LOCAL LastValue := bAskMoreSamples

   IF VALTYPE(bAskSample) == 'B'
      bAskMoreSamples := bAskSample
   ENDIF
   
   RETURN LastValue


/***
*  __LabelForm( ... ) --> NIL
*  Print the specified (.lbl) definition for specified records
*  meeting specified scope and condition
*/
FUNCTION __LabelForm( cLBLName, lPrinter, cAltFile, lConsole, bFor, ;
                       bWhile, nNext, nRecord, lRest, lSample )
 
   LOCAL lPrintOn := .F.            // PRINTER status
   LOCAL lConsoleOn := .T.          // CONSOLE status
   LOCAL cExtraFile, lExtraState    // EXTRA status
   LOCAL xBreakVal, lBroke := .F.
   LOCAL err
   Local OldMargin 

   IF bAskMoreSamples == NIL 
      bASKMORESAMPLES := {|nNb| DefAskMoreSpl(nNb)}
   ENDIF

   // Resolve parameters
   IF cLBLName == NIL
      err := ErrorNew()
      err:severity := 2
      err:genCode := EG_ARG
      err:subSystem := "FRMLBL"
      Eval(ErrorBlock(), err)
   ELSE
      IF AT( ".", cLBLName ) == 0
         cLBLName := TRIM( cLBLName ) + ".LBL"
      ENDIF
   ENDIF
 
   IF lPrinter == NIL
      lPrinter := .F.
   ENDIF
 
   IF lConsole == NIL
      lConsole := .T.
   ENDIF
 
   IF lSample == NIL
      lSample := .F.
   ENDIF
 
   // Set output devices
   IF lPrinter                                  // To the printer
      lPrintOn   := SET( _SET_PRINTER, lPrinter )
   ENDIF
   IF lConsole                                  // To the screen
*      lConsoleOn := SET( _SET_CONSOLE, !lConsole )
      lConsoleOn := SET( _SET_CONSOLE )
   ENDIF
 
   IF (!Empty(cAltFile))                        // To file
      cExtraFile  := SET( _SET_EXTRAFILE, cAltFile )
      lExtraState := SET( _SET_EXTRA, .T. )
   ENDIF
 
   OldMargin := SET( _SET_MARGIN, 0)
   
   BEGIN SEQUENCE
 
      aLabelData := __LblLoad( cLBLName )        // Load the (.lbl) into an array
 
      // Add to the left margin if a SET MARGIN has been defined
      aLabelData[ LB_LMARGIN ] += OldMargin
 
      // Size the aBandToPrint array to the number of fields
      ASIZE( aBandToPrint, LEN( aLabelData[ LB_FIELDS ] ) )
      AFILL( aBandToPrint, SPACE( aLabelData[ LB_LMARGIN ] ) )
 
      // Create enough space for a blank record
      cBlank := SPACE( aLabelData[ LB_WIDTH ] + aLabelData[ LB_SPACES ] )
 
      // Handle sample labels
      IF lSample
                SampleLabels()
      ENDIF
 
      // Execute the actual label run based on matching records
      DBEval( { || ExecuteLabel() }, bFor, bWhile, nNext, nRecord, lRest )
 
      // Print the last band if there is one
      IF lOneMoreBand
         // Print the band
         AEVAL( aBandToPrint, ;
            { | BandLine | ; // PrintIt(SPACE(aLabelData[LB_LMARGIN])+ BandLine);
                        PrintIt( BandLine ) ;
                } ;
           )
      ENDIF
 
 
   RECOVER USING xBreakVal
 
      lBroke := .T.
 
   END  // Sequence
 
 
   // Clean up and leave
   aLabelData   := {}                    // Recover the space
   aBandToPrint := {}
   nCurrentCol  := 1
   cBlank       := ""
   lOneMoreBand :=.T.
 
   // clean up
   SET( _SET_PRINTER, lPrintOn )         // Set the printer back to prior state
   SET( _SET_CONSOLE, lConsoleOn )// Set the console back to prior state
 
   IF (!Empty(cAltFile))                 // Set extrafile back
      SET( _SET_EXTRAFILE, cExtraFile )
      SET( _SET_EXTRA, lExtraState )
   ENDIF
 
   IF lBroke
      // keep the break value going
      BREAK xBreakVal
   ENDIF
 
   SET( _SET_MARGIN, OldMargin)   
   
   RETURN NIL
 
/***
*  ExecuteLabel() --> NIL
*  Process the label array using the current record
*/
STATIC FUNCTION ExecuteLabel
   LOCAL nField, nMoreLines, aBuffer := {}, cBuffer
 
   // Load the current record into aBuffer
   FOR nField := 1 TO LEN( aLabelData[ LB_FIELDS ] )
      cBuffer := PADR(TRANSFORM( EVAL( aLabelData[ LB_FIELDS,nField,;
           LF_EXP ] ), ;
           REPLICATE( "X", aLabelData[ LB_WIDTH ] ) ), ;
                 aLabelData[ LB_WIDTH ] + aLabelData[ LB_SPACES ])
      IF aLabelData[ LB_FIELDS, nField, LF_BLANK ]
         IF lConcatEmpty .and. RTRIM( cBuffer ) == "" // If it's a blank field
         ELSE                                         // and we must skip it !
            AADD( aBuffer, cBuffer )
         ENDIF
      ELSE
         AADD( aBuffer, cBuffer )
      ENDIF
   NEXT
   ASIZE( aBuffer, LEN( aLabelData[ LB_FIELDS ] ) )
 
   // Add aBuffer to aBandToPrint
   FOR nField := 1 TO LEN( aLabelData[ LB_FIELDS ] )
      IF aBuffer[ nField ] == NIL
         aBandToPrint[ nField ] += cBlank
      ELSE
         aBandToPrint[ nField ] += aBuffer[ nField ]
      ENDIF
   NEXT
 
   IF nCurrentCol == aLabelData[ LB_ACROSS ]
      lOneMoreBand := .F.
      nCurrentCol  := 1
 
      // Print the band
      AEVAL( aBandToPrint, ;
                { | BandLine | ; //PrintIt(SPACE(aLabelData[LB_LMARGIN])+ BandLine);
                        PrintIt( BandLine ) ;
                } ;
           )
      nMoreLines := aLabelData[ LB_HEIGHT ] - LEN( aBandToPrint )
      IF nMoreLines > 0
         FOR nField := 1 TO nMoreLines
            PrintIt()
         NEXT
      ENDIF
      IF aLabelData[ LB_LINES ] > 0
 
         // Add the spaces between the label lines
         FOR nField := 1 TO aLabelData[ LB_LINES ]
            PrintIt()
         NEXT
 
      ENDIF
 
      // Clear out the band
      AFILL( aBandToPrint, SPACE( aLabelData[ LB_LMARGIN ] ) )
   ELSE
      lOneMoreBand := .T.
      nCurrentCol++
   ENDIF
 
   RETURN NIL
 
/***
*  SampleLabels() --> NIL
*  Print sample labels
*/
STATIC FUNCTION SampleLabels
   LOCAL nGetKey, lMoreSamples := .T., nField
   LOCAL aBand := {}
   LOCAL nSamplesPrinted := 0
 
   // Create the sample label row
   ASIZE( aBand, aLabelData[ LB_HEIGHT ] )
   AFILL( aBand, SPACE( aLabelData[ LB_LMARGIN ] ) +;
              REPLICATE( REPLICATE( "*", ;
              aLabelData[ LB_WIDTH ] ) + ;
              SPACE( aLabelData[ LB_SPACES ] ), ;
              aLabelData[ LB_ACROSS ] ) )
 
   // Prints sample labels
   DO WHILE lMoreSamples
 
      // Print the samples
       AEVAL( aBand, ;
                  { | BandLine | ;
                        PrintIt( BandLine ) ;
                } ;
           )
      IF aLabelData[ LB_LINES ] > 0
         // Add the spaces between the label lines
         FOR nField := 1 TO aLabelData[ LB_LINES ]
            PrintIt()
         NEXT
      ENDIF
      
      lMoreSamples := eval(bAskMoreSamples,++nSamplesPrinted)

   ENDDO
   RETURN NIL


/***
*  DefAskMoreSpl( <nSamples> ) --> Logical .T. if more samples to print
*  S'87-like 'Do you want more samples ? (Y/N)
*/
STATIC FUNCTION DefAskMoreSpl( nSamples )
   LOCAL nGetKey

   // Prompt for more
   @ ROW(), 0 SAY "Do you want more samples? (Y/N)"
   nGetKey := INKEY(0)
   @ ROW(), COL() SAY CHR(nGetKey)
   IF ROW() == MAXROW()
      SCROLL( 0, 0, MAXROW(), MAXCOL(), 1 )
      @ MAXROW(), 0 SAY ""
   ELSE
      @ ROW()+1, 0 SAY ""
   ENDIF
   IF UPPER(CHR(nGetKey)) == "N"
      RETURN .F.
   ELSE
      RETURN .T.
   ENDIF



/***
*  PrintIt( <cString> ) --> NIL
*  Print a string, then send a CRLF
*/
STATIC FUNCTION PrintIt( cString )
   IF cString == NIL
      cString := ""
   ENDIF
   QQOUT( rtrim(cString) )
   QOUT()
   RETURN NIL
