/********************************
        FIELDVAL.C

Written by:   David J. Sieber (CIS 76662,1632) in response to some requests
              on NANFORUM for such a function.

Uploaded as a free utility to the CI$ NANFORUM.

Update History:

03/16/90      First version, give me some feedback! (bugs?, problems?)

*********************************/

#include "nandef.h"
#include "extend.h"

/*      The SYMBOL structure
 *      --------------------
 *      Each field in a dbf (see FIELD structure below) has a pointer to
 *      a structure wherein can be found the field name.  The name occurs
 *      10 bytes into the structure, as a pointer to the string.
 */
typedef struct _symbol {
        char  unknown[10];
        char *name;
} SYMBOL;

/*      The FIELD structure
 *      -------------------
 *      Each dbf has an array of these structures. The 'type' is
 *      not the usual 'C', 'N', etc., but the values used are given in
 *      #define's below.  The 'len' and 'dec' fields are as expected.
 *      'offset' is the offset from the beginning of the record to this field.
 *      The bytes I don't know are labeled 'unknown'.  See the CLIPPER_DBF
 *      structure below for how to get at this structure.
 */
typedef struct _field {
        int   type;
        int   len;
        int   dec;
        int   unknown1;
        int   offset;
        int   unknown2;
        int   unknown3;
        SYMBOL *symbol;
} FIELD;

#define NFLD        0x0010         /* values for FIELD->type element */
#define DFLD        0x0040
#define LFLD        0x0080
#define CFLD        0x0100
#define MFLD        0x0300

/*      Clipper's internal dbf structure
 *      --------------------------------
 *      Each open dbf in Clipper has a structure of data in memory.
 *      I made up the names for these and haven't figured out what all
 *      of the fields are yet, but many useful things are here.  When
 *      possible, I named the fields after the corresponding function
 *      in clipper which returns the value (i.e. deleted, eof, bof, recno,
 *      and found are the values returned by those functions).  The hex
 *      offsets are what you'll see being referenced by the various functions,
 *      and the way they are referenced tells you what they are (int, char *,
 *      long, etc.). The 'unknown' fields are groups of bytes which I
 *      haven't tracked down yet.  If you like challenges, chase after
 *      them and fill 'em in <g>!
 *
 *      Pointers to other structures
 *      ----------------------------
 *      The 'fields' element points to an list of FIELD structures (above).
 *      There's another at 'idxtbl', which could be useful, but is unknown
 *      right now.  Starting at offset 0xbc is info about the relations (i.e.
 *      DBRELATION(), DBRSELECT()).
 *
 *      The 'recptr' field
 *      ------------------
 *      The MOST useful element of this structure is 'recptr', which points
 *      to the record buffer.  You can reach in and grab any field from this
 *      area, as long as you know where to reach.  That's what FIELDVAL does.
 *      A slicker way to access it is to define a C structure which matches
 *      your dbf record exactly, and cast 'recptr' into a pointer to your
 *      structure, then just grab anything you want.  Always remember that
 *      each field starts with a 1-byte deletion indicator, which contains
 *      '*' if the record is marked for deletion.  String fields are _not_
 *      zero-terminated, and numerics are in ASCII, so you'll have to convert
 *      them.  When done properly, however, you can give an amazing performance
 *      boost to your program.
 */
typedef struct _clipper_dbf {
        char   signature;        /* 0x00 */           /* 03=dbf, 83=dbf+dbt */
        char   ymd[3];           /* 0x01,02,03 */     /* y+1900 m d */
        long   lastrec;          /* 0x04,05,06,07 */
        int    data_off;         /* 0x08,09 */
        int    recsize;          /* 0x0a,0b */
        char   pad[20];          /* 0x0c-1f */
        long   reclen;           /* 0x20,21,22,23 */
        long   header;           /* 0x24,25,26,27 */
        long   recno;            /* 0x28,29,2a,2b */
        char   unknown1[8];      /* 0x2c-33 */
        int    dbf_handle;       /* 0x34,35 */
        int    has_dbt;          /* 0x36,37 */
        int    dbt_handle;       /* 0x38,39 */
        char   unknown2[12];     /* 0x3a-45 */
        int    deleted;          /* 0x46,47 */
        int    eof;              /* 0x48,49 */
        int    bof;              /* 0x4a,4b */
        int    found;            /* 0x4c,4d */
        char   unknown3[2];      /* 0x4e,4f */
        long   lock_offset;      /* 0x50,51,52,53 */
        char   unknown4[6];      /* 0x54-59 */
        char  *recptr;           /* 0x5a,5b,5c,5d */
        char   unknown5[4];      /* 0x5e-61 */
        int    indexord;         /* 0x62,63 */
        int    indexcnt;         /* 0x64,65 */
        char  *idxtbl;           /* 0x66,67,68,69 */
        char   unknown6[60];     /* 0x6a-a5 */
        int    fcount;           /* 0xa6,a7 */
        FIELD *fields;           /* 0xa8,a9,aa,ab */
        char   unknown7[4];      /* 0xac-af */
        int    hasfilter;        /* 0xb0,b1 */
        char  *dbfilter;         /* 0xb2,b3,b4,b5 */
        char   unknown8[4];      /* 0xb6-b9 */
        int    numrelations;     /* 0xba,bb */
        char   unknown9[2];      /* 0xbc,bd,... */
} CLIPPER_DBF;

/*      the _dbf pointer
 *      ----------------
 *      _dbf is a public symbol in clipper.lib.  It always points
 *      to a pointer to a ClIPPER_DBF structure for the current dbf.
 *      if no dbf is open in the selected area, _dbf points to NULL.
 */
extern CLIPPER_DBF ** near _dbf;   /* ptr to ptr to currently selected dbf */

#define DBF         (*_dbf)        /* ptr to current dbf, NULL if none open */

/*      easy access to elements of the fields */
#define FNAME(n)    (DBF->fields[(n)-1].symbol->name)
#define FTYPE(n)    (DBF->fields[(n)-1].type)
#define FLEN(n)     (DBF->fields[(n)-1].len)
#define FDEC(n)     (DBF->fields[(n)-1].dec)

/* clipper internals */
extern void _aton(char *s, int slen, double *dest, int *dlen, int *ddec);
extern void _putd(double x, int len, int dec);
extern void _SAVE_RET(void);

/********************************
        fldarg
 ********************************/

/* gets and validates the field number parameter.
 * returns 0 if:
 *      no DBF is open in the current work area
 *      no argument was passed (we don't care if more than one was passed)
 *      the argument was not numeric
 *      the argument is an illegal field number for the current dbf
 * otherwise the field number is returned
 */

static int fldarg(void)
{
        int   n;

        if (!DBF || PCOUNT < 1 || !ISNUM(1))
           return (0);

        n = _parni(1);        /* desired field number */

        /* make sure it's legal */
        return (n >= 1 && n <= DBF->fcount ? n : 0);
}

/********************************
        get_numeric
 ********************************/

/*      return a numeric field to clipper.
 *      this routine uses some calls into clipper's guts to get
 *      the job done.  if you'd rather not do this, you'll need to
 *      write code which converts ASCII to numeric, or call a supplied
 *      library function (as in MSC's library).  Also, the standard
 *      _retnd isn't used since it doesn't allow us to specify the length
 *      and decimal places in a numeric (you could also check the value
 *      of the field, and use either _retni or _retnl if you like, but
 *      you'll find that this makes no _real_ difference).
 */

static void get_numeric(FIELD *fp, char *bp)
{
        double   numeric;       /* used to compute numeric return */
        int      len, dec;      /* temporary for numeric compute */

/*      this is a call to an internal, _aton, in CLIPPER.LIB.  it takes
 *      a character pointer and length, and stuffs supplied locations
 *      with the double value, length, and decimal places of the string.
 *      we'll ignore the length and decimal places it gives us and use
 *      the values given in the dbf definition when we return it to Clipper.
 */
        _aton(bp, fp->len, &numeric, &len, &dec);

/*      these two lines do what _retnd() does, except that we use
 *      the dbf's length and decimal places to return a more
 *      displayable representation to Clipper.
 *      _retnd() uses the current value of SET DECIMALS TO for
 *      the decimal places returned, and the length is max'd out.
 *      this wouldn't do for values which are really int's, etc.
 *      (just CV the _retnd function to see how it does this)
 *      _putd puts a numeric onto clipper's stack (not the same as
 *       the CPU stack).  there's a version of _put for all the data types.
 */
        _putd(numeric, fp->len, fp->dec);
        _SAVE_RET();
}

/********************************
        get_character
 ********************************/

/*      just return the string, making sure to use the _retclen version,
 *      since strings in the dbf record aren't 0-terminated
 */

static void get_character(FIELD *fp, char *bp)
{
        _retclen(bp, fp->len);
}

/********************************
        get_logical
 ********************************/

/*      return .T. if field is either a 'Y' or 'T', otherwise .F. */

static void get_logical(FIELD *fp, char *bp)
{
        fp;   /* not used */
        _retl(*bp == 'Y' || *bp == 'T');
}

/********************************
        get_date
 ********************************/

/*      copy the date field to a local buffer, 0-terminate it, and
 *      return it to clipper
 */

static void get_date(FIELD *fp, char *bp)
{
        char     datestr[9];    /* holds 0-terminated date string */
        int      i;
        char    *cp;

        fp;   /* not used */

        for (i = 0, cp = datestr; i < 8; i++)    /* need to put a '\0' on it */
           *cp++ = *bp++;

        *cp = '\0';

        _retds(datestr);
}

/********************************
        FIELDVAL
 ********************************/

CLIPPER FIELDVAL(void)
{
        FIELD   *fp;            /* points to current FIELD structure */
        char    *bp;            /* points to field within dbf buffer */
        int      fld;           /* desired field number */

        if (!(fld = fldarg())) {
           _retc("");        /* what to do if bad args? */
           return;
        }

/*      point fp to the FIELD structure we want.
 *      the array is zero-based, while our arg is one-based,
 *      so we subtract one to get the proper offset.
 */
        fp = DBF->fields + fld - 1;

/*      point bp to the beginning of the field in the dbf buffer.
 */
        bp = DBF->recptr + fp->offset;

        switch (fp->type) {
        case NFLD:
           get_numeric(fp, bp);
           break;

        case CFLD:
           get_character(fp, bp);
           break;

        case LFLD:
           get_logical(fp, bp);
           break;

        case DFLD:
           get_date(fp, bp);
           break;

        case MFLD:                 /* memo field, just return null string */
        default:                   /* unknown field type also comes here */
           _retc("");
           break;
        }
}

/********************************
        BONUS SECTION

        FLDNAME, FLDTYPE, FLDLEN, & FLDDEC - basically what you would
        get by calling AFIELDS, but without having to make arrays, etc.

 ********************************/

/********************************
        FLDNAME
 ********************************/

/* returns name of field <n>, "" if anything wrong */
/* same as FIELD/FIELDNAME */

CLIPPER FLDNAME(void)
{
        int   fld;

        _retc((fld = fldarg()) ? FNAME(fld) : "");
}

/********************************
        FLDTYPE
 ********************************/

/* returns field type <n> as a one character string, "" if anything wrong */

CLIPPER FLDTYPE(void)
{
        int   fld;

        if (fld = fldarg()) {
           switch (FTYPE(fld)) {
           case NFLD:
              _retc("N");
              break;
           case DFLD:
              _retc("D");
              break;
           case LFLD:
              _retc("L");
              break;
           case CFLD:
              _retc("C");
              break;
           case MFLD:
              _retc("M");
              break;
           default:
              _retc("");
              break;
           }
        } else {
           _retc("");
        }
}

/********************************
        FLDLEN
 ********************************/

/* returns the length of field <n>, -1 if anything wrong */

CLIPPER FLDLEN(void)
{
        int   fld;

        _retni((fld = fldarg()) ? FLEN(fld) : -1);
}

/********************************
        FLDDEC
 ********************************/

/* returns the decimals for field <n>, -1 if anything wrong */

CLIPPER FLDDEC(void)
{
        int   fld;

        _retni((fld = fldarg()) ? FDEC(fld) : -1);
}

/*      end of fieldval.c */
