**********************
**   ErrorSys.PRG   **
*****************************************************************************
* Copyright (c) 1992 Delcom International Software Engineering
*****************************************************************************
//
//  This file contains code for run-time error-handling.
//

#include "Demo.CH"
#include "Error.ch"
#include "Set.ch"

// put messages to STDERR
#command ? <list,...>   =>  ?? Chr(13) + Chr(10) ; ?? <list>
#command ?? <list,...>  =>  OutErr(<list>)

// used below
#define NTRIM(n)        ( ALLTRIM( STR( n )) )


PROCEDURE ErrorSys

ERRORBLOCK( {|e| MainError(e)} )

IF MEMORY(0) > 35
	InitMouse()
	SetMousePos( ( MAXROW() / 2 ), ( MAXCOL() / 2 ) )
	SetMouse( MOUSE_WAIT )
ELSE
	MemLowMessage()
ENDIF

RETURN
//
// EOP: ErrorSys
//


****************************
**   FUNCTION MainError   **
*****************************************************************************
//
//  Default error handler.
//
//  Parameters: oError - The error object created by the error.
//
//     Returns: xRetCode
//

FUNCTION MainError ( oError )

LOCAL xRetCode,   ; // Function's return code
		cMessage      // Error message text

DISPEND()  // Just in case!

DO CASE
	//
	//  By default, division by zero yields zero
	//
	CASE oError:genCode == EG_ZERODIV
		xRetCode := 0

	//
	// For network open error, set NETERR() and subsystem default
	//
	CASE oError:genCode == EG_OPEN .AND. oError:osCode == 32 ;
			.AND. oError:canDefault

		NETERR( .T. )
		xRetCode := .F.

	//
	// For lock error during APPEND BLANK, set NETERR() and subsystem default
	//
	CASE oError:genCode == EG_APPENDLOCK .AND. oError:canDefault
		NETERR( .T. )
		xRetCode := .F.

	//
	// Low memory condition!
	//
	CASE oError:genCode == EG_MEM
		MemLowMessage()

		xRetCode := .F.

	//
	// Too many files open!
	//
	CASE oError:genCode == EG_OPEN .AND. oError:osCode == 4
		TooManyFiles( oError )

		xRetCode := .F.

	//
	// Corrupt database or index!
	//
	CASE oError:genCode == EG_CORRUPTION
		Corruption( oError )

		xRetCode := .F.

ENDCASE

IF VALTYPE( xRetCode ) <> "N"
	ERRORLEVEL( 1 )

	//
	// Build error message and display.
	//
	cMessage := ErrorMessage( oError )

	BadNews( oError, cMessage )

	Quit_Sys()
ENDIF

RETURN xRetCode
//
// EOP: MainError
//


*******************************
**   FUNCTION ErrorMessage   **
*****************************************************************************
//
//  Builds the error message associated with the error.
//
//  Parameters: e - The error object created by the error.
//
//     Returns: cMessage - The message text.
//

FUNCTION ErrorMessage(e)

LOCAL cMessage     // Short message sent to screen

//
// Start error message
//
cMessage := IF( e:severity > ES_WARNING, "Error ", "Warning " )

//
// Add subsystem name if available
//
IF ValType(e:subsystem) == "C"
   cMessage += e:subsystem()
ELSE
   cMessage += "???"
ENDIF

//
// Add subsystem's error code if available
//
IF ValType(e:subCode) == "N"
   cMessage += ("/" + NTRIM( e:subCode ))
ELSE
   cMessage += "/???"
ENDIF

//
// Add error description if available
//
IF ValType(e:description) == "C"
   cMessage += ("  " + e:description )
ENDIF

//
// Add either filename or operation
//
IF .NOT. EMPTY( e:filename )
   cMessage += (": " + e:filename )

ELSEIF .NOT. EMPTY( e:operation )
   cMessage += (": " + e:operation )

ENDIF

RETURN cMessage
//
// EOP: ErrorMessage
//


***************************
**   PROCEDURE BadNews   **
*****************************************************************************
//
//  Prints the error message associated with the error.
//
//  Parameters: oError   - The error object.
//              cMessage - The error message.
//

STATIC PROCEDURE BadNews ( oError, cMessage )

LOCAL cDetailMsg,     ; // Detailed message for the Error database
		cErrorDet,      ; // Error object details (DEBUG only)
		lWaited,        ; // .T. if already 'waited' for error message
		i                 // Loop counter

cMessage := IF( cMessage == NIL, "(No message)", cMessage )
lWaited  := .F.

//
// Build the detailed message for the database.
//
cDetailMsg := cMessage + _CRLF + _CRLF
i := 3

DO WHILE !EMPTY( ProcName( i ))
	cDetailMsg += "Called from " + ALLTRIM( ProcName( i )) + ;
		 "(" + NTRIM( ProcLine( i )) + ")" + CHR(13) + CHR(10)
	i++
ENDDO

DISPEND()

?

FOR i := 1 TO MLCOUNT( cDetailMsg, 254 )
	? ALLTRIM( MEMOLINE( cDetailMsg, 254, i ))

	IF ROW() >= 23 .AND. !lWaited
		lWaited := .T.
		WAIT
	ENDIF
NEXT

?
WAIT

RETURN
//
// EOP: BadNews
//


*********************************
**   PROCEDURE MemLowMessage   **
*****************************************************************************
//
//  Prints the "Memory Low" message, and provides user with exit options.
//

PROCEDURE MemLowMessage

LOCAL cMessage,   ; // Message text
		aOptions,   ; // Array storing options
		nChoice       // User's selection

cMessage := "There is not enough memory to run DUtil safely.;;" + ;
				"Space: " + ALLTRIM( STR( MEMORY(0) )) + "K;"

aOptions := { "Quit", "Resume" }

DISPEND()

nChoice := ALERT( cMessage, aOptions )

//
// If the user selected 'Quit', get outta there!
//
IF nChoice == 1
	DBUNLOCKALL()
	DBCOMMITALL()

	SETPOS( MAXROW(), 0 )
	@ MAXROW(),0 CLEAR

	Quit_Sys()
ENDIF

RETURN
//
// EOP: MemLowMessage
//


******************************
**   PROCEDURE Corruption   **
*****************************************************************************
//
//  Prints the file corrupted message, and provides user with exit options.
//
//  Parameters: oError - The error object.
//

PROCEDURE Corruption ( oError )

LOCAL cMessage,  ; // Message text
		aOptions,  ; // Array storing options
		nChoice      // User's selection

cMessage := "DUtil has detected corruption in the file;" + ;
				ALLTRIM( oError:filename ) + ";"

aOptions := { "Quit", "Resume" }

DISPEND()

TONE( 250, 1 )

nChoice := ALERT( cMessage, aOptions )

IF nChoice == 1
	DBUNLOCKALL()
	DBCOMMITALL()

	SETPOS( 24, 0 )
	@ 24,0 CLEAR

	Quit_Sys()
ENDIF

RETURN
//
// EOP: Corruption
//


********************************
**   PROCEDURE TooManyFiles   **
*****************************************************************************
//
//  Prints the "Too Many Files" message, and provides user with exit options.
//
//  Parameters: oError - The error object.
//

PROCEDURE TooManyFiles ( oError )

LOCAL cMessage,  ; // Message text
		aOptions,  ; // Array storing options
		nChoice      // User's selection

cMessage := "There are too many files open!.;" + ;
				"Increase the FILES=xxx environment variable in CONFIG.SYS.;;" + ;
				ALLTRIM( oError:filename )

aOptions := { "Quit", "Resume" }

DISPEND()

TONE( 250, 1 )

nChoice := ALERT( cMessage, aOptions )

IF nChoice == 1
	DBUNLOCKALL()
	DBCOMMITALL()

	SETPOS( 24, 0 )
	@ 24,0 CLEAR

	Quit_Sys()
ENDIF

RETURN
//
// EOP: TooManyFiles
//
