// CLASSic OINI Class
// By John D. Van Etten
// Written in CLASSic

// INI file manager

#include "Classic.ch"
#include "FileIO.ch"

#DEFINE pTYPE    1
#DEFINE pHEADER  2
#DEFINE pNAME    2
#DEFINE pVALUE   3
#DEFINE pCOMMENT 4


Begin Class oIni

  Global:
    Var FileName      READONLY  Type Character
    Var Error                   Type Logical
    Var Locked        READONLY  Type Logical

    Method New        Constructor        // Creates an Empty Ini File
    Method Open                          // Read INI file
    Method Save                          // Save Ini File
    Method Save       Message SaveAs     // Save Ini File As
    Method Close                         // Save then Clear Object
    Method INILock    Message Lock       // Lock the INI File
    Method INIUnLock  Message UnLock     // Unlock the INI file
    Method Comment                       // Write a Comment
    Method IniSet     Message Set        // Set a Value
    Method IniGet     Message Get        // Get a Value
    Method GetSet     Message sGet       // Get a Value;If not found, Set it
    Method GetSet     Message Default    // If not found, Set it
    Method Remove     Message Remove     // Delete an entry

  Local:
    Var aIni
    Var oLock

    Method ParseLine
    Method ReadIni
    Method WriteIni
    Method AddHeader
    Method MakeType
    Method MakeChar

End Class

Method New( cFile )
  ::Error    := .t.
  ::aIni     := NIL
  ::Locked   := .f.
  ::Filename := cFile
Return( Self )

Method Open( cName )

  if !::ReadIni( cName )
    ::Error := .f.
    ::aIni := NIL
  endif

Return( Self )

Method Save( cSaveAs := ::FileName )
  Local oFile

  ::Error := ::WriteIni( cSaveAs )
Return( Self )

Method INILock( cName )
  ::oLock := oFILEIO():Open( ::FileName, FO_EXCLUSIVE )
  ::Locked := !::oLock:Error
  ::Error := ::oLock:Error
Return( ::Locked )

Method INIUnLock( cName )
  ::oLock:Close()
  ::Error := ::oLock:Error
  ::Locked := .f.
Return( Self )


Method Close()
  ::Save()
  ::New()
Return( Self )

Method ReadIni( cName )
  Local cLine
  Local aData
  Local oFile

  ::Error := .t.
  If !File( cName )
    ::New()
    ::Error := .f.
    ::FileName := cName
  else
    oFile := oFileIO():Open( cName )
    If !oFile:Error
      ::FileName := cName
      oFile:Top()

      Do while !oFile:EOF() .and. !oFile:Error
        cLine := oFile:ReadLine()
        aData := ::ParseLine( cLine )

        If oFile:Error
          Exit // ERROR!
        elseIf aData[ pTYPE ] == "H"
          If empty( ::aIni )
            ::aIni := { { aData[ pHEADER ], {}, aData[ pCOMMENT ] } }
          else
            aadd( ::aIni, { aData[ pHEADER ], {}, aData[ pCOMMENT ] })
          endif
        elseIf aData[ pTYPE ] == "V"
          If empty( ::aIni )
            ::aIni := { { "", {}, NIL } }
          Endif
          aadd( atail( ::aIni )[ 2 ], { aData[ pNAME ], aData[ pVALUE ], aData[ pCOMMENT ] })
        else
          If empty( ::aIni )
            ::aIni := { { chr(255), {}, aData[ pCOMMENT ] } }
          Else
            aadd( atail( ::aIni )[ 2 ], { chr(255), "", aData[ pCOMMENT ] })
          endif
        Endif
      Enddo
      ::Error := oFile:Error
      oFile:Close()
      ::Error := ::Error .or. oFile:Error
    Else
      ::FileName := ""
      ::Error := .t.
    Endif
  endif

Return( !::Error )

Method ParseLine( cLine )
  Local cTrim := Alltrim( cLine )
  Local aData := { "C", cLine, "", NIL }
  Local nSpaces

  If Left( cTrim, 1 ) == "[" .and. "]" $ cTrim
    aData[ pTYPE ] := "H"
    aData[ pHEADER ] := substr( Left( cTrim, at( "]", cTrim ) - 1 ), 2 )
    aData[ pCOMMENT ] := rtrim( substr( cTrim, at( "]", cTrim ) + 1 ))
  elseIf Left( cTrim, 2 ) == "//" .or. Left( cTrim, 1 ) == ";"
    aData[ pTYPE ] := "C"
    aData[ pCOMMENT ] := rtrim( cLine )
  Elseif "=" $ substr( cTrim, 2 )
    aData[ pTYPE ] := "V"
    aData[ pNAME ] := Left( cLine, at( "=", cLine ) - 1 )
    aData[ pVALUE ] := rtrim( Substr( cLine, at( "=", cLine ) + 1 ) )
    If "//" $ aData[ pVALUE ]
      aData[ pCOMMENT ] := Substr( aData[ pVALUE ], rat( "//", aData[ pVALUE ] ))
      aData[ pVALUE ] := Left( aData[ pVALUE ], rat( "//", aData[ pVALUE ] ) - 1 )
      nSpaces := Len( aData[ pVALUE ] ) - Len( rtrim( aData[ pVALUE ] ))
      aData[ pCOMMENT ] := rTrim( Space( nSpaces ) + aData[ pCOMMENT ])
      aData[ pVALUE ] := rtrim( aData[ pVALUE ] )
    Endif
  Endif
Return( aData )

Method WriteIni( cName )
  Local oFile := oFileIO()
  Local nHeader
  Local nLine
  Local cComment
  Local lWriteComment := .f.
  Local lLocked := ::Locked

  If upper( oFile:Ext( cName )) != ".BAK" .and. file( cName )
    oFile:Move( cName, oFile:Path( cName ) + oFile:Name( cName ) + ".BAK" )
  Endif

  if lLocked
    ::UnLock()
  endif

  oFile:Create( cName )
  ::Error := .t.
  If !oFile:Error
    ::FileName := cName

    For nHeader := 1 to Len( ::aIni )
      cComment := ::aIni[ nHeader ][ 3 ]
      Default cComment to ""

      If nHeader != 1 .and. !lWriteComment
        oFile:WriteLine( "" )
        If oFile:Error
          exit
        Endif
      Endif

      If ::aIni[ nHeader ][ 1 ] != chr( 255 )
        oFile:WriteLine( "[" + ::aIni[ nHeader ][ 1 ] + "]"  + cComment )
        lWriteComment := .f.
      Else
        oFile:WriteLine( cComment )
        lWriteComment := .t.
      Endif
      If oFile:Error
        exit
      Endif

      For nLine := 1 to Len( ::aIni[ nHeader ][ 2 ] )
        cComment := ::aIni[ nHeader ][ 2 ][ nLine ][ 3 ]
        Default cComment to ""

        If ::aIni[ nHeader ][ 2 ][ nLine ][ 1 ] == chr(255)
          oFile:WriteLine( cComment )
          lWriteComment := .t.
        Else
          oFile:WriteLine( ::aIni[ nHeader ][ 2 ][ nLine ][ 1 ] + "=" + ;
                           ::aIni[ nHeader ][ 2 ][ nLine ][ 2 ] + cComment )
          lWriteComment := .f.
        Endif
        If oFile:Error
          exit
        Endif
      Next
      If oFile:Error
        Exit
      Endif
    Next
    ::Error := oFile:Error
    oFile:Close()
    ::Error := ::Error .or. oFile:Error
  Else
    ::Error := .t.
  Endif

  If lLocked
    ::Lock()
  Endif
Return( ::Error )

Method DelHeader( cHeader )
  Local nHeader
  Local cFind

  If !Empty( ::aIni )
    cFind := Upper( alltrim( cHeader ))
    nHeader := Ascan( ::aIni, {|x| Upper( Alltrim( x[ 1 ])) == cFind })

    If nHeader > 0
      aKill( ::aIni, nHeader )
    Endif
  Endif
Return (Self)

Method IniSet( cHeader, cName, xValue, cComment )
  Local cFind
  Local nHeader := 0
  Local nName

  If cComment != NIL
    cComment := " // " + cComment
  Endif

  If Empty( ::aIni )
    If xValue != NIL
      ::aIni := {}
      nHeader := 0
    Endif
  else
    cFind := Upper( alltrim( cHeader ))
    nHeader := Ascan( ::aIni, {|x| Upper( Alltrim( x[ 1 ])) == cFind })
  Endif

  if nHeader == 0
    if xValue != NIL
      nHeader := ::AddHeader( cHeader )
      aadd( ::aIni[ nHeader ], { cName, ::MakeChar( xValue ), cComment })
    Endif
  Elseif full( cName )
    cFind := Upper( alltrim( cName ))
    nName := Ascan( ::aIni[ nHeader ][ 2 ], ;
                      {|x| Upper( Alltrim( x[ 1 ])) == cFind })

    If nName != 0
      if xValue == NIL
        aKill( ::aIni[ nHeader ][ 2 ], nName )
      Else
        ::aIni[ nHeader ][ 2 ][ nName ][2] := ::MakeChar( xValue )
        if cComment != NIL
          ::aIni[ nHeader ][ 2 ][ nName ][3] := cComment
        elseif "//" $ ::aIni[ nHeader ][ 2 ][ nName ][2] .and. ;
               ::aIni[ nHeader ][ 2 ][ nName ][3] == NIL
          ::aIni[ nHeader ][ 2 ][ nName ][3] := " //"
        endif
      Endif
    elseif xValue != NIL
      aadd( ::aIni[ nHeader ][ 2 ], { cName, ::MakeChar( xValue ), cComment })
    elseif cComment != NIL
      aadd( ::aIni[ nHeader ][ 2 ], { cName, "", cComment })
    Endif
  Else
    aKill( ::aIni, nHeader )
  Endif
Return (Self)

Method AddHeader( cHeader )
  Local nPos := 1
  If empty( cHeader )
    aSize( ::aIni, len( ::aIni ) + 1 )
    aIns( ::aIni, 1 )
    ::aIni[ 1 ] := { "", {}, NIL }
  else
    if empty( ::aIni )
      ::aIni := {}
    endif
    aadd( ::aIni, { cHeader, {}, NIL } )
    nPos := len( ::aIni )
  endif
Return( nPos )

Method Comment( cComment, cHeader, cName )
  Local cFind
  Local nHeader := 0
  Local nName := 0

  If cComment != NIL
    cComment := " // " + cComment
  Endif

  cFind := Upper( alltrim( cHeader ))
  nHeader := Ascan( ::aIni, {|x| Upper( Alltrim( x[ 1 ])) == cFind })

  If nHeader == 0
    nHeader := ::AddHeader( cHeader )
  Endif

  If cName == NIL
    ::aIni[ nHeader ][ 3 ] := cComment
  Else
    cFind := Upper( alltrim( cName ))
    nName := Ascan( ::aIni[ nHeader ][ 2 ], ;
                      {|x| Upper( Alltrim( x[ 1 ])) == cFind })

    If nName != 0
      ::aIni[ nHeader ][ 2 ][ nName ][3] := cComment
    else
      aadd( ::aIni[ nHeader ][ 2 ], { cName, "", cComment })
    Endif
  Endif
Return( Self )

Method GetSet( cHeader, cName, xDefault, lSave )
  Local xValue := ::Get( cHeader, cName,, Valtype( xDefault ) )
  Local Message := CallingMsg()

  Default lSave to MESSAGE == "DEFAULT" // If Default, the save it!
  If xValue == NIL
    xValue := xDefault
    Default xDefault to ""
    ::Set( cHeader, cName, xDefault )
    If lSave
      ::Save()
    Endif
  Endif
Return( xValue )

Method IniGet( cHeader, cName, xDefault, cType := Valtype( xDefault ) )
  Local xValue
  Local cFind
  Local nHeader := 0
  Local nName

  If Empty( ::aIni )
    xValue := xDefault
  else
    cFind := Upper( alltrim( cHeader ))
    nHeader := Ascan( ::aIni, {|x| Upper( Alltrim( x[ 1 ])) == cFind })
  Endif

  if nHeader == 0
    xValue := xDefault
  Else
    cFind := Upper( alltrim( cName ))
    nName := Ascan( ::aIni[ nHeader ][ 2 ], ;
                      {|x| Upper( Alltrim( x[ 1 ])) == cFind })

    If nName == 0
      xValue := xDefault
    Else
      xValue := ::aIni[ nHeader ][ 2 ][ nName ][ 2 ]
    Endif
  Endif

  xValue := ::MakeType( xValue, Upper( Left( cType, 1 )) )
Return( xValue )

Method MakeType( xValue, cType )
  If Valtype( xValue ) == "C"
    If cType == "D"
      xValue := ctod( alltrim( xValue ))
    elseIf cType == "N"
      xValue := val( alltrim( xValue ))
    elseIf cType == "L"
      xValue := ( 0 != aScan( { "YES", ".T.", "TRUE" }, ;
                              upper( alltrim( xValue ))))
    Endif
  Endif
Return( xValue )

Method MakeChar( xValue )
  Local cType := Valtype( xValue )

  If cType == "D"
    xValue := dtoc( xValue )
  elseIf cType == "N"
    xValue := ltrim( str( xValue ))
  elseIf cType == "L"
    xValue := IIF( xValue, "True", "False" )
  elseIf cType != "C"
    xValue := "** ERROR ** Can't Save Type: `" + cType + "'"
    ::Error := .t.
  Endif
Return( rTrim( xValue ))

Method Remove( cHeader, cEntry )
  ::Set( cHeader, cEntry, NIL )
Return( Self )
