// CLASSic OFILEIO Class
// By John D. Van Etten
// Written in CLASSic

#include "Classic.ch"
#include "FileIO.ch"

#xTranslate pEOL  => Chr(13)+Chr(10)

Begin Class oFileIO( bDrive )

  Global:
    Var Cargo
    Var BufferSize        Type Numeric // Internal Buffer Size
    Var Handle   READONLY              // The File Handle
    Var FileName READONLY              // The filename of the file opened
    Var fError   READONLY              // The File Error Number
    Var cError   READONLY              // Description of Error
    Var Opened   READONLY
    Method Var FileError Message Error Type Logical // Was their an error?

    // For Opened Files
    Method Var Pointer Type Numeric           // The Current file pointer
    Method Var Size READONLY                  // The Size of the File
    Method Var fDate Message Date READONLY    // Get the File's Date
    Method Var fTime Message Time READONLY    // Get The File's time

    Method Init        Constructor   // Default object settings
    Method Open                      // Open a File
    Method Create                    // Creates a new file
    Method Use                       // Use a file already opened
    Method Clone()    Type Object    // Returns New object, Same file
    Method fileClose() Message Close // Close the file
    Method Top()                     // Goto the top of the file
    Method Bottom()                  // Goto the bottom of the file
    Method fSkip      Message Skip   // Skip x Bytes in the file
    Method fEOF()     Message EOF    // Is it at the EOF
    Method fileSeek   Message Goto   // Goto a file position
    Method fFind      Message Find   // Search for a string
    Method ReadLine()                // Read a line from the file
    Method Read                      // Read x bytes from the file
    Method WriteLine                 // Write a Line to the file
    Method Write                     // Write a string to a file
    Method LineEval                  // Simular a DBEVAL, uses lines
    Method ByteCopy                  // Copy bytes from this object to another

    // Closed File
    Method FileDate                  // Get the date stamp of a file
    Method FileTime                  // Get the time stamp of a file
    Method FileCopy    Message Copy  // Copy a file
    Method Move                      // Move a file
    Method FileRename  Message Rename// Rename a file
    Method FileDelete  Message Erase // Delete a file
    Method FileDelete  Message Delete// Delete a file
    Method GetDir      Message Directory // same as Clipper's DIRECTORY()
    Method PathEval                  // Like aEval but using a path list
    Method TreeEval                  // Like PathEval but uses DOS Dir tree
    Method DirEval                   // Like PathEval but uses files
    Method TreeDirEval               // Combines TreeEval and DirEval
    Method FullName                  // Gets just the filename
    Method Name                      // Gets just the filename w/o extension
    Method Ext                       // Gets just the file extension
    Method Path                      // Gets just the file path
    Method Drive                     // Gets just the file drive
    Method Volume                    // Gets a Drive volume
    Method TempName                  // Create a Unique filename

    Static Var bDrive Type Block                    // Code block to get the current drive
    Method Var CurrDir Message CurDir Type Char     // The Current Directory
    Method Var CurrDrive Message CurDrive Type Char // The Current Drive using bDrive
    Method Var CurrPath Message CurPath Type Char   // The Current path ( Drive + Path )

  Local:
    Method SetPointer                // Store the current file pointer
    Method CharError                 // Return Error Number as a string

End Class:Init( bDrive )

Method Init( bDrive )
  ::Close()
  ::BufferSize := 40
  ::FileName := ""
  ::Handle := -1
  ::fError := 0
  ::Error := .f.
  ::cError := ""
  ::Pointer := 0
  If Valtype( bDrive ) == "B"
    ::bDrive := bDrive
  Endif
  ::Opened := .f.
Return( Self )

Method Open( cFile, nMode := FO_READ )

  ::Close()
  ::Filename := cFile
  ::Handle := fopen( cFile, nMode )
  ::Error := NIL
  ::Opened := !::Error
  ::SetPointer()
Return( Self )

Method Use( nHandle, cFile )
  ::Handle := nHandle
  ::FileName := cFile
  ::Opened := .t.
  ::SetPointer()
Return( Self )

Method Create( cFile, nMode := FC_NORMAL )

  ::Close()
  ::Filename := cFile
  ::Handle := fcreate( cFile, nMode )
  ::Error := NIL
  ::Opened := !::Error
  ::SetPointer()
Return( Self )

Method Clone()
  Local oCopy := oFileIO():Use( ::Handle, ::FileName )
  oCopy:Pointer := ::Pointer
  oCopy:Error   := ::Error
  oCopy:fError  := ::fError
  oCopy:cError  := ::cError
  oCopy:Opened  := ::Opened
Return( oCopy )

Method FileClose()
  If ::Handle != NIL
    If ::Handle > 0
      ::Error := !fclose( ::Handle )
      If !::Error
        ::Handle := -1
        ::Opened := .f.
      Endif
    Endif
  Endif
Return( Self )

Method Top()
  fseek( ::Handle, 0, FS_SET )
  ::SetPointer()
Return( Self )

Method Var Size()
  Local nSize := fseek( ::Handle, 0, FS_END )
  ::Goto()
Return( nSize )

Method Bottom()
  fseek( ::Handle, 0, FS_END )
  ::SetPointer()
Return( Self )

Method fileSeek( nSeek )
  ::Pointer := nSeek
Return( Self )

Method fSkip( nSkip )
  ::Goto()
  fseek( ::Handle, nSkip, FS_RELATIVE )
  ::SetPointer()
Return( Self )

Method Var Pointer( nSeek )
  If SET
    Default nSeek to METHODVAR
    METHODVAR := fseek( ::Handle, nSeek, FS_SET )
  Endif
Return( METHODVAR )

Method fEof()
  Local lEof := .f.
  Local nPointer := ::Pointer
  ::Bottom()

  If nPointer >= ::Pointer
    lEof := .t.
  Endif
  ::Pointer := nPointer
Return( lEof )

Method ReadLine()
  Local cLine := ""
  Local lEOL := .f.
  Local nHandle := ::Handle
  Local nLastPos
  Local cBuffer := Space( ::BufferSize )
  Local nAt
  Local cRead
  Local nRead

  ::Goto()
  Do while !lEOL .and. !::EOF()
    nLastPos := ::Pointer
    nRead := fread( nHandle, @cBuffer, ::BufferSize )
    ::Error := NIL
    ::SetPointer()
    If ::Error
      lEOL := .t.
    Else
      cRead := Left( cBuffer, nRead )
      nAt := At( pEOL, cRead )
      If nAt == 0 .and. nRead < ::BufferSize
        nAt := Len( cRead ) + 1
      Endif
      If nAt != 0
        cRead := Left( cRead, nAt - 1 )
        ::Goto( nLastPos + nAt + 1 )
        lEOL := .t.
      Endif
      cLine += cRead
    Endif
  Enddo
Return( cLine )

Method Read( nBytes := ::BufferSize )
  Local cBuffer := Space( nBytes )
  Local nRead

  if ::eof()
    ::Error := .t.
    cBuffer := ""
  else
    ::Goto():Error := .f.
    nRead := fRead( ::Handle, @cBuffer, nBytes )
    ::Error := NIL
    ::SetPointer()
    If nRead != nBytes
      //If ::EOF() .and. nRead != 0
      //  ::Error := .f.
      //Endif
      cBuffer := Left( cBuffer, nRead )
    Endif
  endif
Return ( cBuffer )

Method Var FileError( lError )
  If Set
    If lError == NIL
     ::fError := ferror()
     METHODVAR := ( ::fError != 0 )
    Else
      METHODVAR := lError
      If lError
        ::fError := ferror()
      Else
        ::fError := 0
      Endif
    Endif
    ::cError := ::CharError( ::fError )
  Endif
Return( METHODVAR )

Method SetPointer
  ::Pointer := fseek( ::Handle, 0, FS_RELATIVE )
Return( Self )

Method WriteLine( cLine := "" )
Return( ::Write( cLine + pEOL ) )

Method Write( cString )
  Local nRet
  ::Goto()
  nRet := fwrite( ::Handle, cString )
  ::Error := NIL
  ::SetPointer()
Return( nRet )

Method LineEval( bBlock, bFor, bWhile, ;
                 nNext := -1, ;
                 lRest := full( bWhile ) )
  Local cLine
  Local lEOF := .F.

  If !lRest
    ::Top()
  Else
    ::Goto()
  Endif

  Do while !::EOF() .and. nNext-- != 0
    cLine := ::ReadLine()
    If ::Error .or. ( bWhile != NIL .and. !Eval( bWhile, cLine, Self ))
      Exit
    Endif

    If bFor == NIL .or. EVAL( bFor, cLine, Self )
      EVAL( bBlock, cLine, Self )
    Endif
  Enddo
  ::SetPointer()
Return( Self )

Method fFind( cFind, ;
              lRest := full( bWhile ), ;
              lCase := .f., bWhile := {|| .t. } )
  Local nFound
  Local nBufferSize := Max( ::BufferSize, Len( cFind ))
  Local cBuffer
  Local cRead
  Local nLastPos

  If !lRest
    ::Top()
  Endif
  If !lCase
    cFind := Upper( cFind )
  Endif

  ::Error := .f.

  nLastPos := ::Pointer
  cBuffer := ::Read( nBufferSize )
  If !lCase
    cBuffer := Upper( cBuffer )
  Endif

  nFound := at( cFind, cBuffer )
  Do while nFound == 0 .and. !::Error
    nLastPos := ::Pointer - ( Len( cFind ) - 1 )
    cBuffer := Right( cBuffer, Len( cFind ) - 1 ) + ::Read( nBufferSize )
    If !::Error
      if ( nFound := at( cFind, iif( lCase, cBuffer, Upper( cBuffer )))) > 0
        exit
      elseif !EVAL:bWhile( Self, cBuffer )
        nFound := -1
        exit
      endif
    Endif
  Enddo

  If nFound > 0
    ::Pointer := nLastPos + nFound - 1
  Endif

Return( nFound > 0 )

Method CharError( nError )
  Local cError := "Reserved"
  Static aError := { {  1, "Invalid function number" }, ;
                     {  2, "File not found" },;
                     {  3, "Path not found" },;
                     {  4, "Too many open files (no handles left)" },;
                     {  5, "Access denied" },;
                     {  6, "Invalid handle" },;
                     {  7, "Memory control blocks destroyed" },;
                     {  8, "Insufficient memory" },;
                     {  9, "Invalid memory block address" },;
                     { 10, "Invalid environment" },;
                     { 11, "Invalid format" },;
                     { 12, "Invalid access code" },;
                     { 13, "Invalid data" },;
                     { 15, "Invalid drive was specified" },;
                     { 16, "Attempt to remove the current directory" },;
                     { 17, "Not same device" },;
                     { 18, "No more files" },;
                     { 19, "Attempt to write on write-protected diskette" },;
                     { 20, "Unknown unit" },;
                     { 21, "Drive not ready" },;
                     { 22, "Unknown command" },;
                     { 23, "Data error (CRC)" },;
                     { 24, "Bad request structure length" },;
                     { 25, "Seek error" },;
                     { 26, "Unknown media type" },;
                     { 27, "Sector not found" },;
                     { 28, "Printer out of paper" },;
                     { 29, "Write fault" },;
                     { 30, "Read fault" },;
                     { 31, "General failure" },;
                     { 32, "Sharing violation" },;
                     { 33, "Lock violation" },;
                     { 34, "Invalid disk change" },;
                     { 35, "FCB unavailable" },;
                     { 36, "Sharing buffer overflow" },;
                     { 50, "Network request not supported" },;
                     { 51, "Remote computer not listening" },;
                     { 52, "Duplicate name on network" },;
                     { 53, "Network name not found" },;
                     { 54, "Network busy" },;
                     { 55, "Network device no longer exists" },;
                     { 56, "Network BIOS command limit exceeded" },;
                     { 57, "Network adapter hardware error" },;
                     { 58, "Incorrect response from network" },;
                     { 59, "Unexpected network error " },;
                     { 60, "Incompatible remote adapter" },;
                     { 61, "Print queue full" },;
                     { 62, "Not enough space for print file" },;
                     { 63, "Print file deleted (not enough space)" },;
                     { 64, "Network name deleted" },;
                     { 65, "Access denied" },;
                     { 66, "Network device type incorrect" },;
                     { 67, "Network name not found" },;
                     { 68, "Network name limit exceeded" },;
                     { 69, "Network BIOS session limit exceeded" },;
                     { 70, "Temporarily paused" },;
                     { 71, "Network request not accepted" },;
                     { 72, "Print or disk redirection paused" },;
                     { 80, "File already exists" },;
                     { 82, "Cannot make directory entry" },;
                     { 83, "Fail on INT 24H" },;
                     { 84, "Too many redirections" },;
                     { 85, "Duplicate redirection" },;
                     { 86, "Invalid password" },;
                     { 87, "Invalid parameter" },;
                     { 88, "Network device fault" } }

  If nError == 0
    cError := ""
  Else
    nError := ascan( aError, {|x| x[1] == nError })
    If nError > 0
      cError := aError[ nError ][ 2 ]
    Endif
  Endif
Return( cError )

Method Var fDate()
Return( ::FileDate() )

Method Var fTime()
Return( ::FileTime() )

Method FileDate( cFile := ::FileName )
  Local dDate := ctod("")
  Local aDir

  if full( cFile )
    aDir := ::directory( cFile )
    If !::Error
      dDate := aDir[1][3]
    Endif
  endif
Return( dDate )

Method FileTime( cFile := ::FileName )
  Local cTime := ""
  Local aDir

  if full( cFile )
    aDir := ::directory( cFile )
    If !::Error
      cTime := aDir[1][4]
    Endif
  endif
Return( cTime )

Method fileRename( cFile, cTo )
  ::Error := fRename( cFile, cTo ) != 0
Return( Self )

Method fileDelete( cFile )
  Local cPath := ::Path( cFile )
  Local aDir := ::Directory( cFile )
  Local aError := {}
  Local nCount

  If !::Error
    For nCount := 1 to len( aDir )
      ::Error := ( ferase( cPath + aDir[ nCount ][ 1 ] ) != 0 )
      If ::Error
        aadd( aError, cPath + aDir[ nCount ][ 1 ])
      Endif
    Next
  Endif

  ::Error := len( aError ) > 0 .and. len( aDir ) == 0
Return( aError )

Method fileCopy( cFile, ;
                 cTo := ::FileName, ;
                 nBufferSize := 30000 )
  Local oFrom := oFileIO():Open( cFile )
  Local oTo
  Local cRead
  Local lError := .f.

  If oFrom:Error
    lError := .t.
  Else
    oTo := oFileIO():Create( cTo )
    If oTo:Error
      lError := .t.
    Else
      Do while !oTo:Error .and. !oFrom:EOF()
        cRead := oFrom:Read( nBufferSize )
        oTo:Write( cRead )
        If oFrom:Error
          lError := .t.
          Exit
        Endif
      Enddo
      oTo:Close()
      If lError
        oTo:Delete()
      Endif
    Endif
    oFrom:Close()
  Endif
  ::Error := lError
Return( Self )

Method Move( cFile, cTo, nBufferSize )
  ::Copy( cFile, cTo, nBufferSize )

  If !::Error
    ::Delete( cFile )
  Endif
Return (Self)

Method DirEval( bBlock, ;
                cFileSpec := "*.*", ;
                bFor, bWhile, bExitTree )
  Local nDir := 1
  Local cPath := ::Path( cFileSpec )
  Local aDir := ::Directory( cFileSpec )

  For nDir := 1 to Len( aDir )
    If bWhile != NIL .and. !Eval( bWhile, cPath, aDir[ nDir ] )
      If Full( bExitTree )
        Eval( bExitTree )
      Endif
      Exit
    Endif
    If bFor == NIL .or. EVAL( bFor, cPath, aDir[ nDir ] )
      EVAL( bBlock, cPath, aDir[ nDir ] )
    Endif
  Next
Return( Self )

Method TreeEval( bBlock, cPath := "", bFor, bWhile, bExit )
  Local lExit := .f.
  Local aDir
  Local nDir := 1

  cPath := ::Path( cPath )

  If bWhile == NIL .or. Eval( bWhile, cPath )
    If bFor == NIL .or. EVAL( bFor, cPath )
      EVAL( bBlock, cPath )
    Endif
    aDir := ::Directory( cPath + "*.*", "D" )
    For nDir = 1 to len( aDir )
      If "D" $ aDir[ nDir ][ 5 ] .and. !aDir[ nDir ][ 1 ] $ ".."
        ::TreeEval( cPath + aDir[ nDir ][ 1 ] + "\", @bBlock, @bFor, @bWhile, {|| lExit := .t.  } )
        If lExit
          if Full( bExit ) // If not first call
            Eval( bExit )  // Tell previous call to exit
          Endif
          Exit
        Endif
      Endif
    Next
  Elseif Full( bExit ) // If not first call
    Eval( bExit ) // Tell previous call to exit
  Endif
Return( Self )

Method TreeDirEval( bBlock, cFileSpec := "*.*", bFor, bWhile )
  Local cStartPath
  Local cSpec
  Local lExitTreeDir := .f.

  cStartPath := ::Path( cFileSpec )
  cSpec      := ::FullName( cFileSpec )

  ::TreeEval( cStartPath, ;
              {|cPath| ::DirEval( bBlock, cPath + cSpec, bFor, bWhile, {|| lExitTreeDir := .t. } ) },,;
              {|| !lExitTreeDir })
Return( Self )

Method Path( cFile := ::FileName )
  Local nFind
  Local cPath := ""

  nFind := Max( Rat( "\", cFile ), Rat( ":", cFile ) )
  If nFind != 0
    if !right( alltrim( cFile ), 1 ) $ ":\"
      cPath := Left( cFile, nFind )
    endif
  Endif
Return( cPath )

Method Name( cFile := ::FileName )
  Local nFind

  cFile := ::FullName( cFile )

  If ( nFind := At( ".", cFile )) > 0
    cFile := Left( cFile, nFind - 1 )
  Endif
Return( cFile )

Method FullName( cFile := ::FileName )
  Local nFind

  nFind := Max( Rat( "\", cFile ), Rat( ":", cFile ) )
  If nFind != 0
    if !right( alltrim( cFile ), 1 ) $ ":\"
      cFile := Substr( cFile, nFind + 1 )
    endif
  Endif
Return( cFile )

Method Ext( cFile := ::FileName, ;
            lDot := .t. )
  Local nFind

  cFile := ::FullName( cFile )
  If ( nFind := At( ".", cFile )) > 0
    cFile := Substr( cFile, nFind + iif(lDot, 0, 1 ))
  Else
    cFile := ""
  Endif
Return( cFile )

Method Drive( cFile := ::FileName )
  Local nFind

  nFind := at( ":", cFile )
Return( Left( cFile, nFind ) )

Method PathEval( bBlock, ;
                 cPath := Upper( Alltrim( GetEnv( "PATH" ))), ;
                 bFor, bWhile )
  Local cDir := ""
  Local nFind

  Do while Full( cPath )
    nFind := at( ";", cPath )
    If nFind == 0
      cDir := cPath
      cPath := ""
    else
      cDir := rtrim( Left( cPath, nFind - 1 ))
      cPath := ltrim( Substr( cPath, nFind + 1 ))
    Endif

    If Full( cDir ) .and. !Right( cDir, 1 ) $ "\:"
      cDir += "\"
    Endif

    If bWhile == NIL .or. Eval( bWhile, cDir )
      If bFor == NIL .or. Eval( bFor, cDir )
        Eval( bBlock, cDir )
      Endif
    Else
      Exit
    Endif
  Enddo
Return( Self )

Method Var CurrDir( cPath )
  If Set
    Set Default to ( cPath )
  Else
    cPath := "\" + CurDir()
  Endif
Return( cPath )

Method Var CurrDrive( cDrive )
  If Set
    Set Default to ( cDrive )
  Elseif Valtype( ::bDrive ) == 'B'
    cDrive := left( Eval( ::bDrive ), 2 )
    if len( cDrive ) == 1
      cDrive += ":"
    endif
  Else
    cDrive := ""
  Endif
Return( cDrive )

Method Var CurrPath( cPath )
  If Set
    Set Default to ( cPath )
  Else
    cPath := CurDir()
    If cPath == ""
      cPath := "\"
    Endif
    cPath := ::CurDrive() + cPath
  Endif
Return( cPath )

Method Volume( cDrive := "" )
  Local aDir
  Local cVolume := ""

  If len( cDrive ) == 1
    cDrive += ":"
  Endif

  ::Directory( cDrive, "V" )
  If !::Error
    cVolume :=  aDir[ 1 ][ 1 ]
  Endif
Return( cVolume )

Method GetDir( cFile, cAttr )
  Local aDir := Directory( cFile, cAttr )
  ::Error := ( len( aDir ) == 0 )
Return( aDir )

Method TempName()
  Static nSeed
  Local cFile := "Z"
  Local nCount

  Default nSeed to Seconds()

  Do while .T.
    Do while Len( cFile ) < 8
      nSeed := Max( 3, nSeed ^ 2 / 2 ) % 999999999
      cFile += str( int( nSeed ) % 10, 1, 0 )
    Enddo
    If File( cFile )
      cFile := "Z"
      loop
    Endif
    exit
  Enddo
Return( cFile + ".TMP" )

Method ByteCopy( nStart, nLen, oTo )
  Local nBuffer
  Local cBuffer
  Local nTotal := 0

  if oTo == SELF // can't copy the the same object!
    ::Error := .t.
  else
    ::Pointer := nStart
    Default nLen to ::Size - nStart + 1

    ::Error := .f.
    oTo:Error := .f.

    Do while nLen > 0 .and. !oTo:Error .and. !::Error
      nBuffer := Min( nLen, 30000 )
      cBuffer := ::Read( nBuffer )
      if !::Error
        nTotal += oTo:Write( cBuffer )
        nLen -= nBuffer
      endif
    Enddo
  endif
Return( nTotal )



