*****************************************************************************
*  Program..: get_file.prg
*****************************************************************************
*
*  Purpose..: Simple program to import a definition for an existing
*             DBF or NTX into a CDD data dictionary.
*
*             NO ERROR CHECKING!  If a file definition already exists
*             it will be overwritten!
*
*  Syntax...: get_file cdd_name, cdd_path, dbfname [, ntxname]
*
*  Arguments: cdd_name is the CDD data dictionary system name of the target
*
*             cdd_path is the path to the CDD data dictionary.  Must end
*             with ":" or "\".
*
*             dbfname is the filename (no extension) of the dbf file
*             definition to import.  To import a datafile definition,
*             do not pass an ntxname parameter.  The dbf file must be in
*             the current directory.
*
*             ntxname is the optional filename (no extension) of the ntx
*             file definition to import.  Import the datafile definition
*             BEFORE you import its indexes.  The ntx file must be in the
*             current directory.
*
*  Author...: Vick Perry,   Wed  02-24-1993.  This file may be freely
*             distributed.
*
*  Notes....: Compile:    "CLIPPER GET_FILE /A /W"
*
*             You should pack the DD files afterwards.
*             Does not identify unique indexes.
*
*  Revisions:
*
*
*  Examples.: To import a DBF def:
*
*                   get_file  tutor  \cddtutor\  myfile
*
*             To import an index def for myfile.dbf
*
*                   get_file  tutor  \cddtutor\  myfile  myfile1
*
*****************************************************************************
local struct_ := {}
local alen := 0
local pos := 0
local i:= 0

parameters cdd_name, cdd_path, dbfname, ntxname
external transform    // function needed by CDD indexes
set deleted on

* get enough parameters?
if pcount() > 2

   * open CDD files exclusive, no error checking or recovery
   use (cdd_path+cdd_name+"D") alias dfile exclusive new
   set index to (cdd_path+cdd_name+"D1"), (cdd_path+cdd_name+"D2")

   use (cdd_path+cdd_name+"F") alias ffile exclusive new
   set index to (cdd_path+cdd_name+"F1"), (cdd_path+cdd_name+"F2")

   use (cdd_path+cdd_name+"I") alias ifile exclusive new
   set index to (cdd_path+cdd_name+"I1"), (cdd_path+cdd_name+"I2")


   * open dbf (and index if requested)
   dbfname := padr(upper(dbfname),8)
   if ntxname <> nil
      use (dbfname) index (ntxname) alias source exclusive new
      ntxname := padr(upper(ntxname),8)
   else
      use (dbfname) alias source exclusive new
   endif

   * get source structure
   select source
   struct_ := dbstruct()

   if pcount() == 3

      * delete existing dbf def record if found
      select dfile
      dbseek(dbfname)
      if dfile->(found())
         dbdelete()
      endif

      * add new dbf def
      append blank
      replace dfile->dalias with dbfname
      replace dfile->dname with dbfname
      replace dfile->open with .t.
      replace dfile->is_index with .t.

      * delete existing field defs
      select ffile
      dbseek(dbfname)
      if ffile->(found())
         do while ffile->dalias == dbfname
            dbdelete()
            dbseek(dbfname)
         enddo
      endif

      * add new field defs
      pos := 10
      alen := len(struct_)
      for i = 1 to alen
         append blank
         replace ffile->dalias with dbfname
         replace ffile->pos with pos
         replace ffile->fname with struct_[i,1]
         replace ffile->ftype with struct_[i,2]
         replace ffile->flen  with struct_[i,3]
         replace ffile->fdec  with struct_[i,4]
         pos := pos + 10
      next

   else
      * import an index def

      * delete existing index def
      select ifile
      dbseek(dbfname+ntxname)
      if ifile->(found())
         dbdelete()
      endif

      * add new ntx def
      append blank
      replace ifile->dalias with dbfname
      replace ifile->ialias with ntxname
      replace ifile->iname  with ntxname
      replace ifile->ikey   with source->(indexkey(0))
      replace ifile->unique with .f.
   endif
else
   ?
   ?  "Syntax...: GET_FILE <cdd_name>, <cdd_path>, <dbfname> [, <ntxname>]"
   ?
   ?  "Arguments: cdd_name is the CDD data dictionary system name of the target"
   ?
   ?  "           cdd_path is the path to the CDD data dictionary.  Must end"
   ?  "           with ':' or '\'."
   ?
   ?  "           dbfname is the filename (no extension) of the dbf file"
   ?  "           definition to import.  To import a datafile definition,"
   ?  "           DO NOT pass an ntxname parameter.  The datafile must be"
   ?  "           in the current directory."
   ?
   ?  "           ntxname is the optional filename (no extension) of the ntx"
   ?  "           file definition to import.  If you are importing an index"
   ?  "           you must have ALREADY imported the datafile!  The index file"
   ?  "           must be in the current directory."
   ?
   tone(880,3)
   inkey(5)
endif

return



