*****************************************************************************
*  Program..: CDDFUNCS.PRG Version 1.03 for Clipper 5.01a
*****************************************************************************
*
*  Purpose..: Supporting functions for the CDDLIB runtime data dictionary
*             system.
*
*             This is the entire set of CDDLIB Clipper functions except for
*             v_menu(), v_browse(), and v_screen().
*
*             The source code for the core functions v_menu(), v_browse(),
*             and v_screen() is included with the CDD 1.03 Professional
*             Edition only.
*
*             Compile CDDFUNCS.PRG with the following command line:
*
*                CLIPPER CDDFUNCS
*
*             Note: No compiler parameters are used!
*
*
*  Author...: Vick Perry - Phoenix Systems, Inc.
*             Copyright 1992, 1993 Phoenix Systems, Inc.
*
*  Revisions:
*
*****************************************************************************
#include "cddlib.ch"

* list externals needed but not explicitly called
* these can also be commonly used EXTEND.LIB functions used in an application
* but not already used by CDDLIB.LIB functions
external descend
external errorlevel
external v_menu_std
external isprinter


*****************************************************************************
*  Function.: v_init()
*****************************************************************************
*
*  Purpose..: Initialize CDD's variables, open dd files as shared,
*             load globals, open data files.  Does not reindex dd files.
*             A call to V_INIT() should be the VERY FIRST LINE in your
*             program!
*
*  Syntax...: v_init(sysname [, ddpath  [, vidcode]])
*
*  Arguments: sysname is the name of the program (Six characters max.)
*
*             ddpath is the path to the data dictionary files, ending
*             in "\" or ":".  Optional.  If not passed, use current dir.
*
*             vidcode - code to set video type detection, optional
*                       Vidcode will override the data dictionary
*                       setting.
*
*                 "A" - autodetect video mode (default if not passed)
*                 "C" - force color
*                 "M" - force mono
*
*  Returns..: .t. if initialization and dd file opens are successful.
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....: If mono monitor is detected or vidcode=="M"
*             then colors are from the CDD hardcoded default.
*
*             PUBLICs are used for the system variables for the
*             following reasons:
*
*                - PUBLICs do not require writing and maintenance of
*                  static variables and their get/set functions.
*                - Changing PUBLICs via the data dictionary conditions
*                  and procs is simplified.
*                - PUBLICs are always visible.
*                - The system-wide PUBLICs are rarely changed.
*                - PUBLICs are immediately viewable via the debugger.
*
*  Revisions:
*
*  NG Short.: Initialize CDDLIB, open DD file, open app files, etc.
*****************************************************************************
function v_init(sysname,ddpath,vidcode)
   local l := .f.
   local mono_full := "W+/N,N/W,,,W/N"
   local mono_say  := "W+/N"               // std color only
   local mono_fade := "W/N"                // mono fade default
   local color_fade := "BG/B"              // color fade default

   * misc global variables, v_stack_ must be declared and assigned
   * at beginning of v_init because if an open or other fatal error
   * occurs, then v_stack_ must be available for errorsys to show problem
   public v_exitflag := .f.
   public v_shadow   := .f.    // shadows for color only
   public v_stack_   := {}     // declare load stack array
   aadd(v_stack_," "+padr("V_INIT",10))   // add first entry

   * close all files, just in case
   dbcloseall()

   * default params
   default ddpath to ""           // "" is current directory

   * declare/clean/load publics from v_init() parameters
   public v_sysname := upper(alltrim(sysname))
   public v_ddpath := upper(alltrim(ddpath))

   * open dd files and dd indexes needed at runtime
   open_dd(v_sysname,v_ddpath)

   * read configuration file
   * DO NOT TRIM BOX STRINGS - 9th character is often an intentional space!
   public v_datapath  := upper(alltrim(v_sys->datapath))
   public v_idxpath   := upper(alltrim(v_sys->idxpath ))
   public v_header0   := alltrim(v_sys->header0)   // header line 0
   public v_header1   := alltrim(v_sys->header1)   // header line 1
   public v_backchar  := alltrim(v_sys->backchar)  // backdrop fill char
   public v_brobox    := v_sys->brobox             // browse box
   public v_brosepc   := alltrim(v_sys->brosepc)   // browse col sep.
   public v_broseph   := alltrim(v_sys->broseph)   // browse header/data sep.
   public v_menbox    := v_sys->menbox             // menu box
   public v_scrbox    := v_sys->scrbox             // screen box
   public v_lstbox    := v_sys->lstbox             // key list box
   public v_lsttitle  := strtran(alltrim(v_sys->lsttitle),"~"," ")  // key list title
   public v_sound_m   := v_sys->sound_m            // menu sound
   public v_sound_b   := v_sys->sound_b            // browse sound
   public v_sound_s   := v_sys->sound_s            // screen sound
   public v_sound_z   := v_sys->sound_z            // CDDLIB system warnings
   public v_lock_ed   := v_sys->lock_ed            // lock rec before edit
   public v_infokey   := v_sys->infokey            // hotkey to call v_info()
   public v_is_info   := v_infokey<>0              // flag to turn on hotkey
   public v_s_corner  := v_sys->shad_quad          // shadow quadrant
   public v_vidcode  := upper(alltrim(v_sys->vidcode))

   * is video code overridden by v_init() video parameter?
   if !vidcode==nil
      v_vidcode := vidcode
   endif

   * declare colors, first default to mono
   public v_syscolor  := mono_full   // default colors
   public v_titcolor  := mono_say    // default color
   public v_statcolor := mono_say    // status line color
   public v_brocolor  := mono_full   // browse colors
   public v_brofade   := mono_fade   // browse fade color
   public v_hlpcolor  := mono_say    // help color
   public v_mencolor  := mono_full   // menu colors
   public v_menfade   := mono_fade   // menu fade color
   public v_findcolor  := mono_full  // find colors
   public v_scrcolor  := mono_full   // screen colors
   public v_scrfade   := mono_fade   // screen fade color
   public v_msgcolor  := mono_full   // message v_msg(), v_alert(), v_disp()
   public v_lstcolor  := mono_full   // key list colors

   * autodetection
   if v_vidcode == "A"
      v_vidcode := if(iscolor(),"C","M")
   endif

   * set colors if color mode
   if v_vidcode == "C"
      v_syscolor  := alltrim(v_sys->syscolor)  // default colors
      v_titcolor  := alltrim(v_sys->titcolor)  // backdrop title color
      v_statcolor := alltrim(v_sys->statcolor) // status line color
      v_brocolor  := alltrim(v_sys->brocolor)  // browse colors
      v_brofade   := if(empty(v_sys->brofade),color_fade,alltrim(v_sys->brofade))   // browse fade color
      v_hlpcolor  := alltrim(v_sys->hlpcolor)  // help colors
      v_mencolor  := alltrim(v_sys->mencolor)  // menu colors
      v_menfade   := if(empty(v_sys->menfade),color_fade,alltrim(v_sys->menfade))
      v_findcolor  := alltrim(v_sys->findcolor) // find colors
      v_scrcolor  := alltrim(v_sys->scrcolor)   // screen colors
      v_scrfade   := if(empty(v_sys->scrfade),color_fade,alltrim(v_sys->scrfade))
      v_msgcolor  := alltrim(v_sys->msgcolor)   // message function colors
      v_lstcolor  := alltrim(v_sys->lstcolor)   // key list colors
      v_shadow    := .t.
   endif

   * other initializations
   setenv()
   set key 28 to v_help()
   if v_is_info
      set key (v_infokey) to v_info()
   endif
   set color to (v_syscolor)
   if v_sys->show_back
      v_backdrop()                // display backdrop
      v_stat24(" Opening files, please wait...")
   endif

   * open all data files and indexes
   open_files()

   if v_sys->show_back
      v_stat24()                 // clear line 24 message
   endif

   * load first function
   if !empty(v_sys->start_id)
      * load first func and loop until user wants to exit
      do while .t.
         v_load(v_sys->start_id, v_sys->idtype)
         if v_sys->exitprompt
            if v_yesno("Do you wish to exit? (Y/N)")
               exit
            endif
         else
            exit
         endif
      enddo
   else
      v_fatal("V_INIT - No starting function ID specified")
   endif
   set cursor on
   set color to
   cls
return l


*****************************************************************************
*  Function : v_stdKEYS()
*****************************************************************************
*
*  Purpose..: Process standard TBROWSE navigation keys for v_dbrow()
*             and v_browse().
*
*  Syntax...: v_stdKEYS(<objTBrowse>, <nKey>, full_bar)
*
*  Arguments: <objTBrowse> is the TBrowse object.
*
*             <nKey> is the inkey value of the key to be processed.
*
*              full_bar is the flag indicating whether the browse is
*              showing a full width highlight bar or a single cell
*              highlight .t. = full width highlight
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Thu  06-20-1991
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Standard tbrowse navigation keys
*****************************************************************************
function v_stdkeys(tobj, nkey, full_bar)
   do case
      case nkey == K_DOWN
         tobj:down()
      case nkey == K_UP
         tobj:up()
      case nkey == K_LEFT
         if full_bar
            tobj:panleft()
         else
            tobj:left()
         endif
      case nkey == K_RIGHT
         if full_bar
            tobj:panright()
         else
            tobj:right()
         endif
      case nkey == K_CTRL_LEFT
         tobj:panleft()
      case nkey == K_CTRL_RIGHT
         tobj:panright()
      case nkey == K_HOME
         if full_bar
            tobj:panhome()
         else
            tobj:home()
         endif
      case nkey == K_END
         if full_bar
            tobj:panend()
         else
            tobj:end()
         endif
      case nkey == K_CTRL_HOME
         tobj:panhome()
      case nkey == K_CTRL_END
         tobj:panend()
      case nkey == K_PGDN
         tobj:pagedown()
      case nkey == K_PGUP
         tobj:pageup()
      case nkey == K_CTRL_PGUP
         tobj:gotop()
      case nkey == K_CTRL_PGDN
         tobj:gobottom()
   endcase
return nil


*****************************************************************************
*  Function : V_GOTOTOP()
*****************************************************************************
*
*  Purpose..: Replacement TBrowse index-filtered GOTOP() method
*
*  Syntax...: V_GOTOTOP(<Pattern>)
*
*  Arguments: <Pattern> is the search pattern
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Fri  07-12-1991
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Replacement TBrowse index-filtered GOTOP() method
*****************************************************************************
function v_gototop( LimitValue )
   seek LimitValue
return nil


*****************************************************************************
*  Function : V_GOTOBOTT()
*****************************************************************************
*
*  Purpose..: Replacement TBrowse index-filtered GOBOTTOM() method
*
*  Syntax...: V_GOTOBOTT(<Pattern>)
*
*  Arguments: <Pattern> is the search pattern
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Fri  07-12-1991
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Replacement TBrowse index-filtered GOBOTTOM() method
*****************************************************************************
function v_gotobott( LimitValue )
   local is_soft
   is_soft = SET(_SET_SOFTSEEK, .T.)
   seek substr(LimitValue, 1, len(LimitValue) - 1) + ;
      chr(asc(substr(LimitValue, len(LimitValue))) + 1)
   skip -1
   set(_SET_SOFTSEEK, is_soft)
return nil


*****************************************************************************
*  Function : V_MOVEPTR()
*****************************************************************************
*
*  Purpose..: Replacement TBrowse index-filtered SKIPBLOCK method
*
*  Syntax...: V_MOVEPTR(<nSkip>,<LimitValue>,<bIndex>,<oBrowse>) --> nSkipped
*
*  Arguments: <nSkip> is the number of records to skip.  The sign indicates
*             the direction.
*
*             <LimitValue> is the search pattern
*
*             <bIndex> is the index key code block
*
*             <oBrowse> is the active browse object
*
*  Returns..: A numeric value for the direction and number or records skipped
*
*  Author...: Vick Perry,   Fri  07-12-1991
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Replacement TBrowse index-filtered SKIPBLOCK method
*****************************************************************************
function v_moveptr( nskip, LimitValue, bIndex, tobj )
   local nskipped := 0

   /*
     Loop and skip the requested number of records until eof(), bof(),
     out of filter range, or skips are completed.
   */
   do while ! eof() .and. ! bof() .and. nskipped != nskip;
         .and. eval(bIndex) = LimitValue     // partial index match allowed
      if nskip > 0
         skip
         nskipped++
      else
         skip -1
         nskipped--
      endif
   enddo

   /*
     After skipping is finished, check the record/file status.
     Make adjustments to the record pointer if currently eof(),
     bof(), or filter is out of range.
   */
   do case
      case eof()
         skip -1                        // try previous record
         if !eval(bIndex) = LimitValue
            goto lastrec() + 1          // no records match filter - go to eof()
         else
            nskipped--
         endif
         tobj:hitbottom := .t.
         tobj:hittop := .f.

      case bof()
         nskipped++
         goto recno()
         tobj:hitbottom := .f.
         tobj:hittop := .t.

     case !eval(bIndex) = LimitValue     // out of range
        if nskip > 0
           /*
             Attempted to skip +1 but failed
           */
           skip -1
           if !eval(bIndex) = LimitValue // does rec match filter?
              goto lastrec() + 1        // no - go to eof()
           else
              nskipped--
           endif
           tobj:hitbottom := .t.
           tobj:hittop := .f.
        else
           /*
             Attempted to skip -1 but failed
           */
           skip
           if !eval(bIndex) = LimitValue // does rec match filter?
              goto lastrec() + 1        // no - go to eof()
           else
              nskipped++
           endif
           tobj:hitbottom := .f.
           tobj:hittop := .t.
        endif

     otherwise
        tobj:hitbottom := .f.
        tobj:hittop := .f.
   endcase
return nskipped


*****************************************************************************
*  Function.: v_list_udf()
*****************************************************************************
*
*  Purpose..: Achoice() udf for v_browse() and v_screen() key list.
*
*  Syntax...: v_list_udf(status, elem, winrel)
*
*  Arguments: status, elem, winrel - see achoice() section in Clipper
*             manual.
*
*  Returns..: A numeric value
*
*  Author...: Vick Perry,   Sun  08-23-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Achoice() udf for v_browse() and v_screen() key list
*****************************************************************************
function v_list_udf(status,elem,winrel)
   local n := AC_CONT   // default to continue

   do case
      case status == AC_IDLE
         * do nothing

      case status == AC_HITTOP .OR. status == AC_HITBOTTOM
         * do nothing - no sounds, please

      CASE status == AC_EXCEPT
         // Keystroke exception
         DO CASE
            CASE LASTKEY() == K_ESC
               n := AC_ABORT

            CASE LASTKEY() == K_HOME
               KEYBOARD CHR( K_CTRL_PGUP )
               n := AC_CONT

            CASE LASTKEY() == K_END
               KEYBOARD CHR( K_CTRL_PGDN )
               n := AC_CONT

            CASE LASTKEY() == K_LEFT
               KEYBOARD CHR( K_UP )
               n := AC_CONT

            CASE LASTKEY() == K_RIGHT
               KEYBOARD CHR( K_DOWN )
               n := AC_CONT

            CASE LASTKEY() == K_ENTER
               n := AC_SELECT

            CASE UPPER(CHR(LASTKEY())) $ ;
               "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890 "

               * automatically select item if reqd
               KEYBOARD CHR( K_ENTER )

               n := AC_GOTO
         ENDCASE
   endcase
return n


*****************************************************************************
*  Function.: v_menu_std()
*****************************************************************************
*
*  Purpose..: Achoice udf, uses privates line23_ ,line24_, v_oldelem
*             from v_menu().
*
*  Syntax...: v_menu_std(status, elem, winrel)
*
*  Arguments: status, elem, winrel - see achoice() section in Clipper
*             manual.
*
*  Returns..: A numeric value
*
*  Author...: Vick Perry,   Sun  08-23-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Achoice() udf for v_menu()
*****************************************************************************
function v_menu_std(status,elem,winrel)
   local n := AC_CONT   // default to continue

   * update status lines if needed
   if !elem==v_oldelem
      v_stat23(line23_[elem],v_center)
      v_stat24(line24_[elem],v_center)
      v_oldelem := elem
   endif

   do case
      case status == AC_IDLE
         * do nothing

      case status == AC_HITTOP .OR. status == AC_HITBOTTOM
         if v_sound_m
            v_beep("M")
         endif

      CASE status == AC_EXCEPT
         // Keystroke exception
         DO CASE
            CASE LASTKEY() == K_ESC
               n := AC_ABORT

            CASE LASTKEY() == K_HOME
               KEYBOARD CHR( K_CTRL_PGUP )
               n := AC_CONT

            CASE LASTKEY() == K_END
               KEYBOARD CHR( K_CTRL_PGDN )
               n := AC_CONT

            CASE LASTKEY() == K_LEFT
               KEYBOARD CHR( K_UP )
               n := AC_CONT

            CASE LASTKEY() == K_RIGHT
               KEYBOARD CHR( K_DOWN )
               n := AC_CONT

            CASE LASTKEY() == K_ENTER
               n := AC_SELECT

            CASE UPPER(CHR(LASTKEY())) $ ;
               "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890 "

               * automatically select item if first char is found
               if v_autokey .and. ascan(items_,{|x| upper(left(x,1))==upper(chr(lastkey())) }) > 0
                  KEYBOARD CHR( K_ENTER )
               endif

               n := AC_GOTO
         ENDCASE
   endcase
return n


*****************************************************************************
*  Function : V_LOAD()
*****************************************************************************
*
*  Purpose..: Load a CDDLIB.LIB function.  All calls to CDD core functions
*             v_menu(), v_browse(), v_screen(), and v_find() MUST be made
*             via v_load().
*
*             V_load() also maintains the CDD stack array v_stack_.
*
*  Syntax...: v_load(id,idtype [,param])
*
*  Arguments: id is the function id
*
*             idtype is the single character function type identifier where:
*
*                B - Browse
*                F - Find
*                M - Menu
*                S - Screen
*                U - UDF, CDDLIB, Other
*
*             param is the optional startup parameter for an "S"-type
*             (screen) function only.
*
*  Returns..: A value from the loaded function upon exiting - normally this
*             return value is meaningless.
*
*  Author...: Vick Perry,   Fri  08-21-1992
*
*  Notes....:
*
*
*  Revisions:
*
*  NG Short.: Always use this to call v_browse(), v_menu(), v_screen()
*****************************************************************************
function v_load(id,idtype,param)
   local ret_val := .t.
   local paren_pos := 0

   if !(id == nil .or. empty(id))

      * clean the function parameters
      default param to ""
      param := alltrim(param)
      idtype := upper(idtype)

      * clean id string, DO NOT CLEAN, or UPCASE "U" types
      if !idtype == "U"
         id := padr(upper(ltrim(id)),ID_LEN)
      else
         * you may ltrim() U-types
         id := ltrim(id)
      endif

      * push load stack, uppercase and 10 chars max for all id types
      * load stack identifies the id type and id of next function
      * to be loaded, used by debugger, help, and program tree documenter
      * for U-types truncate function name after "(" or 10 chars whichever
      * is first
      paren_pos := at("(",id)
      if paren_pos > 0
         aadd(v_stack_,idtype+padr(left(upper(id),paren_pos-1),10))
      else
         aadd(v_stack_,idtype+padr(upper(id),10))
      endif

      do case
         * menu
         case idtype == "M"
            ret_val := v_menu(id)

         * browse
         case idtype == "B"
            ret_val := v_browse(id)

         * screen
         case idtype == "S"
            ret_val := v_screen(id,param)  //  A=Add, E=Edit, otherwise view

         * find
         case idtype == "F"
            ret_val := v_find(id)

         * user defined or external function call
         case idtype == "U"
            ret_val := &id
      endcase

      * pop load stack, resize
      asize(v_stack_,len(v_stack_)-1)
   else
      v_alert("V_LOAD() -  Nothing was specified to load")
   endif
return ret_val


*****************************************************************************
*  Function.: v_open()
*****************************************************************************
*
*  Purpose..: Open a data file and its indexes.  Create public index alias
*             variables each containing the index order of the index.
*             V_OPEN() also positions V_DBF dd file on the proper record.
*             The newly opened file will be selected if it is opened.
*
*             ALWAYS USE v_db_close() TO CLOSE A FILE OPENED BY v_open() !
*
*  Syntax...: v_open(alias,excl,open_ntx)
*
*  Arguments: alias is the data file alias to open, a character string.
*
*             excl is the excl file open flag (default is .f. for shared)
*
*             open_ntx is the flag indicating whether indexes are opened
*             (default is .t. to open all indexes)
*
*  Returns..: a logical, .t. if opened successfully
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....: If file is already open, it will be closed and reopened
*             in the new mode.
*
*  Revisions: dvp 12/11/92 save/set/restore SET DELETED state.
*
*  NG Short.: Open an app data file and indexes.  Paired with v_db_close().
*****************************************************************************
function v_open(alias,excl,open_ntx)
   local l := .t.
   local ipath := ""
   local dpath := ""
   local order := 0
   local tempstr := ""
   local o_deleted := set(_SET_DELETED,.t.)
   default excl to .f.
   default open_ntx to .t.
   alias := padr(upper(alltrim(alias)),len(v_dbf->dalias))

   * close data file if currently open
   if select(alias) > 0
      select (alias)
      close
   endif

   select v_dbf
   seek alias
   if v_dbf->(found())
      * get true data file path
      if empty(v_dbf->altpath)
         dpath := v_datapath
      else
         dpath := upper(alltrim(v_dbf->altpath))
      endif

      if open_ntx
         * open data file, assign aliases
         if v_net_use(dpath+v_dbf->dname,excl,4,v_dbf->dalias,.t.)
            * open indexes, assign index aliases
            select v_idx
            seek v_dbf->dalias
            if v_idx->(found())
               order := 0
               do while v_idx->dalias == v_dbf->dalias .and. !v_idx->(eof())
                  * get true index path
                  if empty(v_idx->altpath)
                     ipath := v_idxpath
                  else
                     ipath := upper(alltrim(v_idx->altpath))
                  endif

                  * open the index in proper workarea
                  select (v_dbf->dalias)
                  dbsetindex(ipath+v_idx->iname)

                  * quit and give error message if index alias
                  * conflicts with dbf alias.  If the index alias
                  * conflicts with an already existing memvar, too bad.
                  if select(v_idx->ialias) > 0
                     v_fatal(rtrim(v_idx->ialias)+" - Index alias conflicts with DBF alias")
                  endif

                  * declare public variables for index alias
                  * if the /a compiler switch is used this causes
                  * a warning.
                  tempstr := v_idx->ialias
                  public &tempstr
                  &tempstr := ++order

                  select v_idx
                  skip
               enddo
            endif
         else
            l := .f.
         endif
      else
         * open data file only - no indexes
         if !v_net_use(dpath+v_dbf->dname,excl,4,v_dbf->dalias,.t.)
            l := .f.
         endif
      endif

      * change to the newly opened file, if open is successful
      if select(alias) > 0
         select (alias)
      endif
   else
      v_fatal("Alias: "+alias+" is not listed in data dictionary")
   endif
   set(_SET_DELETED,o_deleted)
return l


*****************************************************************************
*  Function.: v_db_close()
*****************************************************************************
*
*  Purpose..: Close a data file and indexes.  Release public index alias
*             variables.
*
*  Syntax...: v_db_close(alias)
*
*  Arguments: alias is the data file alias string to close
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Close an app data file and indexes.  Paired with v_open().
*****************************************************************************
function v_db_close(alias)
   alias := padr(upper(alltrim(alias)),len(v_idx->dalias))
   select v_idx
   seek alias
   if v_idx->(found())
      do while v_idx->dalias = alias .and. !v_idx->(eof())
         release &(v_idx->ialias)
         skip
      enddo
   endif
   select (alias)
   close
return .t.


*****************************************************************************
*  Function.: v_order()
*****************************************************************************
*
*  Purpose..: Return the current order and optionally set a new
*             index order.  An index alias is passed as a parameter
*             to v_order().
*
*             Index aliases are public variables created by v_open()
*             and contain a numeric index order.
*
*  Syntax...: v_order([ind_alias])
*
*  Arguments: ind_alias is a numeric variable containing the index order.
*
*  Returns..: The current index order - numeric
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Set index order using an index alias
*****************************************************************************
function v_order(ind_order)
   old_order := indexord()
   if ind_order<>nil
      set order to (ind_order)
   endif
return old_order


*****************************************************************************
*  Function.: v_index()
*****************************************************************************
*
*  Purpose..: Create indexes for a data file.  The data file is closed upon
*             exiting v_index()
*
*  Syntax...: v_index(alias,show_stat)
*
*  Arguments: alias is the data file alias string to open
*
*             show_stat - .t. to display status line
*
*  Returns..: .f. if unable to reindex file
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....: It is advisable to close the file, call v_index(), then
*             reopen it.
*
*  Revisions:
*
*  NG Short.: Index a data file.
*****************************************************************************
function v_index(alias,show_stat)
   local buf := savescreen(22,0,24,79)
   local l := .f.
   local ipath := ""
   alias := padr(upper(alltrim(alias)),len(v_idx->dalias))
   if v_open(alias,.t.,.f.)  // no indexes - v_open() also positions v_dbf
      select v_idx
      seek alias
      if v_idx->(found())
         do while v_idx->dalias = alias .and. !v_idx->(eof())

            if show_stat
               v_stat24("Indexing "+rtrim(v_dbf->dname)+".DBF -- "+rtrim(v_idx->iname)+".NTX")
            endif

            * get true index path
            if empty(v_idx->altpath)
               ipath := alltrim(v_idxpath)
            else
               ipath := upper(alltrim(v_idx->altpath))
            endif

            select (alias)
            if !v_idx->unique
               index on &(v_idx->ikey) to &(ipath+v_idx->iname)
            else
               index on &(v_idx->ikey) to &(ipath+v_idx->iname) unique
            endif

            * close indexes to flush buffers
            dbclearind()
            select v_idx
            skip
         enddo
      endif

      * close file that you just indexed
      select (alias)
      v_db_close(alias)
      l := .t.
   endif
   restscreen(22,0,24,79,buf)
return l


*****************************************************************************
*  Function.: v_indexall()
*****************************************************************************
*
*  Purpose..: Reindex all data files listed in the dd.  All data files
*             are closed upon exiting.
*
*  Syntax...: v_indexall(show_stat)
*
*  Arguments: show_stat - .t. to display status line
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Index all data files listed in DD.
*****************************************************************************
function v_indexall(show_stat)
   local buf := savescreen(22,0,24,79)
   v_stat24()
   select v_dbf
   go top
   do while !v_dbf->(eof())
      v_index(v_dbf->dalias,show_stat)
      select v_dbf
      skip
   enddo
   restscreen(22,0,24,79,buf)
return .t.


*****************************************************************************
*  Function : v_net_use()
*****************************************************************************
*
*  Purpose..: Open a file, exclusive or shared mode, and set an alias
*
*  Syntax...: v_net_use(<cFile>, <lOpenMode>, <nSeconds>, [<cAlias>], [<lNew>]);
*             --> lOpened
*
*  Arguments: <cFile> - Path and name of the data file to open
*
*             <lOpenMode> - .T. = exclusive, .F. = shared mode
*
*             <nSeconds>  Number of seconds to attempt an open
*
*             <cAlias> - Optional alias name
*
*             <lNew> - Optional flag to open in new workarea.  The default
*                      is .f.
*
*  Returns..: .T. if a successful open
*
*  Author...: Vick Perry,      4-29-1990
*
*  Notes....: If an alias name is not passed, the alias name will
*             be the same as the data file name.
*
*  Revisions:
*
*  NG Short.: Open a data file
*****************************************************************************
function v_net_use(datafile, excl_flag, wait_time, aliasname, is_new)
   local forever
   local force_alias := .f.         // flag if alias name will be used
   default excl_flag to .f.         // shared
   default wait_time to 2
   default aliasname to ""
   default is_new to .f.
   force_alias := if(empty(aliasname),.f.,.t.)
   forever := (wait_time = 0)
   do while (forever .or. wait_time > 0)
      if excl_flag
         if force_alias
            if is_new
               use (datafile) exclusive alias (aliasname) new
            else
               use (datafile) exclusive alias (aliasname)
            endif
         else
            if is_new
               use (datafile) exclusive new
            else
               use (datafile) exclusive
            endif
         endif
      else
         if force_alias
            if is_new
               use (datafile) alias (aliasname) shared new
            else
               use (datafile) alias (aliasname) shared
            endif
         else
            if is_new
               use (datafile) shared new
            else
               use (datafile) shared
            endif
         endif
      endif
      if ! neterr()                // use succeeds
         return (.t.)
      endif
      inkey(1)                     // wait 1 second
      wait_time = wait_time - 1
   enddo
return (.f.)                       // use fails


*****************************************************************************
*  Function : V_REC_LOCK()
*****************************************************************************
*
*  Purpose..: To lock a record in a data file.
*
*  Syntax...: V_REC_LOCK(<nWaitSec>)
*
*  Arguments: <nWaitSec> is the number of seconds to retry.  The default
*             is 2 seconds.
*
*  Returns..: A boolean value.  .T. if successfully locked the record.
*
*  Author...: Vick Perry,   Fri  06-07-1991
*
*  Notes....: Based upon Clipper's example RecLock()
*
*  Revisions:
*
*  NG Short.: Lock a record
*****************************************************************************
function v_rec_lock( nseconds )
   local lforever
   default nseconds to 2
   if rlock()
      return (.t.)        // locked
   endif
   lforever = (nseconds = 0)
   do while (lforever .or. nseconds > 0)
      if rlock()
         return (.t.)     // locked
      endif
      inkey(.5)           // wait 1/2 second
      nseconds = nseconds - .5
   enddo
return (.f.)              // not locked


*****************************************************************************
*  Function : V_FIL_LOCK()
*****************************************************************************
*
*  Purpose..: To lock a data file.
*
*  Syntax...: V_FIL_LOCK([<nWaitSec>])
*
*  Arguments: <nWaitSec> is the number of seconds to retry.  The default
*             is 2 seconds.
*
*  Returns..: A boolean value.  .T. if successfully locked the file.
*
*  Author...: Vick Perry,   Fri  06-07-1991
*
*  Notes....: Based upon Clipper's example FilLock()
*
*  Revisions:
*
*  NG Short.: Lock a file
*****************************************************************************
function v_fil_lock( nseconds )
   local lforever
   default nseconds to 2
   if flock()
      return (.t.)      // locked
   endif
   lforever = (nseconds = 0)
   do while (lforever .or. nseconds > 0)
      inkey(.5)         // wait 1/2 second
      nseconds = nseconds - .5
      if flock()
         return (.t.)   // locked
      endif
   enddo
return (.f.)            // not locked


*****************************************************************************
*  Function : V_ADD_REC()
*****************************************************************************
*
*  Purpose..: To add a blank record to a data file.  If the data file is
*             opened shared, the new record will be locked.
*
*  Syntax...: V_ADD_REC([<nWaitSec>])
*
*  Arguments: <nWaitSec> is the number of seconds to retry.  The default
*             is 2 seconds.
*
*  Returns..: A boolean value.  .T. if successfully added and locked a new
*             record
*
*  Author...: Vick Perry,   Fri  06-07-1991
*
*  Notes....: Based upon Clipper's example ADDREC()
*
*  Revisions:
*
*  NG Short.: Add and lock a record
*****************************************************************************
function v_add_rec( nwaitseconds )
   local lforever
   default nwaitseconds to 2
   append blank
   if .not. neterr()
      return (.t.)
   endif
   lforever = (nwaitseconds = 0)
   do while (lforever .or. nwaitseconds > 0)
      append blank
      if .not. neterr()
         return .t.
      endif
      inkey(.5)         // wait 1/2 second
      nwaitseconds  = nwaitseconds  - .5
   enddo
return (.f.)            // not locked


*****************************************************************************
*  Function.: open_dd()
*****************************************************************************
*
*  Purpose..: Open dd files and some indexes, set aliases
*
*  Syntax...:open_dd(sysname,ddpath [,is_excl])
*
*  Arguments: sysname - name of system
*             ddpath - path for dd files
*             is_excl - optional dd open exclusive flag, default is .f. shared
*
*  Returns..: a logical, .t. if successful open
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....: Not all indexes are opened, only those needed for operation.
*
*  Revisions:
*
*  NG Short.: Open the data dictionary files
*****************************************************************************
function open_dd(sysname,ddpath,is_excl)
   local l := .t.
   local sfile
   local dfile
   local ffile
   local ifile
   local bfile
   local cfile
   local kfile
   local mfile
   local nfile
   local ofile
   local pfile
   local tfile
   local ufile
   local vfile
   local xfile

   default is_excl to .f.
   sfile := ddpath+sysname+SYS_SUF
   dfile := ddpath+sysname+DBF_SUF
   ffile := ddpath+sysname+FLD_SUF
   ifile := ddpath+sysname+IDX_SUF
   bfile := ddpath+sysname+BRO_SUF
   cfile := ddpath+sysname+COL_SUF
   kfile := ddpath+sysname+KEY_SUF
   mfile := ddpath+sysname+MEN_SUF
   nfile := ddpath+sysname+MNK_SUF
   ofile := ddpath+sysname+FID_SUF
   pfile := ddpath+sysname+FND_SUF
   tfile := ddpath+sysname+SCR_SUF
   ufile := ddpath+sysname+SGT_SUF
   vfile := ddpath+sysname+SKY_SUF
   xfile := ddpath+sysname+SEQ_SUF

   begin sequence

   * the file may be already open
   select (select("v_sys"))
   if !v_net_use(sfile,is_excl,4,"V_SYS")
      l := .f.
      break
   endif

   select (select("v_dbf"))
   if v_net_use(dfile,is_excl,4,"V_DBF")
      dbsetindex(dfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_fld"))
   if v_net_use(ffile,is_excl,4,"V_FLD")
      dbsetindex(ffile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_idx"))
   if v_net_use(ifile,is_excl,4,"V_IDX")   // alias is V_NTX in CDD.EXE!!
      dbsetindex(ifile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_bro"))
   if v_net_use(bfile,is_excl,4,"V_BRO")
      dbsetindex(bfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_col"))
   if v_net_use(cfile,is_excl,4,"V_COL")
      dbsetindex(cfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_key"))
   if v_net_use(kfile,is_excl,4,"V_KEY")
      dbsetindex(kfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_men"))
   if v_net_use(mfile,is_excl,4,"V_MEN")
      dbsetindex(mfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_mnk"))
   if v_net_use(nfile,is_excl,4,"V_MNK")
      dbsetindex(nfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_fid"))
   if v_net_use(ofile,is_excl,4,"V_FID")
      dbsetindex(ofile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_fnd"))
   if v_net_use(pfile,is_excl,4,"V_FND")
      dbsetindex(pfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_scr"))
   if v_net_use(tfile,is_excl,4,"V_SCR")
      dbsetindex(tfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_get"))
   if v_net_use(ufile,is_excl,4,"V_GET")
      dbsetindex(ufile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_skey"))
   if v_net_use(vfile,is_excl,4,"V_SKEY")
      dbsetindex(vfile+"1")
   else
      l := .f.
      break
   endif

   select (select("v_seq"))
   if v_net_use(xfile,is_excl,4,"V_SEQ")
      dbsetindex(xfile+"1")
   else
      l := .f.
      break
   endif

   * break to here upon failure
   end

   if !l
      quitmsg("Program is being maintained.;Please try again later.")
   endif
return l


*****************************************************************************
*  Function.: setenv()
*****************************************************************************
*
*  Purpose..: Set the default environment
*
*  Syntax...:
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Set the default CDD environment
*****************************************************************************
function setenv
   set(_SET_EXACT     ,.f.)
   set(_SET_PATH      ,"")
   set(_SET_DEFAULT   ,"")
   set(_SET_SOFTSEEK  ,.f.)
   set(_SET_DELETED   ,.t.)
   set(_SET_CURSOR    ,0)
   set(_SET_CONSOLE   ,.t.)
   set(_SET_ALTERNATE ,.f.)
   set(_SET_ALTFILE   ,"")
   set(_SET_DEVICE    ,"")
   set(_SET_PRINTER   ,.f.)
   set(_SET_PRINTFILE ,"")
   set(_SET_MARGIN    ,.f.)
   set(_SET_BELL      ,.f.)
   set(_SET_CONFIRM   ,.f.)
   set(_SET_ESCAPE    ,.t.)
   set(_SET_INTENSITY ,.t.)
   set(_SET_SCOREBOARD,.f.)
   set(_SET_WRAP      ,.t.)
return .t.


*****************************************************************************
*  Function.: quitmsg()
*****************************************************************************
*
*  Purpose..: Display alert message and quit.  Generally used for
*             non-serious errors or to report to the user that the
*             program is currently unavailable.
*
*  Syntax...: quitmsg(msg)
*
*  Arguments: msg is the message to display
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Show a message and quit
*****************************************************************************
function quitmsg(msg)
   tone(880,4)
   tone(880,4)
   tone(880,4)
   alert(msg)
   v_quit()
return nil


*****************************************************************************
*  Function : v_fatal()
*****************************************************************************
*
*  Purpose..: Abort the program and generate an error via the error handler.
*             V-fatal() Always uses alternate error handler v_e_alt(), so
*             that the error message will also be recorded in error.txt.
*
*
*  Syntax...: v_fatal(msg)
*
*  Arguments: msg is the error message to display and write to error.txt.
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Fri  11-27-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Show a message, write to error.txt, and quit
*****************************************************************************
function v_fatal(msg)
   local eobj := errornew()  // create new error object

   * replace current error handler with v_e_alt()
   errorblock({|e| v_e_alt(e)})

   * load description into error object
   eobj:description := msg

   * force error
   eval({|e| v_e_alt(e) }, eobj )
return nil


*****************************************************************************
*  Function.: v_quit()
*****************************************************************************
*
*  Purpose..: Immediate closing of files and quit program.  Optional
*             execution of a cleanup function just prior to quitting.
*
*  Syntax...: v_quit([errorlevel] [, cFunc])
*
*  Arguments: errorlevel is the optional numeric DOS ERRORLEVEL to set
*             before returning to DOS
*
*             cCleanFunc is the name of an optional function for execution.
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Sat  09-19-1992
*
*  Notes....: v_quit() may also be called as a browse or screen action key
*
*  Revisions:
*
*  NG Short.: Close files and quit
*****************************************************************************
function v_quit(errorlevel,funcname)
   local ret_val
   if !funcname == nil
      ret_val := &(funcname)
   endif
   default errorlevel to 0
   close all
   set color to
   set cursor on
   cls
   errorlevel(errorlevel)
   quit
return nil


*****************************************************************************
*  Function.: v_stat23()
*****************************************************************************
*
*  Purpose..: Write status line message on line 23
*
*  Syntax...: v_stat23(msg, is_centered)
*
*  Arguments: msg is a string
*
*             is_centered is a flag indicating whether string is centered
*             (the default is .f. - no centering)
*
*  Returns..: .t. because the primative v_stat() always returns true
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Write status line message on line 23
*****************************************************************************
function v_stat23(msg,is_centered)
return v_stat(23,msg,if(is_centered==nil, .f., is_centered))


*****************************************************************************
*  Function.: v_stat24()
*****************************************************************************
*
*  Purpose..: Write status line message on line 24
*
*  Syntax...: v_stat24(msg, is_centered)
*
*  Arguments: msg is a string
*
*             is_centered is a flag indicating whether string is centered
*             (the default is .f. - no centering)
*
*  Returns..:  .t. because the primative v_stat() always returns true
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Write status line message on line 24
*****************************************************************************
function v_stat24(msg,is_centered)
return v_stat(24,msg,if(is_centered==nil, .f., is_centered))


*****************************************************************************
*  Function.: v_stat()
*****************************************************************************
*
*  Purpose..: Write status line message on designated line
*
*  Syntax...: v_stat(lineno, msg, is_centered)
*
*  Arguments: lineno is the line no for display
*
*             msg is a string
*
*             is_centered is a flag indicating whether string is centered
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Write status line message
*****************************************************************************
function v_stat(lineno,msg,is_centered)
   if msg==nil .or. empty(msg)
      msg := space(80)
   endif
   if !is_centered
      @ lineno, 0 say padr(msg,80) color v_statcolor
   else
      msg := alltrim(msg)
      @ lineno, 0 say space(80) color v_statcolor   // clear line
      @ lineno, 40-(len(msg)/2) say msg color v_statcolor
   endif
return .t.


*****************************************************************************
*  Function : v_yesno()
*****************************************************************************
*
*  Purpose..: Yes/no prompt.
*
*  Syntax...: v_yesno(msg)
*
*  Arguments: msg - a string
*
*  Returns..: A logical - .t. if the answer is YES.
*
*  Author...: Vick Perry,   Fri  08-21-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Yes/no prompt with msg
*****************************************************************************
function v_yesno(msg)
return v_alert(msg,{"Yes","No"})==1


*****************************************************************************
*  Function.: v_help()
*****************************************************************************
*
*  Purpose..: Display the current help, uses v_stack_ array to seek to
*             help memo
*
*  Syntax...: v_help()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Sat  08-22-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Display the current help
*****************************************************************************
function v_help()
   local buf := savescreen(0,0,24,79)
   local xcolor := setcolor(v_hlpcolor)
   local xcursor := setcursor(0)
   local F1block := setkey(K_F1)
   local alias := ""
   local xarea := select()
   local xrec := 0
   local idtype := ""
   local seekstr := ""
   local id := ""
   local cur_ord := indexord()  // ord of current workarea, used for find help
   local has_help := .t.
   set key K_F1 to

   * if v_help() is called as a hotkey, the last stack element is
   * the help to show.  if v_help() is called via a browse or screen
   * key, then it was loaded via v_load(), therefore use the next to
   * last element for showing help
   if substr(v_stack_[len(v_stack_)],2,6) == "V_HELP"
      * next to last elem
      idtype := left(v_stack_[len(v_stack_)-1],1)
      alias := help_alias(idtype)
      seekstr := substr(v_stack_[len(v_stack_)-1],2)
   else
      * last elem
      idtype := left(v_stack_[len(v_stack_)],1)
      alias := help_alias(idtype)
      seekstr := substr(v_stack_[len(v_stack_)],2)
   endif

   * does the dd file actually contain a helpmemo field?
   if idtype $ "MBSF" .and. !empty(seekstr)

      @ 0,0 clear to 23, 79
      @ 0,0 to 23, 79 double
      v_stat24(space(30)+"<Esc> to exit help")

      select (alias)
      xrec := recno()
      dbseek(seekstr)
      if idtype == "F"
         * find does not have complete enough index key to find
         * exact rec needed
         if found()
            id = v_fnd->id
            do while v_fnd->id == id .and. !v_fnd->(eof())
               if &(v_fnd->ialias) == cur_ord
                  * found it and do help memoedit
                  if empty(field->helpmemo)
                     has_help := .f.
                  endif
                  exit  // end loop
               endif
               dbskip()
            enddo
         else
            has_help := .f.
         endif
      else
         if !found() .or. empty(field->helpmemo)
            has_help := .f.
         endif
      endif

      if !has_help
         memoedit("                  Sorry, no help is available at this point.",1,1,22,78,.f.)
      else
         memoedit(field->helpmemo,1,1,22,78,.f.)
      endif

      dbgoto(xrec)
      select (xarea)
      setcolor(xcolor)
      setcursor(xcursor)
      restscreen(0,0,24,79,buf)
   endif
   setkey(K_F1,F1block)
return .t.


*****************************************************************************
*  Function : help_alias()
*****************************************************************************
*
*  Purpose..: Return the dd alias for the function idtype passed
*
*  Syntax...: help_alias(idtype)
*
*  Arguments: idtype is the type of CDDLIB function "MBSF"
*
*  Returns..: The dd alias or "" if not found
*
*  Author...: Vick Perry,   Tue  09-29-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Return the dd alias for the function idtype passed
*****************************************************************************
function help_alias(idtype)
   local s := ""
   idtype := upper(alltrim(idtype))
   do case
      case idtype == "B"
         s := "V_BRO"
      case idtype == "M"
         s := "V_MEN"
      case idtype == "F"
         s := "V_FND"     // not v_fid
      case idtype == "S"
         s := "V_SCR"
   endcase
return s


*****************************************************************************
*  Function.: v_ghelp()
*****************************************************************************
*
*  Purpose..: Display the current get help, uses v_stack_ array to seek to
*             help memo
*
*  Syntax...: v_ghelp()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Sat  08-22-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Display the help for the active get
*****************************************************************************
function v_ghelp()
   local buf := savescreen(0,0,24,79)
   local xcolor := setcolor(v_hlpcolor)
   local xcursor := setcursor(0)
   local F1block := setkey(K_F1)
   local xarea := select()
   local xrec := 0
   local seekstr := ""
   local cur_get := readvar()

   set key K_F1 to

   * help from active gets is called via hotkey, v_stack_
   * indicates current screen
   seekstr := substr(v_stack_[len(v_stack_)],2)

   @ 0,0 clear to 23, 79
   @ 0,0 to 23, 79 double
   v_stat24(space(30)+"<Esc> to exit help")

   select v_get
   xrec := recno()
   dbseek(seekstr)
   if found()
      * no index on id+internal is available seek first, then loop until found
      do while v_get->id = seekstr .and. !v_get->(eof())
         if upper(alltrim(v_get->internal)) == cur_get
            * found it, do help
            if empty(field->helpmemo)
               memoedit("                  Sorry, no help is available at this point.",1,1,22,78,.f.)
            else
               memoedit(field->helpmemo,1,1,22,78,.f.)
            endif
            exit
         endif
         dbskip()
      enddo
   else
      memoedit("                  Sorry, no help is available at this point.",1,1,22,78,.f.)
   endif

   dbgoto(xrec)
   select (xarea)
   setcolor(xcolor)
   setcursor(xcursor)
   restscreen(0,0,24,79,buf)
   setkey(K_F1,F1block)
return .t.


*****************************************************************************
*  Function.: v_backdrop()
*****************************************************************************
*
*  Purpose..: Display the standard backdrop with optional titles.
*             Uses v_syscolor, v_header0 and v_header1 public vars.
*
*  Syntax...: v_backdrop()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Sun  08-23-1992
*
*  Notes....:
*
*  Revisions: dvp 1/31/92 - if no titles then no box
*
*  NG Short.: Display the standard backdrop with optional titles.
*****************************************************************************
function v_backdrop
   local xcolor := setcolor(v_syscolor)
   local i
   local numhead := 0
   local s := replicate(v_backchar,80)
   for i = 0 to 24
      @ i, 0 say s
   next


   * display header 1
   if !empty(v_header0)
      @ numhead+1, 0 say space(80) color v_titcolor
      @ numhead+1, 40-(len(v_header0)/2) say v_header0 color v_titcolor
      ++numhead
   endif

   * display header 2
   if !empty(v_header1)
      @ numhead+1, 0 say space(80) color v_titcolor
      @ numhead+1, 40-(len(v_header1)/2) say v_header1 color v_titcolor
      ++numhead
   endif

   * draw box around title (if there is a title - dvp 1/31/92)
   if numhead > 0
      dispbox(0,0,numhead+1,79,,v_titcolor)
   endif

   * clear status lines
   v_stat23()
   v_stat24()
   setcolor(xcolor)
return .t.


*****************************************************************************
*  Function.: v_alert()
*****************************************************************************
*
*  Purpose..: Float function for alert(), intended to be spiffed up later
*             using some function other than Clipper's alert().
*
*  Syntax...: v_alert(msg [,choice_] [,is_sound])
*
*  Arguments: msg is the message to display
*             choice_ is an array of user choice strings
*             is_sound is the flag to sound a beep, default is .f.
*
*  Returns..: The numeric choice if an array was passed, otherwise .f.
*
*  Author...: Vick Perry,   Sun  08-23-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Display alert()-like prompt
*****************************************************************************
function v_alert(msg,a_,is_sound)
   local x
   local maxw := 0
   local s_current, s_remain
   local alen := 0
   local i := 0
   local numlines := 0
   local t,l,b,r
   local buf

   default is_sound to .f.
   msg := alltrim(msg)

   * get max width of text
   if ";" $ msg
      s_remain := msg
      do while at(";", s_remain) > 0
         ++numlines
         s_current := alltrim(left(s_remain,at(";", msg)-1))
         maxw := max(maxw,len(s_current))
         s_remain := alltrim(substr(s_remain,at(";", msg)+1))
      enddo
      if len(s_remain)>0
         ++numlines
         maxw := max(maxw,len(s_remain))
      endif
   else
      numlines := 1
      maxw := len(msg)
   endif

   * get max width of array line
   if !a_ == nil
      alen := len(a_)
      s_current := ""
      for i = 1 to alen
         s_current += alltrim(a_[i]) + space(4)
      next
      maxw := max(maxw,len(alltrim(s_current)))
   endif

   * calc alert box coords
   if numlines % 2 == 0
      * even number of lines
      t := 11-(numlines/2)-1
      b := 11+(numlines/2)+2
   else
      * odd number of lines
      t := 11-((numlines-1)/2)-1
      b := 11+((numlines-1)/2)+3
   endif

   l := 40-(maxw/2)-4
   r := 40+(maxw/2)+3

   * save screen and shadow area
   buf := savescreen(0,0,24,79)

   * paint shadow, at least until alert() msg string exceeds 52
   * after 52 alert() does some unfathomable box size changes
   if maxw < 53
      v_shadow(t,l,b,r)
   endif

   * sound
   if is_sound
      v_beep("2")
   endif

   * call Clipper's alert()
   if a_ == nil
      alert(msg,,v_msgcolor)
      x := .f.               // no array passed, return .f.
   else
      x := alert(msg,a_,v_msgcolor)
   endif

   * restore screen
   restscreen(0,0,24,79,buf)
return x


*****************************************************************************
*  Function : v_exit()
*****************************************************************************
*
*  Purpose..: Set v_exitflag variable.  Generally called as "U" type
*             function from a menu or browse key to exit from the active
*             browse or active menu
*
*  Syntax...: v_exit([new])
*
*  Arguments: new is the optional new setting for v_exitflag.  The default
*             is .t. (to signal exit).
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  08-25-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Cause a menu, browse, or screen to terminate
*****************************************************************************
function v_exit(new)
   default new to .t.
   v_exitflag := new
return .t.


*****************************************************************************
*  Function : v_shadow
*****************************************************************************
*
*  Purpose..: Float function for vattr() assembly routine.  For Color only.
*
*  Syntax...: v_shadow(t,l,b,r,colornum)
*
*  Arguments: t,l,b,r are the BOX coordinates, the shadow is painted OUTSIDE
*             the box area.
*
*             colornum is the decimal color attribute
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Tue  08-25-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Display a shadow
*****************************************************************************
function v_shadow(t,l,b,r,colornum)
   local w_vert := 0
   local is_even := .f.
   external vattr
   if v_shadow
      default colornum to 8
      do case
         case v_s_corner == 0
            w_vert := min(l, 2)
            vattr(t+1,l-w_vert,b,l-1,colornum)   // left
            vattr(b+1,l-w_vert,b+1,r-2,colornum) // bottom
         case v_s_corner == 1
            w_vert := min(l, 2)
            vattr(t,l-w_vert,b-1,l-1,colornum)   // left
            vattr(t-1,l-w_vert,t-1,r-2,colornum) // top
         case v_s_corner == 2
            w_vert := min(79-r, 2)
            vattr(t,r+1,b-1,r+w_vert,colornum)   // rt
            vattr(t-1,l+2,t-1,r+w_vert,colornum) // top
         case v_s_corner == 3
            w_vert := min(79-r, 2)
            vattr(t+1,r+1,b,r+w_vert,colornum)   // rt
            vattr(b+1,l+2,b+1,r+w_vert,colornum) // bottom
      endcase
   endif
return nil


*****************************************************************************
*  Function : v_highbar()
*****************************************************************************
*
*  Purpose..: Float function for vattr() assembly routine to display full
*             browse highlight bar.  Same as v_shadow() except paints
*             video attribute for both color and mono.
*
*  Syntax...: v_highbar(t,l,b,r,colornum)
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Tue  08-25-1992
*
*  Notes....: I see that tbrowse has a color rectangle method.  Change this
*             function later. - dvp
*
*  Revisions:
*
*  NG Short.: Paint highlight bar on browse
*****************************************************************************
function v_highbar(t,l,b,r,colornum)
   vattr(t+1,l-1,b,l-1,colornum)   // left
   vattr(b+1,l-1,b+1,r-1,colornum) // bottom
return nil


*****************************************************************************
*  Function : v_vieword()
*****************************************************************************
*
*  Purpose..: Change the viewing index order while in a browse or screen.
*             V_vieword() is designed to be called from a menu and does
*             the following:
*
*                    1.  Select desired area
*                    2.  Set new order
*                    3.  Signal menu to terminate
*
*  Syntax...: v_vieword(cAlias, nOrder)
*
*  Arguments: cAlias is the alias
*
*             nOrder is the new index order (an index alias)
*
*  Returns..: the original index order number
*
*  Author...: Vick Perry,   Thu  08-27-1992
*
*  Notes....: If called from a browse key, set the key to REFRESHALL = "Y".
*             If called from a screen key, set the key to RELOAD = "Y" and
*             REDISPLAY = "Y".
*
*  Revisions:
*
*  NG Short.: Change the viewing index order while in a browse or screen.
*****************************************************************************
function v_vieword(cAlias,nOrder)
   local old_order
   local xrec
   local o_area := select()
   select (cAlias)
   xrec := recno()
   old_order := v_order(nOrder)
   dbgoto(xrec)    // don't know why, but a reposition is sometimes needed
   select (o_area)
   v_exit(.t.)
return old_order


*****************************************************************************
*  Function : parse_color()
*****************************************************************************
*
*  Purpose..: Return a single foreground/background color pair from a
*             Clipper color string.
*
*  Syntax...: parse_color(mode [,colorstr])
*
*  Arguments: mode is the color pair to return. 0=standard, 1=enhanced,
*             2=border, 3=background, 4=unselected.  Default is 0.
*
*             colorstr is the optional colorstring to parse, otherwise
*             the current color setting is used
*
*  Returns..: a string containing the extracted color pair
*
*  Author...: Vick Perry,   Fri  08-28-1992
*
*  Notes....:
*               * parse_color() constants
*               #define C_STD 0      // standard
*               #define C_ENH 1      // enhanced
*               #define C_BRD 2      // border
*               #define C_BKG 3      // background (no supported)
*               #define C_UNS 4      // unselected
*
*  Revisions:
*
*  NG Short.: Extract a color pair from a color string
*****************************************************************************
function parse_color(mode, colorstr)
   local xcolor := setcolor()
   local pos := 0
   if colorstr == nil
      s:= setcolor()
   else
      s := colorstr      // optional passed color string
   endif
   do case
      case mode == C_STD
         pos := at(",", s)
         if pos > 0
            s := left(s, at(",", s)-1)
         endif
      case mode == C_ENH
         s := substr(s, at(",", s)+1)  // peel off first (std) color
         pos := at(",", s)
         if pos > 0
            s := left(s, at(",", s)-1)
         endif
      case mode == C_BRD
         s := substr(s, at(",", s)+1)  // peel off first (std) color
         s := substr(s, at(",", s)+1)  // peel off second (enh) color
         pos := at(",", s)
         if pos > 0
            s := left(s, at(",", s)-1)
         endif
      case mode == C_BKG
         s := ""
      case mode == C_UNS
         s := substr(s, rat(",", s)+1)
      otherwise
         s := ""
   endcase
   setcolor(xcolor)
return s


*****************************************************************************
*  Function : color2dec()
*****************************************************************************
*
*  Purpose..: Convert Clipper color string pairs into decimal color numbers
*
*  Syntax...: color2dec(colorpair)
*
*  Arguments: colorpair is a string containing a clipper color pair.
*             E.g. "W+/B"
*
*  Returns..: a decimal number
*
*  Author...: Vick Perry,   Fri  08-28-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Convert fore/back color string to decimal color
*****************************************************************************
function color2dec(colorpair)
   local n := 0
   local fore, back, multiplier, highbits, lowbits
   colorpair := upper(alltrim(colorpair))

   * is it a mono only color?
   if !("U" $ colorpair .or. "I" $ colorpair .or. "X" $ colorpair)

      * get foreground letter
      fore := alltrim(left(colorpair, at("/", colorpair)-1))

      * ignore high intensity designator + in foreground color
      if right(fore) == "+"
         fore := left(fore,len(fore)-1)
      endif

      * get background letter
      back := alltrim(substr(colorpair, at("/", colorpair)+1))

      do case
         case back == "N" .or. back == ""   //blk
            highbits := 0
         case back == "B"
            highbits := 1
         case back == "G"
            highbits := 2
         case back == "BG"
            highbits := 3
         case back == "R"
            highbits := 4
         case back == "RB"
            highbits := 5
         case back == "GR"
            highbits := 6
         case back == "W"
            highbits := 7
      endcase

      do case
         case fore == "N" .or. fore == ""   //blk
            lowbits := 0
         case fore == "B"
            lowbits := 1
         case fore == "G"
            lowbits := 2
         case fore == "BG"
            lowbits := 3
         case fore == "R"
            lowbits := 4
         case fore == "RB"
            lowbits := 5
         case fore == "GR"
            lowbits := 6
         case fore == "W"
            lowbits := 7
         case fore == "N+"
            lowbits := 8
         case fore == "B+"
            lowbits := 9
         case fore == "G+"
            lowbits := 10
         case fore == "BG+"
            lowbits := 11
         case fore == "R+"
            lowbits := 12
         case fore == "RB+"
            lowbits := 13
         case fore == "GR+"
            lowbits := 14
         case fore == "W+"
            lowbits := 15
      endcase

      * convert to decimal color
      n := (highbits * 16) + lowbits
   else
      * a mono color
      do case
         case colorpair == "U"
            n := 1                 // dim underline
         case colorpair == "I"
            n := 120               // reverse
         case colorpair == "X"
            n := 0                 // blank
      endcase
   endif
return n


*****************************************************************************
*  Function.: v_fade()
*****************************************************************************
*
*  Purpose..: Float function for vattr() - used to fade browse and menu
*             boxes
*
*  Syntax...: v_fade(t,l,b,r,n)
*
*  Arguments: screen coordinated and decimal color
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-04-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Fade a browse, menu, or screen
*****************************************************************************
function v_fade(t,l,b,r,n)
   external vattr
   vattr(t,l,b,r,n)
return .t.


*****************************************************************************
*  Function.: v_find()
*****************************************************************************
*
*  Purpose..: To provide find capability for a browse
*
*             WARNING: V_FIND() should never be called directly.  It
*                      must be called via a browse or screen key.
*
*  Syntax...: v_find(id) --> success
*
*  Arguments: id is the find function id
*
*  Returns..: a boolean value - .t. if successful find
*
*  Author...: Vick Perry,   Sat  09-05-1992
*
*  Notes....:
*
*  Revisions: Remember to set refreshall switch to "Y" in browse/screen
*
*  NG Short.: Find function called as browse or screen key
*****************************************************************************
function v_find(id)
   local getlist   := {}
   local did_find  := .f.
   local i         := 0
   local elem      := 0
   local curord    := 0
   local oconfirm  := .f.
   local osoftseek := .f.
   local orec      := 0
   local alen      := 0
   local xbuf      := savescreen(23,0,24,79)
   local xcolor    := setcolor(v_findcolor)
   local xcursor   := setcursor(1)
   local xarea     := select()
   local temppict  := ""
   local xrec      := 0

   * find arrays
   local xialias_     := {}
   local xtitle_      := {}
   local xftype_      := {}
   local xflen_       := {}
   local xfdec_       := {}
   local xpict_       := {}
   local xinit_expr_  := {}
   local xcond_val_   := {}
   local xclean_expr_  := {}
   local xapnd_expr_  := {}
   local xis_confirm_ := {}
   local xis_soft_    := {}

   * declare the get variable private so that the post process and append
   * expression can modify it if needed.
   private v_findget

   * get find information from dd
   select v_fid
   seek id
   if v_fid->(found())
      * position find get items file
      select v_fnd
      seek id
      if v_fnd->(found())
         * restore to orig workarea (workarea to find )
         select (xarea)

         * does the alias listed in the find match this workarea?
         if alias() == upper(alltrim(v_fid->dalias))
            * load find get arrays
            select v_fnd
            do while v_fnd->id == v_fid->id .and. !v_fnd->(eof())
               aadd(xialias_   ,  v_fnd->ialias    )
               aadd(xtitle_    ,  v_fnd->title     )
               aadd(xftype_    ,  v_fnd->ftype     )
               aadd(xflen_     ,  v_fnd->flen      )
               aadd(xfdec_     ,  v_fnd->fdec      )
               aadd(xpict_     ,  v_fnd->pict      )
               aadd(xinit_expr_,  alltrim(v_fnd->init_expr  ))
               aadd(xcond_val_ ,  alltrim(v_fnd->cond_val   ))
               aadd(xclean_expr_, alltrim(v_fnd->clean_expr ))
               aadd(xapnd_expr_,  alltrim(v_fnd->apnd_expr  ))
               aadd(xis_confirm_, v_fnd->is_confirm)
               aadd(xis_soft_  ,  v_fnd->is_soft   )

               select v_fnd
               skip
            enddo

            * how many find items?
            alen := len(xialias_)

            * back to orig workarea again
            select (xarea)

            * save current recno
            orec := recno()

            * is there an active index?
            curord := indexord()
            if curord > 0
               * is there a find defined for the current index order?
               for i = 1 to alen
                  if &(xialias_[i]) == curord
                     elem := i
                     exit
                  endif
               next

               * if no find defined for this index, simply return
               if elem > 0

                  * clear status lines
                  @ 23, 0 say space(80) color v_findcolor
                  @ 24, 0 say space(80) color v_findcolor

                  * preload with init expr if required
                  if !empty(xinit_expr_[elem])
                     v_findget := &(xinit_expr_[elem])
                  endif

                  * init/pad v_findget
                  * set default pictures for each data type
                  do case

                      case xftype_[elem] == "C"
                         if v_findget == nil
                            v_findget := space(xflen_[elem])
                         else
                            v_findget := padr(v_findget,xflen_[elem])
                         endif
                         temppict := '"@K!"'

                      case xftype_[elem] == "N"
                         if v_findget == nil
                            v_findget := 0
                         endif
                         temppict := '"@K ' + replicate("9",xflen_[elem]) + '"'

                      case xftype_[elem] == "L"
                         if v_findget == nil
                            v_findget := .t.
                         endif
                         temppict := '"Y"'

                      case xftype_[elem] == "D"
                         if v_findget == nil
                            v_findget := ctod("")
                         endif
                         temppict := '"@D"'
                  endcase

                  * override default picture if programmer specified another
                  if !empty(xpict_[elem])
                     temppict := '"'+alltrim(xpict_[elem])+'"'
                  endif

                  * save/set confirm
                  oconfirm := set(_SET_CONFIRM, xis_confirm_[elem])

                  * get user input
                  * different get stmts depending upon presence of pict or valid
                  if !empty(xcond_val_[elem])
                     @ 24, 1 say "Search for " + alltrim(xtitle_[elem])+":" get v_findget picture &temppict valid &(xcond_val_[elem])
                  else
                     @ 24, 1 say "Search for " + alltrim(xtitle_[elem])+":" get v_findget picture &temppict
                  endif
                  read

                  * restore confirm
                  set(_SET_CONFIRM, oconfirm)

                  * got something to search on?
                  if !(lastkey() == 27 .or. empty(v_findget))

                     * save softseek
                     osoftseek := set(_SET_SOFTSEEK, xis_soft_[elem])

                     * cleanup expression returns cleaned v_findget
                     if !empty(xclean_expr_[elem])
                        v_findget := &(xclean_expr_[elem])
                     endif

                     * append expression to front of v_findget
                     if !empty(xapnd_expr_[elem])
                        v_findget := &(xapnd_expr_[elem]) + v_findget
                     endif

                     * find the pattern
                     dbseek(rtrim(v_findget))

                     * clear status lines in case of "not found" message
                     @ 23, 0 say space(80) color v_findcolor
                     @ 24, 0 say space(80) color v_findcolor

                     if !xis_soft_[elem]
                        * softseek OFF
                        if found()
                           did_find := .t.
                        else
                           * back to original record
                           v_alert("Not found")
                           go orec
                        endif
                     else
                        * softseek ON
                        if !eof()
                           did_find := .t.
                        else
                           * back to original record
                           v_alert("Not found")
                           go orec
                        endif
                     endif

                     * restore softseek setting
                     set(_SET_SOFTSEEK, osoftseek)
                  endif
               else
                  v_alert("No FIND defined for index order = "+str(indexord(),2))
               endif
            else
               * cannot find with no index
               v_alert("An index must be active for find: "+id)
            endif
         else
            * wrong workarea
            v_alert("Incorrect workarea for this find: "+id)
         endif
      else
         * no items
         v_alert("No finds defined for find id: "+id)
      endif
   else
      * unknown find id
      v_alert("Unknown find id: "+id)
   endif

   * restores
   select (xarea)
   setcursor(xcursor)
   setcolor(xcolor)
   restscreen(23,0,24,79,xbuf)
return did_find


*****************************************************************************
*  Function.: v_nil()
*****************************************************************************
*
*  Purpose..: Does absolutely nothing and returns nil.  Used for browse
*             keys defined as visual spacers but that have no actual use
*
*  Syntax...: v_nil()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Sun  09-06-1992
*
*  Notes....: Is this still needed? - dvp
*
*  Revisions:
*
*  NG Short.: Does absolutely nothing
*****************************************************************************
function v_nil
return nil


*****************************************************************************
*  Function.: open_files()
*****************************************************************************
*
*  Purpose..: Open all data files and indexes listed in dd as shared.
*             Only opens files where "Open at Startup = Y".
*             Called at startup by v_init().
*
*  Syntax...: open_files()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Open application data files and indexes in shared mode.
*****************************************************************************
function open_files
   local xarea := select()
   * open data files and indexes
   select v_dbf
   go top
   do while !v_dbf->(eof())
      if v_dbf->open
         if !v_open(v_dbf->dalias)
            v_fatal("Cannot open file "+v_dbf->dname+" as shared.  Aborting...")
         endif
      endif
      select v_dbf
      skip
   enddo
   select (xarea)
return nil


*****************************************************************************
*  Function.: v_reindex()
*****************************************************************************
*
*  Purpose..: Simple file reindexing utility. This utility closes and
*             reopens all application data files during operation - do not
*             call v_reindex() from within an active browse or similar
*             file-dependent function.
*
*             This function is generally called from a utilities menu.
*
*  Syntax...: v_reindex([cmdstr] [,is_pack])
*
*  Arguments: cmdstr is the command parameter string where:
*
*             cmdstr = nil or "" - show taggable pick list
*             cmdstr = "ALL"     - index all application files
*             cmdstr = "CDD"     - index all CDD data dictionary files
*             cmdstr = DBF alias - index the dbf specified by the alias
*
*             is_pack is the optional pack flag, default is .f. - no pack
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Reindexing function for users
*****************************************************************************
function v_reindex(cmdstr,is_pack)
   local sfile := v_ddpath+v_sysname+SYS_SUF
   local tempstr
   local xrec
   local n
   local sys_area := 0

   default cmdstr to ""
   default is_pack to .f.
   cmdstr := upper(alltrim(cmdstr))

   * use dd v_sys as exclusive - this is also semaphore file
   * that will prohibit access to v_reindex() if other users are
   * still in the system and will prevent other users from
   * running the program if reindexing is operating.
   select v_sys
   sys_area := select("v_sys")
   if v_net_use(sfile,.t.,4,"V_SYS")
      * close all data files - but not the data dictionary files!
      * dd files can be detected by their alias = "V_"
      for i = 1 to 250
         dbselectarea(i)
         tempstr := alias()
         do case
            case empty(tempstr)
               * do nothing

            case left(tempstr,2) == "V_"    // "V_" indicates dd file
               * do nothing

            otherwise
               * close the non-dd file
               v_db_close(tempstr)
         endcase
      next

      * reindex
      do case
         case empty(cmdstr)
            * user selects files to reindex
            v_dbrow()

         case cmdstr == "CDD"
            * reindex the data dictionary files

            * reopen dd files as exclusive, close the runtime indexes
            * before reindexing each file
            open_dd(v_sysname,v_ddpath, .t.)

            v_stat23('Sorting Files. Please wait...')
            select v_dbf
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+DBF_SUF+"1.NTX","UPPER(dalias)",{|| UPPER(dalias)},)
            DBCREATEINDEX(v_ddpath+v_sysname+DBF_SUF+"2.NTX","UPPER(dname)",{|| UPPER(dname)},)
            close

            select v_bro
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+BRO_SUF+"1.NTX","UPPER(id)",{|| UPPER(id)},)
            DBCREATEINDEX(v_ddpath+v_sysname+BRO_SUF+"2.NTX","UPPER(dalias)",{|| UPPER(dalias)},)
            close

            select v_men
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+MEN_SUF+"1.NTX","UPPER(id)",{|| UPPER(id)},)
            DBCREATEINDEX(v_ddpath+v_sysname+MEN_SUF+"2.NTX","UPPER(title)",{|| UPPER(title)},)
            close

            select v_fid
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+FID_SUF+"1.NTX","UPPER(id)",{|| UPPER(id)},)
            DBCREATEINDEX(v_ddpath+v_sysname+FID_SUF+"2.NTX","UPPER(dalias)",{|| UPPER(dalias)},)
            close

            select v_scr
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+SCR_SUF+"1.NTX","UPPER(id)",{|| UPPER(id)},)
            DBCREATEINDEX(v_ddpath+v_sysname+SCR_SUF+"2.NTX","UPPER(dalias)",{|| UPPER(dalias)},)
            close

            select v_fld
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+FLD_SUF+"1.NTX","UPPER(dalias)+TRANS(pos,'999')",{|| UPPER(dalias)+TRANS(pos,'999')},)
            DBCREATEINDEX(v_ddpath+v_sysname+FLD_SUF+"2.NTX","UPPER(dalias)+UPPER(fname)",{|| UPPER(dalias)+UPPER(fname)},)
            close

            select v_idx
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+IDX_SUF+"1.NTX","UPPER(dalias)+UPPER(ialias)",{|| UPPER(dalias)+UPPER(ialias)},)
            DBCREATEINDEX(v_ddpath+v_sysname+IDX_SUF+"2.NTX","UPPER(dalias)+UPPER(iname)",{|| UPPER(dalias)+UPPER(iname)},)
            close

            select v_col
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+COL_SUF+"1.NTX","UPPER(id)+TRANS(pos,'999')",{|| UPPER(id)+TRANS(pos,'999')},)
            close

            select v_key
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+KEY_SUF+"1.NTX","UPPER(id)+TRANS(pos,'999')",{|| UPPER(id)+TRANS(pos,'999')},)
            DBCREATEINDEX(v_ddpath+v_sysname+KEY_SUF+"2.NTX","UPPER(id)+UPPER(desc)",{|| UPPER(id)+UPPER(desc)},)
            close

            select v_mnk
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+MNK_SUF+"1.NTX","UPPER(id)+TRANS(pos,'999')",{|| UPPER(id)+TRANS(pos,'999')},)
            close

            select v_fnd
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+FND_SUF+"1.NTX","UPPER(id)+UPPER(ialias)",{|| UPPER(id)+UPPER(ialias)},)
            close

            select v_skey
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+SKY_SUF+"1.NTX","UPPER(id)+TRANS(pos,'999')",{|| UPPER(id)+TRANS(pos,'999')},)
            DBCREATEINDEX(v_ddpath+v_sysname+SKY_SUF+"2.NTX","UPPER(id)+UPPER(desc)",{|| UPPER(id)+UPPER(desc)},)
            close

            select v_get
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+SGT_SUF+"1.NTX","UPPER(id)+TRANS(pos,'999')",{|| UPPER(id)+TRANS(pos,'999')},)
            close

            select v_seq
            dbclearindex()
            DBCREATEINDEX(v_ddpath+v_sysname+SEQ_SUF+"1.NTX","UPPER(id)",{|| UPPER(id)},)
            close

            * reopen dd files
            * in theory, there is no need to save file pointers for the
            * dd files, because each active function has already loaded
            * the information that it needs to operate.
            * Note that v_sys is now opened as shared after this but you
            * are exiting the function anyway.
            open_dd(v_sysname,v_ddpath)

            v_stat23()
            clear typeahead

         case cmdstr == "ALL"
            * index all files
            dbselectarea("v_dbf")
            go top
            do while !v_dbf->(eof())
               xrec := recno()
               if is_pack
                  if !v_pack(v_dbf->dalias,.t.)
                     v_fatal("Unable to pack "+v_dbf->dname+".DBF - Aborting")
                  endif
               endif
               if !v_index(v_dbf->dalias,.t.)
                  v_fatal("Unable to index "+v_dbf->dname+".DBF - Aborting")
               endif
               dbselectarea("v_dbf")
               dbgoto(xrec)
               dbskip()
            enddo
            clear typeahead

         otherwise
            * reindex one file
            if !v_index(cmdstr,.t.)
               v_fatal("Unable to index "+v_dbf->dname+".DBF - Aborting")
            endif
            clear typeahead
      endcase

      * reopen v_sys as shared
      select v_sys
      if !v_net_use(sfile,.f.,4,"V_SYS")
         v_fatal("Unable to reopen data dictionary - Aborting")
      endif

      * reopen data files and indexes
      open_files()
   else
      * since a failed open excl closes file, reopen v_sys as shared
      select (sys_area)
      if !v_net_use(sfile,.f.,4,"V_SYS")
         v_fatal("Unable to reopen data dictionary - Aborting")
      endif
      v_alert("Cannot reindex - Other users still in program.")
   endif
return .t.


*****************************************************************************
*  Function.: v_pack()
*****************************************************************************
*
*  Purpose..: Pack a data file (with no indexes open).  The data file is
*             closed upon exiting v_pack()
*
*  Syntax...: v_pack(alias,[show_stat])
*
*  Arguments: alias is the data file alias string to pack
*
*             show_stat - .t. to display status line
*
*  Returns..: .f. if unable to pack file
*
*  Author...: Vick Perry,   Tue  08-18-1992
*
*  Notes....: It is advisable to close the file, call v_pack(), then
*             reopen it.
*
*  Revisions:
*
*  NG Short.: Pack a data file
*****************************************************************************
function v_pack(alias,show_stat)
   local buf := savescreen(22,0,24,79)
   local l := .f.
   local ipath := ""
   local tempfile := ""
   local path := ""
   alias := padr(upper(alltrim(alias)),len(v_idx->dalias))
   if v_open(alias,.t.,.f.)  // no indexes - v_open() also positions v_dbf
      if show_stat
         v_stat24("Packing "+rtrim(v_dbf->dname)+".DBF")
      endif

      * is there a DBT?
      if v_is_memo()
         path := if(empty(v_dbf->altpath),alltrim(v_sys->datapath),alltrim(v_dbf->altpath))
         is_del := set(_SET_DELETED,.t.)
         copy to (path+v_sysname+"!")     // sysname + "!" is tempfile
         set(_SET_DELETED,is_del)

         * close and erase orig file, rename tempfiles
         select (alias)
         v_db_close(alias)

         * normally I would check return values for ferase() and frename();
         * however, there are naming conflicts between Funcky 1.5 and
         * Clipper 5.01a for these functions.  The Funcky 1.5 functions
         * return boolean values, and the Clipper 5.01a functions return
         * numerics
         ferase(path+alltrim(v_dbf->dname)+".DBF")
         ferase(path+alltrim(v_dbf->dname)+".DBT")
         frename(path+v_sysname+"!.DBF", path+alltrim(v_dbf->dname)+".DBF")
         frename(path+v_sysname+"!.DBT", path+alltrim(v_dbf->dname)+".DBT")

         * file is already closed
      else
         * pack the DBF, there is no DBT
         pack

         * close file that you just packed
         select (alias)
         v_db_close(alias)
      endif

      l := .t.
   endif

   restscreen(22,0,24,79,buf)
return l


*****************************************************************************
*  Function.: v_dbrow()
*****************************************************************************
*
*  Purpose..: Browse for v_reindex()
*
*  Syntax...: v_dbrow()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions: dvp 12/5/92 - needed to exclude records where v_dbf->is_index
*                           is .f.  Since no index with is_index is available,
*                           and the total records are commonly less than 60,
*                           I used a SET FILTER TO.  So shoot me...
*
*  NG Short.: Browse used by v_reindex()
*****************************************************************************
function v_dbrow
   local buf := savescreen(0,0,24,79)
   local t := 7                                   // TBROWSE coordinates
   local l := 3
   local b := t + 14
   local r := l + 45
   local o_color := setcolor(v_brocolor)          // save, set color
   local d := tbrowsedb( t+1, l+1, b-1, r-1)      // create a TBROWSE object
   local tmp_column                               // temp column object
   local nkey                                     // input keystroke
   local nchar                                    // key converted to char
   local tag_ := {}
   local choice := 0
   local xtitle := "INDEX/PACK DATA FILES"
   local buf2

   /*
     Size and fill the data tag array
   */
   v_tag_init(v_dbf->(lastrec()))

   /*
     Select data control workarea, position pointer
   */
   select v_dbf

   * exclude recs where v_dbf->is_index == .f.
   set filter to is_index

   go top

   /*
     paint backdrop and draw titled box
   */
   v_backdrop()                // display backdrop
   @ t-2,l clear to b,r
   dispbox(t-2,l,b,r)
   v_shadow(t-2,l,b,r)

   * draw title
   @ t-1,l+((r-l)/2)-(len(xtitle)/2) say xtitle

   * draw line below title, same type as top line of box
   @ t,l+1 say replicate("",r-l-1)

   v_stat23("<Space> to Tag/Untag    <Enter> to Index    <Esc> to Exit",.t.)
   v_stat24()

   /*
     Define the TBROWSE columns and get/set blocks
   */
   tmp_column := tbcolumnnew("",{|| v_tag(recno()) } )
   d:addColumn(tmp_column)
   tmp_column := tbcolumnnew(" Filename",{|| dname})
   d:addColumn(tmp_column)
   tmp_column := tbcolumnnew(" Alias",{|| dalias})
   d:addColumn(tmp_column)
   tmp_column := tbcolumnnew(" Description",{|| desc})
   d:addColumn(tmp_column)

   /*
     After columns are defined, define TBROWSE global instances
   */
   d:colSep  := ""          // column separator
   d:headSep := ""         // header/data separator
   d:freeze  := 1            // freeze tag, alias, and filename columns

   do while .t.
      /*
        keep out of first column (tag column)
      */
      if d:colpos == 1
         d:colpos := 2
      endif

      /*
        Loop until stabilized or key pressed
      */
      do while ( !d:stabilize() )   // loop until stabilized or key pressed
         nkey := inkey()
         if ( nkey != 0 )
            exit
         endif
      enddo

      // Wait for keypress if browse is stable
      if d:stable
         nkey := inkey(0)             // wait for user keystroke
      endif
      nchar := upper(chr(nkey))       // convert to char for some comparisons

      // Process keystrokes -  exception keys first, then navigation keys
      do case

         // Escape
         case nkey == K_ESC
            exit

         // Help
         case nkey == K_F1

         // do the reindexing
         case nkey == K_ENTER
            if v_any_tagged()
               buf2 := savescreen(0,0,24,79)

               * fade browse screen area
               v_fade(t-2,l,b,r,color2dec(v_brofade))

               * clear status lines
               v_stat23("Please choose the desired process",.t.)
               v_stat24()

               choice := v_alert("Choose process:",{"Index","Pack & Index","Exit"})
               if !(choice == 0 .or. choice == 3)
                  v_stat23()
                  xrec := recno()
                  go top
                  do while !v_dbf->(eof())
                     if v_tag(recno()) == ""
                        if choice == 2
                           if !v_pack(v_dbf->dalias,.t.)
                              v_fatal("Unable to pack "+v_dbf->dname+".DBF - Aborting")
                           endif
                        endif
                        if !v_index(v_dbf->dalias,.t.)
                           v_fatal("Unable to index "+v_dbf->dname+".DBF - Aborting")
                        endif
                     endif

                     select v_dbf
                     skip
                  enddo
                  goto xrec
                  v_tag_init(v_dbf->(lastrec()))
                  d:refreshall()
               endif
               restscreen(0,0,24,79,buf2)
            endif

         // Toggle Tag
         case nchar == " "
            if v_tag(recno()) == ""
               v_tag(recno()," ")
            else
               v_tag(recno(),"")
            endif
            d:refreshcurrent()
            d:down()

         // Process standard TBROWSE keys
         otherwise
            v_stdkeys(d,nkey,.f.)
      endcase
   enddo

   * filter off
   select v_dbf
   set filter to

   setcolor(o_color)
   restscreen(0,0,24,79,buf)
return nil


*****************************************************************************
*  Function.: v_tag()
*****************************************************************************
*
*  Purpose..: Return and optionally set a tag character element
*
*  Syntax...: v_tag(pos,tag_char)
*
*  Arguments: pos is the tag position 1..64k
*
*             tag_char is the tag character to set (optional)
*
*  Returns..: the tag_char
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Get/Set a tag
*****************************************************************************
function v_tag(pos,tag_char)
   local s := ""
   if tag_char == nil
     s:=v_tagstr(3,,pos)
   else
     s:=v_tagstr(1,tag_char,pos)
   endif
return s


*****************************************************************************
*  Function.: v_tag_init()
*****************************************************************************
*
*  Purpose..: Initialize tag string to a given size
*
*  Syntax...:v_tag_init(size)
*
*  Arguments: size - the number of elements  1..64k
*
*  Returns..: " " - ignore the return value
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Initialize the tag string
*****************************************************************************
function v_tag_init(size)
return v_tagstr(0,,,size)


*****************************************************************************
*  Function.: v_tag_add()
*****************************************************************************
*
*  Purpose..: Add new elem to tag string
*
*  Syntax...: v_tag_add(tag_char)
*
*  Arguments: tag_char is the tag character to set for the new elem
*
*  Returns..: the tag_char
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Add a new tag element to the tag string
*****************************************************************************
function v_tag_add(tag_char)
return v_tagstr(2,tag_char)


*****************************************************************************
*  Function.: v_any_tagged()
*****************************************************************************
*
*  Purpose..: Is anything tagged?
*
*  Syntax...: v_any_tagged()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Is anything tagged
*****************************************************************************
function v_any_tagged
return v_tagstr(4)


*****************************************************************************
*  Function : v_tag_ret()
*****************************************************************************
*
*  Purpose..: To return the current tag string
*
*  Syntax...: v_tag_ret()
*
*  Arguments: None
*
*  Returns..: A string of tag characters
*
*  Author...: Vick Perry,   Thu  01-21-1993
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Return the entire tag string
*****************************************************************************
function v_tag_ret
return v_tagstr(5)


*****************************************************************************
*  Function.: v_tagstr()
*****************************************************************************
*
*  Purpose..: Tag handler function, never called directly - only called by
*             v_tag_init(), v_tag_add(), v_any_tagged(), v_tag_ret(),
*             and v_tag().
*
*  Syntax...: v_tagstr(mode,tag_char,pos,num_elems)
*
*  Arguments: mode is the mode of operation
*                0 = init and size the tag string
*                1 = set tag char at a position
*                2 = add char to end of tag string representing a new record
*                3 = return the tag char at a position
*                4 = return whether any tags exist
*                5 = return the entire tag string
*
*             tagchar is the tag character
*
*             pos is the tag position
*
*             num_elems is the initial len of tag string
*
*
*  Returns..: the original tag_char (or a boolean if mode==4)
*
*  Author...: Vick Perry,   Mon  09-07-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Tag - Master tag function, not called directly
*****************************************************************************
function v_tagstr(mode,tag_char,pos,num_elems)
   static tagstr := ""
   local ret_val := " "
   do case
      * init and size
      case mode == 0
         tagstr := ""
         tagstr := space(num_elems)  // space =  not tagged

      * set a tag at a position
      case mode == 1
         ret_val := substr(tagstr,pos,1)
         tagstr := left(tagstr,pos-1) + tag_char + substr(tagstr,pos+1)

      * add a new position and set
      case mode == 2
         tagstr += tag_char

      * read a position
      case mode == 3
         ret_val := substr(tagstr,pos,1)

      * return whether anything is tagged at all
      case mode == 4
         ret_val := !empty(tagstr)

      * return the entire string to the caller
      case mode == 5
         ret_val := tagstr
   endcase
return ret_val


*****************************************************************************
*  Function : s_paint()
*****************************************************************************
*
*  Purpose..: Display the screen box, shadow, and text from a screen-type
*             memo field.  Uses current colors for painting.
*
*  Syntax...: s_paint()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Mon  09-14-1992
*
*  Notes....: Does dispbegin/end really do anything here? - dvp
*
*  Revisions:
*
*  NG Short.: Screen - Display the screen text and box
*****************************************************************************
function s_paint()
   local i := 0
   local slen := s_rt-s_left+1

   * begin screen buffering
   dispbegin()

   * clear box
   @ s_top, s_left clear to s_bot, s_rt

   * draw screen text
   for i := s_top to s_bot
      @i,s_left say substr(memoline(s_screen,80,i),s_left,slen)
   next

   * draw box around text
   dispbox(s_top, s_left, s_bot, s_rt, left(s_scrbox,8))   // ignore fill char in box

   * paint shadow around box
   v_shadow(s_top, s_left, s_bot, s_rt)

   * end buffering
   dispend()

return .t.


*****************************************************************************
*  Function.: s_binit()
*****************************************************************************
*
*  Purpose..: Make code blocks of field and memvar initialization expressions
*             and load them into arrays.  There are three types of blocks
*             created, add mode init,  edit/view init, and a get/set block
*             for the screen memvar
*
*  Syntax...: s_binit()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Sat  10-10-1992
*
*  Notes....: Some block expressions in this function require early
*             evaluation of macros.  See NG 5.01a Release Notes about
*             code blocks
*
*  Revisions:
*
*  NG Short.: Screen - Create the various screen code blocks
*****************************************************************************
function s_binit
   local i := 0
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   private tempstr := ""

   for i = 1 to getlen

      * create and assign ADD init block for the get
      * has init expr? -
      if !empty(flds_[GINIT_EXPR, i])
         * has init expr
         * create add mode init block from init_expr
         * force early eval of macro in block expression
         tempstr := alltrim(flds_[GINIT_EXPR, i])
         flds_[GINIT_ADD_B,i]  :=  &("{||"+tempstr+" }")
      else
        * no init expression
        * create block according to type/len/dec defaults
        * block :=  empty initialization block
        do case

           * dvp 1/22/93
           * Variable declaration type is "N" (No Declaration)
           * Create the init block to return the original value.
           * "N" type variables are declared and assigned before
           * the screen is called and must not be wiped out by
           * the new screen
           case flds_[GDECLARE, i] == "N"
              tempstr := flds_[GINTERNAL,i]
              flds_[GINIT_ADD_B, i] := &("{||"+tempstr+" }")

           case flds_[GFTYPE, i] == "C"
              tempstr := "space(" + str(flds_[GFLEN,i]) + ")"
              flds_[GINIT_ADD_B, i] := &("{||"+tempstr+" }")

           case flds_[GFTYPE, i] == "N"
              flds_[GINIT_ADD_B, i] := {|| 0 }

           case flds_[GFTYPE, i] == "D"
              flds_[GINIT_ADD_B, i] := {|| ctod("") }

           case flds_[GFTYPE, i] == "L"
              flds_[GINIT_ADD_B, i] :=  {|| .f. }

           case flds_[GFTYPE, i] == "M"
              flds_[GINIT_ADD_B, i] := {|| "" }
        endcase
      endif

      * create and assign EDIT/VIEW init block for the get
      * is dbf field or memvar?
      if !empty(flds_[GDALIAS, i]) .and. !empty(flds_[GFIELD, i])
         * dbf field, use fieldwblock()
         * the dbf field must be available at this time!!!
         flds_[GINIT_VIEW_B,i] := fieldwblock(flds_[GFIELD, i], select(flds_[GDALIAS, i]) )
      else
         * memvar - same as ADD mode block
         flds_[GINIT_VIEW_B,i] := flds_[GINIT_ADD_B,i]
      endif

      * create a memvar get/set block for each screen variable to be
      * evaluated in s_load()
      * MEMVARBLOCKS are used instead of simple assignments to assign values
      * to the screen variables!
      flds_[GMEMVAR_B,i] := memvarblock(flds_[GINTERNAL,i])

   next
return nil


*****************************************************************************
*  Function : s_fldnum()
*****************************************************************************
*
*  Purpose..: Return the element number in the array flds_ for an internal
*             variable
*
*  Syntax...: s_fldnum(cVarname)
*
*  Arguments: cVarname
*
*  Returns..: a positive numeric value or 0 if not found
*
*  Author...: Vick Perry,   Tue  12-01-1992
*
*  Notes....: Assumes that flds_[GINTERNAL] elements are uppercase,
*             left justified, and right padded to a width of ten.
*
*  Revisions:
*
*  NG Short.: Screen - Return element number of a variable in FLDS_ array
*****************************************************************************
function s_fldnum(cVarname)
   cVarname := padr(upper(ltrim(cVarname)),10)
return ascan(flds_[GINTERNAL],cVarname)


*****************************************************************************
*  Function : s_load()
*****************************************************************************
*
*  Purpose..: Load one screen internal variable.  Depending upon the
*             s_mode setting, the internal variable is assigned the
*             actual dbf field values or empty values.  Also stores
*             initial value for later comparison by s_changed().
*
*             NOTE: Instead of assigning values directly to the screen
*                   variables, assignments are made by evaluating the
*                   get/set code blocks for the variables created by
*                   s_binit()
*
*  Syntax...: s_load(int_var_name)
*
*  Arguments: int_var_name is the name of the screen internal variable
*             to load.  All screen internals must have previously
*             been declared by v_screen() or at a higher level.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: varnum is declared as private for visibility in v_e_alt()
*             in case of variable initialization error.
*
*  Revisions:
*
*  NG Short.: Screen - Initialize a screen variable from field or init expr
*****************************************************************************
function s_load(int_var_name)
   private varnum:= 0

   * find the element for the intvarname
   varnum := s_fldnum(int_var_name)

   * if found, load it into internal variable, and reset init_val element
   if varnum > 0
      * assign the screen variables, via the get/set block
      * assign new value to init_val array for later comparison by s_changed()
      if s_mode == "A"
         flds_[GINIT_VAL, varnum] := eval(flds_[GMEMVAR_B,varnum], eval(flds_[GINIT_ADD_B, varnum]) )
      else
         flds_[GINIT_VAL, varnum] := eval(flds_[GMEMVAR_B,varnum], eval(flds_[GINIT_VIEW_B, varnum]) )
      endif
   endif
return .t.


*****************************************************************************
*  Function : s_loadall()
*****************************************************************************
*
*  Purpose..: Load all internal screen variables
*
*  Syntax...: s_loadall()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Initialize all screen variables
*****************************************************************************
function s_loadall
   local i := 0
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   for i = 1 to getlen
      s_load(flds_[GINTERNAL,i])
   next
return .t.


*****************************************************************************
*  Function : s_refresh()
*****************************************************************************
*
*  Purpose..: Redisplay one screen internal variable.  Uses the current
*             s_mode and current color to determine field color.
*
*  Syntax...: s_refresh(int_var_name)
*
*  Arguments: int_var_name is the name of the screen internal variable
*             to redisplay.  All screen internals must have previously
*             been declared and assigned.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Colors used will either be the current color, or the alt. colors
*             for the screen field
*
*             ASSUMES that the alternate GET color string is a full
*             color string!
*
*  Revisions:
*
*  NG Short.: Screen - Redisplay a screen variables
*****************************************************************************
function s_refresh(int_var_name)
   local i := 0
   local the_color := ""

   * find the element for the intvarname, flds_ is declared private in
   * v_screen()
   * find the element for the intvarname
   i := s_fldnum(int_var_name)

   * if found, display it
   if i > 0

      * skip non-displayed fields, these are
      * set to 99,99 in screen designer
      if flds_[GROW, i] < 24

         * define colors according to mode
         if s_mode == "V"
            the_color := flds_[GUNS_VIEW,i]
         else
            the_color := flds_[GUNS_EDIT,i]
         endif

         *  trim picture
         flds_[GPICT, i] := alltrim(flds_[GPICT, i])

         * paint field
         if empty(flds_[GPICT, i])
            * no picture
            @ flds_[GROW, i], flds_[GCOL, i] say eval(flds_[GMEMVAR_B,i]) color the_color
         else
            * has picture
            @ flds_[GROW, i], flds_[GCOL, i] say eval(flds_[GMEMVAR_B,i]) picture flds_[GPICT, i] color the_color
         endif
      endif
   endif
return .t.


*****************************************************************************
*  Function : s_refreshall()
*****************************************************************************
*
*  Purpose..: Load all internal screen variables
*
*  Syntax...: s_refreshall()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Redisplay all screen variables
*****************************************************************************
function s_refreshall
   local i := 0
   local getlen := len(flds_[GINTERNAL])   // num of subarray elements = num fields

   * begin screen buffering
   dispbegin()

   * paint each screen field
   for i = 1 to getlen
      s_refresh(flds_[GINTERNAL,i])
   next

   * end buffering
   dispend()
return .t.


*****************************************************************************
*  Function : s_gotop()
*****************************************************************************
*
*  Purpose..: Go to logical top and optionally reload memvars and refresh
*             the screen
*
*  Syntax...: s_gotop([refresh_now])
*
*  Arguments: refresh_now is a flag to indicate whether the screen variables
*             are loaded and the screen refreshed immediately.  The
*             default is .t. = immediate screen refresh.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Go to logical (or filtered) top
*****************************************************************************
function s_gotop(refresh_now)
   default refresh_now to .t.
   if !empty(s_filter_exp)
      * position on first rec that meets
      * filter condition
      dbseek(s_limit)
   else
      dbgotop()
   endif
   if refresh_now
      s_loadall()
      s_refreshall()
   endif
return .t.


*****************************************************************************
*  Function : s_gobottom()
*****************************************************************************
*
*  Purpose..: Go to logical bottom and immediately reload vars and
*             refresh screen
*
*  Syntax...: s_gobottom([refresh_now])
*
*  Arguments: refresh_now is a flag to indicate whether the screen variables
*             are loaded and the screen refreshed immediately.  The
*             default is .t. = immediate screen refresh.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Go to logical (or filtered) bottom
*****************************************************************************
function s_gobottom(refresh_now)
   local old_ss
   local last_char := ""
   local bottomlimit := ""

   default refresh_now to .t.

   if !empty(s_filter_exp)
      * position on last rec that meets
      * filter condition
      old_ss := set(_SET_SOFTSEEK,.t.)
      last_char := chr(asc(right(s_limit,1))+1)
      bottomlimit := left(s_limit,len(s_limit)-1) + last_char
      dbseek(bottomlimit)

      * still in range?
      if !eval(s_bIndex) = s_limit .or. eof()
         * no, out of range
         dbskip(-1)

         * in range now?
         if !eval(s_bIndex) = s_limit .or. eof()
            * no still not in range, therefore there are NO records in
            * the appropriate range at all, goto eof()
            dbgoto(lastrec()+1)
         endif
      endif

      set(_SET_SOFTSEEK,old_ss)
   else
      dbgobottom()
   endif

   if refresh_now
      s_loadall()
      s_refreshall()
   endif
return .t.


*****************************************************************************
*  Function : s_up()
*****************************************************************************
*
*  Purpose..: Go up one record and immediately reload vars and
*             refresh screen
*
*  Syntax...: s_up([refresh_now])
*
*  Arguments: refresh_now is a flag to indicate whether the screen variables
*             are loaded and the screen refreshed immediately.  The
*             default is .t. = immediate screen refresh.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Go up one logical (or filtered) record
*****************************************************************************
function s_up(refresh_now)
   default refresh_now to .t.
   if !empty(s_filter_exp)
      dbskip(-1)
      * still in range?
      if !eval(s_bIndex) = s_limit .or. bof()
         * position on first rec that meets
         * filter condition
         dbseek(s_limit)
         s_hittop := .t.
      endif
   else
      dbskip(-1)
      if bof()
         s_hittop := .t.
         dbgotop()
      endif
      if deleted()
         dbgotop()
         if deleted()
            dbgoto(lastrec()+1)   // goto eof
         endif
      endif
   endif

   if refresh_now
      s_loadall()
      s_refreshall()
   endif
return .t.


*****************************************************************************
*  Function : s_down()
*****************************************************************************
*
*  Purpose..: Go down one record and immediately reload vars and
*             refresh screen
*
*  Syntax...: s_down([refresh_now])
*
*  Arguments: refresh_now is a flag to indicate whether the screen variables
*             are loaded and the screen refreshed immediately.  The
*             default is .t. = immediate screen refresh.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Go down one logical (or filtered) record
*****************************************************************************
function s_down(refresh_now)
   default refresh_now to .t.
   if !empty(s_filter_exp)
      dbskip()
      * still in range?
      if !eval(s_bIndex) = s_limit .or. eof()
         * not in range, go back up one
         dbskip(-1)

         * in range now?
         if !eval(s_bIndex) = s_limit .or. eof()
            * no still not in range, therefore there are NO records in
            * the appropriate range at all, goto eof()
            dbgoto(lastrec()+1)
         endif
         s_hitbottom := .t.
      endif
   else
      dbskip()
      if eof()
         s_hitbottom := .t.
         dbskip(-1)
      endif
   endif

   if refresh_now
      s_loadall()
      s_refreshall()
   endif
return .t.


*****************************************************************************
*  Function : s_add()
*****************************************************************************
*
*  Purpose..: Signal v_screen() to edit a new record.  Usually called
*             as a key proc from a screen key.
*
*  Syntax...: s_add()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Add a new record and edit it
*****************************************************************************
function s_add
   s_mode := "A"
return .t.


*****************************************************************************
*  Function : s_edit()
*****************************************************************************
*
*  Purpose..: Signal v_screen() to edit the current record.  Usually called
*             as a key proc from a screen key.
*
*  Syntax...: s_edit()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: Usually called as a screen key proc.
*
*  Revisions:
*
*  NG Short.: Screen - Edit a record
*****************************************************************************
function s_edit
   s_mode := "E"
return .t.


*****************************************************************************
*  Function : s_readgets()
*****************************************************************************
*
*  Purpose..: Builds the getlist and does the full screen editing.
*
*  Syntax...: s_readgets([cInitGet]) --> .t.
*
*  Arguments: cInitGet is an optional name of the initial get in the
*             getlist to activate.  cInitGet is generally used by the
*             full screen field validation to resume editing at a
*             designated field.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Activate full screen editing
*****************************************************************************
function s_readgets(cInitGet)
   local i := 0
   local the_color := ""
   local cur_colors := setcolor()
   local color_type
   local tempget
   local tempvalid
   local tempwhen
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   local o_confirm := set(_SET_CONFIRM,s_confirm)
   local F1block := setkey(K_F1)
   local rx := readexit(.t.)          // turn on up/down arrows to exit read
   local o_insert

   * make visible since you cannot pass v_reader() more params
   private v_initget := ""            // default to null string
   if !cInitGet==nil
      v_initget := upper(alltrim(cInitGet))
   endif

   * reassign current help to help for get
   * cannot use cargo, it is used for get prompt
   setkey(K_F1,{||v_ghelp()})

   * set INSERT key to toggle cursor to indicate state of readinsert()
   o_insert := setkey(K_INS,{||v_flipcurs()})

   * set cursor size according to readinsert()
   if readinsert()
      setcursor(SC_INSERT)
   else
      setcursor(SC_NORMAL)
   endif

   * private to make available externally
   private getlist := {}

   if s_mode == "V"
      color_type := C_STD
   else
      color_type := C_ENH
   endif

   * create get objects and load into getlist array
   for i = 1 to getlen

      * make new get object
      oGet := getnew()

      * assign row, col
      oGet:row := flds_[GROW, i]
      oGet:col := flds_[GCOL, i]

      * assign get var name for readval(), etc
      oGet:name := rtrim(flds_[GINTERNAL, i])  // rtrim same as default getsys

      * get block
      tempget := "{|x| if( pcount() > 0," + flds_[GINTERNAL, i] + ":= x," + flds_[GINTERNAL, i] +")}"
      oGet:block := &tempget

      * picture
      if !empty(flds_[GPICT, i])
         oGet:picture := alltrim(flds_[GPICT, i])
      endif

      * colorspec ALWAYS defined from precalculated values
      oGet:colorspec := flds_[GUNS_EDIT, i] + "," + flds_[GENH_EDIT, i]

      * valid
      if !empty(flds_[GCOND_VAL, i])
         tempvalid := "{||"+ flds_[GCOND_VAL, i] +"}"
         oGet:postblock := &tempvalid
      endif

      * when pre proc
      if !empty(flds_[GCOND_ACC, i])

         * tempwhen  := "{|x|" + flds_[GCOND_ACC, i] +"}"
         * Manual says parameter is passed - I never found it with debugger.
         * Also it appears that the preBlock gets called twice in
         * rapid succession if the preblock evals to .f., but only
         * once if it evals to .t.

         tempwhen  := "{||" + flds_[GCOND_ACC, i] +"}"
         oGet:preblock := &tempwhen
      endif

      * load get prompt string into cargo
      oget:cargo := flds_[GPROMPT, i]

      * replace std getreader()
      oget:reader := {|objget|v_reader(objget)}

      * add get object to the getlist array
      aadd(getlist,oGet)

   next

   * do the read
   readmodal(getlist)

   * restore environment
   set(_SET_CONFIRM,o_confirm)
   setkey(K_F1,F1block)
   setkey(K_INS,o_insert)
   readexit(rx)
   setcursor(SC_NONE)
return .t.


*****************************************************************************
*  Function.: v_reader()
*****************************************************************************
*
*  Purpose..: Replacement reader, simply does a few things and returns
*             control back to the standard get reader.  The variable
*             v_initget is created in s_readgets() and is used to
*             position the initial active get.
*
*  Syntax...: v_reader(objGet)
*
*  Arguments: objGet is the current get object
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Fri  09-25-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Replacement reader, chains back to default reader
*****************************************************************************
function v_reader(objget)
   if empty(v_initget)
      * v_initget is empty, no initial get was specified
      * process this get normally

      * paint new status line using cargo contents
      v_stat23(objget:cargo)

      * pass control back to Clipper's standard get reader
      getreader(objget)

   else
      * an initial get is specified -  skip all gets until you arrive
      * at the designated get, then null out v_initget so that
      * the editing will resume in a normal manner
      if v_initget == objget:name
         v_initget := ""

         * paint new status line using cargo contents
         v_stat23(objget:cargo)

         * pass control back to Clipper's standard get reader
         getreader(objget)

      else
         * not on designated get, skip this one and signal to
         * readmodal() to move down to the next get
         objget:exitState := GE_DOWN
      endif
   endif
return nil


*****************************************************************************
*  Function.: v_flipcurs()
*****************************************************************************
*
*  Purpose..: Toggle readinsert() and change the cursor size
*
*  Syntax...: v_flipcurs()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Sun  12-20-1992
*
*  Notes....: Assumes cursor is visible
*
*  Revisions:
*
*  NG Short.: Screen - Toggle cursor size, insert or overwrite mode
*****************************************************************************
function v_flipcurs
   if readinsert()
      readinsert(.f.)
      setcursor(SC_NORMAL)
   else
      readinsert(.t.)
      setcursor(SC_INSERT)
   endif
return nil


*****************************************************************************
*  Function : v_accept()
*****************************************************************************
*
*  Purpose..: Accept prompt
*
*  Syntax...: v_accept()
*
*  Arguments: None
*
*  Returns..: A numeric value  0=Cancel or Escaped, 1=Accept, 2=Re-try
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Accept prompt before saving record
*****************************************************************************
function v_accept
   local choice := v_alert("Action",{"Accept","Re-Try","Cancel"})

   * make Esc same as cancel
   if choice == 0
      choice := 3
   endif
return choice


*****************************************************************************
*  Function : s_changed()
*****************************************************************************
*
*  Purpose..: Test whether screen memvars have changed since loading
*
*  Syntax...: s_changed()
*
*  Arguments: None
*
*  Returns..: A boolean value  .t. if internal variables have changed
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Has anything changed?
*****************************************************************************
function s_changed
   local l := .f.
   local i := 0
   local tempint
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   for i = 1 to getlen
      tempint := flds_[GINTERNAL, i]
      if !flds_[GINIT_VAL, i] == &tempint
         l := .t.
         exit
      endif
   next
return l


*****************************************************************************
*  Function : s_replace()
*****************************************************************************
*
*  Purpose..: Replace all data fields with new values.  The record(s) MUST
*             be locked before calling s_replace()
*
*  Syntax...: s_replace()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....: This function has no effect on memory variable gets.
*             TEMPINT is declared as private for visibility in v_e_alt()
*             in case of replace error.
*
*  Revisions:
*
*  NG Short.: Screen - Replace memvars to their associated fields
*****************************************************************************
function s_replace
   local l := .t.
   local i := 0
   local tempfldname
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   private tempint
   for i = 1 to getlen
      * is this an actual field?
      if !empty(flds_[GDALIAS, i]) .and. !empty(flds_[GFIELD, i])

         * do the replace for all fields.  ALL fields are replaced
         * because the fields may have been initialized by a non-editable
         * internal screen variable or function that has changed.
         * For example, the parent-child link fields in a child database
         * may never be editable - but they do need to be written
         * to the child record
         tempint := flds_[GINTERNAL, i]
         tempfldname := alltrim(flds_[GDALIAS, i]) + "->" + alltrim(flds_[GFIELD, i])
         replace &tempfldname with &tempint
      endif
   next
return .t.


*****************************************************************************
*  Function : s_restore()
*****************************************************************************
*
*  Purpose..: Restore the originally loaded values
*
*  Syntax...:
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Restore original memvars values
*****************************************************************************
function s_restore
   local l := .t.
   local i := 0
   local tempint
   local getlen := len(flds_[GINTERNAL])  // how many elements in a subarray?
   for i = 1 to getlen
      &(flds_[GINTERNAL, i]) := flds_[GINIT_VAL, i]
   next
return .t.

*****************************************************************************
*  Function.: s_origval()
*****************************************************************************
*
*  Purpose..: Return the original (initialization) value for a screen
*             variable.
*
*  Syntax...: s_origval(int_var_name)
*
*  Arguments: int_var_name is the name of the screen internal variable
*             original value to return.
*
*  Returns..: A value depending upon the data type of the screen memvar.
*
*  Author...: Vick Perry,   Sat  10-03-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Return the original value for a screen memvar
*****************************************************************************
function s_origval(int_var_name)
   local i := 0
   local ret_val := ""

   * find the element for the intvarname, flds_ is declared private in
   * v_screen()
   i := s_fldnum(int_var_name)

   * if found, return it
   if i > 0
      ret_val := flds_[GINIT_VAL,i]
   endif
return ret_val


*****************************************************************************
*  Function : v_defkeys()
*****************************************************************************
*
*  Purpose..: Process standard v_screen() navigation keys
*
*  Syntax...: v_defkeys(<nKey>) --> lMovedPtr
*
*  Arguments: <nKey> is the inkey value of the key to be processed.
*
*  Returns..: a boolean value indicating whether the record pointer was
*             changed so that v_screen() knows to reload and refresh
*
*  Author...: Vick Perry,   Sat  09-26-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Standard screen navigation keys
*****************************************************************************
function v_defkeys(nkey)
   local lMovedPtr := .t.
   do case
      case nkey == K_DOWN
         s_down()
      case nkey == K_UP
         s_up()
      case nkey == K_LEFT
         s_up()
      case nkey == K_RIGHT
         s_down()
      case nkey == K_HOME
         s_gotop()
      case nkey == K_END
         s_gobottom()
      case nkey == K_PGDN
         s_gobottom()
      case nkey == K_PGUP
         s_gotop()
      case nkey == K_CTRL_PGUP
         s_gotop()
      case nkey == K_CTRL_PGDN
         s_gobottom()
      otherwise
         lMovedPtr := .f.
   endcase
return lMovedPtr


*****************************************************************************
*  Function.: v_delrec()
*****************************************************************************
*
*  Purpose..: Lock and delete the current record in the current workarea
*
*  Syntax...: v_delrec(is_msg)
*
*  Arguments: is_msg is the flag to display an error message if the record
*             cannot be locked for deletion.  Default is .t. (show msg)
*
*  Returns..: a boolean value - .t. if successful
*
*  Author...: Vick Perry,   Sun  09-20-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Screen - Lock and delete current screen record
*****************************************************************************
function v_delrec(is_msg)
   local success := .f.
   default is_msg to .t.
   if v_rec_lock(2)
      dbdelete()
      success := .t.
   else
      if is_msg
         v_msg("Sorry, Cannot lock this record for deletion.",,,v_sound_z)
      endif
   endif
return success


*****************************************************************************
*  Function : v_match()
*****************************************************************************
*
*  Purpose..: Return whether a matching record is found in a file.  Often
*             used for table validation or to determine whether child
*             records exist before allowing deletion of their parent record.
*
*             The target record pointer may be left sitting on the target
*             record (or eof()) if desired.
*
*  Syntax...: v_match(cAlias, nIndexOrd, cSeekStr [,restore_rec])
*
*  Arguments: calias is the datafile alias
*
*             nIndexOrd is the index order number
*
*             cseekstr is the seek string
*
*             restore_rec is the optional flag to force restoration
*             of the record pointer in the target file. The default
*             is .t. (restore pointer)
*
*
*
*  Returns..: a boolean value, .t. if a matching record is found
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Seek for a match in a file
*****************************************************************************
function v_match(cAlias, nIndexOrd, cSeekStr, restore_rec)
   local l := .f.
   local xarea := select()
   local oorder
   local orec
   default restore_rec to .t.

   * switch to target area, save environment, select order, seek, and restore
   select &calias
   orec := recno()
   oorder := indexord()
   dbsetorder(nindexord)
   dbseek(cseekstr)
   if found()
      l := .t.
   endif
   dbsetorder(oorder)

   * restore target rec ptr?
   if restore_rec
      dbgoto(orec)
   endif

   * back to original workarea
   select &xarea
return l


*****************************************************************************
*  Function.: v_write()
*****************************************************************************
*
*  Purpose..: Display text to the screen.  Often used by a CDD condition
*             to substitute for @ SAY command.  Always returns true.
*
*  Syntax...: v_write(nRow, nCol, cText [, cColorStr])
*
*  Arguments: nrow, ncol are screen position
*             ctext is the text to display
*             ccolorstr is the optional color string
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....:  Used in a condition for validation to update the screen
*              based upon the value of another field:
*
*  Revisions:
*
*  NG Short.: Write text to the screen
*****************************************************************************
function v_write(nRow, nCol, cText, cColorStr)
   default ccolorstr to setcolor()
   @ nrow, ncol say ctext color ccolorstr
return .t.


*****************************************************************************
*  Function : v_seekval()
*****************************************************************************
*
*  Purpose..: Seek and return a field value from a datafile.  Often
*             used to display descriptions that are controlled by a
*             code field.  Does NOT restore target file record pointer
*             unless save_rec parameter is set.
*
*  Syntax...: v_seekval(cAlias, nOrder, cSeekStr, cField [, save_rec])
*
*  Arguments: cAlias is the alias for the file to use as lookup
*
*             nOrder is the index order
*
*             cSeekStr is the seek string
*
*             cField is the field (in cAlias'ed file ) whose contents
*             will be returned  (fieldname only - not alias->field !!!).
*
*             save_rec is optional flag to force restoration of original
*             target rec position.  Default is .f. - no restore, record
*             pointer stays at new location or eof() if nothing found.
*
*  Returns..: A found value if seek is successful, an empty value if not
*
*  Author...: Vick Perry,   Tue  09-01-1992
*
*  Notes....:
*
*  Revisions:
*
*  xxamples.:  name := v_seekval("company", 1, mcompno, "compname1")
*                                   or
*              name := v_seekval("company", company1, mcompno, "compname1")
*                  (where company1 is the index alias of the desired index)
*
*  NG Short.: Seek and obtain a value from a file
*****************************************************************************
function v_seekval(cAlias, nOrder, cSeekStr, cField, save_rec)
   local s := ""
   local type
   local orig_area
   local ord_target
   local xrec

   default save_rec to .f.

   orig_area := select()
   select (cAlias)
   ord_target := indexord()
   xrec := recno()
   dbsetorder(nOrder)
   dbseek(cSeekStr)
   if found()
      s := &(cField)
   else
      * nothing found, return empty value of same type as field
      type := valtype(&cField)
      do case
         case type == "C"
            s:= space(len(&cField))
         case type == "N"
            s:= 0
         case type == "D"
            s:= ctod("")
         case type == "L"
            s:= .f.
         case type == "M"
            s:= ""
      endcase
   endif
   dbsetorder(ord_target)
   if save_rec
      dbgoto(xrec)
   endif

   * back to orig area
   select (orig_area)
return s


*****************************************************************************
*  Function.: v_assign()
*****************************************************************************
*
*  Purpose..: To assign a memvar by reference.  Designed for calling from
*             a CDD condition field with a programmer controlled return
*             value.  Often used to reassign a second variables based upon
*             the current or new value of the first variable.
*
*  Syntax...: v_assign(cVarName, value [,ret_val])
*
*  Arguments: cVarName is the variable to assign, in quotes.
*
*             value is the value to assign to the cVarName
*
*             ret_val is the optional return value for v_assign()
*             If ret_val is not passed v_assign() returns .t.
*
*  Returns..: .t. or the ret_val
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Assign a value to a memvar or field
*****************************************************************************
function v_assign(cVarName, value ,ret_val)
   default ret_val to .t.
   &cVarName := value
return ret_val


*****************************************************************************
*  Function.: v_shell()
*****************************************************************************
*
*  Purpose..: Provides a shell in which a function may be called that
*             returns a value incompatible with one of CDD's condition
*             fields.  v_shell() allows the programmer to call
*             the other function and provide a return value of the
*             desired type.  This is the moral equivalent of the
*             shell game.
*
*  Syntax...: v_shell(cExpr [,ret_val])
*
*  Arguments: cExpr is the the expression, in quotes, to execute
*             E.g. "MyFunc()" or "dbsetindex(1)"
*
*             ret_val is the return value for v_shell()
*             The default is .t.
*
*  Returns..: .t. or the passed ret_val
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Function shell to control return value of another function
*****************************************************************************
function v_shell(cExpr,ret_val)
   local temp
   default ret_val to .t.
   temp := &cExpr
return ret_val


*****************************************************************************
*  Function : v_look1()
*****************************************************************************
*
*  Purpose..: Lookup table function called as a condition for valid.
*             The type of validation is selectable (Required, Optional...)
*
*             V_look1() was designed to meet most lookup needs.  If you
*             need something different, copy, rename, and modify v_look1().
*
*             Operates by keyboarding the return value into the key buffer
*             and by optionally loading additional fields into designated
*             memvars.  The actual lookup table is a browse.  The option
*             of simultaneously loading additional memvars from other
*             table fields is intended to provide the programmer with
*             the ability to update dependent fields on the screen.
*             For example, the description from a code/description
*             screen field pair can now be easily updated upon
*             selection of a new code.
*
*             Does NOT restore original record pointer in the lookup
*             table; however, since the original table record position
*             is stored in the private variable o_rec, the optional
*             post_proc can call dbgoto(o_rec) to restore original
*             position.  This is useful if you are doing a lookup into
*             the same file.
*
*             NOTE: If you are lookup up in a filtered browse -
*                   the cSeekstr MUST contain the filter expression
*                   also.
*
*             The primary field value, cFldExpr1, is stuffed into the
*             keyboard buffer upon selection.  The two other optional
*             field values are loaded into their respective screen memory
*             variables.
*
*             A post procedure may be specified to refresh the display
*             or to do other processes.
*
*             By default, the found or selected target record pointer is not
*             restored to its pre-lookup position.  This allows
*             additional reference to the table fields.
*
*             SEE NOTES about setting up the browse definition!
*
*  Syntax...: v_look1(mode_code, id, cAlias, nIndexOrd, [cSeekStr], fx1;
*                     [,post_proc] [[,fx2, v2] [,fx3, v3] ......, fx9 , v9 ]
*
*  Arguments: mode_code is the validation method:
*
*                "R"=Required field, value must validate before exiting
*
*                "RU"=Required field but allow exit if uparrow is pressed
*                     and field is empty
*
*                "O"=Optional, no validation if empty, press <Enter> for
*                    lookup, arrows to bypass field
*
*             id is the browse function id used for lookup
*
*             calias is the datafile alias string
*
*             nIndexOrd is the index order number
*
*             cseekstr is the seek string. optional
*
*             fx1 is the string containing the field or expression to
*             return as a selected value.  Alias and field should
*             be used where possible.  Since this parameter is
*             macroed, expressions to limit length such as
*             "left(table->code,1)" are allowed.
*
*             post_proc is the optional post process to execute.
*             post_proc is often a screen or field refresh function call.
*             Post_proc is optional.
*
*             fx2, fx3,...fx9 are additional fields to load into
*             memory variables.  Complex expressions are allowed.
*             Each field expression MUST be paired with a memory variable.
*             Optional.  In quotes.
*
*             v2, v3,...v9 are additional memory variables that are
*             loaded by fx2, fx3,...fx9.  Optional. In quotes.
*
*  Returns..: A boolean value. .t. if a match is found.  The ret_val
*             variable may be altered externally via the post proc
*             to permit such things as an <Esc> with no selection to
*             return .t. allowing a blank field to valid.
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....: The lookup browse definition must adhere to the following:
*
*                If a datafile alias is specified, it must be the
*                same as cAlias.
*
*                Any index for the data file may be specified as
*                the initial order.
*
*                Any filter conditions are set in the lookup browse
*                definition.  cSeekstr MUST be within the filter range!
*
*                Initial GOTOP must be "N" - v_lookup_b() does the
*                initial file positioning NOT the browse.
*
*                At least two action keys must be specified, Escape and
*                Return.  Both of these keys should call v_exit() to
*                force immediate termination.
*
*                Find and order functions may be used in the lookup browse.
*
*  Example..: v_look1("R", "l_table", "table", table1, mcode, "code")
*
*  Revisions:
*
*  NG Short.: Generic lookup for CDD.  Copy and modify it for your needs.
*****************************************************************************
function v_look1(mode_code,id,cAlias,nIndexOrd,cSeekStr,fx1,post_proc,;
                 fx2,v2,fx3,v3,fx4,v4,fx5,v5,fx6,v6,fx7,v7,fx8,v8,fx9,v9)
   local xarea := select()
   local o_order
   local no_return
   local load_vars := .f.
   local do_valid := .f.
   local get_value := ""

   private o_rec := 0            // defined private for post_proc visibility
   private ret_val := .f.        // defined private for post_proc visibility

   default mode_code to "R"      // default to required field validation
   default cseekstr to ""        // optional
   default post_proc to ""
   mode_code := upper(mode_code)
   get_value := &(readvar())     // current get value

   do case
      * required field, must valid
      case mode_code == "R"
         do_valid := .t.

      * required field, K_UP and K_SH_TAB allow upward exit on blank field
      case mode_code == "RU"
         if empty(get_value)
            if lastkey() == K_UP .or. lastkey() == K_SH_TAB
               do_valid := .f.
               ret_val := .t.
            else
               do_valid := .t.
            endif
         else
            do_valid := .t.
         endif

      * optional field, allow exit on blank field, valid if press K_ENTER
      * on blank field. Nonblank always valid.
      case mode_code == "O"
         if empty(get_value)
            if lastkey() == K_ENTER
               do_valid := .t.
            else
               do_valid := .f.
               ret_val := .t.
            endif
         else
            do_valid := .t.
         endif

      otherwise
         do_valid := .t.
   endcase

   if do_valid
      * switch to target area, save environment, select order, seek, and restore
      select &calias
      o_order := indexord()
      o_rec := recno()
      dbsetorder(nindexord)
      dbseek(cseekstr)
      if found()
         * user typed in a valid value

         * signal to load the additional memvars
         load_vars := .t.

         * return .t. to validation
         ret_val := .t.
      else
         * go top whether filtered or unfiltered browse is called next.
         * if the browse is filtered, then the browse logic will
         * reposition the record pointer to within proper range
         * At this point you have no knowledge of what the
         * filter for the browse will be.
         dbgotop()

         * avoid an oft-reported "bug" - if the user holds down
         * the <Enter> key, it causes what seems to be an automatic
         * selection of a lookup item.
         clear typeahead

         * load the lookup browse
         v_load(id,"B")

         * did the user select something (by pressing return)?
         if lastkey()==13

            * return fx1 via the keyboard
            keyboard &fx1

            * signal to load the additional memvars
            load_vars := .t.
         endif
         * return .f. to validation to force revalidation based
         * upon key stuffed value
         ret_val := .f.
      endif

      if load_vars
         * load the additional memvars from the specified fields or expr
         if !fx2 == nil
            &v2 := &fx2
         endif
         if !fx3 == nil
            &v3 := &fx3
         endif
         if !fx4 == nil
            &v4 := &fx4
         endif
         if !fx5 == nil
            &v5 := &fx5
         endif
         if !fx6 == nil
            &v6 := &fx6
         endif
         if !fx7 == nil
            &v7 := &fx7
         endif
         if !fx8 == nil
            &v8 := &fx8
         endif
         if !fx9 == nil
            &v9 := &fx9
         endif
      endif
      dbsetorder(o_order)

      * do post proc if specified - often used to refresh screen variables
      * or to restore the original lookup rec pointer
      * or to alter return value
      if !empty(post_proc)
         no_return := &(post_proc)
      endif

      * back to original workarea
      select &xarea
   endif

return ret_val


*****************************************************************************
*  Function : v_inc()
*****************************************************************************
*
*  Purpose..: Increment a NUMERIC field in a database and return the
*             original value.  Generally used to update counter fields
*             in a configuration file.
*
*  Syntax...: v_inc(cAlias, cField)
*
*  Arguments: calias is the data file alias
*             cField is the data field to increment
*
*  Returns..: the original number before incrementing
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....: A record is appended if file is eof.
*
*  Revisions: dvp 10/13/92 - per PJS, lock rec before getting number
*
*  NG Short.: Increment a numeric field and return original value
*****************************************************************************
function v_inc(cAlias,cField)
   local n := 0
   local xarea := select()
   select &cAlias
   if eof()
      v_add_rec(10)
   endif
   if v_rec_lock(10)
      n := &cField
      replace &cField with n+1
      dbunlock()
   endif
   select &xarea
return n


*****************************************************************************
*  Function : v_dec()
*****************************************************************************
*
*  Purpose..: Decrement a NUMERIC field in a database and return the
*             original value.  Generally used to update counter fields
*             in a configuration file.
*
*  Syntax...: v_dec(cAlias, cField)
*
*  Arguments: calias is the data file alias
*             cField is the data field to decrement
*
*  Returns..: the original number before decrementing
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....: A record is appended if file is eof.
*
*  Revisions: dvp 10/13/92 - per PJS, lock rec before getting number
*
*  NG Short.: Decrement a numeric field and return original value
*****************************************************************************
function v_dec(cAlias,cField)
   local n := 0
   local xarea := select()
   select &cAlias
   if eof()
      v_add_rec(10)
   endif
   if v_rec_lock(10)
      n := &cField
      replace &cField with n-1
      dbunlock()
   endif
   select &xarea
return n


*****************************************************************************
*  Function : v_inc_s()
*****************************************************************************
*
*  Purpose..: Increment a CHAR field in a database and return the
*             original value.  Generally used to update counter fields
*             in a configuration file.
*
*  Syntax...: v_inc_s(cAlias, cField)
*
*  Arguments: calias is the data file alias
*             cField is the data field to increment
*
*  Returns..: the original number before incrementing padded with zeros
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....: A record is appended if file is eof.
*
*  Revisions: dvp 10/13/92 - per PJS, lock rec before getting number
*
*  NG Short.: Increment a string field and return original value
*****************************************************************************
function v_inc_s(cAlias,cField)
   local n := 0
   local s := ""
   local xarea := select()
   local flen := len(&cfield)
   select &cAlias
   if eof()
      v_add_rec(10)
   endif
   if v_rec_lock(10)
      n := val(&cField)
      s := strzero(n,flen)
      ++n
      replace &cField with strzero(n,flen)
      dbunlock()
   else
      s := replicate("0",flen)
   endif
   select &xarea
return s


*****************************************************************************
*  Function : v_dec_s()
*****************************************************************************
*
*  Purpose..: Decrement a CHAR field in a database and return the
*             original value.  Generally used to update counter fields
*             in a configuration file.
*
*  Syntax...: v_dec_s(cAlias, cField)
*
*  Arguments: calias is the data file alias
*             cField is the data field to decrement
*
*  Returns..: the original number before decrementing padded with zeros
*
*  Author...: Vick Perry,   Tue  09-22-1992
*
*  Notes....: A record is appended if file is eof.
*
*  Revisions: dvp 10/13/92 - per PJS, lock rec before getting number
*
*  NG Short.: Decrement a string field and return original value
*****************************************************************************
function v_dec_s(cAlias,cField)
   local n := 0
   local s := ""
   local xarea := select()
   local flen := len(&cfield)
   select &cAlias
   if eof()
      v_add_rec(10)
   endif
   if v_rec_lock(10)
      n := val(&cField)
      s := strzero(n,flen)
      --n
      if n < 0
         n := 0
      endif
      replace &cField with strzero(n,flen)
      dbunlock()
   else
      s := replicate("0",flen)
   endif
   select &xarea
return s


*****************************************************************************
*  Function.: v_sweep_i()
*****************************************************************************
*
*  Purpose..: Indexed sweep to search and replace values in a file on a
*             record by record basis.  v_sweep_i() is generally used to
*             maintain data relational integrity when a link field(s) in
*             a parent-child relationship is changed or to globally
*             update files if a table lookup value is changed.
*
*  Syntax...: v_sweep_i(cAlias, nIndexOrd, cSeekStr,;
*                       fld1, new1,;
*                     [ fld2, new2,;
*                       .
*                       .
*                       .
*                       fld9, new9 ])
*
*  Arguments: calias is the datafile alias string
*
*             nIndexOrd is the index order number
*
*             cseekstr is the seek string to find matching record.
*             The fld1 field contents MUST match the first part of
*             cseekstr in order for a replace to be done!
*
*             fld1 is the string representing a key field to search,
*             in quotes.
*
*             new1 is the new value to replace.
*
*             The remaining parameters are optional.
*             If used, these additional parameters
*             must be specified as pairs...fldn, newn.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Wed  09-23-1992
*
*  Notes....:
*
*  Example..: v_sweep_i("myfile", myfile1, oldcode, "code", newcode)
*
*  Revisions:
*
*  NG Short.: Search and replace values in a file using an index
*****************************************************************************
function v_sweep_i(cAlias, nIndexOrd, cSeekStr, fld1, new1,;
                   fld2, new2,;
                   fld3, new3,;
                   fld4, new4,;
                   fld5, new5,;
                   fld6, new6,;
                   fld7, new7,;
                   fld8, new8,;
                   fld9, new9)

   local xarea := select()
   local o_order
   local o_rec
   local bIndex               // code block of index expr

   * switch to target workarea
   select &calias
   o_rec   := recno()
   o_order := indexord()
   dbsetorder(nindexord)
   bIndex := compile(indexkey())   // make code block out of index key expr
   dbseek(cseekstr)

   * loop through matching records until out of range
   do while eval(bIndex) = cseekstr .and. !eof()
      if v_rec_lock(2)

         * the current index key MUST change when fld1
         * is replaced with new1!!
         replace &fld1 with new1

         * linear sequence of code with break
         begin sequence

         if !fld2 == nil
            replace &fld2 with new2
         else
            break
         endif

         if !fld3 == nil
            replace &fld3 with new3
         else
            break
         endif

         if !fld4 == nil
            replace &fld4 with new4
         else
            break
         endif

         if !fld5 == nil
            replace &fld5 with new5
         else
            break
         endif

         if !fld6 == nil
            replace &fld6 with new6
         else
            break
         endif

         if !fld7 == nil
            replace &fld7 with new7
         else
            break
         endif

         if !fld8 == nil
            replace &fld8 with new8
         else
            break
         endif

         if !fld9 == nil
            replace &fld9 with new9
         endif

         end
      endif


      * Next, please.
      dbseek(cSeekstr)
   enddo
   dbcommit()
   dbsetorder(o_order)
   dbgoto(o_rec)

   * back to original workarea
   select &xarea
return .t.


*****************************************************************************
*  Function.: v_memo_m()
*****************************************************************************
*
*  Purpose..: Simple memvar memo editor/viewer.  Generally called as a cond.
*             for validation by a dummy screen memvar.
*
*             WARNING: v_memo_m() does not edit fields directly!
*
*  Syntax...: v_memo_m(var_by_ref,[title],[ret_val],[is_edit],[color],;
*                     [is_accept],[t],[l],[b],[r])
*
*  Arguments: var_by_ref is the char or memo var passed by reference
*
*             title is the optional memo window title
*
*             ret_val is the optional return value for v_memo_m()
*             If not passed, the ret_val is .t.
*
*             is_edit is optional flag indicating edit mode. .t. = edit
*             .f. = view only, the default is .t. (edit)
*
*             color is the optional color string
*
*             is_accept is the optional flag indicating whether to
*             trap with an accept prompt upon exiting.
*
*             t,l,b,r are the optional coordinates.  The defaults
*             display the window in the bottom half of the screen
*
*  Returns..: The specified ret_val or .t. if ret_val is not passed.
*             The actual memo memvar is updated directly because it is
*             passed by reference.
*
*  Author...: Vick Perry,   Thu  09-24-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Memoedit for a memvar
*****************************************************************************
function v_memo_m(var_by_ref,title,ret_val,is_edit,color,is_accept,t,l,b,r)
   local buf := savescreen(0,0,24,79)
   local xcolor := setcolor()
   local xcursor := setcursor()
   local orig_memo := ""
   local accept := 0
   local esc_block := setkey(K_ESC)
   local o_insert
   default title to ""
   default ret_val to .t.
   default is_edit to .t.
   default is_accept to .t.
   default color to xcolor
   default t to 9
   default l to 0
   default b to 22
   default r to 79

   * lines 23, 24 prohibited
   if b > 22
      b := 22
   endif

   * set color
   setcolor(color)

   * set INSERT key to toggle cursor to indicate state of readinsert()
   * of course this insert-cursor toggle can also be done in a
   * memoedit() udf.
   o_insert := setkey(K_INS,{||v_flipcurs()})

   * draw box, title, shadow, and status lines
   @ t,l to b, r double
   if !empty(title)
      @ t,l+((r-l)/2)-(len(title)/2) say title
   endif
   v_shadow(t,l,b,r)
   v_stat23()
   if is_accept
      v_stat24("<Esc> to exit",.t.)
   else
      v_stat24("<Esc> to exit without saving     <Ctrl_W> to save and exit",.t.)
   endif

   * set cursor for editing
   if is_edit
      setcursor(1)
   endif

   * save original
   orig_memo := var_by_ref

   * remap Esc key to Ctrl_W if accept msg is desired
   if is_accept
      setkey(K_ESC,{||memosave()})
   endif

   do while .t.
      * do memoedit
      var_by_ref := memoedit(var_by_ref,t+1,l+1,b-1,r-1,is_edit)

      if is_edit
         if is_accept
            * trap exit with an accept msg
            if ! orig_memo == var_by_ref
               accept := v_accept()
               do case
                  case accept == ACCEPT_A   // save
                     exit

                  case accept == ACCEPT_R   // retry - reedit
                     loop

                  case accept == ACCEPT_C   // cancel or escaped
                     * restore original contents
                     var_by_ref := orig_memo
                     exit
               endcase
            else
               exit
            endif
         else
            exit
         endif
      else
         exit
      endif
   enddo

   if is_accept
      setkey(K_ESC,esc_block)
   endif

   setkey(K_INS,o_insert)
   setcolor(xcolor)
   setcursor(xcursor)
   restscreen(0,0,24,79,buf)
return ret_val


*****************************************************************************
*  Function.: memosave()
*****************************************************************************
*
*  Purpose..: Stuff Ctrl_W into keyboard buffer - called by v_memo_m()
*
*  Syntax...: memosave()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Thu  09-24-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Internal to v_memo_m()
*****************************************************************************
function memosave
   keyboard chr(K_CTRL_W)
return nil


*****************************************************************************
*  Function.: v_memo_f()
*****************************************************************************
*
*  Purpose..: Simple dbf field memo editor/viewer.  Generally called as a
*             browse or screen action key or a menu item.
*
*             WARNING: v_memo_f() unlocks the record upon exit!!!
*
*  Syntax...: v_memo_f(cfield,[title],[ret_val],[is_edit],[color],;
*                     [is_accept],[t],[l],[b],[r])
*
*  Arguments: cfield is the fieldname
*
*             title is the optional memo window title
*
*             ret_val is the optional return value for v_memo_f()
*             If not passed, the ret_val is .t.
*
*             is_edit is optional flag indicating edit mode. .t. = edit
*             .f. = view only, the default is .t. (edit)
*
*             color is the optional color string
*
*             is_accept is the optional flag indicating whether to
*             trap with an accept prompt upon exiting.
*
*             t,l,b,r are the optional coordinates.  The defaults
*             display the window in the bottom half of the screen
*
*  Returns..: The specified ret_val or .t. if ret_val is not passed.
*             The actual memo memvar is updated directly because it is
*             passed by reference.
*
*  Author...: Vick Perry,   Thu  09-24-1992
*
*  Notes....: Assumes the correct workarea is selected.
*
*  Revisions:
*
*  NG Short.: Memoedit for a memo field.  Handles rec locking.
*****************************************************************************
function v_memo_f(cField,title,ret_val,is_edit,color,is_accept,t,l,b,r)
   local the_memo := ""
   default title to ""
   default ret_val to .t.
   default is_edit to .t.
   default is_accept to .t.
   default color to setcolor()
   default t to 9
   default l to 0
   default b to 22
   default r to 79
   the_memo := &(cField)
   if v_lock_ed
      if !v_rec_lock(2)
         is_edit := .f.
         v_msg("Sorry, cannot lock record for editing.  View mode only.",,,v_sound_z)
      endif
   endif
   v_memo_m(@the_memo,title,ret_val,is_edit,color,is_accept,t,l,b,r)
   if !the_memo == &(cField)
      if is_edit
         * lock again, just in case
         if v_rec_lock(2)
            replace &(cField) with the_memo
            dbunlock()
         else
            v_msg("Sorry, cannot lock record for update.  Please try again later.",,,v_sound_z)
         endif
      endif
   endif
return ret_val


*****************************************************************************
*  Function.: v_make_pub()
*****************************************************************************
*
*  Purpose..: Create and optionally initialize a public variable.  Generally
*             paired with v_rel_pub() when the variable is no longer needed.
*
*  Syntax...: v_make_pub(cVarName [, init_val] [,ret_val])
*
*  Arguments: cVarname is the variable name to create.
*             init_val is the optional initialization value, default is .f.
*             ret_val is the optional return value, default is .t.
*
*  Returns..: .t. or ret_val if defined.
*
*  Author...: Vick Perry,   Fri  09-25-1992
*
*  Notes....: To help with debugging and to avoid memvar-field naming
*             conflicts:
*
*                 1. Begin all created public memvars with "P_"
*                 2. Do NOT begin any data field names with "P_"
*
*  Revisions:
*
*  NG Short.: Create and assign a PUBLIC variable
*****************************************************************************
function v_make_pub(cVarName, init_val, ret_val)
   local tempstr := left(cVarname,10)
   default ret_val to .t.
   default init_val to .f.

   * create PUBLIC, (with /a compiler switch this results in a warning)
   public &tempstr := init_val
return ret_val


*****************************************************************************
*  Function.: v_rel_pub()
*****************************************************************************
*
*  Purpose..: Release a public (or visible private) variable. Generally
*             called after v_make_pub() when the variable is no longer needed.
*
*  Syntax...: v_rel_pub(cVarName [,ret_val])
*
*  Arguments: cVarname is the variable name to release.
*             ret_val is the optional return value, default is .t.
*
*  Returns..: .t. or ret_val if defined.
*
*  Author...: Vick Perry,   Fri  09-25-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Release a PUBLIC variable
*****************************************************************************
function v_rel_pub(cVarName, init_val, ret_val)
   local tempstr := left(cVarname,10)
   default ret_val to .t.
   release &tempstr
return ret_val


*****************************************************************************
*  Function.: v_sequence()
*****************************************************************************
*
*  Purpose..: Execute a predefined sequence.  The value of the private
*             variable SEQ is used to determine the function return value.
*             The sequencer may be halted by setting the private variable
*             STOP_SEQ to .t.
*
*  Syntax...: v_sequence(id)
*
*  Arguments: id is the sequence id
*
*  Returns..: .t. or the value of SEQ upon exiting.
*
*  Author...: Vick Perry,   Fri  09-25-1992
*
*  Notes....: v_sequence() is NOT to be considered a replacement for
*             compiled source code.  The sequencer is slow and does not
*             provide for looping; however, it is ideal for one time
*             processes such as initialization and cleanup.
*
*             The sequence memo is 256 characters wide.
*
*  Revisions:
*
*  NG Short.: Execute a sequence
*****************************************************************************
function v_sequence(id)
   local i := 0
   local xsequence := ""
   local totlines := 0
   local leftchar := ""
   local no_return       // return value from a function that is ignored
   local xarea := select()

   * seq is the return value, it may be set within the sequence itself
   private seq := .t.
   private stop_seq := .f.

   * declare privates used for display by alternate errorsys
   * if a sequence bombs
   private v_id := id
   private v_curline := ""

   * clean id
   id := padr(upper(alltrim(id)),10)

   * position sequence file
   select v_seq
   seek id

   * back to caller environ before running sequence
   select &xarea

   * run sequence in caller's environment
   if v_seq->(found())
      xsequence := hardcr(v_seq->sequence)


      totlines := mlcount(xsequence)
      for i = 1 to totlines
         v_curline := alltrim(memoline(xsequence,254,i))
         leftchar := left(v_curline,1)
         if !(empty(v_curline) .or. leftchar $ "*/{")
            no_return := &(v_curline)
         endif

         * signal to stop the sequencer?
         if stop_seq
            exit
         endif
      next
   else
      * unknown sequence
      v_alert("Unknown sequence id: "+id)
      seq := .f.
   endif
return seq


*****************************************************************************
*  Function.: v_msg()
*****************************************************************************
*
*  Purpose..: Display a single line boxed message.  Timeout and
*             return value parameters are optional
*
*  Syntax...: v_msg(cMsg [,seconds] [,ret_val] [,is_sound])
*
*  Arguments: cmsg is the one line message
*
*             seconds is the optional timeout, 0=wait for key, default is 3
*
*             ret_val is optional return value, defaulted is .f. because
*             v_msg() is often used to return an error message
*
*             is_sound is the flag to sound a beep, default is .f.
*
*  Returns..: .f. or the specified return value
*
*  Author...: Vick Perry,   Fri  09-25-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Display a timed message.  Returns false for use as error msg.
*****************************************************************************
function v_msg(cMsg ,seconds ,ret_val, is_sound)
   local o_buf := savescreen(0,0,24,79)
   local mlen := len(cmsg)
   local t := 10
   local b := 14
   local l,r
   local xcolor := setcolor(v_msgcolor)
   local xcursor := setcursor(0)
   default ret_val to .f.
   default seconds to 2.25
   default is_sound to .f.
   cmsg := left(cmsg,75)
   l := 40-(mlen/2)-2
   r := 40+(mlen/2)+2
   @ t,l clear to b, r
   @ t,l to b, r
   @ t+2,l+2 say cmsg
   v_shadow(t,l,b,r)
   if is_sound
      v_beep("1")
   endif
   clear typeahead
   inkey(seconds)
   setcolor(xcolor)
   restscreen(0,0,24,79,o_buf)
   setcursor(xcursor)
return ret_val


*****************************************************************************
*  Function.: v_sweep_l()
*****************************************************************************
*
*  Purpose..: Very slow unindexed search and replace for values in a file.
*             This function is used when no appropriate indexes are available.
*
*             v_sweep_l() is used when there is no appropriate index available.
*             v_sweep_l() is generally used to maintain data relational
*             integrity when a table lookup value is changed.
*
*  Syntax...: v_sweep_l(cAlias, cLocateExpr,;
*                       fld1, new1,;
*                     [ fld2, new2,;
*                       .
*                       .
*                       .
*                       fld9, new9 ])
*
*  Arguments: calias is the datafile alias string
*
*             cLocateExpr is the expression sent to LOCATE.
*
*             fld1 is the string representing the field to search, in quotes.
*             new1 is the new value to replace.
*
*
*             The remaining parameters are optional.  If used, they must be
*             specified as pairs...fldn, newn.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Wed  09-23-1992
*
*  Notes....:
*
*  Example..: v_sweep_l("myfile","code=oldcode","code",newcode)
*
*  Revisions:
*
*  NG Short.: Search and replace values in a file (No index is used!)
*****************************************************************************
function v_sweep_l(cAlias, cLocateExpr, fld1, new1,;
                   fld2, new2,;
                   fld3, new3,;
                   fld4, new4,;
                   fld5, new5,;
                   fld6, new6,;
                   fld7, new7,;
                   fld8, new8,;
                   fld9, new9)

   local xarea := select()
   local o_rec
   local o_order := indexord()

   * switch to target workarea
   select &calias
   o_rec   := recno()
   dbsetorder(0)             // natural order
   dbgotop()

   * do the first locate
   locate for &cLocateExpr

   * is there a match?
   if found()
      * loop through matching records until out of range
      do while found()

         if v_rec_lock(2)
            replace &fld1 with new1

            begin sequence

            if !fld2 == nil
               replace &fld2 with new2
            else
               break
            endif

            if !fld3 == nil
               replace &fld3 with new3
            else
               break
            endif

            if !fld4 == nil
               replace &fld4 with new4
            else
               break
            endif

            if !fld5 == nil
               replace &fld5 with new5
            else
               break
            endif

            if !fld6 == nil
               replace &fld6 with new6
            else
               break
            endif

            if !fld7 == nil
               replace &fld7 with new7
            else
               break
            endif

            if !fld8 == nil
               replace &fld8 with new8
            else
               break
            endif

            if !fld9 == nil
               replace &fld9 with new9
            endif

            end
         endif
         continue
      enddo
   endif
   dbcommit()
   dbsetorder(o_order)
   dbgoto(o_rec)

   * back to original workarea
   select &xarea
return .t.


*****************************************************************************
*  Function.: v_keyput()
*****************************************************************************
*
*  Purpose..: Float function for procedure KEYBOARD, Char 1...255 only
*
*  Syntax...: v_keyput(string)
*
*  Arguments: string is any character string
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Sat  09-26-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Function version of KEYBOARD command
*****************************************************************************
function v_keyput(s)
   keyboard s
return .t.


*****************************************************************************
*  Function : v_is_delete()
*****************************************************************************
*
*  Purpose..: If not at eof, display delete prompt for user
*
*  Syntax...: v_is_delete()
*
*  Arguments: None
*
*  Returns..: A boolean value  .t. to delete, .f. to exit
*
*  Author...: Vick Perry,   Fri  09-18-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Ask user if they want to delete this record.
*****************************************************************************
function v_is_delete
return !eof() .and. v_alert("Delete this record?",{"No","Yes"}) == 2


*****************************************************************************
*  Function.: v_getempty()
*****************************************************************************
*
*  Purpose..: Empty field validation and error message function
*             based upon current get - by default will allow upward
*             exit from field if field is empty.
*
*  Syntax...: v_getempty(msg [, is_up])
*
*  Arguments: msg is the error message
*
*             is_up is the flag to set to allow user to press uparrow
*             and bypass validation if field is blank, default is .t.
*             to allow exit upward
*
*  Returns..: .t. if valid
*
*  Author...: Vick Perry,   Thu  10-01-1992
*
*  Notes....:
*
*
*  Revisions:
*
*  NG Short.: Check for empty value, give error message.  Used in VALIDs.
*****************************************************************************
function v_getempty(msg, is_up)
   local l := .t.
   local var := ""
   local type := ""
   local key_up := if(lastkey()==K_UP,.t.,.f.)
   var := &(readvar())
   type := valtype(var)
   default msg to "This field"
   default is_up to .t.
   do case
      case type == "C"
         if empty(var) .and. !(is_up .and. key_up)
            v_msg(msg+" cannot be empty.",,,v_sound_z)
            l := .f.
         endif
      case type == "N"
         if empty(var) .and. !(is_up .and. key_up)
            v_msg(msg+" cannot be zero.",,,v_sound_z)
            l := .f.
         endif
      case type == "D"
         if empty(var) .and. !(is_up .and. key_up)
            v_msg(msg+" cannot be empty.",,,v_sound_z)
            l := .f.
         endif
      case type == "L"
         if empty(var) .and. !(is_up .and. key_up)
            v_msg(msg+" cannot be false.",,,v_sound_z)
            l := .f.
         endif
   endcase
return l


*****************************************************************************
*  Function.: v_getljust()
*****************************************************************************
*
*  Purpose..: Left justify function based upon current get.  Generally
*             called as an ANDed condition for valid for a screen memvar.
*             For CHARACTER type data ONLY!
*
*  Syntax...: v_getljust()
*
*  Arguments: None
*
*  Returns..: .t., the memvar is ltrimmed and padded directly
*
*  Author...: Vick Perry,   Thu  10-01-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Left justify a string.  Often used in screen memvar VALID.
*****************************************************************************
function v_getljust
   local varname := readvar()
   local var := ""
   local varlen := 0
   var := &varname
   varlen := len(var)
   &varname := padr(ltrim(var),varlen)
return .t.


*****************************************************************************
*  Function.: v_clrtype()
*****************************************************************************
*
*  Purpose..: Function version of clear typeahead
*
*  Syntax...: v_clrtype()
*
*  Arguments: None
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Sat  10-03-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Function version of CLEAR TYPEAHEAD
*****************************************************************************
function v_clrtype
   clear typeahead
return .t.


*****************************************************************************
*  Function.: v_is_dup()
*****************************************************************************
*
*  Purpose..: Check for duplicate key fields.  Generally used when adding
*             or editing a record to prevent duplicate keys.  This function
*             must be evaluated as a validation occuring BEFORE SAVING
*             the record.  If you are editing and duplicates already
*             exist v_is_dup() will not detect them.
*
*             The cSeekStr must reflect the value that the record
*             WILL be become.
*
*  Syntax...: v_is_dup(cAlias,nOrder,cSeekStr, cMode [, cErrMsg]])
*
*  Arguments: calias is the data file alias to search
*
*             norder is the index order to use
*
*             cseekstr is the seek string
*
*             cMode is "A" for add mode or "E" for editing an existing
*             record.  If "E" mode the current record in the current
*             workarea will be ignored when checking for a duplicate.
*             The private screen variable "S_MODE" is often passed.
*
*             cerrmsg is the optional error msg to display if a dup is found
*             The message is inserted into the phrase "This ???????? is already
*             on file, please re-try."
*
*  Returns..: A logical, .t. if a duplicate is found, .f. if not found.
*
*  Author...: Vick Perry,   Sat  10-03-1992
*
*  Notes....:
*
*  Example..: !v_is_dup("myfile", myfile1, mcode, s_mode)
*
*  Revisions:
*
*  NG Short.: Check for duplicate key fields
*****************************************************************************
function v_is_dup(cAlias,nOrder,cSeekStr, cmode, cerrmsg)
   local dup := .f.
   local o_area := select()
   local o_order := 0
   local o_rec := 0

   default cerrmsg to ""

   select (calias)
   o_rec := recno()
   o_order := indexord()
   dbsetorder(norder)
   dbseek(cseekstr)
   if found()
      * you found a match, is it the same record that you are currently
      * editing?
      if cmode == "A"
         * adding, no previous match should exist
         dup := .t.
      else
         * editing
         if !recno()==o_rec
            dup := .t.
         else
            * you found original rec, now try next one
            dbskip()
            if &(indexkey(0)) = cseekstr
               dup := .t.
            endif
         endif
      endif
   endif
   if dup
      if !empty(cerrmsg)
         v_msg(cerrmsg+" is already on file.  Please re-try.",,,v_sound_z)
      endif
   endif
   select (o_area)
   dbsetorder(o_order)
   dbgoto(o_rec)
return dup


*****************************************************************************
*  Function.: v_getzpad()
*****************************************************************************
*
*  Purpose..: Left pad the current active get with zeros.  The get variable
*             must be a char type with numeric characters only e.g. "8" will
*             be converted to "08" characters.  If get is blank no action
*             is taken. i.e "0000000" will not be returned if get is blank.
*
*  Syntax...: v_getzpad([len])
*
*  Arguments: len is the optional string length to pad, if not passed the full
*             length of the incoming variable will be returned
*
*  Returns..: .T. always, but the get memvar is modified directly
*
*  Author...: Vick Perry,   Sun  10-04-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Left pad the current active get with zeros.  Used as VALID.
*****************************************************************************
function v_getzpad(len)
   local varname := readvar()
   local var := ""
   local varlen := 0
   local number := 0
   var := &varname
   if len == nil
      varlen := len(var)
   else
      varlen := len
   endif
   number := val(var)

   * don't do anything if number is 0 (get was blank or contained all nonnumerics)
   if number > 0
      &varname := strzero(number,varlen)
   endif
return .t.


*****************************************************************************
*  Function : v_is_memo()
*****************************************************************************
*
*  Purpose..: Determine whether the file open in the current workarea
*             contains a memo field.  Used by v_pack() to know when to
*             pack DBTs.
*
*  Syntax...: v_is_memo()
*
*  Arguments: None
*
*  Returns..: .t. if a memo field exists in currently selected file
*
*  Author...: Vick Perry,   Mon  11-16-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Does the file in current workarea have a memo field?
*****************************************************************************
function v_is_memo
   local l := .f.
   local struct_ := dbstruct()
   local alen := len(struct_)
   local i := 0
   for i = 1 to alen
      if struct_[i,2] == "M"
         l := .t.
         exit
      endif
   next
return l


*****************************************************************************
*  Function : v_seek()
*****************************************************************************
*
*  Purpose..: Seek and position to a record in data file.
*             Does NOT restore target file record pointer if seek is
*             successful.  If seek fails, record pointer is restored.
*
*  Syntax...: v_seek(cAlias, nOrder, cSeekStr)
*
*  Arguments: cAlias is the alias
*
*             nOrder is the index order
*
*             cSeekStr is the seek string
*
*  Returns..: .t. if found, otherwise .f.
*
*  Author...: Vick Perry,   Tue  09-01-1992
*
*  Notes....:
*
*  Revisions:
*
*  xxamples.:  success := v_seek("company", 1, mcompno)
*
*  NG Short.: Seek and position in a file
*****************************************************************************
function v_seek(cAlias, nOrder, cSeekStr)
   local orig_area
   local ord_target
   local xrec
   local l := .f.

   orig_area := select()
   select (cAlias)
   ord_target := indexord()
   xrec := recno()
   dbsetorder(nOrder)
   dbseek(cSeekStr)
   if found()
      l := .t.
   else
      dbgoto(xrec)
   endif
   dbsetorder(ord_target)

   * back to orig area
   select (orig_area)
return l


*****************************************************************************
*  Function.: v_go_eof()
*****************************************************************************
*
*  Purpose..: Position a file pointer to the phantom record at the
*             logical eof.
*
*  Syntax...: v_go_eof(cAlias)
*
*  Arguments: cAlias is the alias to position.
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Tue  11-24-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Go to eof().  Positions on the "phantom" eof() record
*****************************************************************************
function v_go_eof(cAlias)
   select (cAlias)
   dbgobottom()
   dbskip()
return .t.


*****************************************************************************
*  Function.: v_beep()
*****************************************************************************
*
*  Purpose..: Sound several different tones, used internally by CDDLIB
*             and may be called by the programmer.
*
*  Syntax...: v_beep(code)
*
*  Arguments: code is the type of beep to sound
*
*  Returns..: .t.
*
*  Author...: Vick Perry,   Thu  11-26-1992
*
*  Notes....:
*
*  Revisions:
*
*  NG Short.: Beep, beep
*****************************************************************************
function v_beep(code)
   code := upper(code)
   do case
      case code == "M" // menu
         tone(880,1)
      case code == "B" // browse
         tone(880,1)
      case code == "S" // screen
         tone(880,1)
      case code == "1" // v_msg()
         tone(880,1)
      case code == "2" // v_alert()
         tone(880,1)
      otherwise
         tone(800,7)
         tone(220,5)
   endcase
return .t.


*****************************************************************************
*  Function : v_info()
*****************************************************************************
*
*  Purpose..: Display runtime information about the application and CDDLIB
*             internals.  The inkey value of the hotkey used to call v_info()
*             is set in the config file "Information Key..." field.
*             V_info() is usually called via the <Alt-F5> key.
*
*             Programmer's Little Helper....
*
*  Syntax...: v_info()
*
*  Arguments: None
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Tue  12-01-1992
*
*  Notes....: Not pretty, but very useful.
*
*  Revisions:
*
*  NG Short.: Display CDD internal information, current records, etc.
*****************************************************************************
function v_info
   local o_buf := savescreen(0,0,24,79)
   local slen := len(v_stack_)
   local stype := ""
   local sid := ""
   local pick := 1
   local alias_ := {}
   local o_area := 0
   local i,j,k
   cls

   ?
   ?
   ?
   ? 'Alias()....:',alias()
   ? 'IndexOrd().:',ltrim(str(indexord()))
   ? 'IndexKey(0):',rtrim(indexkey(0))
   ? 'Dbfilter().:',rtrim(dbfilter())
   ? 'Recno()....:',ltrim(str(recno()))
   ? 'Deleted()..:',deleted()
   ? 'Found()....:',found()
   ? 'Eof()......:',eof()
   ? 'Bof()......:',bof()
   ? 'Ferror()...:',ltrim(str(ferror()))
   ? 'IsPrinter():',isprinter()
   ? 'LastRec()..:',ltrim(str(lastrec()))
   ? 'ReadVar()..:',readvar()


   pick := alert("Select something.",{"CDD Stack","View Record","Exit"})
   cls

   do case
      case pick == 0  .or. pick == 3
         * do nothing

      case pick == 1
         * WHERE AM I? - show current CDDLIB stack
         ? " CDDLIB Stack"
         ? "Type     ID"
         ? "----------------"
         for i = slen to 1 step -1
            stype := left(v_stack_[i],1)
            sid := upper(ltrim(substr(v_stack_[i],2)))

            * don't show a call to yourself
            if !left(sid,6) == "V_INFO"
               ? stype + "     " + sid
            endif
         next
         inkey(0)

      case pick == 2
         * build an array of aliases for all workareas in use
         o_area := select()
         for i = 1 to 250
            if !empty(alias(i))
               aadd(alias_,alias(i))
            endif
         next
         * sort array
         if len(alias_) > 0
            asort(alias_)
            pick := max(ascan(alias_,alias()),1)
            do while .t.
               cls
               @0,0 say replicate("",80)
               @0,24 say " Select a workarea....<Esc> to exit "
               pick := achoice(1,32,24,42,alias_,,,pick)
               if pick > 0
                  select (alias_[pick])
                  j := fcount()
                  cls
                  ? "Record #: ",recno()
                  if eof()
                     ? "<<<< End-Of-File >>>>"
                     k := 2
                  else
                     k := 1
                  endif
                  for i = 1 to j
                     if valtype(fieldget(i))=="C" .or. valtype(fieldget(i))=="M"
                        * strip out cr/lf pairs
                        ? padr(fieldname(i),10) + ":  ", strtran(ltrim(left(fieldget(i),50)), chr(13)+chr(10), "  " )
                     else
                        ? padr(fieldname(i),10) + ":  ", fieldget(i)
                     endif
                     if k > 21
                        ? "More....."
                        inkey(0)
                        cls
                        k := 0
                     else
                        ++k
                     endif
                  next
                  inkey(0)
               else
                  exit
               endif
            enddo
         else
            * hardly possible, the DD files are open, right?
            alert("No files open")  
         endif
         select (o_area)
   endcase

   restscreen(0,0,24,79,o_buf)
return nil


*****************************************************************************
*  Function : v_write2err()
*****************************************************************************
*
*  Purpose..: Append a line to the ERROR.TXT file in the current directory
*             Can be used to record non-fatal error messages in ERROR.TXT.
*
*  Syntax...: v_write2err(cMsg)
*
*  Arguments: cMsg is the string to write
*
*  Returns..: NIL
*
*  Author...: Vick Perry,   Thu  12-31-1992
*
*  Notes....: If the previous file ends with eof marker, it is removed.
*
*  Revisions:
*
*  NG Short.: Write a line to ERROR.TXT
*****************************************************************************
function v_write2err(msg)
   local handle := 0
   local onechar  := space(1)
   local twochars := space(2)
   local crlf := chr(13)+chr(10)
   local flen := 0

   * open or create ERROR.TXT
   if file("ERROR.TXT")
      * file already exists
      handle := fopen("ERROR.TXT",2)
   else
      * create the file
      handle := fcreate("ERROR.TXT",0)
   endif

   * get length of file
   flen := fseek(handle,0,2)

   do case
      case flen == 0
         * do nothing, empty file

      case flen = 1
         * read the character
         fseek(handle,0)
         fread(handle,@onechar,1)
         if !onechar == chr(26)
            * char is not eof marker, write cr+lf to begin new line
            fwrite(handle,crlf)
         else
            * char is eof marker, position to overwrite it
            fseek(handle,0)
         endif

      case flen > 1
         * read the last two characters
         fseek(handle,-2,2)
         fread(handle,@twochars,2)
         do case
            case twochars == crlf
               * do nothing, you are positioned on a new line

            case right(twochars,1) == chr(26)
               * eof marker, overwrite it with crlf
               fseek(handle,-1,2)
               fwrite(handle,crlf)

            otherwise
               * start new line
               fwrite(handle,crlf)
         endcase
   endcase

   * append new line
   fwrite(handle,alltrim(msg)+crlf)
   fclose(handle)
return nil

