INCLUDE     MACRO.INC
INCLUDE     VECTOR.INC


_STKLEN         EQU     200                     ; Length of TSR's Stack

_TSR_DORMANT    EQU     000H                    ; Flag TSR's dormant
_TSR_REQUESTED  EQU     001H                    ; Flag TSR's requested
_TSR_ACTIVATED  EQU     002H                    ; Flag TSR's active


.MODEL TINY, C
.CODE
        ORG     2CH
EnvSeg  label   word                            ; Environment Segment

.STARTUP
        jmp     InitTSR                         ; Jmp to initialization


;[]-------------------[ RESIDENT VARIABLES DECLARATION ]---------------------[]
;|  The following section declares the various variables used by the resident |
;|  portion of the TSR... They reside in the CODE segment...                  |
;[]--------------------------------------------------------------------------[]
TsrStack        dw      (_STKLEN SHR 1) dup (00); TSR's Stack
TsrStackEnd     label   word                    ; End of Stack Label
                dw      00                      ;

TsrStateFlg     db     _TSR_DORMANT             ; Hotkey's Status Flag
TsrSaveSS       dw      00                      ; To Save Foreground SS
TsrSaveSP       dw      00                      ; To Save Foreground SP
TsrPSP          dw      00                      ; The TSR's PSP
ForegroundPSP   dw      00                      ; Save Foreground PSP
DosBusyFlag     label   dword                   ; Label to INDOS Flag
DosBusyOff      dw      00                      ; Offset of INDOS Flag
DosBusySeg      dw      00                      ; Segment of INDOS Flag
@DefVect        MainVectors, 05,08,13,28        ; *Permanent* Vectors



;[]-------------------[ NEW INTERRUPT 05/PrtScreen ISR ]---------------------[]
;|  The following is triggered when the Print Screen Key is pressed....       |
;|  It merely sets a request flag and lets TIMER/28H take care of launching   |
;|  the actual TSR                                                            |
;[]--------------------------------------------------------------------------[]
New05Handler    PROC    FAR
        sti                                     ; Set Interrupt Flag
        cmp     cs:TsrStateFlg, _TSR_DORMANT    ; Waiting to be waken up ?
        jne     @@L00                           ; Nope...
        inc     cs:TsrStateFlg                  ; Yep...  flag we're requested
@@L00:  iret                                    ; End the Interrupt
New05Handler    ENDP                            ;




;[]-------------------[ NEW TIMER/08 INTERRUPT HANDLER ]---------------------[]
;|  The following section contains our Interrupt Service Routines...  They    |
;|  include the ones which are active throughout the whole time the TSR is    |
;|  loaded in memory and  also the ones which are only active when the TSR    |
;|  itself is activated (popped up)                                           |
;[]--------------------------------------------------------------------------[]
New08Handler    PROC    FAR
        sti                                     ; Set Interrupt Flag
        pushf                                   ; Mimic an INT instruction
        call    cs:Old08Handler                 ; Chain to old Timer
        cmp     cs:TsrStateFlg, _TSR_REQUESTED  ; Has TSR been requested ?
        jne     @@L01                           ; No... just return
        @PUSH   es, bx                          ; Save registers
        les     bx, cs:DosBusyFlag              ; Load Pointer to INDOS Flag
        cmp     byte ptr es:[bx], 00            ; Is Flag clear ?
        jne     @@L00                           ; No... just return
        cmp     cs:Vect13.VectFlag, 00          ; Any DISK I/O taking place ?
        jne     @@L00                           ; Yes.. just return
        call    TsrEntry                        ; Call TSR <<
        dec     cs:TsrStateFlg                  ; Back to DORMANT Mode..
@@L00:  @POP    es, bx                          ; Restore registers
@@L01:  iret                                    ; Terminate Interrupt...
New08Handler    ENDP




;[]------------------[ 13 HANDLER TO MONITOR DISK ACTIVITY ]-----------------[]
;|  The purpose of the INT 13H handler is to prevent any activity while       |
;|  some type of DISK I/O is taking place.  Since our TSR can be launched     |
;|  from a Hardware Interrupt (TIMER), this allows for safe PopUps..          |
;[]--------------------------------------------------------------------------[]
New13Handler    PROC    FAR                     ; DISK I/O Int. Serv. Rout.
        inc     cs:Vect13.VectFlag              ; Flag we're in INT 13H
        pushf                                   ; Push the Flags
        call    cs:Old13Handler                 ; Call the Old Handler
        pushf                                   ; Save Flags
        dec     cs:Vect13.VectFlag              ; Clear BusyFlag
        popf                                    ; Restore prior flags
        retf    2                               ; Ret. without disturbing Flgs
New13Handler    ENDP                            ; End of DISK I/O Handler




;[]-----------------[ DOS IDLE/28H INTERRUPT HANDLER  ]----------------------[]
;|   The 28H handler verifies the TSRFlag, if the latter has been requested,  |
;|   it is safe to activate...                                               |
;[]--------------------------------------------------------------------------[]
New28Handler    PROC    FAR
        cmp     cs:TsrStateFlg, _TSR_REQUESTED  ; Has TSR been requested ?
        jne     @@L00                           ; No... just chain
        call    TsrEntry                        ; Wake up TSR
        dec     cs:TsrStateFlg                  ; Bring back to Dormancy State
@@L00:  jmp     cs:Old28Handler                 ; Chain to Old Handler
New28Handler    ENDP



;[]----------------[ ACTUAL TSR ENTRY POINT - SCREEN SAVER ]-----------------[]
;|  The following is the -Entry Point- to the TSR... Registers are saved, a   |
;|  new stack put in place, PID switched and off we go...                     |
;[]--------------------------------------------------------------------------[]
TsrEntry        PROC
        inc     cs:TsrStateFlg                          ; Flag we're Active
        mov     cs:TsrSaveSS, ss                        ; Save Stack Segment
        mov     cs:TsrSaveSP, sp                        ; Save Stack Pointer
        push    cs                                      ; Make Stack equal
        pop     ss                                      ;      to Code Segment
        mov     sp, OFFSET TsrStackEnd                  ; Point SP to end Stack
        @PUSH   ax, bx, cx, dx, es, si, di, ds, bp      ; Save Registers
        push    cs                                      ; Make DS equal
        pop     ds                                      ;      to Code Segment

        assume  ds:_TEXT                                ; Set up assumption

        mov     ah, 62H                                 ; Get PID Function
        int     21H                                     ; Call DOS
        mov     ForegroundPSP, bx                       ; Save current PID

        mov     bx, TsrPSP                              ; Load TSR's PSP
        mov     ah, 50h                                 ; Set PSP function
        int     21h                                     ; Call DOS

       ;[]-----------------------[ ACTUAL TSR CODE ]------------------------ []
       ;|  The actual Resident portion of the TSR's code should be inserted   |
       ;|  here;  the outer portion of the Code remaining resident should be  |
       ;|  kept as a generic Resident Shell to allow for easy reusability...  |
       ;[]-------------------------------------------------------------------[]
        jmp     @VerifyMode                             ; Jmp over Variables

        fName   db  'SNAPFILE.000',0                    ; File Name...
        PlnCnt  db   00h                                ; Video Plane Counter
        EGADAC  db  17 dup (0)
        DAC     db  3*100h dup (0)
    @VerifyMode:
        mov     ah, 0Fh                                 ; Get MODE ( BIOS )
        int     10h                                     ; Call Video BIOS
        cmp     al, 10h                                 ; EGA ( Mode 10h? )
        mov     cx, 28000                               ; Assume EGA  (10h)
        je      @PrepareFile                            ; Goto setup File
        cmp     al, 12h                                 ; VGA ( Mode 12h? )
        jne     @VModeError                             ; Nope...
        mov     cx, 38400                               ; Load Plane Size

    @PrepareFile:
        lea     dx, fName                               ; DS:DX -> FileName
        push    cx                                      ; Save Plane Size
        sub     cx, cx                                  ; Clear CX-Attr
        mov     ah, 3Ch                                 ; Create File Function
        pop     cx                                      ; Restore Plane Size
        int     21h                                     ; Call DOS
        jc      @CreateError                            ; Cannot create file
        push    ax

    @GetPalette:
        push    cx

        mov     ax, 1009h                               ; Get EGA pal regs
        push    cs
        pop     es
        mov     dx, offset EGADAC
        int     10h

        mov     ax, 1017h                               ; Get VGA Pal Regs
        mov     bx, 0
        mov     cx, 100h
        mov     dx, offset DAC
        int     10h

     @WritePalette:
        pop     cx
        pop     bx
        push    cx
        mov     ah, 40h
        mov     cx, 100h*3 + 17
        mov     dx, offset EGADAC
        int     21h
        pop     cx
        jc      @CloseFile

    @InitPlanes:
        mov     PlnCnt, 3                               ; Init Plane Counter
        mov     dx, 3CEh                                ; Select RMS Reg.
        mov     al, 4                                   ; Graphics Controller
        out     dx, al                                  ; Send Byte
        mov     ax, 0A000h                              ; Load Video Segment
        mov     ds, ax                                  ; Set DS to Segment

        assume  ds: NOTHING                             ; DS -> Video Memory

    @SavePlanes:
        mov     al, cs:PlnCnt                           ; Load Plane
        mov     dx, 3CFh                                ;
        out     dx, al                                  ;
        sub     dx, dx                                  ; DS:DX Video Buffer
        mov     ah, 40h                                 ; Write Function
        int     21h                                     ; Write to File
        jnc     @@3                                     ; Verify any errors

        jmp     @CloseFile                              ; Close the File

    @@3:
        cmp     ax, cx                                  ; Verify Write Count
        je      @@4                                     ;

        jmp     @CloseFile                              ; Close the File

    @@4:
        dec     cs:PlnCnt                               ; Decrement Plane Count
        jge     @SavePlanes                             ; Loop through Planes
        push    cs                                      ; Set Code Segment
        pop     ds                                      ; Restore DS register

    @CloseFile:
        push    cs                                      ; Push Code Segment
        pop     ds                                      ; Restore Data Segment

        assume  ds:_TEXT                                ; Set Assumption

        mov     ah, 3Eh                                 ; Close File Function
        int     21h                                     ; Call DOS
        inc     fName[11]                               ; Use .001, .002, etc
        jmp     @Done


    @CreateError:

    @VModeError:


    @Done:


       ;[]-----------------------[ END OF TSR CODE ]------------------------ []
       ;|  This is the End of the TSR code... It takes care of restoring      |
       ;|  registers, stack, PID, *Temporary* Interrupt Vectors (Enhancments  |
       ;|  should include restoring Extended Error Info, DTA... if those are  |
       ;|  used by the TSR)                                                   |
       ;[]-------------------------------------------------------------------[]

        mov     bx, ForegroundPSP                       ; Load Previous PID
        mov     ah, 50h                                 ; Set PSP Function
        int     21h                                     ; Call Upon DOS

        @POP    ax, bx, cx, dx, es, si, di, ds, bp      ; Restore registers
        assume  ds: NOTHING                             ; Clear assumption
        mov     ss, cs:TsrSaveSS                        ; Restore SS register
        mov     sp, cs:TsrSaveSP                        ; Restore SP register
        dec     cs:TsrStateFlg                          ; Flag we're done
        ret                                             ; Return to caller
TsrEntry        ENDP                                    ; End of TSREntryPoint



;[]==========================================================================[]
;|                                                                            |
;|          >>>>  END      OF      RESIDENT      PORTION  <<<<                |
;|                                                                            |
;[]--------------------------------------------------------------------------[]

Transient:

        Msg     db '<Shift-PrintScreen> to take SnapShots..',10,13,'$'

InitTSR:   
        
        mov     TsrPSP, es                      ; Save PSP in Variable

        mov     es, EnvSeg                      ; Load Environment Segment
        mov     ah, 49H                         ; Function 49H: Free Memory
        int     21H                             ; Call DOS

        mov     ah, 34H                         ; Get DOS INDOS Flag
        int     21H                             ; Call DOS
        mov     DosBusyOff, bx                  ; Save Offset to Flag
        mov     DosBusySeg, es                  ; Save Segment to Flag

        @GetVect MainVectors                    ; Save Old Interrupt Vectors
        @SetVect MainVectors                    ; Install our ISRs

        mov     dx, OFFSET Msg                  ; Load Message Offset
        mov     ah, 9                           ; Function 9
        int     21h                             ; Display Msg using DOS

        mov     dx, OFFSET Transient            ; Point to end of Resident
        int     27H                             ; Stay Resident
END     @Startup
