;****************************************************************************
; UMBFILES extends the system file table (SFT) that DOS uses to track
; the state of open files by creating an extension of the SFT in upper
; memory. The syntax is
;
;       UMBFILES[=]nn
;
; where "nn" is the number of entries to be created in the SFT extension.
; For best results, boot your system with the statement FILES=8 in CONFIG.-
; SYS, and then allocate room for additional FILES with UMBFILES. For exam-
; ple, if you now boot with the statement FILES=40, change it to FILES=8 and
; add the statement UMBFILES=32 to your CONFIG.SYS file. The sum of FILES
; and UMBFILES may not exceed 255.
;
; NOTE: On some PCs, Windows will not start in 386 enhanced mode unless
; FILES is 10 or 12 or higher. If Windows refuses to start when UMBFILES
; is loaded, increase the FILES setting.
;****************************************************************************

code            segment
                assume  cs:code,ds:code
                org     100h
begin:          jmp     main

copyright       db      "UMBFILES 1.3 Copyright (c) 1993 Jeff Prosise",13,10
                db      "From: PC Magazine DOS 6 Memory Management "
                db      "with Utilities",13,10,13,10
                db      "System file table extended",13,10
                db      "FILES is now equal to $"

helpmsg         db      "Builds an extension to the system file table (SFT) "
                db      "in upper memory.",13,10,13,10
                db      "UMBFILES[=]nn",13,10,13,10
                db      "  nn    Number of new SFT entries to create."
                db      13,10,13,10
                db      "UMBFILES requires DOS 5 or DOS 6 and a PC configured "
                db      "to load programs and",13,10
                db      "device drivers high. The sum of FILES and UMBFILES "
                db      "may not exceed 255.",13,10
crlf            db      13,10,"$"

errmsg1         db      "Requires DOS 5 or DOS 6",13,10,"$"
errmsg2         db      "Syntax: UMBFILES[=]nn",13,10,"$"
errmsg3         db      "Invalid parameter or parameter out of range"
                db      13,10,"$"
errmsg4         db      "FILES plus UMBFILES may not exceed 255",13,10,"$"
errmsg5         db      "There is no upper memory available",13,10,"$"
errmsg6         db      "Insufficient upper memory",13,10,"$"

umbfiles        dw      0                       ;Number of UMBFILES
filecount       dw      0                       ;Resultant number of FILES
strategy        dw      ?                       ;Memory allocation strategy
linkstate       dw      0                       ;Upper memory link status
lastblock       dd      ?                       ;Address of final SFT block
sftlen          dw      ?                       ;Length of SFT extension

;****************************************************************************
; Procedure MAIN
;****************************************************************************

main            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    scanhelp                ;Scan for "/?" switch
                jnc     main1                   ;Branch if not found

                mov     ah,09h                  ;Display help text and exit
                mov     dx,offset helpmsg       ;with ERRORLEVEL=0
                int     21h
                mov     ax,4C00h
                int     21h
;
; Check the DOS version and abort if it's not DOS 5 or DOS 6.
;
main1:          mov     ah,30h                  ;Get DOS version
                int     21h
                mov     dx,offset errmsg1       ;Load error message address
                cmp     al,5                    ;Branch if the major version
                je      main2                   ;number is 5 or 6
                cmp     al,6
                je      main2

error:          mov     ah,09h                  ;Display error message
                int     21h
                mov     ax,4C01h                ;Exit with ERRORLEVEL=1
                int     21h
;
; Parse the command line to determine how many SFT entries to create.
;
main2:          call    findchar                ;Advance to next character
                mov     dx,offset errmsg2       ;Error if EOL encountered
                jc      error

                call    asc2bin                 ;Convert ASCII to binary
                mov     dx,offset errmsg3       ;Error if carry is set on
                jc      error                   ;return
                or      al,al                   ;Error if number was 0
                jz      error
                mov     byte ptr umbfiles,al    ;Save UMBFILES number
;
; Walk the chain of SFT headers to determine how many FILES already exist.
;
                xor     cx,cx                   ;Zero SFT entry count
                mov     ah,52h                  ;Get address of DOS's List
                int     21h                     ;of Lists
                les     bx,es:[bx+4]            ;Get SFT address in ES:BX

main3:          mov     word ptr lastblock,bx   ;Save the header address
                mov     word ptr lastblock[2],es
                add     cx,es:[bx+4]            ;Add entries in this block
                les     bx,es:[bx]              ;Get address of next block
                cmp     bx,0FFFFh               ;Loop if the offset address
                jne     main3                   ;is other than FFFFh

                mov     dx,offset errmsg4       ;Verify that FILES plus
                add     cx,umbfiles             ;UMBFILES is less than
                cmp     cx,255                  ;or equal to 255 and
                ja      error                   ;abort if it's not
                mov     filecount,cx            ;Save new FILES count
;
; Allocate an upper memory block to hold the SFT extension.
;
                call    save_state              ;Save the current state

                mov     ax,5803h                ;Set the UMB link
                mov     bx,1
                int     21h
                mov     dx,offset errmsg5       ;Error if carry set
                jc      error

                mov     ax,5801h                ;Change to high-only, best-
                mov     bx,41h                  ;fit allocation strategy
                int     21h
                mov     dx,offset errmsg5       ;Error if carry set
                jc      error

                mov     al,59                   ;Compute the number of
                mul     byte ptr umbfiles       ;paragraphs of memory
                mov     sftlen,ax               ;required for the SFT
                add     ax,21                   ;extension
                mov     cl,4
                shr     ax,cl
                mov     bx,ax
                mov     ah,48h                  ;Request the memory through
                int     21h                     ;DOS function 48h
                jnc     main4                   ;Branch if call succeeded

                call    restore_state           ;Restore the memory state
                mov     dx,offset errmsg6       ;Load error message address
                jmp     error                   ;Jump to error handler

main4:          dec     ax                      ;Point ES to MCB preceding
                mov     es,ax                   ;the memory block
                mov     word ptr es:[01h],ax    ;Change owner ID to segment
                inc     word ptr es:[01h]       ;address of the UMB
                mov     si,offset copyright     ;Copy "UMBFILES" to the MCB
                mov     di,8                    ;so that DOS will know
                mov     cx,8                    ;who the owner is
                rep     movsb
                inc     ax                      ;Point ES back to the block
                mov     es,ax                   ;itself

                call    restore_state           ;Restore the memory state
;
; Initialize the new SFT block and link it into the chain.
;
                mov     word ptr es:[00h],0FFFFh        ;Initialize the
                mov     word ptr es:[02h],0000h         ;SFT header
                mov     ax,umbfiles
                mov     word ptr es:[04h],ax

                mov     cx,sftlen               ;Load CX with SFT length
                xor     al,al                   ;Fill the new SFT with
                mov     di,06h                  ;zeroes from start to
                rep     stosb                   ;finish

                mov     ax,es                   ;Place the address of the
                les     di,lastblock            ;new SFT header in what
                mov     word ptr es:[di],00h    ;was formerly the final
                mov     word ptr es:[di+2],ax   ;SFT block
;
; Display message verifying that the operation is completed, and then exit.
;
                mov     ah,09h                  ;Display message
                mov     dx,offset copyright
                int     21h
                mov     ax,filecount            ;Display FILES count
                call    bin2asc
                mov     ah,09h                  ;Next line
                mov     dx,offset crlf
                int     21h
                mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
main            endp

;****************************************************************************
; SAVE_STATE saves the current memory allocation strategy and link state.
;****************************************************************************

save_state      proc    near
                mov     ax,5800h                ;Get the current memory
                int     21h                     ;allocation strategy
                mov     strategy,ax             ;code and save it
                mov     ax,5802h                ;Get the current link
                int     21h                     ;status and save it
                mov     byte ptr linkstate,al
                ret
save_state      endp

;****************************************************************************
; RESTORE_STATE restores the memory allocation strategy and link state
; saved earlier.
;****************************************************************************

restore_state   proc    near
                mov     ax,5801h                ;Restore original memory
                mov     bx,strategy             ;allocation strategy
                int     21h
                mov     ax,5803h                ;Restore original link
                mov     bx,linkstate            ;status
                int     21h
                ret
restore_state   endp

;****************************************************************************
; SCANHELP scans the command line for a /? switch. If found, carry returns
; set and SI contains its offset. If not found, carry returns clear.
;****************************************************************************

scanhelp        proc    near
                push    si                      ;Save SI
scanloop:       lodsb                           ;Get a character
                cmp     al,0Dh                  ;Exit if end of line
                je      scan_exit
                cmp     al,"?"                  ;Loop if not "?"
                jne     scanloop
                cmp     byte ptr [si-2],"/"     ;Loop if not "/"
                jne     scanloop

                add     sp,2                    ;Clear the stack
                sub     si,2                    ;Adjust SI
                stc                             ;Set carry and exit
                ret

scan_exit:      pop     si                      ;Restore SI
                clc                             ;Clear carry and exit
                ret
scanhelp        endp

;****************************************************************************
; FINDCHAR advances SI to the next non-white-space character. On return,
; carry set indicates EOL was encountered; carry clear indicates it was not.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,09h                  ;Loop if tab
                je      findchar
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                cmp     al,3Dh                  ;Loop if equal sign
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL. Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

;****************************************************************************
; BIN2ASC converts a binary value in AX to ASCII form and displays it.
;****************************************************************************

bin2asc         proc    near
                mov     bx,10                   ;Initialize divisor word and
                xor     cx,cx                   ;digit counter
b2a1:           inc     cx                      ;Increment digit count
                xor     dx,dx                   ;Divide by 10
                div     bx
                push    dx                      ;Save remainder on stack
                or      ax,ax                   ;Loop until quotient is zero
                jnz     b2a1
b2a2:           pop     dx                      ;Retrieve a digit from stack
                add     dl,30h                  ;Convert it to ASCII
                mov     ah,2                    ;Display it
                int     21h
                loop    b2a2                    ;Loop until done
                ret
bin2asc         endp

code            ends
                end     begin
