(* The string of text cannot be longer than 255 characters. *)

Type LongString=String[255];

(* Constants. *)

Const MaxEndpoints=7; (* Maximum number of end points in any character. *)
      MaxStrokes  =6; (* Maximum number of strokes in any character.	*)

(* Types. *)

Type SupportedChars   =(Uns,Spc,DQu,SQu,Cma,Dsh,Per,
			D0,D1,D2,D3,D4,D5,D6,D7,D8,D9,
			UA,UB,UC,UD,UE,UF,UG,UH,UI,UJ,UK,UL,UM,
			UN,UO,UP,UQ,UR,US,UT,UU,UV,UW,UX,UY,UZ);
     Info	      =0..255;
     Endpoint	      =-255..255;
     Stroke	      =0..MaxEndpoints;
     CharInfo	      =(EndpointCount,StrokeCount,Width);
     CharCoors	      =(CharX,CharY);
     FromTo	      =(StrokeFrom,StrokeTo);
     CharArrayType    =Array[Char] Of SupportedChars;
     InfoArrayType    =Array[SupportedChars,CharInfo] Of Info;
     EndpointArrayType=Array[SupportedChars,1..MaxEndpoints,CharCoors]
			     Of Endpoint;
     StrokeArrayType  =Array[SupportedChars,1..MaxStrokes,FromTo] Of Stroke;

(* Array of 256 elements that tells what character should be drawn for each *)
(* ASCII code.								    *)

Const CharArray:CharArrayType=(
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Spc,Uns,DQu,Uns,Uns,Uns,Uns,SQu,Uns,Uns,Uns,Uns,Cma,Dsh,Per,Uns,
 D0, D1, D2, D3, D4, D5, D6, D7, D8, D9,Uns,Uns,Uns,Uns,Uns,Uns,
Uns, UA, UB, UC, UD, UE, UF, UG, UH, UI, UJ, UK, UL, UM, UN, UO,
 UP, UQ, UR, US, UT, UU, UV, UW, UX, UY, UZ,Uns,Uns,Uns,Uns,Uns,
Uns, UA, UB, UC, UD, UE, UF, UG, UH, UI, UJ, UK, UL, UM, UN, UO,
 UP, UQ, UR, US, UT, UU, UV, UW, UX, UY, UZ,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,
Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns,Uns);

(* Each of the elements of this array gives three pieces of information about *)
(* each supported character: the number of endpoints in the character, the    *)
(* number of lines in the character, and the width of the character.  The     *)
(* width is given as though the character was drawn in a 121x121 box whose    *)
(* coordinates range from 0 to 120 along each side.  The width includes some  *)
(* extra space so that adjacent characters don't touch.                       *)

InfoArray: InfoArrayType = (
     (* *) (0,0,80),
     (*Space*) (0,0,80),
     (*DoubleQuote*) (4,2,60),
     (*SingleQuote*) (2,1,40),
     (*Comma*) (5,4,40),
     (*Dash*) (2,1,120),
     (*Period*) (4,4,40),
     (*D0*) (4,5,130),
     (*D1*) (5,3,130),
     (*D2*) (5,4,130),
     (*D3*) (6,4,130),
     (*D4*) (4,3,130),
     (*D5*) (6,5,130),
     (*D6*) (6,5,130),
     (*D7*) (3,2,130),
     (*D8*) (6,5,130),
     (*D9*) (5,4,130),
     (*UA*) (5,3,150),
     (*UB*) (7,6,120),
     (*UC*) (4,3,135),
     (*UD*) (6,6,130),
     (*UE*) (6,4,125),
     (*UF*) (5,3,120),
     (*UG*) (6,5,135),
     (*UH*) (6,3,130),
     (*UI*) (2,1,30),
     (*UJ*) (4,3,120),
     (*UK*) (6,3,135),
     (*UL*) (3,2,120),
     (*UM*) (5,4,160),
     (*UN*) (4,3,130),
     (*UO*) (4,4,150),
     (*UP*) (5,4,120),
     (*UQ*) (6,5,150),
     (*UR*) (7,5,120),
     (*US*) (6,5,130),
     (*UT*) (4,2,150),
     (*UU*) (4,3,150),
     (*UV*) (3,2,150),
     (*UW*) (5,4,190),
     (*UX*) (4,2,150),
     (*UY*) (4,3,145),
     (*UZ*) (4,3,145));

(* This array contains the endpoints in each of the supported characters. *)
(* If a character has less than MaxEndpoints endpoints, then the extra	  *)
(* endpoints are set to (0,0).						  *)

EndpointArray: EndpointArrayType = (
     (* *) ((0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Space*) ((0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*DoubleQuote*) ((0,120),(0,80),(30,120),(30,80),(0,0),(0,0),(0,0)),
     (*SingleQuote*) ((0,120),(0,80),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Comma*) ((0,0),(0,10),(10,10),(10,0),(10,-20),(0,0),(0,0)),
     (*Dash*) ((0,60),(100,60),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Period*) ((0,0),(0,10),(10,10),(10,0),(0,0),(0,0),(0,0)),
     (*D0*) ((0,0),(0,120),(100,120),(100,0),(0,0),(0,0),(0,0)),
     (*D1*) ((30,0),(70,0),(50,0),(50,120),(30,100),(0,0),(0,0)),
     (*D2*) ((10,120),(95,120),(95,70),(0,0),(100,0),(0,0),(0,0)),
     (*D3*) ((10,120),(100,120),(100,60),(100,0),(45,60),(0,0),(0,0)),
     (*D4*) ((90,0),(90,120),(0,40),(110,40),(0,0),(0,0),(0,0)),
     (*D5*) ((95,120),(0,120),(0,60),(100,60),(100,0),(0,0),(0,0)),
     (*D6*) ((95,120),(0,120),(0,0),(100,0),(100,60),(0,60),(0,0)),
     (*D7*) ((0,120),(100,120),(55,0),(0,0),(0,0),(0,0),(0,0)),
     (*D8*) ((0,0),(0,120),(100,120),(100,0),(0,60),(100,60),(0,0)),
     (*D9*) ((100,0),(100,120),(0,120),(0,60),(100,60),(0,0),(0,0)),
     (*UA*) ((0,0),(60,120),(120,0),(20,40),(100,40),(0,0),(0,0)),
     (*UB*) ((0,0),(0,120),(85,120),(85,60),(90,0),(90,60),(0,60)),
     (*UC*) ((105,0),(0,0),(0,120),(100,120),(0,0),(0,0),(0,0)),
     (*UD*) ((0,0),(0,120),(70,120),(100,90),(100,30),(70,0),(0,0)),
     (*UE*) ((0,0),(0,60),(0,120),(90,120),(70,60),(95,0),(0,0)),
     (*UF*) ((0,0),(0,60),(0,120),(90,120),(70,60),(0,0),(0,0)),
     (*UG*) ((100,120),(0,120),(0,0),(105,0),(105,60),(65,60),(0,0)),
     (*UH*) ((0,0),(0,60),(0,120),(100,0),(100,60),(100,120),(0,0)),
     (*UI*) ((0,0),(0,120),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*UJ*) ((0,30),(0,0),(90,0),(90,120),(0,0),(0,0),(0,0)),
     (*UK*) ((0,0),(0,120),(0,40),(96,120),(42,75),(105,0),(0,0)),
     (*UL*) ((0,120),(0,0),(90,0),(0,0),(0,0),(0,0),(0,0)),
     (*UM*) ((0,0),(0,120),(65,0),(130,120),(130,0),(0,0),(0,0)),
     (*UN*) ((0,0),(0,120),(100,0),(100,120),(0,0),(0,0),(0,0)),
     (*UO*) ((0,0),(0,120),(120,120),(120,0),(0,0),(0,0),(0,0)),
     (*UP*) ((0,0),(0,120),(90,120),(90,60),(0,60),(0,0),(0,0)),
     (*UQ*) ((0,0),(0,120),(120,120),(120,0),(65,15),(80,-10),(0,0)),
     (*UR*) ((0,0),(0,120),(90,120),(90,60),(0,60),(60,60),(90,0)),
     (*US*) ((0,0),(100,0),(100,60),(10,60),(10,120),(90,120),(0,0)),
     (*UT*) ((0,120),(120,120),(60,120),(60,0),(0,0),(0,0),(0,0)),
     (*UU*) ((0,120),(0,0),(120,0),(120,120),(0,0),(0,0),(0,0)),
     (*UV*) ((0,120),(60,0),(120,120),(0,0),(0,0),(0,0),(0,0)),
     (*UW*) ((0,120),(40,0),(80,120),(120,0),(160,120),(0,0),(0,0)),
     (*UX*) ((0,0),(110,120),(10,120),(120,0),(0,0),(0,0),(0,0)),
     (*UY*) ((5,120),(60,60),(60,0),(115,120),(0,0),(0,0),(0,0)),
     (*UZ*) ((10,120),(110,120),(0,0),(115,0),(0,0),(0,0),(0,0)) );

(* This array contains the lines in each of the supported characters.  Each *)
(* line is described by a pair of numbers indicating which endpoints should *)
(* be joined together.	If a character has less than the maximum number of  *)
(* lines, then the pair of numbers is set to (0,0).			    *)

StrokeArray: StrokeArrayType = (
     (* *) ((0,0),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Space*) ((0,0),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*DoubleQuote*) ((1,2),(3,4),(0,0),(0,0),(0,0),(0,0)),
     (*SingleQuote*) ((1,2),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Comma*) ((4,1),(1,2),(2,3),(3,5),(0,0),(0,0)),
     (*Dash*) ((1,2),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*Period*) ((1,2),(2,3),(3,4),(4,1),(0,0),(0,0)),
     (*D0*) ((1,2),(2,3),(3,4),(4,1),(1,3),(0,0)),
     (*D1*) ((1,2),(3,4),(4,5),(0,0),(0,0),(0,0)),
     (*D2*) ((1,2),(2,3),(3,4),(4,5),(0,0),(0,0)),
     (*D3*) ((1,2),(2,4),(3,5),(4,6),(0,0),(0,0)),
     (*D4*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)),
     (*D5*) ((1,2),(2,3),(3,4),(4,5),(5,6),(0,0)),
     (*D6*) ((1,2),(2,3),(3,4),(4,5),(5,6),(0,0)),
     (*D7*) ((1,2),(2,3),(0,0),(0,0),(0,0),(0,0)),
     (*D8*) ((1,2),(2,3),(3,4),(4,1),(5,6),(0,0)),
     (*D9*) ((1,2),(2,3),(3,4),(4,5),(0,0),(0,0)),
     (*UA*) ((1,2),(2,3),(4,5),(0,0),(0,0),(0,0)),
     (*UB*) ((1,2),(2,3),(3,4),(1,5),(5,6),(6,7)),
     (*UC*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)),
     (*UD*) ((1,2),(2,3),(3,4),(4,5),(5,6),(6,1)),
     (*UE*) ((1,3),(3,4),(2,5),(1,6),(0,0),(0,0)),
     (*UF*) ((1,3),(3,4),(2,5),(0,0),(0,0),(0,0)),
     (*UG*) ((1,2),(2,3),(3,4),(4,5),(5,6),(0,0)),
     (*UH*) ((1,3),(4,6),(2,5),(0,0),(0,0),(0,0)),
     (*UI*) ((1,2),(0,0),(0,0),(0,0),(0,0),(0,0)),
     (*UJ*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)),
     (*UK*) ((1,2),(3,4),(5,6),(0,0),(0,0),(0,0)),
     (*UL*) ((1,2),(2,3),(0,0),(0,0),(0,0),(0,0)),
     (*UM*) ((1,2),(2,3),(3,4),(4,5),(0,0),(0,0)),
     (*UN*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)),
     (*UO*) ((1,2),(2,3),(3,4),(4,1),(0,0),(0,0)),
     (*UP*) ((1,2),(2,3),(3,4),(4,5),(0,0),(0,0)),
     (*UQ*) ((1,2),(2,3),(3,4),(4,1),(5,6),(0,0)),
     (*UR*) ((1,2),(2,3),(3,4),(4,5),(6,7),(0,0)),
     (*US*) ((1,2),(2,3),(3,4),(4,5),(5,6),(0,0)),
     (*UT*) ((1,2),(3,4),(0,0),(0,0),(0,0),(0,0)),
     (*UU*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)),
     (*UV*) ((1,2),(2,3),(0,0),(0,0),(0,0),(0,0)),
     (*UW*) ((1,2),(2,3),(3,4),(4,5),(0,0),(0,0)),
     (*UX*) ((1,2),(3,4),(0,0),(0,0),(0,0),(0,0)),
     (*UY*) ((1,2),(2,3),(2,4),(0,0),(0,0),(0,0)),
     (*UZ*) ((1,2),(2,3),(3,4),(0,0),(0,0),(0,0)) );

(* This function returns the natural length of a string of text, assuming *)
(* that its height is 1.						  *)

Function TextLength(TextString:LongString):Real;
Var CurrentLength:Real;
    I:Integer;
Begin
CurrentLength:=0;
For I:=1 To Length(TextString) Do
    CurrentLength:=CurrentLength+InfoArray[CharArray[TextString[I]],Width];
TextLength:=CurrentLength/120.0;
End;

(* This subroutine generates the endpoints and lines needed to display text  *)
(* characters.	It only supports the characters needed in the objects used   *)
(* in the tutorial, though you can add your own characters if you wish.  The *)
(* supported characters are the space, double quote, period, digits, single  *)
(* quote, dash, comma, and uppercase letters.  The calling sequence is:      *)
(*									     *)
(*  WriteText( TextColor,						     *)
(*	       XScale, YScale,						     *)
(*	       XAngle, YAngle, ZAngle,					     *)
(*	       InitialX, InitialY, InitialZ,				     *)
(*	       TextString		     ); 			     *)
(*									     *)
(* The arguments are:							     *)
(*									     *)
(* TextColor : Integer;     Color to draw the text in.			     *)
(* XScale    : Real;	    Amount to scale the x dimension by. 	     *)
(* YScale    : Real;	    Amount to scale the y dimension by. 	     *)
(* XAngle    : Real;	    Amount to rotate the text around the x axis.     *)
(* YAngle    : Real;	    Amount to rotate the text around the y axis.     *)
(* ZAngle    : Real;	    Amount to rotate the text around the z axis.     *)
(* InitialX  : Real;	    X coordinate of lower left corner of text.	     *)
(* InitialY  : Real;	    Y coordinate of lower left corner of text.	     *)
(* InitialZ  : Real;	    Z coordinate of lower left corner of text.	     *)
(* TextString: String[255]; String of text to write.			     *)
(*									     *)
(* The XScale and YScale parameters can be used to make tall, skinny	     *)
(* characters or short, fat ones.  The characters are initially scaled so    *)
(* their nominal height is 1.  The XAngle, YAngle, and ZAngle parameters     *)
(* enable you to draw the text at any angle in 3 dimensions.  The angles     *)
(* should be given in degrees.	The InitialX, InitialY, and InitialZ	     *)
(* parameters enable you to position the text anywhere in the coordinate     *)
(* space.  Characters in TextString which aren't supported are drawn as      *)
(* blank spaces.							     *)
(*						 David B. Parker	     *)

(* Declaration of the WriteText procedure. *)

Procedure WriteText(TextColor		      : Integer;
		    XScale,YScale	      : Real;
		    XAngle,YAngle,ZAngle      : Real;
		    InitialX,InitialY,InitialZ: Real;
		    TextString		      : LongString);

(* Number of points used for text strings. *)

Const PointNumber:Integer=0;

(* Variables. *)

Var I,J 	     :Integer;			(* Looping variables.	 *)
    CurrentX	     :Real;			(* Current position.	 *)
    TempCos,Temp     :Real;			(* Temporary variables.  *)
    NullString	     :Boolean;
    XFlag,YFlag,ZFlag:Boolean;			(* Coordinate flags.	 *)
    AR		     :Array[1..3,1..2] Of Real; (* Transformation array. *)
    CurrentChar      :SupportedChars;		(* Current character.	 *)
    CurrentPoint     :Integer;			(* Points so far.	 *)

(* Make sure that something in the string is displayable. *)

Begin

NullString:=True;
For I:=1 To Length(TextString) Do
    If (CharArray[TextString[I]]<>Uns) And (CharArray[TextString[I]]<>Spc)
    Then NullString:=False;
If Not NullString Then Begin

(* Initialize some variables. *)

   CurrentX:=0.0;

(* Convert angles from degrees to radians. *)

   XAngle:=XAngle*Pi/180.0;
   YAngle:=YAngle*Pi/180.0;
   ZAngle:=ZAngle*Pi/180.0;

(* Start calculating the transformation with the x axis rotation. *)

   AR[2,2]:=Cos(XAngle);
   AR[3,2]:=Sin(XAngle);

(* Add the y axis rotation to the transformation. *)

   AR[1,1]:=Cos(YAngle);
   AR[3,1]:=Sin(YAngle);
   AR[1,2]:=-AR[3,1]*AR[3,2];
   AR[3,2]:=AR[1,1]*AR[3,2];

(* Add the z axis rotation to the transformation. *)

   AR[2,1]:=Sin(ZAngle);
   TempCos:=Cos(ZAngle);
   Temp:=TempCos*AR[1,2]-AR[2,1]*AR[2,2];
   AR[2,2]:=AR[2,1]*AR[1,2]+TempCos*AR[2,2];
   AR[1,2]:=Temp;
   AR[2,1]:=AR[2,1]*AR[1,1];
   AR[1,1]:=TempCos*AR[1,1];

(* Divide the scaling factors by 120, which is the size of the box that we *)
(* entered the endpoints of the characters in.				   *)

   XScale:=XScale/120.0;
   YScale:=YScale/120.0;

(* Add the scaling factors to the transformation. *)

   For I:=1 To 3 Do Begin
       AR[I,1]:=AR[I,1]*XScale;
       AR[I,2]:=AR[I,2]*YScale; End;

   XFlag:=(AR[1,1]<>0.0) Or (AR[1,2]<>0.0);
   YFlag:=(AR[2,1]<>0.0) Or (AR[2,2]<>0.0);
   ZFlag:=(AR[3,1]<>0.0) Or (AR[3,2]<>0.0);

   Write(Output,'Set Color ',TextColor);
   If Not XFlag Then Write(Output,' X ',RealStr(InitialX));
   If Not YFlag Then Write(Output,' Y ',RealStr(InitialY));
   If Not ZFlag Then Write(Output,' Z ',RealStr(InitialZ));
   Writeln(Output);

(* Loop over the characters in the string, transforming them from the box *)
(* that we initially entered the coordinates in to AcroSpin's coordinate  *)
(* space, and then write out the endpoints for all of the characters in   *)
(* the string.								  *)

   Write(Output,'EndpointList');
   If XFlag Then Write(Output,' X');
   If YFlag Then Write(Output,' Y');
   If ZFlag Then Write(Output,' Z');
   Writeln(Output,' Name');
   CurrentPoint:=PointNumber;
   For I:=1 To Length(TextString) Do Begin
       CurrentChar:=CharArray[TextString[I]];
       For J:=1 To InfoArray[CurrentChar,EndpointCount] Do Begin
	   CurrentPoint:=CurrentPoint+1;
	   If XFlag Then Write(Output,
	      RealStr(AR[1,1]*(EndpointArray[CurrentChar,J,CharX]+CurrentX)
		     +AR[1,2]*EndpointArray[CurrentChar,J,CharY]+InitialX));
	   If YFlag Then Write(Output,' ',
	      RealStr(AR[2,1]*(EndpointArray[CurrentChar,J,CharX]+CurrentX)
		     +AR[2,2]*EndpointArray[CurrentChar,J,CharY]+InitialY));
	   If ZFlag Then Write(Output,' ',
	      RealStr(AR[3,1]*(EndpointArray[CurrentChar,J,CharX]+CurrentX)
		     +AR[3,2]*EndpointArray[CurrentChar,J,CharY]+InitialZ));
	   Writeln(Output,' S',CurrentPoint); End;
       CurrentX:=CurrentX+InfoArray[CurrentChar,Width]; End;

(* Write out the lines for all of the characters in the string. *)

   WriteLn(Output,'LineList From To');
   CurrentPoint:=PointNumber;
   For I:=1 To Length(TextString) Do Begin
       CurrentChar:=CharArray[TextString[I]];
       For J:=1 To InfoArray[CurrentChar,StrokeCount] Do
	  Writeln(Output,'S',CurrentPoint+StrokeArray[CurrentChar,J,StrokeFrom],
			' S',CurrentPoint+StrokeArray[CurrentChar,J,StrokeTo]);
       CurrentPoint:=CurrentPoint+InfoArray[CurrentChar,EndpointCount]; End;
   PointNumber:=CurrentPoint; End;
End;
