Program Surface;

(* This program generates an AcroSpin file called SURFACE.ACD that displays a *)
(* 3-dimensional surface plot.	It has been tested with Turbo Pascal Versions *)
(* 3 and 5.  You can plot your own function by changing the Z subroutine      *)
(* below.  By changing the following constants, you can control a number of   *)
(* other parameters.							      *)
(*									      *)
(*						      David B. Parker	      *)

(* Constants. *)

Const AutoScaling =True; (* Automatically scale Z to look nice. *)
      MinX	  =-1.0; (* Minimum x value.			*)
      MaxX	  =1.0;  (* Maximum x value.			*)
      MinY	  =-1.0; (* Minimum y value.			*)
      MaxY	  =1.0;  (* Maximum y value.			*)
      XDivisions  =10;	 (* Number of x divisions.		*)
      YDivisions  =10;	 (* Number of y divisions.		*)
      SurfaceColor=13;	 (* Color to use for the surface.	*)
      AxisColor   =15;	 (* Color to use for the axes.		*)
      TextColor   =15;	 (* Color to use for the text.		*)

(* Variables. *)

Var ZArray		:Array[0..XDivisions,0..YDivisions] Of Real;
    I,J 		:Integer; (* Looping variables. 	   *)
    MinZ,MaxZ		:Real;	  (* Minimum and maximum z values. *)
    XRange,YRange,ZRange:Real;	  (* Range of x, y, and z values.  *)
    MaxRange		:Real;	  (* Maximum of the three ranges.  *)
    MinMaxFlag		:Boolean; (* Flag variable.		   *)
    Output		:Text;	  (* Output file.		   *)

(* Function to be plotted.  Another good function is Z:=X*Y. *)

Function Z(X,Y:Real):Real;
Begin
Z:=Exp(-X*X-Y*Y)*Cos(2.0*(X*X+Y*Y));
End;

(* Include the subroutines that write out text and convert reals to strings. *)

{$I REALSTR.PAS}
{$I WRITETEX.PAS}

(* Open the output file. *)

Begin
Assign(Output,'SURFACE.ACD');
Rewrite(Output);

(* Calculate the points on the surface,        *)
(* remembering the minimum and maximum values. *)

XRange:=MaxX-MinX;
YRange:=MaxY-MinY;
MinMaxFlag:=False;
For I:=0 to XDivisions Do
    For J:=0 To YDivisions Do Begin
	ZArray[I,J]:=Z(MinX+I*XRange/XDivisions,MinY+J*YRange/YDivisions);
	If Not MinMaxFlag Then Begin
	   MinMaxFlag:=True;
	   MinZ:=ZArray[I,J];
	   MaxZ:=MinZ; End
	Else If ZArray[I,J]<MinZ Then MinZ:=ZArray[I,J]
	Else If ZArray[I,J]>MaxZ Then MaxZ:=ZArray[I,J]; End;
ZRange:=MaxZ-MinZ;

(* Calculate the maximum range of the x, y, and z coordinates.	 *)
(* If autoscaling was requested, scale the z coordinates so that *)
(* their range is the same as the maximum of the x and y ranges. *)

MaxRange:=XRange;
If MaxRange<YRange Then MaxRange:=YRange;
If AutoScaling Then Begin
   For I:=0 To XDivisions Do
       For J:=0 To YDivisions Do ZArray[I,J]:=ZArray[I,J]*MaxRange/ZRange;
   MaxZ:=MaxZ*MaxRange/ZRange;
   MinZ:=MinZ*MaxRange/ZRange; End
Else If MaxRange<ZRange Then MaxRange:=ZRange;

(* Write out the points for the surface. *)

Writeln(Output,'EndpointList X Y Z Name');
For I:=0 to XDivisions Do
    For J:=0 To YDivisions Do
	Writeln(Output,RealStr(13000.0*(2.0*I/XDivisions-1.0)*XRange/MaxRange),
		   ' ',RealStr(13000.0*(2.0*ZArray[I,J]-MaxZ-MinZ)/MaxRange),
		   ' ',RealStr(13000.0*(1.0-2.0*J/YDivisions)*YRange/MaxRange),
		   ' X',I*(YDivisions+1)+J);

(* Write out the lines for the surface. *)

Writeln(Output,'Set Color ',SurfaceColor);
Writeln(Output,'LineList From To');
For I:=0 To XDivisions Do
    For J:=0 To YDivisions Do Begin
	If I<XDivisions Then
	  Writeln(Output,'X',I*(YDivisions+1)+J,' X',(I+1)*(YDivisions+1)+J);
	If J<YDivisions Then
	  Writeln(Output,'X',I*(YDivisions+1)+J,' X',I*(YDivisions+1)+J+1); End;

(* Write out the x, y, and z axes. *)

Writeln(Output,'EndpointList X Y Z Name');
Writeln(Output,'-15000 -15000 15000 X0Y0Z1');
Writeln(Output,'15000 -15000 15000 X1Y0Z1');
Writeln(Output,'-15000 15000 15000 X0Y1Z1');
Writeln(Output,'-15000 -15000 -15000 X0Y0Z0');
Writeln(Output,'Set Color ',AxisColor,' Layer 1');
Writeln(Output,'LineList From To');
Writeln(Output,'X0Y0Z1 X1Y0Z1');
Writeln(Output,'X0Y0Z1 X0Y1Z1');
Writeln(Output,'X0Y0Z1 X0Y0Z0');

(* Put little arrows on the axes. *)

Writeln(Output,'EndpointList X Y Z Name');
Writeln(Output,'14500 -14550 15000 XA1');
Writeln(Output,'14500 -15450 15000 XA2');
Writeln(Output,'-14550 14500 15000 ZA1');
Writeln(Output,'-15450 14500 15000 ZA2');
Writeln(Output,'-15000 -14550 -14500 YA1');
Writeln(Output,'-15000 -15450 -14500 YA2');
Writeln(Output,'LineList From To');
Writeln(Output,'XA1 X1Y0Z1');
Writeln(Output,'XA2 X1Y0Z1');
Writeln(Output,'ZA1 X0Y1Z1');
Writeln(Output,'YA2 X0Y0Z0');
Writeln(Output,'YA1 X0Y0Z0');
Writeln(Output,'ZA2 X0Y1Z1');

(* Write out the legends for the axes. *)

WriteText(TextColor,1500.0,1500.0,0.0,0.0,0.0,13000.0,-18000.0,15000.0,'X');
WriteText(TextColor,1500.0,1500.0,0.0,-90.0,0.0,-15000.0,-18000.0,-13000.0,'Y');
WriteText(TextColor,1500.0,1500.0,0.0,0.0,0.0,-18000.0,13000.0,15000.0,'Z');

(* Close the output file. *)

Close(Output);
End.
