/**************************************************************
*                                                             *
*   P.C                                                       *
*                                                             *
*   PrintScreen for OS/2 PM 1.3.1                             *
*                                                             *
**************************************************************/

#define INCL_DEV

#define INCL_DOSERRORS
#define INCL_DOSMODULEMGR
#define INCL_DOSPROCESS

#define INCL_GPIBITMAPS
#define INCL_GPICONTROL
#define INCL_GPIERRORS
#define INCL_GPIPRIMITIVES

#define INCL_WIN
#define INCL_WINERRORS
#define INCL_WINSHELLDATA

#include <os2.h>

#include <float.h>
#include <io.h>
#include <malloc.h>
#include <math.h>
#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "p.h"

/**************************************************************
*                                                             *
*   DEFINITIONS                                               *
*                                                             *
**************************************************************/

#define WM_PRINT_OK     (WM_USER + 1)
#define WM_PRINT_ERROR  (WM_USER + 2)

#define CHARS_PER_LINE  80
#define LINES_PER_PAGE  66

#define MARGIN_TOP      2       // Lines
#define MARGIN_BOTTOM   2       // Lines
#define MARGIN_LEFT     2       // Chars (ave. char. width)
#define MARGIN_RIGHT    2       // Chars (ave. char. width)

#define CHANGE_PRINT_PARAMS             0
#define RETURN_PRINTER_DEVICE_CONTEXT   1

#define INFOMSG(x) WinMessageBox (HWND_DESKTOP, \
                      WinQueryActiveWindow (HWND_DESKTOP, FALSE), \
                      x, \
                      "Print Screen", \
                      0, \
                      MB_OK | MB_ICONASTERISK | MB_APPLMODAL | MB_MOVEABLE)

/**************************************************************
*                                                             *
*   GLOBAL VARIABLES                                          *
*                                                             *
**************************************************************/

HAB     hab ;
HMQ     hmq ;
HWND    hwndMainFrame ;
HWND    hwndMainClient ;

CHAR szPrintScreenMsg[] = "PrintScreen" ;

CHAR szClassName[] = "Print Screen" ;

/**************************************************************
*                                                             *
*   Local Function Prototypes                                 *
*                                                             *
**************************************************************/

MRESULT EXPENTRY ClientWndProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2) ;

MRESULT EXPENTRY AboutDlgProc  (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2) ;

HDC              OpenDefaultPrinterDC (HAB hab, SHORT sAction) ;

BOOL EXPENTRY    InitializePrint (HAB hab, PHDC phdc, PHPS phps,
                                  PSIZEL psizlPage, PSIZEL psizlChar) ;

HBITMAP          ScreenToBitmap (HAB hab) ;

VOID             PrintTheBitmap (HBITMAP hbm) ;

SHORT  BitmapToPrinter (HBITMAP hbm, HPS hpsPrinter, HDC hdcPrinter,
                        HAB habPrinter, SIZEL *psizlPage, SIZEL *psizlChar) ;

void             ScaleToWindowSize (SHORT sXTarget, SHORT sYTarget,
                                    SHORT sXSource, SHORT sYSource,
                                    float *pfltResRatio) ;

BOOL CALLBACK    PrintScreenHook (HAB hab, PQMSG pQmsg, USHORT fs) ;

/**************************************************************
*                                                             *
*   MAIN                                                      *
*                                                             *
**************************************************************/

int main (int argc, char *argv[])
    {
    static ULONG flFrameFlags = FCF_TITLEBAR      | FCF_SYSMENU |
                                FCF_SIZEBORDER    | FCF_MINMAX  |
                                FCF_MENU          | FCF_ICON    |
                                FCF_SHELLPOSITION | FCF_TASKLIST ;

    QMSG    qmsg ;

    hab = WinInitialize (0);
    hmq = WinCreateMsgQueue (hab, 0);

    WinRegisterClass (hab, szClassName, ClientWndProc, CS_SIZEREDRAW, 0) ;

    hwndMainFrame = WinCreateStdWindow (HWND_DESKTOP, WS_VISIBLE,
                                        &flFrameFlags, szClassName, NULL,
                                        0L, (HMODULE) NULL, ID_RESOURCE,
                                        &hwndMainClient) ;

    while (WinGetMsg (hab, &qmsg, NULL, 0, 0))
        WinDispatchMsg (hab, &qmsg) ;

    WinDestroyWindow (hwndMainFrame) ;
    WinDestroyMsgQueue (hmq) ;
    WinTerminate (hab) ;
    return 0 ;
    }

/**************************************************************
*                                                             *
*   ClientWndProc                                               *
*                                                             *
**************************************************************/

MRESULT EXPENTRY ClientWndProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)

    {
    HPS     hps;
    RECTL   rcl ;
    USHORT  usRC ;

    switch (msg)
	{
        case WM_CREATE:

            // Set the print screen hook

            WinSetHook (hab, HMQ_CURRENT, HK_INPUT, (PFN) PrintScreenHook, (HMODULE) NULL) ;
            return 0 ;

	case WM_COMMAND:

	    switch (COMMANDMSG(&msg)->cmd)
                {
                case IDM_EXIT:
                    WinPostMsg (hwnd, WM_QUIT, 0L, 0L) ;
                    return 0 ;

		case IDM_ABOUT:
                    usRC = WinDlgBox (HWND_DESKTOP,
                                      hwnd,
                                      AboutDlgProc,
                                      (HMODULE) NULL,
                                      IDD_ABOUT,
                                      NULL);

                    return 0;
                }

            break ;

        case WM_DESTROY:

            // Release the hook

            WinReleaseHook (hab, hmq, HK_INPUT, (PFN) PrintScreenHook, (HMODULE) NULL) ;
            return 0L ;

        case WM_PAINT:

            hps = WinBeginPaint (hwnd, NULL, NULL);
            WinQueryWindowRect (hwnd, &rcl) ;
            WinFillRect (hps, &rcl, CLR_CYAN) ;
            WinEndPaint (hps);

            return 0;

        case WM_PRINT_OK:
            INFOMSG ("Screen printed") ;
            return 0 ;

        case WM_PRINT_ERROR:
            INFOMSG ("Error printing screen") ;
            return 0 ;
	}

    return WinDefWindowProc (hwnd, msg, mp1, mp2);
    }

/**************************************************************
*                                                             *
*   AboutDlgProc                                              *
*                                                             *
**************************************************************/

MRESULT EXPENTRY AboutDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
    {
    switch (msg)
	{
        case WM_COMMAND:

	    switch (COMMANDMSG(&msg)->cmd)
		{
                case DID_CANCEL:
                case IDD_ABOUT_OK:
		    WinDismissDlg (hwnd, TRUE);
		    return 0;
		}
	    break;
	}
    return WinDefDlgProc (hwnd, msg, mp1, mp2);
    }

/**************************************************************
*                                                             *
*   PrintScreenHook                                           *
*                                                             *
*   When the Print Screen key is pressed, call the            *
*   PrintScreen function.                                     *
*                                                             *
**************************************************************/

BOOL CALLBACK PrintScreenHook (HAB hab, PQMSG pQmsg, USHORT fs)

    {
    HBITMAP hbm ;

    // If it's our key, do it

    if ( pQmsg->msg == WM_CHAR )
        if ( ( SHORT1FROMMP(pQmsg->mp1) & KC_KEYUP) &&
             ( SHORT1FROMMP(pQmsg->mp1) & KC_VIRTUALKEY ) &&
             ( SHORT2FROMMP(pQmsg->mp2) == VK_PRINTSCRN) )

            {
            // Save the screen

            hbm = ScreenToBitmap (hab) ;

            // Print it

            PrintTheBitmap (hbm) ;
            }

    return FALSE;
    }

/**************************************************************
*                                                             *
*   ScreenToBitmap                                            *
*                                                             *
*   Save the screen display to a bitmap.                      *
*                                                             *
**************************************************************/

HBITMAP ScreenToBitmap (HAB hab)

    {
    BITMAPINFOHEADER bmp ;
    HBITMAP          hbm ;
    HDC              hdcMemory ;
    HPS              hpsScreen, hpsMemory ;
    LONG             alBitmapFormats [2] ;
    POINTL           aptl[3] ;
    SIZEL            sizl ;
    SHORT            cxScreen;
    SHORT            cyScreen;
    BOOL             fMonochrome = FALSE;

    // Create memory DC and PS

    cxScreen = (SHORT) WinQuerySysValue (HWND_DESKTOP, SV_CXSCREEN);
    cyScreen = (SHORT) WinQuerySysValue (HWND_DESKTOP, SV_CYSCREEN);

    hdcMemory = DevOpenDC (hab, OD_MEMORY, "*", 0L, NULL, NULL) ;

    sizl.cx = sizl.cy = 0 ;
    hpsMemory = GpiCreatePS (hab, hdcMemory, &sizl,
                             PU_PELS    | GPIF_DEFAULT |
                             GPIT_MICRO | GPIA_ASSOC) ;

    // Create bitmap for destination

    bmp.cbFix = sizeof bmp ;

    if (fMonochrome)
         {
         bmp.cPlanes   = 1 ;
         bmp.cBitCount = 1 ;
         }
    else
         {
         GpiQueryDeviceBitmapFormats (hpsMemory, 2L, alBitmapFormats) ;

         bmp.cPlanes   = (USHORT) alBitmapFormats[0] ;
         bmp.cBitCount = (USHORT) alBitmapFormats[1] ;
         }

    bmp.cx = cxScreen ;
    bmp.cy = cyScreen ;

    hbm = GpiCreateBitmap (hpsMemory, &bmp, 0L, NULL, NULL) ;

    // Copy from screen to bitmap

    if (hbm != NULL)
         {
         GpiSetBitmap (hpsMemory, hbm) ;
         hpsScreen = WinGetScreenPS (HWND_DESKTOP) ;

         aptl[0].x = 0 ;
         aptl[0].y = 0 ;
         aptl[1].x = cxScreen ;
         aptl[1].y = cyScreen ;
         aptl[2].x = 0 ;
         aptl[2].y = 0 ;

         WinLockVisRegions (HWND_DESKTOP, TRUE) ;

         GpiBitBlt (hpsMemory, hpsScreen, 3L, aptl,
                    fMonochrome ? ROP_NOTSRCCOPY : ROP_SRCCOPY, BBO_IGNORE) ;

         WinLockVisRegions (HWND_DESKTOP, FALSE) ;

         WinReleasePS (hpsScreen) ;
         GpiDestroyPS (hpsMemory) ;
         DevCloseDC (hdcMemory) ;
         }

    return hbm ;
    }

/**************************************************************
*                                                             *
*   PrintTheBitmap                                            *
*                                                             *
*   Perform the actual printing of the stored bitmap to       *
*   the printer.                                              *
*                                                             *
**************************************************************/

VOID PrintTheBitmap (HBITMAP hbm)

    {
    HAB       hab ;
    HDC       hdc ;
    HPS       hps ;

    SHORT     msgReturn = WM_PRINT_OK ;

    SIZEL     sizlChar ;
    SIZEL     sizlPage ;

    // Get a handle to an anchor block

    hab = WinInitialize (0) ;

    // Initialize the printer according to our page setup parameters

    if (InitializePrint (hab, &hdc, &hps, &sizlPage, &sizlChar))
        {
        // Start the document

        if (DevEscape (hdc, DEVESC_STARTDOC, sizeof (szPrintScreenMsg),
                       szPrintScreenMsg, NULL, NULL) != DEVESC_ERROR)
            {
            // Print the bitmap on the printer

            BitmapToPrinter (hbm, hps, hdc, hab, &sizlPage, &sizlChar) ;

            // Close the document

            DevEscape (hdc, DEVESC_ENDDOC, 0L, NULL, NULL, NULL) ;
            }
        else
            {
            // DevEscape failed

            msgReturn = WM_PRINT_ERROR ;
            }

        // Done printing, so clean up

        GpiAssociate (hps, NULL) ;
        GpiDestroyPS (hps) ;
        DevCloseDC (hdc) ;
        }
    else
        {
        // Couldn't initialize printer

        msgReturn = WM_PRINT_ERROR ;
        }

    // Delete the bitmap

    GpiDeleteBitmap (hbm) ;

    // Post message

    WinPostMsg (hwndMainClient, msgReturn, NULL, NULL) ;

    // We're done interacting with PM, so quit the local HAB

    WinTerminate (hab) ;
    }

/***************************************************************
*                                                              *
*   InitializePrint                                            *
*                                                              *
*   Open printer and determine boundaries.  psizlChar is       *
*   used to determine what boundary should be used around      *
*   the printed image.                                         *
*                                                              *
***************************************************************/

BOOL EXPENTRY InitializePrint (HAB        hab,          // Anchor block
                               PHDC       phdc,         // ->printer DC
                               PHPS       phps,         // ->printer PS
                               PSIZEL     psizlPage,    // ->SIZEL of page
                               PSIZEL     psizlChar)    // ->SIZEL of char

    {
    // Open the default printer device context

    if ((*phdc = OpenDefaultPrinterDC (hab, 1)) == DEV_ERROR)
        return FALSE ;

    // Initialize our page size to zero

    psizlPage->cx = 0 ;
    psizlPage->cy = 0 ;

    // Create our printer presentation space

    *phps = GpiCreatePS (hab,                 // For this hab, ...
                         *phdc,               // ... using this device context, ...
                         psizlPage,           // ... tell me how big the page is, ...
                         PU_ARBITRARY   |     // ... start with PEL's, but let me change, ...
                           GPIF_DEFAULT |     // ... points are 2 byte integers, ...
                           GPIT_NORMAL  |     // ... use a normal presentation space, ...
                           GPIA_ASSOC) ;      // ... and associate with the device context.

    // Get the page size

    GpiQueryPS (*phps, psizlPage) ;

    // Compute how large any font should be

    psizlChar->cx = (psizlPage->cx - 1) / CHARS_PER_LINE ;
    psizlChar->cy = (psizlPage->cy - 1) / LINES_PER_PAGE ;
    }

/**************************************************************
*                                                             *
*   OpenDefaultPrinterDC                                      *
*                                                             *
*   Derived From Charles Petzold                              *
*                                                             *
**************************************************************/

#pragma pack(1)

HDC OpenDefaultPrinterDC (HAB hab, SHORT sAction)
    {
    static CHAR      achPrnData[256] ;
    static DRIVDATA  driv = { sizeof (DRIVDATA) } ;

    CHAR             achDefPrnName[34], *pchDelimiter ;
    DEVOPENSTRUC     dop ;
    ERRORID          eid ;
    HDC              hdcReturn ;
    PDRIVDATA        pdriv ;
    SEL              sel ;
    ULONG            ulSize ;
    USHORT           usResult ;

    // Obtain default printer name and remove semicolon

    WinQueryProfileString (hab, "PM_SPOOLER", "PRINTER", ";",
                           achDefPrnName, sizeof achDefPrnName) ;

    if ((pchDelimiter = strchr (achDefPrnName, ';')) != NULL)
        *pchDelimiter = '\0' ;

    if (achDefPrnName[0] == '\0')
        return DEV_ERROR ;

    // Obtain information on default printer

    WinQueryProfileString (hab, "PM_SPOOLER_PRINTER", achDefPrnName,
                           ";;;;", achPrnData, sizeof achPrnData) ;

    // Parse printer information string

    if ((pchDelimiter = strchr (achPrnData, ';')) == NULL)
        return DEV_ERROR ;

    dop.pszDriverName = pchDelimiter + 1 ;

    if ((pchDelimiter = strchr (dop.pszDriverName, ';')) == NULL)
        return DEV_ERROR ;

    dop.pszLogAddress = pchDelimiter + 1 ;

    *(dop.pszLogAddress + strcspn (dop.pszLogAddress, ",;")) = '\0' ;
    *(dop.pszDriverName + strcspn (dop.pszDriverName, ",;")) = '\0' ;

    // Fill DRIVDATA structure if necessary

    if ((pchDelimiter = strchr (dop.pszDriverName, '.')) != NULL)
        {
        *pchDelimiter = '\0' ;
        strncpy (driv.szDeviceName, pchDelimiter + 1,
                                    sizeof (driv.szDeviceName)) ;
        dop.pdriv = &driv ;
        }
    else
        {
        strcpy (driv.szDeviceName, "") ;
        dop.pdriv = NULL ;
        }

    // Set data type to "raw"

    dop.pszDataType = "PM_Q_RAW" ;

    // If this is a request to change the printer, do it ...

    if (sAction == CHANGE_PRINT_PARAMS)
        {
        // Update the printer options

        ulSize = DevPostDeviceModes (hab, NULL, dop.pszDriverName,
                                     driv.szDeviceName, achDefPrnName, 0L) ;

        // NOTE: DosAllocSeg takes an USHORT, yet DevPosDeviceModes
        //       produces a LONG!!! A potential problem exists here.

        usResult = DosAllocSeg( (USHORT) ulSize, &sel, 0) ;

        pdriv = MAKEP (sel, 0) ;

        ulSize = DevPostDeviceModes (hab, pdriv, dop.pszDriverName,
                                     driv.szDeviceName, achDefPrnName, 1L) ;
        hdcReturn = NULL ;
        }
    else
        {
        // Obtain printer device context

        hdcReturn = DevOpenDC (hab, OD_QUEUED, "*", 4L, (PDEVOPENDATA) &dop, NULL) ;

        // Multiple calls cause errors -- let's see what kind (13-Nov-1990)

        if (hdcReturn == DEV_ERROR)
            {
            eid = WinGetLastError (hab) ;

            if (eid == PMERR_INV_DC_TYPE)
                {
                usResult = 0 ;
                }
            else if (eid == PMERR_INV_LENGTH_OR_COUNT)
                {
                usResult = 1 ;
                }
            else if (eid == PMERR_INV_DC_DATA)
                {
                usResult = 2 ;
                }
            else if (eid == PMERR_INV_HDC)
                {
                usResult = 3 ;
                }
            }
        }

    return hdcReturn ;
    }

/**************************************************************
*                                                             *
*   BitmapToPrinter                                           *
*                                                             *
*   Print the bitmap.  Bitmap is scaled according to the      *
*   size of the printer.  No distortion is allowed.           *
*                                                             *
*   Known problems:                                           *
*                                                             *
*       On an HP LaserJet, areas with black foreground and    *
*       gray background are completely black when printed.    *
*       This is a bug in the HP LaserJet printer driver,      *
*       not this program.                                     *
*                                                             *
**************************************************************/

SHORT  BitmapToPrinter (HBITMAP hbm,
                        HPS     hpsPrinter,
                        HDC     hdcPrinter,
                        HAB     habPrinter,
                        SIZEL   *psizlPage,
                        SIZEL   *psizlChar)

    {
    HDC             hdcPrinterMemory;
    HPS             hpsPrinterMemory;
    POINTL          ptl;
    SHORT           sPage = 1;
    RECTL           rcl;            // Coordinates of region

    long            lCapsHRes;
    long            lCapsVRes;
    float           fYAspectRatio;
    float           fXAspectRatio;
    SIZEL           sizl;

    POINTL          aptl [4] ;
    SHORT           cxScreen;
    SHORT           cyScreen;
    float           fltScale;

    // Position to top left of page

    ptl.x = 0 ;                     // Add margins here, if desired
    ptl.y = psizlPage->cy ;

    // Find the aspect ratio of the printer

    DevQueryCaps (hdcPrinter, CAPS_HORIZONTAL_RESOLUTION, 1L, &lCapsHRes) ;
    DevQueryCaps (hdcPrinter, CAPS_VERTICAL_RESOLUTION,   1L, &lCapsVRes) ;

    if ( (lCapsVRes == 0) || (lCapsHRes == 0) )
        {
        fXAspectRatio = (float) 1 ;
        fYAspectRatio = (float) 1 ;
        }
    else
        {
        fXAspectRatio = (float) ((float) lCapsVRes / (float) lCapsHRes) ;
        fYAspectRatio = (float) ((float) lCapsHRes / (float) lCapsVRes) ;
        }

    // Determine coordinates to print on printer

    rcl.xLeft   = MARGIN_LEFT * psizlChar->cx ;
    rcl.xRight  = psizlPage->cx - (MARGIN_RIGHT * psizlChar->cx) ;
    rcl.yBottom = (MARGIN_BOTTOM + 1) * psizlChar->cy ;
    rcl.yTop    = psizlPage->cy - ((MARGIN_TOP + 1) * psizlChar->cy) ;

    // Get screen dimensions

    cxScreen = (SHORT) WinQuerySysValue (HWND_DESKTOP, SV_CXSCREEN) ;
    cyScreen = (SHORT) WinQuerySysValue (HWND_DESKTOP, SV_CYSCREEN) ;

    ScaleToWindowSize ((SHORT) (rcl.xRight - rcl.xLeft),   // sXtarget
                       (SHORT) (rcl.yTop - rcl.yBottom),   // sYTarget
                       cxScreen,                           // sXSource
                       cyScreen,                           // sYSource
                       &fltScale) ;

    // Create a memory device context
    // Memory device contexts are used to contain bitmaps

    hdcPrinterMemory = DevOpenDC (habPrinter, OD_MEMORY, "*", 0L, NULL, hdcPrinter);

    if ( hdcPrinterMemory == DEV_ERROR )
        return FALSE;

    sizl.cx = 0;
    sizl.cy = 0;

    // Create a presentation space and associate it the memory device context

    hpsPrinterMemory = GpiCreatePS (habPrinter,
                                    hdcPrinterMemory,
                                    &sizl,
                                    PU_PELS     | GPIF_DEFAULT |
                                    GPIT_NORMAL | GPIA_ASSOC) ;

    // Fail if we didn't create the PS

    if (!hpsPrinterMemory)
        {
        DevCloseDC (hdcPrinterMemory);
        return FALSE;
        }

    // Set bitmap into presentation space

    GpiSetBitmap (hpsPrinterMemory, hbm) ;

    // Create transformation "matrix"

    aptl [0].x = rcl.xRight - (long) ((float) cxScreen * fltScale) ;
    aptl [0].y = rcl.yTop - (long) ((float) cyScreen * fltScale * fYAspectRatio) ;
    aptl [1].x = rcl.xRight ;
    aptl [1].y = rcl.yTop ;
    aptl [2].x = 0 ;
    aptl [2].y = 0 ;
    aptl [3].x = cxScreen ;
    aptl [3].y = cyScreen ;

    // Move the bitmap to the printer

    GpiBitBlt (hpsPrinter, hpsPrinterMemory, 4L, aptl, ROP_SRCCOPY, BBO_IGNORE) ;

    // Clean up

    GpiAssociate (hpsPrinterMemory, NULL) ;
    GpiDestroyPS (hpsPrinterMemory) ;
    DevCloseDC   (hdcPrinterMemory) ;

    // Job done

    return (TRUE) ;
    }

/**************************************************************
*                                                             *
*   ScaleToWindowSize                                         *
*                                                             *
*   Calculate the change in scale required to display the     *
*   same portion of the image in a different sized window.    *
*   Used to ensure that no distortion results from printing   *
*   the screen.                                               *
*                                                             *
**************************************************************/

void ScaleToWindowSize (SHORT sXTarget, SHORT sYTarget,
                        SHORT sXSource, SHORT sYSource,
                        float *pfltResRatio)

    {
    float fltRes1 ;
    float fltRes2 ;
    float fltSmallestRes ;

    // Determine the resolution changes for the X and Y sizes

    // Make sure that none of the sides of the window are zero.  This can
    // happen if zoom box is a straight line.  The smallest X and Y size is 1.

    if (sXSource == 0)
        sXSource = 1;

    if (sYSource == 0)
        sYSource = 1 ;

    fltRes1 = (float) ((float) sXTarget / (float) sXSource) ;

    fltRes2 = (float) ((float) sYTarget / (float) sYSource) ;

    // Use the smallest resolution change

    if (fltRes1 < fltRes2)
        fltSmallestRes = fltRes1 ;
    else
        fltSmallestRes = fltRes2 ;

    *pfltResRatio = fltSmallestRes ;
    }
