/* os2comm.c - a simple os/2 comm program - version 1.4
*
* This program uses the multitasking features of OS/2 and the OS/2
* communications device driver to provide a simple communications
* function.  It is not intended to provide complete communications
* capabilities, instead it is intended to demonstrate the use of the
* comm device driver under OS/2.
*
* To use this program, type OS2COMM at the OS/2 prompt,
* followed by the port, baud rate, parity, databits, and stopbits,
* and an optional capture file name:
*
*    OS2COMM  COMx:baud,parity,databits,stopbits  capture-filespec
*
* To end this program, press Alt-X.
*
* While a communications session is in progress, data can be transmitted
* from a file by pressing Alt-F or Alt-A.  The program will prompt for the
* filename, the data in the file will be transmitted, then control will be
* returned to the keyboard.  Using Alt-A performs an ASCII upload - replacing
* CRLF pairs with CRs.  Alt-F sends the file without any translation.
*
* This version of OS2Comm has added logic to receive a file using the
* XModem file transfer protocol.  (Sorry, but there is no corresponding
* capability for transmit.  Maybe you can add one.)  This routine is
* somewhat experimental, in that it is not in keeping with the design
* goals of the program, but it does work reliably.  I include it mostly to
* show how the protocol code might be written.
*
* To receive a file using XModem, first tell the remote system to begin
* sending it, then press Alt-R.  OS2Comm will prompt you for the receive
* filespec.  You may optionally specify a drive letter and path.
*
* This program will run only in the protected mode of OS/2.
* It will not run under DOS.
*
* The program contains 3 threads:
*
*   1. main() - This thread calls a COM initialization routine, then
*      opens the COM port and creates the next two threads.  After they
*      have been created, it waits for the keytocom() thread to complete,
*      then exits back to OS/2.
*
*   2. keytocom() - This thread reads characters from the keyboard and
*      sends them to the COM device driver.  When a Alt-Z is received
*      from the keyboard, it signals main() to exit.  When Alt-F or Alt-A is
*      received from the keyboard, it prompts for a filename, then sends
*      the data from the file to the COM port.  It also handles the Alt-R
*      XModem protocol for receiving files.
*
*   3. comtodsp() - This thread reads characters from the COM device
*      driver and displays them on the screen.  It also writes data to
*      the optional capture file.
*
* Three semaphores are used in this version of OS2Comm.  The first,
* main_sem, is used to release the main thread when the keytocom thread
* has received an Alt-X from the keyboard.  This allows the program to end
* gracefully.
*
* The second, buf_sem, is used by the ascii upload routine, filetocom,
* which uses the OS/2 DosWriteAsync function to send data to the COM
* device driver.  A semaphore is required by this function to allow the
* program to know when the buffer has been freed.  Using DosWriteAsync for
* this routine allows the file transfer to proceed as fast as possible.
*
* The third semaphore, ctd_sem, is used to control the comtodisp thread.
* Requesting this semaphore causes the comtodisp thread to block waiting
* for it.  Clearing it allows comtodisp to run again.  This method of
* controlling comtodisp is used instead of the OS/2 DosSuspendThread and
* DosResumeThread functions because it allows control over where the
* thread is when it is blocked.  DosSuspendThread will block the thread no
* matter what it is doing at the time.
*
* To compile and link this program, at the OS2 prompt, type:
*
*      set PATH=C:\C\BIN;C:\;C:\OS2
*      set INCLUDE=C:\C\INCLUDE
*      set LIB=C:\C\LIB;C:\OS2
*      set TMP=C:\
*      cl os2comm.c /AS
*
* These settings are dependent on the compiler and version.  Yours may
* differ from these.
*
* This program requires the use of the COM0x.SYS device driver
* in CONFIG.SYS.  Use COM01.SYS for AT class machines.  Use
* COM02.SYS for PS/2 machines.
*
* You will probably need the OS/2 Technical Reference Books to
* understand some parts of the code.  In particular, the specifics of
* the DosDevIOCtl calls will require documentation.
*
* OS2Comm.c - Copyright 1988 by Jim Gilliland.
*/
 
#define   LINT_ARGS
#define   INCL_BASE
#define   INCL_DOSDEVICES
#include <os2.h>
#include <stdio.h>
#include <conio.h>
#include <process.h>
#include <malloc.h>
#include <io.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#define   STK_SIZE    2048
 
void main(int, char**, char**);
int      initcomm(int, char, int, int);
void far keytocom(void);
int      gettext(char *, char *, int);
unsigned chkesc(void);
int      filetocom(char);
int      xmodemr(char *);
void far comtodsp(void);
int      parsarg(char*, char*, int*,
                 char*, int*, int*);
unsigned short com, cfile;
unsigned long far main_sem, far buf_sem, far ctd_sem;
unsigned ktcid, ctdid, cflag;
 
char *usage ="\nOS2Comm - usage is:\r\n"
"\n\tOS2Comm COMx:baudrate,parity,"
                "databits,stopbits"
"\n\twhere:"
"\n\t\t COMx     = COM1, COM2, or COM3"
"\n\t\t baudrate = 300,1200,2400,"
                  "4800,9600, or 19200"
"\n\t\t parity   = N,O,E,M, or S"
"\n\t\t databits = 5,6,7, or 8"
"\n\t\t stopbits = 1 or 2\r\n";
 
void main(argc, argv, envp)
int argc;
char **argv;
char **envp;
{
 char far *ctdstack, far *ktcstack;
 int act, baud, dbits, sbits;
 char parity, comport[8];
 
 puts("OS2Comm.c version 1.4");
 puts("Copyright 1988 by Jim Gilliland");
 
 if (argc < 2 || argc > 3)
    {
     puts(usage);
     exit(1);
    }
 
 if (parsarg(argv[1],comport,&baud,
             &parity,&dbits,&sbits))
    {
     puts(usage);
     exit(1);
    }
 
 /* Open com device driver: */
 if (DosOpen(comport,&com,&act,
             0L,0,0x01,0x0012,0L))
     {
     fprintf(stderr,"\nError opening port");
     exit(1);
     }
 
 /* Initialize com device driver: */
 if (initcomm(baud, parity, dbits, sbits))
    {
    fprintf(stderr,"\nPort setup error");
    exit(1);
    }
 
 /* Open capture file, if specified: */
 if (argc > 2)
   if (DosOpen(argv[2],&cfile,&act,
               0L,0,0x12,0x0022,0L))
     {
     fprintf(stderr,"\nErr: %s\n",argv[2]);
     exit(1);
     }
    else cflag = 1;
 else cflag = 0;
 
 /* allocate stack for threads: */
 ctdstack = malloc(STK_SIZE);
 ktcstack = malloc(STK_SIZE);
 
 if (ctdstack == NULL || ktcstack == NULL)
    {
    puts ("Unable to allocate stacks");
    exit(2);
    }
 
 /* Create receive and display thread: */
 if (DosCreateThread(comtodsp,&ctdid,
                     ctdstack+STK_SIZE))
    {
    puts("Can't create COM receive thread");
    exit(1);
    }
 
 /* Set semaphore to block main thread: */
 DosSemSet(&main_sem);
 
 /* Create transmit thread: */
 if (DosCreateThread(keytocom,&ktcid,
                     ktcstack+STK_SIZE))
    {
    puts("Can't create COM transmit thread");
    exit(1);
    }
 
 puts("Alt-X will end this program");
 
 /* Set high priority for COM threads */
 DosSetPrty(2,3,1,ctdid); /* time-critical + 1  */
 DosSetPrty(2,3,2,ktcid); /* time-critical + 2  */
 
/* Wait for clear semaphore (see keytocom) */
 DosSemWait(&main_sem,-1L);
 
/* Suspend the other threads before ending */
 DosSuspendThread(ktcid);
 DosSuspendThread(ctdid);
 
 /* Close com driver and capture file: */
 DosClose(com);
 if (cflag==1) DosClose(cfile);
 
 /* Give the DosClose calls time to finish: */
 DosSleep(100L);
 
 DosExit(1,0); /* exit: end all threads */
}
/*******************************************/
 
void far comtodsp()
/* This routine is run as a separate thread */
{
 char comchar[512];
 unsigned int bytes, readerr, cnt;
 
 while ( -1 )  /* Do forever: */
 {
    DosSemRequest(&ctd_sem,-1L);
    /* read character(s) from COMport: */
    readerr = DosRead(com,comchar,512,&bytes);
 
    if (readerr == 0 && bytes > 0)
       {
        /* Write to screen: */
        VioWrtTTy(comchar,bytes,0);
 
        /* write to capture file: */
        if (cflag == 1)
            DosWrite(cfile,comchar,bytes,&cnt);
       }
    DosSemClear(&ctd_sem);
 }
}
/*******************************************/
 
void far keytocom()
/* This routine is run as a separate thread */
{
 KBDKEYINFO keyinfo;
 unsigned char charcode, scancode;
 unsigned int written, ioctlerr;
 char comerr, xname[64], YNresponse[5];
 
 while ( -1 )  /* Do forever: */
 {
   /* Get character from keyboard: */
   KbdCharIn (&keyinfo,0,0);
   charcode = keyinfo.chChar;
   scancode = keyinfo.chScan;
 
   /* Alt-X indicates End-Of-Processing: */
   if (charcode == 0x00 && scancode == 0x2D)
      {
      DosSemRequest(&ctd_sem,-1L);
      gettext("Exit to DOS? ",YNresponse,4);
      DosSemClear(&ctd_sem);
      YNresponse[0] = toupper(YNresponse[0]);
      if (YNresponse[0] == 'Y')
          /* Clear Main semaphore:        */
             DosSemClear(&main_sem);
      continue;
      }
 
   /* Alt-A: get filename and process ASCII file */
   if (charcode == 0x00 && scancode == 0x1E)
      filetocom('a');
 
   /* Alt-F: get filename and process file */
   if (charcode == 0x00 && scancode == 0x21)
      filetocom('f');
 
   /* Alt-R: get filename and receive XModem file */
   if (charcode == 0x00 && scancode == 0x13)
      {
       DosSemRequest(&ctd_sem,-1L);
       gettext("Enter receive filename: ",xname,64);
       if (xmodemr(xname))
           puts("XModem receive failed");
       DosSemClear(&ctd_sem);
      }
 
   if ((charcode == 0x00 || charcode == 0xE0) && scancode != 0x00)
      continue; /* skip Alt-keys & F-Keys */
 
   /* Write character(s) to com port: */
   DosWrite(com,&charcode,1,&written);
 
   if (written == 0)
      {
       DosDevIOCtl(&comerr,NULL,0x64,01,com);
       printf("\r\nCOM Driver reports error %u\r\n", comerr);
      }
 }
}
 
int filetocom(type)
/* this routine transmits an ASCII file to the remote system */
char type;
{
 char buffer[64], work[64], *cptr;
 unsigned written, writerr, bytes, inhandle, inoflag;
 char infile[64];
 
    DosSemRequest(&ctd_sem,-1L);
    gettext("Enter file name: ",infile,64);
    if (type == 'a')
        inoflag = O_RDONLY | O_TEXT;
    else
        inoflag = O_RDONLY | O_BINARY;
    if ((inhandle=open(infile,inoflag)) == -1)
       {
       fprintf(stderr,"\nOpen error: %s\r\n",infile);
       DosSemClear(&ctd_sem);
       return 1;
       }
 
    DosSemClear(&ctd_sem);
    /* Initialize write semaphore: */
    DosSemClear(&buf_sem);
 
    while ( -1 )   /* Do forever: */
    {
      bytes=read(inhandle,work,64);
      if (bytes == 0)  /* end of file ? */
         {
         close(inhandle); /* close file */
         return 0;
         }
 
      /* Translate newline chars to carriage returns: */
      if (type == 'a')
         for ( cptr=work ; cptr < work+bytes ; cptr++ )
                if (*cptr == '\n') *cptr = '\r';
 
      /* Wait for last write to complete: */
      DosSemWait(&buf_sem, -1L);
 
      strncpy(buffer,work,bytes); /* copy to buffer */
 
      if (chkesc())  /* check keyboard for ESC character */
         {
          fprintf(stderr,"\r\nText upload aborted.\r\n");
          return 1;
         }
 
      DosSemSet(&buf_sem);
      DosWriteAsync(com,&buf_sem,&writerr,
                    buffer,bytes,&written);
 
    }
}
 
int gettext(prompt,text,length)
/* this function issues a prompt and returns the user's input */
char *prompt, *text;
int length;
{
      STRINGINBUF inbuf;
      VioWrtTTy("\r\n\n",3,0);
      VioWrtTTy(prompt,strlen(prompt),0);
      inbuf.cchIn = 0;
      inbuf.cb = length;
      KbdStringIn(text,&inbuf,0,0);
      VioWrtTTy("\r",1,0);
      text[inbuf.cchIn] = '\0';   /* terminate string */
      return 0;
}
 
unsigned chkesc()
/* this function checks the keyboard for the Escape character */
{
      KBDKEYINFO keyinfo;
      while ( -1 )
        {
         if (KbdCharIn(&keyinfo,1,0))
              return 0;   /* return if keyboard error */
         if ((keyinfo.fbStatus & 0xC0) == 0)
              return 0;   /* return if no key pressed */
         if (((keyinfo.fbStatus & 0x40) != 0) &&
              keyinfo.chChar == 0x1B)
             {
              KbdFlushBuffer(0);
              return 1;   /* if escape was pressed  */
             }
        }
}
 
#define  NAK  "\x15"
#define  ACK  "\x06"
#define  SOH   0x01
#define  CAN   0x18
#define  EOT   0x04
#define  BS    0x08
 
int xmodemr(filename)
char *filename;
/* this function processes the XModem file receive */
 
{
    unsigned int fact, ferr, bytes, writ,
                 retcode, nak, ioctlerr;
    unsigned long int totsofar;
    unsigned short fhand;
 
    unsigned char ident, expected, chksum, *cbp, comerr;
 
    struct {
        unsigned char packet;
        unsigned char xpacket;
        unsigned char data[128];
        unsigned char chksum;
        } combuff;
 
    struct {
      unsigned int wtime;
      unsigned int rtime;
      unsigned char flags1;
      unsigned char flags2;
      unsigned char flags3;
      unsigned char errchar;
      unsigned char brkchar;
      unsigned char xonchar;
      unsigned char xoffchar;
      } olddcb, newdcb;
 
    struct {
      unsigned char databits;
      unsigned char parity;
      unsigned char stopbits;
      unsigned char brkstat;
       } oldlc, newlc;
 
    ferr=DosOpen(filename,&fhand,&fact,0L,0,0x12,0x0022,0L);
    if (ferr)
       {
       printf("\r\n\nError Opening file: %s\r\n", filename);
       return 1;
       }
 
    /* get current settings: */
    DosDevIOCtl((char *)&olddcb,NULL,0x73,01,com);
    DosDevIOCtl((char *)&oldlc,NULL,0x62,01,com);
 
    /* Set com device processing parameters:  */
    newdcb.wtime = 10; /* .1sec transmit timeout */
    newdcb.rtime = 10; /* .1sec receive  timeout */
    newdcb.flags1 = 0x01;  /* enable DTR */
    newdcb.flags2 = 0x40;  /* enable RTS, disable XON/XOFF */
    newdcb.flags3 = 0x02;  /* recv timeout mode  */
    newdcb.errchar = olddcb.errchar;
    newdcb.brkchar = olddcb.brkchar;
    newdcb.xonchar = olddcb.xonchar;
    newdcb.xoffchar = olddcb.xoffchar;
    DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
 
    /* Set databits, stopbits, parity: */
    newlc.parity = 0;
    newlc.stopbits = 0;
    newlc.databits = 8;
    DosDevIOCtl(NULL,(char *)&newlc,0x42,01,com);
 
    /* Make sure line is quiet before starting: */
    for (bytes = 131; bytes == 131; )
         DosRead(com,(char *)&combuff,131,&bytes);
 
    /* Set com device processing parameters:  */
    newdcb.rtime = 1000; /* 10sec receive  timeout */
    DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
 
    printf("\r\n");
    expected = 1;
    nak = 0;
    totsofar = 0;
    retcode = 0;
 
    DosWrite(com,NAK,1,&writ);
 
    while ( -1 )
    {
        if (chkesc()) /* check keyboard for ESC character */
           {
            printf("\rXModem download aborted.             \r\n");
            DosWrite(com,"\x18\x18\x18\x08\x08\x08",6,&writ);
            retcode = 1;
            break;
           }
 
        /* Set com device processing parameters:  */
        newdcb.rtime = 1000; /* 10sec receive  timeout */
        DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
 
        if (nak == 8)
           {
            printf("\rError: timed out or too many errors  \r\n");
            DosWrite(com,"\x18\x18\x18\x08\x08\x08",6,&writ);
            retcode = 1;
            break;
           }
 
        /* Read first packet character; should be SOH, CAN, or EOT: */
        DosRead(com,&ident,1,&bytes);
 
        /* Send NAKs until first packet is started: */
        if (bytes == 0)
           {
            DosWrite(com,NAK,1,&writ);
            nak++;
            continue;
           }
 
        if (ident != SOH && ident != EOT && ident != CAN)
            continue;
 
        if (ident == EOT) /* EOT=End Of Transmission */
           {
            printf("\r\nReceived EOT\r\n");
            newdcb.rtime = 10; /* .1sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            for (bytes = 131; bytes == 131; ) /* wait for quiet */
                 DosRead(com,(char *)&combuff,131,&bytes);
            DosWrite(com,ACK,1,&writ);
            retcode = 0;
            break;
           }
 
        if (ident == CAN) /* CAN=Cancel XModem process */
           {
            printf("\rReceived CAN                         \r\n");
            newdcb.rtime = 10; /* .1sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            for (bytes = 131; bytes == 131; ) /* wait for quiet */
                 DosRead(com,(char *)&combuff,131,&bytes);
            retcode = 1;
            break;
           }
 
        /* if not EOT and not CAN, then receive the rest of the packet: */
 
        /* Set com device processing parameters:  */
        newdcb.rtime = 100; /* 1sec receive  timeout */
        DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
 
        DosRead(com,(char *)&combuff,131,&bytes);
 
        if ((bytes != 131) || ((combuff.packet & combuff.xpacket) != 0x00))
           {
            printf("\rPacket Error: bytes %i, packet %x, xpacket %x\r\n",
                                    bytes, combuff.packet, combuff.xpacket);
            newdcb.rtime = 10; /* .1sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            for (bytes = 131; bytes == 131; ) /* wait for quiet */
                 DosRead(com,(char *)&combuff,131,&bytes);
            DosWrite(com,NAK,1,&writ);
            newdcb.rtime = 1000; /* 10sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            nak++;
            continue;
           }
 
        chksum = 0;
        /* Compute checksum: */
        for (cbp = combuff.data ; cbp < combuff.data+128 ; cbp++)
            chksum += *cbp;
 
        if (chksum != combuff.chksum)
           {
            printf("\rChecksum error: received %x, computed %x\r\n",
                               combuff.chksum,     chksum);
            newdcb.rtime = 10; /* .1sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            for (bytes = 131; bytes == 131; ) /* wait for quiet */
                 DosRead(com,(char *)&combuff,131,&bytes);
            DosWrite(com,NAK,1,&writ);
            newdcb.rtime = 1000; /* 10sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            nak++;
            continue;
           }
 
        if ((combuff.packet != expected) && (combuff.packet != expected-1))
           {
            printf("\rSequence failure: expected %u, received %u\r\n",
                                        expected,    combuff.packet);
            newdcb.rtime = 10; /* .1sec receive  timeout */
            DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
            for (bytes = 131; bytes == 131; ) /* wait for quiet */
                 DosRead(com,(char * )&combuff,131,&bytes);
            DosWrite(com,"\x18\x18\x18\x08\x08\x08",6,&writ);
            retcode = 1;
            break;
           }
        else if (combuff.packet == expected-1)
             /* we've already received this packet, so ignore */
           {
            printf("\rBytes received: %lu\r", totsofar);
            DosWrite(com,ACK,1,&writ);
            nak = 0;
            continue;
           }
        else
           {
             /* good packet, write it to disk, increment bytecount */
            totsofar += 128;
            printf("\rBytes received: %lu\r", totsofar);
            DosWrite(com,ACK,1,&writ);
            ferr=DosWrite(fhand,combuff.data,128,&writ);
            if (ferr || writ < 128)
               {
                printf("\r\nFile write error: %u, %u bytes written\r\n",
                                            ferr, writ);
                newdcb.rtime = 10; /* .1sec receive  timeout */
                DosDevIOCtl(NULL,(char *)&newdcb,0x53,01,com);
                for (bytes = 131; bytes == 131; ) /* wait for quiet */
                     DosRead(com,(char *)&combuff,131,&bytes);
                DosWrite(com,"\x18\x18\x18\x08\x08\x08",6,&writ);
                retcode = 1;
                break;
               }
            expected++;
            nak = 0;
            continue;
           }
    }
    DosClose(fhand);
    if (retcode == 0)
        printf("\r%s received successfully.           \r\n", filename);
    ioctlerr=DosDevIOCtl(NULL,(char *)&olddcb,0x53,01,com);
    if (ioctlerr) printf("\r\nDevIOCtl 53h Error: %x\r\n",ioctlerr);
    ioctlerr=DosDevIOCtl(NULL,(char *)&oldlc,0x42,01,com);
    if (ioctlerr) printf("\r\nDevIOCtl 42h Error: %x\r\n",ioctlerr);
    return retcode;
}
/*******************************************/
 
int initcomm(baud,parity,dbits,sbits)
char parity;
int baud,dbits,sbits;
   /* this routine used by main thread */
{
struct {
  unsigned char databits;
  unsigned char parity;
  unsigned char stopbits;
  } linechar;
 
struct {
  unsigned char onmask;
  unsigned char offmask;
  } modemctrl;
 
struct {
  unsigned int wtime;
  unsigned int rtime;
  unsigned char flags1;
  unsigned char flags2;
  unsigned char flags3;
  unsigned char errchar;
  unsigned char brkchar;
  unsigned char xonchar;
  unsigned char xoffchar;
  } dcb;
 
int comerr,act;
 
 /* Set bitrate: */
 if (DosDevIOCtl(NULL,(char *)&baud,
                     0x41,01,com))
     {
     fprintf(stderr,"\nBitrate error");
     return(1);
     }
 
 /* Set databits, stopbits, parity: */
 if (parity == 'N') linechar.parity = 0;
 if (parity == 'O') linechar.parity = 1;
 if (parity == 'E') linechar.parity = 2;
 if (parity == 'M') linechar.parity = 3;
 if (parity == 'S') linechar.parity = 4;
 if (sbits == 2) linechar.stopbits = 2;
 if (sbits == 1) linechar.stopbits = 0;
 linechar.databits = dbits;
 if (DosDevIOCtl(NULL,(char *)&linechar,
                     0x42,01,com))
     {
     puts("Line characteristics error");
     return(1);
     }
 
 /* Set modem control signals: */
 modemctrl.onmask = 0x03;  /* DTR & RTS on */
 modemctrl.offmask = 0xff; /* nothing off */
 if (DosDevIOCtl((char *)&comerr,
       (char *)&modemctrl,0x46,01,com))
     {
     puts("Modem control error");
     return(1);
     }
 
 /* Set com device processing parameters:  */
 dcb.wtime = 100; /* 1sec transmit timeout */
 dcb.rtime = 100; /* 1sec receive  timeout */
 dcb.flags1 = 0x01;  /* enable DTR,        */
 dcb.flags2 = 0x40;  /* enable RTS, disable XON/XOFF  */
 dcb.flags3 = 0x04;  /* recv timeout mode  */
 dcb.errchar = 0x00; /* no error translate */
 dcb.brkchar = 0x00; /* no break translate */
 dcb.xonchar = 0x11;  /* standard XON  */
 dcb.xoffchar = 0x13; /* standard XOFF */
 if (DosDevIOCtl(NULL,(char *)&dcb,
                       0x53,01,com))
     {
     puts("Device control block error");
     return(1);
     }
 
 return(0);
}
 
int parsarg(arg,port,baud,parity,dbits,sbits)
char *arg,*port;
char *parity;
int *baud,*dbits,*sbits;
    /* this routine used by main thread */
{
  int strptr;
  char strhold[8];
  strupr(arg);   /* cvt to uppercase */
 
  /* Parse cmdline for COM port: */
  if ((strptr=strcspn(arg,":")) == 0)
                                  return(1);
  if (strptr > 8) return(1);
  strncpy(port,arg,strptr);
  *(port+strptr) = '\0';
  arg = arg+strptr+1;
 
  /* Parse for cmdline baudrate: */
  if ((strptr=strcspn(arg,",")) == 0)
                                  return(2);
  strncpy(strhold,arg,strptr);
  *(strhold+strptr) = '\0';
  *baud = atoi(strhold);
  if (*baud != 300 &&  *baud != 1200 &&
      *baud != 2400 && *baud != 4800 &&
      *baud != 9600 && *baud != 19200)
                                  return(2);
  arg = arg+strptr+1;
 
  /* Parse cmdline for parity: */
  if ((strptr = strcspn(arg,",")) == 0)
                                  return(3);
  *parity = *(arg+strptr-1);
  if (*parity != 'N' && *parity != 'O' &&
      *parity != 'E' && *parity != 'M' &&
      *parity != 'S')             return(3);
  arg = arg+strptr+1;
 
  /* Parse cmdline for databits: */
  if ((strptr = strcspn(arg,",")) == 0)
                                  return(4);
  *dbits = *(arg+strptr-1) - '0';
  if (*dbits != 5 &&   *dbits != 6 &&
      *dbits != 7 &&   *dbits != 8)
      return(4);
  arg = arg+strptr+1;
 
  /* Parse for stopbit value: */
  if ((strptr = strcspn(arg,",")) == 0)
                                  return(5);
  *sbits = *(arg+strptr-1) - '0';
  if (*sbits != 1 && *sbits != 2) return(5);
 
  return(0);
}
