/**************************************************************************
 *
 *  COMM32.C
 *
 *  This file contains the 32-bit routines which provide the basic
 *  serial support
 *
 *  This file was compiled with the IBM C Set/2 compiler using the
 *  following flags:
 *
 *    To produce a multi-thread version of the library
 *      /W3 /Ss /Gm /Gd- /Ge /O /Re /C /Kbcepr
 *
 *    To produce a single-thread version of the library
 *      /W3 /Ss /Gm- /Gd- /Ge /O /Re /C /Kbcepr
 *_________________________________________________________________________
 *
 *  Copyright (c) 1992 by ASH Software, Inc.
 *
 *  Update History
 *
 *    11/28/1992 - Module created
 *
 **************************************************************************/

#define INCL_COMMOS2_32BIT
#include "COMM.H"

/*------------------------------------------------------------------------*
 *  CommStartSystem
 *
 *  This routine initializes the comm library.  This routine must be
 *  called before other routines in the comm library and must specify the
 *  maximum number of ports to be handled by the library.
 *------------------------------------------------------------------------*/

unsigned long CommStartSystem(int iCommPorts)
{
unsigned long
  ulRC;

int
  iLoop;

if (iCommPorts <= 0)
  return(COMM_ERROR_INVALIDMAXPORTS);      // Invalid number of ports

iLoop=TRUE;

ulRC=DosCreateMutexSem(NULL,&hCommSem,0L,TRUE);
if (ulRC)
  return(COMM_ERROR_SEMAPHOREFAILED);

if (iMaxNumberCommPorts > 0)
  iLoop=FALSE;
else
  iMaxNumberCommPorts = iCommPorts;

if (!iLoop)
  {
  DosReleaseMutexSem(hCommSem);
  return(COMM_ERROR_ALREADYINITIALIZED);   // System already initialized
  }

pCommOS2ErrorCode=malloc(iCommPorts*sizeof(COMM_ERROR_CODE));
if (pCommOS2ErrorCode == NULL)
  {
  DosReleaseMutexSem(hCommSem);
  return(COMM_ERROR_SYSINITFAILED);
  }

//
// Initialize the COMM structure
//

for (iLoop=0; iLoop<iCommPorts; iLoop++)
  {
  pCommOS2ErrorCode[iLoop].ulCommErrorCode = COMM_ERROR_NOERROR;
  pCommOS2ErrorCode[iLoop].ulDosErrorCode  = NO_ERROR;
  pCommOS2ErrorCode[iLoop].hCommPort       = 0;
  pCommOS2ErrorCode[iLoop].ulCommMode      = COMM_MODE_NOTSTARTED;
  pCommOS2ErrorCode[iLoop].fPortOpen       = FALSE;
  pCommOS2ErrorCode[iLoop].fPortOpened     = FALSE;
  }

DosReleaseMutexSem(hCommSem);
return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommStopSystem
 *
 *  This system frees the resources used when initializing the comm
 *  library.  This should be called when the comm library is no longer
 *  needed or during program termination.
 *------------------------------------------------------------------------*/

unsigned long CommStopSystem(void)
{
int
  iLoop;

if (pCommOS2ErrorCode != NULL)
  {
  for (iLoop=0; iLoop<iMaxNumberCommPorts; iLoop++)
    CommClose(pCommOS2ErrorCode[iLoop].hCommPort);

  free(pCommOS2ErrorCode);
  }
iMaxNumberCommPorts = 0;

DosCloseMutexSem(hCommSem);
return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommGetLastError
 *
 *  This routine returns information concerning the last error for the
 *  requested comm port.
 *------------------------------------------------------------------------*/

unsigned long CommGetLastError(HFILE hCommPort,unsigned long *ulCommError,
  unsigned long *ulDosError,unsigned long *ulCommMode)
{
PCOMM_ERROR_CODE
  pCommError;

pCommError=CommGetErrorStruct(hCommPort);

if (pCommError == NULL)
  return(COMM_ERROR_PORTNOTFOUND);   // Unable to locate comm port

(*ulCommError) = pCommError->ulCommErrorCode;
(*ulDosError)  = pCommError->ulDosErrorCode;
(*ulCommMode)  = pCommError->ulCommMode;

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommOpen
 *
 *  This routine invokes DosOpen to open the specified serial comm port.
 *  The error return is the same as for DosOpen.
 *------------------------------------------------------------------------*/

unsigned long CommOpen(int iCommPort,HFILE *phCommPort)
{
int
  iLoop,
  fKeepLooping;

unsigned long
  ulAction,
  ulRC;

char
  cPortName[21];

PCOMM_ERROR_CODE
  pCommError;

//
// Find an empty location in the static structure.
// Use a Mutex semaphore to serialize access.
//

ulRC=DosRequestMutexSem(hCommSem,SEM_INDEFINITE_WAIT);
if (ulRC)
  return(COMM_ERROR_SEMAPHOREFAILED);

fKeepLooping=TRUE;
iLoop=0;
pCommError=NULL;
while(fKeepLooping)
  {
  if (pCommOS2ErrorCode[iLoop].fPortOpened)
    {
    iLoop++;
    if (iLoop == iMaxNumberCommPorts)
      fKeepLooping=FALSE;
    }
  else
    {
    pCommError=&pCommOS2ErrorCode[iLoop];
    fKeepLooping=FALSE;
    pCommError->fPortOpened=TRUE;
    }
  }
DosReleaseMutexSem(hCommSem);

if (pCommError == NULL)
  return(COMM_ERROR_MAXPORTSEXCEEDED);  // Max number of comm ports exceeded

sprintf(cPortName,"COM%d",iCommPort);
ulRC=(unsigned long)DosOpen(cPortName,phCommPort,&ulAction,0L,
     FILE_NORMAL,FILE_OPEN,
     OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE,
     NULL);

if (ulRC)
  {
  pCommError->ulCommErrorCode = COMM_ERROR_OPENFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = -1;
  pCommError->ulCommMode      = COMM_MODE_OPEN;
  return(COMM_ERROR_OPENFAILED);
  }
else
  {
  pCommError->ulCommErrorCode = COMM_ERROR_NOERROR;
  pCommError->ulDosErrorCode  = NO_ERROR;
  pCommError->hCommPort       = (*phCommPort);
  pCommError->ulCommMode      = COMM_MODE_OPEN;
  pCommError->fPortOpen       = TRUE;
  }

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommInit
 *
 *  This routine initializes a serial comm port by calling the
 *  DosDevIOCtl routine with the proper parameters.
 *------------------------------------------------------------------------*/

unsigned long CommInit(HFILE hCommPort,unsigned short usBaudRate,
         unsigned short usParity,unsigned short usDataBits,
         unsigned short usStopBits,unsigned short usTxBreak)
{
unsigned short
  usErrorCode;

unsigned long
  ulParmLength,
  ulDataLength,
  ulRC;

LINECONTROL
  COMParams;

PCOMM_ERROR_CODE
  pCommError;

//
// Set the baud rate
//

ulParmLength=sizeof(short);
ulRC=(unsigned long)DosDevIOCtl(hCommPort,IOCTL_ASYNC,ASYNC_SETBAUDRATE,
     &usBaudRate,ulParmLength,&ulParmLength,
     NULL,0L,NULL);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_BAUDFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_SETBAUD;
  return(COMM_ERROR_BAUDFAILED);
  }

//
// Set the number of bits, parity, stop bits, and Transmit break
//

COMParams.bDataBits  =(unsigned char)usDataBits;  // Set data bits
COMParams.bParity    =(unsigned char)usParity;    // Set parity
COMParams.bStopBits  =(unsigned char)usStopBits;  // Set stop bits
COMParams.fTransBreak=(unsigned char)usTxBreak;   // Set Transmit break
ulParmLength=sizeof(LINECONTROL);
  
ulRC=(unsigned long)DosDevIOCtl(hCommPort,IOCTL_ASYNC,ASYNC_SETLINECTRL,
     &COMParams,ulParmLength,&ulParmLength,
     NULL,0L,NULL);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_LINECNTRLFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_SETLINECNTRL;
  return(COMM_ERROR_LINECNTRLFAILED);
  }

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommClear
 *
 *  This routine clears both the transmit and receive buffers for the
 *  com port
 *------------------------------------------------------------------------*/

unsigned long CommClear(HFILE hCommPort)
{
unsigned long
  ulRC;

ulRC=CommClearTxBuffer(hCommPort);
if (!ulRC)
  ulRC=CommClearRxBuffer(hCommPort);

return(ulRC);
}

/*------------------------------------------------------------------------*
 *  CommClearTxBuffer
 *
 *  This routine clears the transmit buffer for the com port
 *------------------------------------------------------------------------*/

unsigned long CommClearTxBuffer(HFILE hCommPort)
{
unsigned char
  ucZero;

unsigned long
  ulRC,
  ulParmLength;

PCOMM_ERROR_CODE
  pCommError;

ucZero=0;
ulParmLength=sizeof(unsigned char);

ulRC=DosDevIOCtl(hCommPort,IOCTL_GENERAL,DEV_FLUSHOUTPUT,
     &ucZero,ulParmLength,&ulParmLength,
     NULL,0L,NULL);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_CLEARBUFFERFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_CLEARTXBUFFER;
  return(COMM_ERROR_CLEARBUFFERFAILED);
  }

return (COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommClearRxBuffer
 *
 *  This routine clears receive buffer for the com port
 *------------------------------------------------------------------------*/

unsigned long CommClearRxBuffer(HFILE hCommPort)
{
unsigned char
  ucZero;

unsigned long
  ulRC,
  ulParmLength;

PCOMM_ERROR_CODE
  pCommError;

ucZero=0;
ulParmLength=sizeof(unsigned char);

ulRC=DosDevIOCtl(hCommPort,IOCTL_GENERAL,DEV_FLUSHINPUT,
     &ucZero,ulParmLength,&ulParmLength,
     NULL,0L,NULL);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_CLEARBUFFERFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_CLEARRXBUFFER;
  return(COMM_ERROR_CLEARBUFFERFAILED);
  }

return (COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommClose
 *
 *  This routine invokes DosClose to close a serial comm port.
 *------------------------------------------------------------------------*/

unsigned long CommClose(HFILE hCommPort)
{
PCOMM_ERROR_CODE
  pCommError;

pCommError=CommGetErrorStruct(hCommPort);

if (pCommError == NULL)
  return(COMM_ERROR_PORTNOTFOUND);   // Unable to locate comm port

if (pCommError->fPortOpen)
  DosClose(hCommPort);

pCommError->ulCommErrorCode = COMM_ERROR_NOERROR;
pCommError->ulDosErrorCode  = NO_ERROR;
pCommError->hCommPort       = 0;
pCommError->ulCommMode      = COMM_MODE_NOTSTARTED;
pCommError->fPortOpen       = FALSE;
pCommError->fPortOpened     = FALSE;

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommWrite
 *
 *  This routine invokes DosWrite to write the data to the serial comm
 *  port.
 *------------------------------------------------------------------------*/

unsigned long CommWrite(HFILE hCommPort,unsigned char *pucDataArea,
         unsigned short usDataAreaSize,unsigned long *pulWritten)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=(unsigned long)DosWrite(hCommPort,pucDataArea,
     (unsigned long)usDataAreaSize,pulWritten);

if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_WRITEFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_WRITE;
  return(COMM_ERROR_WRITEFAILED);
  }

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommRead
 *
 *  This routine invokes DosRead to read the data from the serial comm
 *  port.  This routine reads whatever data is currently available from
 *  the port buffer.  It does not wait for information to arrive.
 *------------------------------------------------------------------------*/

unsigned long CommRead(HFILE hCommPort,unsigned char *pucDataArea,
         unsigned short usDataAreaSize,unsigned long *pulRead)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=(unsigned long)DosRead(hCommPort,pucDataArea,
     (unsigned long)usDataAreaSize,pulRead);

if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_READFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  pCommError->ulCommMode      = COMM_MODE_READ;
  return(COMM_ERROR_WRITEFAILED);
  }

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommReadTimeOut
 *
 *  This routine invokes DosRead and DosDevIOCtl to either read the
 *  specified number of characters or to return once the timeout specified
 *  has elapsed.
 *------------------------------------------------------------------------*/

unsigned long CommReadTimeOut(HFILE hCommPort,
         unsigned char *pucDataArea,unsigned short usDataAreaSize,
         unsigned long *pulRead,long lTimeOutmSec)
{
int
  fCheck,
  fKeepWaiting;

unsigned long
  ulStartTime,
  ulTimeOut,
  ulCharCount,
  ulSleepTime,
  ulReadRC,
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulStartTime=TimerValue();
ulTimeOut=lTimeOutmSec/10L;
ulSleepTime=1L;
fKeepWaiting=TRUE;
if (lTimeOutmSec == TIMEOUT_INDEFINITE)
  fCheck=FALSE;
else
  fCheck=TRUE;
ulReadRC=COMM_ERROR_NOERROR;
do
  {
  ulRC=CommQueryQueue(hCommPort,COMM_QUERY_RXCOUNT,&ulCharCount);
  if (ulRC)
    {
    pCommError=CommGetErrorStruct(hCommPort);
    pCommError->ulCommMode      = COMM_MODE_READTIMEOUT;
    return(COMM_ERROR_IOCTLFAILED);
    }

  if (ulCharCount >= usDataAreaSize) 
    {

    //
    // The number of characters requested are available.  Read the data
    // and return to the calling program.
    //

    fKeepWaiting=FALSE;
    ulReadRC=CommRead(hCommPort,pucDataArea,usDataAreaSize,pulRead);
    }
  else
    {

    //
    // The requested number of characters are not available at this
    // time.  Test to see if the specified time has elapsed and return
    // to the calling program if it has.
    //

    if (fCheck && (TimerDifference(ulStartTime) > ulTimeOut))
      {
      fKeepWaiting=FALSE;
      (*pulRead)=0;
      pCommError=CommGetErrorStruct(hCommPort);
      pCommError->ulCommErrorCode = COMM_ERROR_TIMEOUTEXCEEDED;
      pCommError->ulDosErrorCode  = NO_ERROR;
      pCommError->hCommPort       = hCommPort;
      pCommError->ulCommMode      = COMM_MODE_READTIMEOUT;
      ulReadRC=COMM_ERROR_TIMEOUTEXCEEDED;
      }
    }
  if (fKeepWaiting)
    {
    DosSleep(ulSleepTime);

    //
    // If port is not responding, slowly increase sleep interval.  Do
    // not exceed 256 (approximately 1/4 second).
    //

    if (ulSleepTime < 256L)
      ulSleepTime *= 2L;
    }
  } while (fKeepWaiting);

return(ulReadRC);
}

/*------------------------------------------------------------------------*
 *  CommReadUntilByte
 *
 *  This routine invokes DosRead and DosDevIOCtl to either read either the
 *  specified number of characters or to return once the character
 *  specified has been read.  The routine will also return if the timeout
 *  specified has elapsed.
 *------------------------------------------------------------------------*/

unsigned long CommReadUntilByte(HFILE hCommPort,
         unsigned char *pucDataArea,unsigned short usDataAreaSize,
         unsigned long *pulRead,unsigned char ucWaitByte,
         long lTimeOutmSec)
{
unsigned short
  usDataPosition;

unsigned char
  ucNextChar;

int
  fCheck,
  fKeepWaiting;

unsigned long
  ulStartTime,
  ulTimeOut,
  ulSleepTime,
  ulCharCount,
  ulReadRC,
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulStartTime=TimerValue();
ulTimeOut=lTimeOutmSec/10L;
ulSleepTime=1L;
fKeepWaiting=TRUE;
if (lTimeOutmSec == TIMEOUT_INDEFINITE)
  fCheck=FALSE;
else
  fCheck=TRUE;
ulReadRC=COMM_ERROR_NOERROR;
usDataPosition=0;
(*pulRead)=0;
do
  {
  ulRC=CommQueryQueue(hCommPort,COMM_QUERY_RXCOUNT,&ulCharCount);
  if (ulRC)
    {
    pCommError=CommGetErrorStruct(hCommPort);
    pCommError->ulCommMode      = COMM_MODE_READUNTILBYTE;
    return(COMM_ERROR_IOCTLFAILED);
    }

  if (ulCharCount > 0) 
    {

    //
    // Some characters are available.  Read them one by one checking
    // to see if the termination character is present or the maximum
    // number of characters have been read.
    //

    do
      {
      ulReadRC=CommRead(hCommPort,&ucNextChar,1,pulRead);
      if (ulReadRC)
        fKeepWaiting=FALSE;
      if (fKeepWaiting)
        {
        pucDataArea[usDataPosition]=ucNextChar;
        usDataPosition++;
        ulCharCount--;
        if (ucNextChar == ucWaitByte)
          fKeepWaiting=FALSE;
        if (usDataPosition >= usDataAreaSize)
          fKeepWaiting=FALSE;
        }
      } while (fKeepWaiting && (ulCharCount > 0));
    (*pulRead)=(unsigned long)usDataPosition;
    }
  else
    {

    //
    // The requested number of characters are not available at this
    // time.  Test to see if the specified time has elapsed and return
    // to the calling program if it has.
    //

    if (fCheck && (TimerDifference(ulStartTime) > ulTimeOut))
      {
      fKeepWaiting=FALSE;
      pCommError=CommGetErrorStruct(hCommPort);
      pCommError->ulCommErrorCode = COMM_ERROR_TIMEOUTEXCEEDED;
      pCommError->ulDosErrorCode  = NO_ERROR;
      pCommError->hCommPort       = hCommPort;
      pCommError->ulCommMode      = COMM_MODE_READUNTILBYTE;
      ulReadRC=COMM_ERROR_TIMEOUTEXCEEDED;
      }
    }
  if (fKeepWaiting)
    {
    DosSleep(ulSleepTime);

    //
    // If port is not responding, slowly increase sleep interval.  Do
    // not exceed 256 (approximately 1/4 second).
    //

    if (ulSleepTime < 256L)
      ulSleepTime *= 2L;
    }
  } while (fKeepWaiting);

return(ulReadRC);
}

/*------------------------------------------------------------------------*
 *  TimerValue
 *
 *  This routine returns a timer value in hundredths of seconds.  This
 *  routine does not return a value which can be used for the current time,
 *  but it does return a value useful for use as a timer.  If the value
 *  returned is less than a previous value, you should add 8,640,000 which
 *  is equivalent to one day in hundredths of seconds.
 *------------------------------------------------------------------------*/

static unsigned long TimerValue()
{
DATETIME
  CurrentTime;

DosGetDateTime(&CurrentTime);

return(360000L*(unsigned long)CurrentTime.hours+
         6000L*(unsigned long)CurrentTime.minutes+
          100L*(unsigned long)CurrentTime.seconds+
               (unsigned long)CurrentTime.hundredths);
}

/*------------------------------------------------------------------------*
 *  TimerDifference
 *
 *  This routine uses a base value obtained from TimerValue and calls 
 *  TimerValue itself to return the difference in hundreths of seconds.
 *------------------------------------------------------------------------*/

static unsigned long TimerDifference(unsigned long ulBaseTimerValue)
{
unsigned long
  ulCurrentTime;

ulCurrentTime=TimerValue();
if (ulCurrentTime < ulBaseTimerValue)
  ulCurrentTime += 8640000L;

return(ulCurrentTime-ulBaseTimerValue);
}                              

/*------------------------------------------------------------------------*
 *  CommGetErrorStruct
 *
 *  This is an internal routine which returns the address of the error
 *  structure for the requested comm port.
 *------------------------------------------------------------------------*/

static PCOMM_ERROR_CODE CommGetErrorStruct(HFILE hCommPort)
{
int
  iLoop,
  fKeepLooping;

PCOMM_ERROR_CODE
  pErrorStruct;

pErrorStruct=NULL;
fKeepLooping=TRUE;
iLoop=0;

while(fKeepLooping)
  {
  if ((pCommOS2ErrorCode[iLoop].hCommPort == hCommPort) &&
      (pCommOS2ErrorCode[iLoop].fPortOpened))
    {
    pErrorStruct=&pCommOS2ErrorCode[iLoop];
    fKeepLooping=FALSE;
    }
  else
    {
    iLoop++;
    if (iLoop == iMaxNumberCommPorts)
      fKeepLooping=FALSE;
    }
  }

return(pErrorStruct);
}

/*------------------------------------------------------------------------*
 *  CommQueryQueue
 *
 *  This routine queries the values of count and size for both the
 *  receive and transmit buffers.
 *------------------------------------------------------------------------*/

static unsigned long CommQueryQueue(HFILE hCommPort,short sMode,
  unsigned long *ulValue)
{
RXQUEUE
  RxQueue;

unsigned long
  ulRC,
  ulRxQueue,
  ulFunction;

PCOMM_ERROR_CODE
  pCommError;

switch (sMode)
  {
  case COMM_QUERY_RXCOUNT:  // Query Rx character count
  case COMM_QUERY_RXBUFFER: // Query Rx buffer size
    ulFunction=ASYNC_GETINQUECOUNT;
    break;

  case COMM_QUERY_TXCOUNT:  // Query Tx character count
  case COMM_QUERY_TXBUFFER: // Query Tx buffer size
    ulFunction=ASYNC_GETOUTQUECOUNT;
    break;
  }

ulRxQueue=sizeof(RXQUEUE);
(*ulValue)=0;
ulRC=(unsigned long)DosDevIOCtl(hCommPort,IOCTL_ASYNC,
  ulFunction,NULL,0L,NULL,
  &RxQueue,ulRxQueue,&ulRxQueue);

if (ulRC)
  {  
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommErrorCode = COMM_ERROR_IOCTLFAILED;
  pCommError->ulDosErrorCode  = ulRC;
  pCommError->hCommPort       = hCommPort;
  return(COMM_ERROR_IOCTLFAILED);
  }

switch (sMode)
  {
  case COMM_QUERY_RXCOUNT:  // Return Rx buffer count
  case COMM_QUERY_TXCOUNT:  // Return Tx buffer count
    (*ulValue)=RxQueue.cch;
    break;

  case COMM_QUERY_RXBUFFER: // Return Rx buffer size
  case COMM_QUERY_TXBUFFER: // Return Tx buffer size
    (*ulValue)=RxQueue.cb;
    break;
  }

return(COMM_ERROR_NOERROR);
}

/*------------------------------------------------------------------------*
 *  CommQueryRxCount
 *
 *  This routine returns the number of characters in the receiver buffer.
 *------------------------------------------------------------------------*/

unsigned long CommQueryRxCount(HFILE hCommPort,unsigned long *ulCount)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=CommQueryQueue(hCommPort,COMM_QUERY_RXCOUNT,ulCount);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommMode      = COMM_MODE_QUERYRXCOUNT;
  }

return ulRC;
}

/*------------------------------------------------------------------------*
 *  CommQueryRxBufSize
 *
 *  This routine returns the size of the receiver buffer.
 *------------------------------------------------------------------------*/

unsigned long CommQueryRxBufSize(HFILE hCommPort,unsigned long *ulSize)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=CommQueryQueue(hCommPort,COMM_QUERY_RXBUFFER,ulSize);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommMode      = COMM_MODE_QUERYRXBUFSIZE;
  }

return ulRC;
}

/*------------------------------------------------------------------------*
 *  CommQueryTxCount
 *
 *  This routine returns the number of characters in the transmit buffer.
 *------------------------------------------------------------------------*/

unsigned long CommQueryTxCount(HFILE hCommPort,unsigned long *ulCount)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=CommQueryQueue(hCommPort,COMM_QUERY_TXCOUNT,ulCount);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommMode      = COMM_MODE_QUERYTXCOUNT;
  }

return ulRC;
}

/*------------------------------------------------------------------------*
 *  CommQueryTxBufSize
 *
 *  This routine returns the size of the transmit buffer.
 *------------------------------------------------------------------------*/

unsigned long CommQueryTxBufSize(HFILE hCommPort,unsigned long *ulSize)
{
unsigned long
  ulRC;

PCOMM_ERROR_CODE
  pCommError;

ulRC=CommQueryQueue(hCommPort,COMM_QUERY_TXBUFFER,ulSize);
if (ulRC)
  {
  pCommError=CommGetErrorStruct(hCommPort);
  pCommError->ulCommMode      = COMM_MODE_QUERYTXBUFSIZE;
  }

return ulRC;
}

