/*****************************************************************************
  Name: lispsql.c

  Description: Lisp - ASI-SQL interface. This programm gets parameters
               from AutoLISP and pass them to ASI functions. Returned
               values are passed to AutoLISP.

  Author: Frumkin Alexander
          Autodesk, Inc.
          St. Peterburg, Russia.
          (812) 274-93-17

 ****************************************************************************

          lispsql.c
          Copyright (C) 1991-1992 by Autodesk, Inc.

          Permission to use, copy, modify, and distribute this software 
          for any purpose and without fee is hereby granted, provided 
          that the above copyright notice appears in all copies and that 
          both that copyright notice and this permission notice appear in 
          all supporting documentation.

          THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
          WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
          PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.
 *****************************************************************************

  Entry Points:

  Modification History:

  Bugs and restriction on use:

  Notes:

*****************************************************************************/

/****************************************************************************/
/*  INCLUDES */
/****************************************************************************/
#include <stdio.h>
#include <limits.h>
#ifndef UNIX
#include <stdlib.h>
#endif  /* UNIX */
#include <string.h>
#include <adslib.h>
#include "asi.h"

/****************************************************************************/
/*   TYPEDEF */
/****************************************************************************/
typedef struct {                      /* function name-code struct */
    char *cmdname;                    /* function name */
    void (*cmdfunc)();                /* function registrated code */
} CMD;

typedef struct {                      /* handle identifier struct */
        ASIHANDLEPTR pt;              /* handle pointer */
        short        NBh,             /* correspondent DBMS handle number */
                     NCh;             /* correspondent CONNECTION handle
                                         number */
} HENDLID;

/****************************************************************************/
/*   DEFINES */
/****************************************************************************/
#define S(x)                (x)->resval.rstring
#define I(x)                (x)->resval.rint
#define L(x)                (x)->resval.rlong
#define EN(x)               (x)->resval.rlname
#define R(x)                (x)->resval.rreal
#define PT(x)               (x)->resval.rpoint

#define STEP                 20     /* step for glabal handle identifier 
                                       array reallocation */ 
#define DRV_H                0      /* DBMS handle flag */
#define CON_H                1      /* CONNECTION handle flag */
#define COM_H                2      /* COMMUNICATION handle flag */
#define MAX_ACAD_PRINT       132    /* max length of string to be 
                                       printed in AutoCAD */ 
#define testHandlNumb(i)     ((i) >= 0 && (i) < n)
                                    /* whether i is number of handle from
                                       global array length n */ 
#define ELEMENTS(x)          (sizeof(x)/sizeof(x[0]))
                                    /* length of array */

/****************************************************************************/
/*   FUNCTION PROTOTYPES */
/****************************************************************************/
void *allocMem              _((unsigned int, size_t));
void *reallocMem            _((void *, size_t));
int handlType               _((int));
int eraseHandlId            _((short, int));
void termAll                _((void));
short searchDrvHandle       _((short));
short putHandlId            _((short, short));
short allocHandlid          _((void));
short handlidNumb           _((void));
struct resbuf *enttail      _((struct resbuf *));
int     rbcmp               _((struct resbuf *, struct resbuf *));
int     testHandle          _((short, int));
struct resbuf *rbColDsc     _((short, short));
int findName                _((short ,COLDSCPTR ,char *));
struct resbuf *getVal       _((short, short));
struct resbuf *getRowVal    _((short));
void print_sql_err          _((char *, int));
void lsp_initdrv            _((void));
void lsp_termdrv            _((void));
void lsp_termsql            _((void));
void lsp_lon                _((void));
void lsp_lof                _((void));
void lsp_ohdl               _((void));
void lsp_chdl               _((void));
void lsp_com                _((void));
void lsp_bnd                _((void));
void lsp_exe                _((void));
void lsp_cex                _((void));
void lsp_ftr                _((void));
void lsp_fbr                _((void));
void lsp_fbk                _((void));
void lsp_fet                _((void));
void lsp_del                _((void));
void lsp_upd                _((void));
void lsp_currow             _((void));
void lsp_rowqty             _((void));
void lsp_cds                _((void));
void lsp_colsdsc            _((void));
void lsp_cmt                _((void));
void lsp_rbk                _((void));
void lsp_cvl                _((void));
void lsp_stm                _((void));
void lsp_opr                _((void));
void lsp_err                _((void));
void lsp_errmsg             _((void));
void lsp_grow               _((void));
void lsp_gettable           _((void));

/****************************************************************************/
/*   GLOBAL VARIABLES */
/****************************************************************************/
HENDLID         *hid    = NULL;     /* global handle identifier array */
short           n       = 0;        /* length of global handle array */

static CMD cmd[]    =   {{"ASI_INITDRV", lsp_initdrv},
                         {"ASI_TERMDRV", lsp_termdrv},
                         {"ASI_TERMSQL", lsp_termsql},
                         {"ASI_LON", lsp_lon},
                         {"ASI_LOF", lsp_lof},
                         {"ASI_OHDL", lsp_ohdl},
                         {"ASI_CHDL", lsp_chdl},
                         {"ASI_COM", lsp_com},
                         {"ASI_BND", lsp_bnd},
                         {"ASI_EXE", lsp_exe},
                         {"ASI_CEX", lsp_cex},
                         {"ASI_FTR", lsp_ftr},
                         {"ASI_FBR", lsp_fbr},
                         {"ASI_FBK", lsp_fbk},
                         {"ASI_FET", lsp_fet},
                         {"ASI_DEL", lsp_del},
                         {"ASI_UPD", lsp_upd},
                         {"ASI_CURROW", lsp_currow},
                         {"ASI_ROWQTY", lsp_rowqty},
                         {"ASI_CDS", lsp_cds},
                         {"ASI_COLSDSC", lsp_colsdsc},
                         {"ASI_CMT", lsp_cmt},
                         {"ASI_RBK", lsp_rbk},
                         {"ASI_CVL", lsp_cvl},
                         {"ASI_STM", lsp_stm},
                         {"ASI_OPR", lsp_opr},
                         {"ASI_ERR", lsp_err},
                         {"ASI_ERRMSG", lsp_errmsg},
                         {"ASI_GROW", lsp_grow},
                         {"ASI_GETTABLE", lsp_gettable}};
                                    /* global function-code array */

/****************************************************************************/
/*.doc loadfuncs(internal)*/
/*+
    This function is called to load external functions into AutoLISP.
    It will returns FALSE if any function cannot be defined.
-*/
/****************************************************************************/
static int
/*FCN*/loadfuncs()
{
    short i = ELEMENTS(cmd);        /* length of function-code array 'cmd' */

    while (i--)                     /* for every declared function from cmd */
    if (ads_defun(cmd[i].cmdname, i) != RTNORM)
            return FALSE;           /* wrong function defenition */

    return TRUE;                    /* all functions defined right */

}                                   /* end of loadfuncs () */

/****************************************************************************/
/*.doc dofuncs(internal)*/
/*+
    This function is called to start external functions from AutoLISP.
-*/
/****************************************************************************/
static int
/*FCN*/dofuncs()
{
    short icommand = ads_getfuncode();
                                    /* function code */

    if (icommand == RTERROR)        ads_abort ("fatal error");
    (cmd[icommand].cmdfunc)();      /* executes function from cmd by code */
    return TRUE;

}                                   /* end of dofuncs() */

/****************************************************************************/
/*.doc main(external)*/
/*+
    Main entry point
-*/
/****************************************************************************/
void
/*FCN*/main(argc, argv)
  int argc;                         /* argument count */
  char *argv[];                     /* argument vector */
{
    short scode = RSRSLT;           /* return code */
    int stat;                       /* work variable */

    ads_init(argc, argv);           /* Initializes interface between AutoLISP 
                                       and this ADS application */ 

    for (;;) {
        if ((stat = ads_link(scode)) < 0)   
            return;                 /* wrong communication with AutoLISP */

        scode = RSRSLT;             /* default return code */

        switch (stat) {
        case RQXLOAD:               /* load function by (xload) command */  
            if ((scode = loadfuncs() ? RSRSLT : RSERR) == RSRSLT)
                asi_initsql();      /* SQL initialization */
            break;

        case RQSUBR:                /* invoke function */
            if (!dofuncs()) scode = RSERR;
            break;

        case RQEND:                 /* terminate options */
        case RQXUNLD:
        case RQQUIT:
            termAll();              /* free all handles */
            asi_termsql();          /* term SQL */
        }
    }
}                                   /* end of main () */

/****************************************************************************/
/*.doc allocMem (internal)*/
/*+
    Memory allocation.
-*/
/****************************************************************************/

void
/*FCN*/*allocMem (n, size)
  unsigned int    n;                /* number of elements */
  size_t          size;             /* size of element */
{
    void        *p;                 /* pointer to allocated memory */

    if ((p = calloc (n, size)) == NULL)     ads_abort ("Out of memory");

    return p;                       /* returns pointer to allocated memeory */

}                                   /* end of allocMem () */

/****************************************************************************/
/*.doc reallocMem(internal) */
/*+
    Memory reallocation.
-*/
/****************************************************************************/
void *
/*FCN*/reallocMem (p, size)
  void        *p;                   /* pointer to memory to be reallocated */
  size_t      size;                 /* new size for reallocation */
{
    if ((p = realloc (p, size)) == NULL)    ads_abort ("Out of memory");

    return p;                       /* returns pointer to allocated memeory */

}                                   /* end of reallocMem () */

/****************************************************************************/
/*.doc handlidNumb (internal) */
/*+
    Find new or use existing handler identifier number.
-*/
/****************************************************************************/
short
/*FCN*/handlidNumb ()
{
    short       i;                  /* work variable */                    

    /* 1. Check all allocated handle's identifiers for free handle in
          'hid' array */
    for (i = 0; i < n; i++)         
    if (hid[i].pt == NULL)   break; /* free identifyer detected */


    if (i == n)                    /* no free identifier detected */
    /* 2. New handle identifier allocation */
    i = allocHandlid();

    /* 3. New handle allocation for i-th identifier */
    hid[i].pt = (ASIHANDLEPTR)allocMem (1, sizeof(ASIHANDLE));

    return i;                       /* returns number of identifier in
                                       'hid' array */

}                                   /* end of handleidNumb () */

/****************************************************************************/
/*.doc allocHandlid (internal) */
/*+
    Global array 'hid' memory manager.
-*/
/****************************************************************************/
short
/*FCN*/allocHandlid ()
{
    static short length;            /* current 'hid' length */

    if (n == 0) {                   /* first 'hid' allocation */
        hid     = (HENDLID *)allocMem (STEP, sizeof(HENDLID));
        length  = STEP;             /* length of array 'hid' */
    }

    if (n == length) {              /* lenth is not enought */
        if (length > SHRT_MAX - STEP)  ads_abort ("Out of memory");
        length  += STEP;            /* new length of array 'hid' */
        hid     = (HENDLID *)reallocMem (hid, 
                        (unsigned int)(length*sizeof(HENDLID)));
    }

    hid[n].pt = NULL;

    return n++;                     /* returns next element of 'hid' number */

}                                   /* end of allocHandlid () */

/****************************************************************************/
/*.doc putHandlId(internal) */
/*+
    Puts handle identifier into global 'hid' array.
-*/
/****************************************************************************/
short
/*FCN*/putHandlId (NBase, NCon)
  short        NBase,               /* DBMS handle number */
               NCon;                /* CONNECTION handle number */
{
    short   i   = handlidNumb();    /* free handle identifier in 'hid'
                                       number */

    hid[i].NCh  = NCon;             /* write connection handle number to
                                       handle identifier */
    hid[i].NBh  = (NCon == SHRT_MAX)/* write correspondent DBMS handle
                                       number to identifier */
                    ? NBase
                    : hid[NCon].NBh;
    return i;                       /* returns current handle identifer
                                       number in 'hid' array */ 
}                                   /* end of putHandleId () */

/****************************************************************************/
/*.doc handlIType (internal) */
/*+
    Returns handl type.
-*/
/****************************************************************************/
int
/*FCN*/handlType (i)
  int         i;                    /* handle identifier number in 'hid'
                                       array */ 
{
    return (hid[i].NCh == SHRT_MAX) 
            ? ((hid[i].NBh == SHRT_MAX) ? DRV_H : CON_H)
            : COM_H;
}

/****************************************************************************/
/*.doc eraseHandlId (internal) */
/*+
    Free handle identifier and correspondent connection and communication
    handles identifiers in 'hid' array.
-*/
/****************************************************************************/

int     eraseHandlId (i, type)
  short             i;              /* handle number in 'hid' array */
  int               type;           /* handle type */
{
    int             j,              /* work variables */
                    ret = ASI_GOOD;
    int             htype;

    if (!testHandlNumb(i))  {       /* check handle number */
        ads_printf ("\nWrong handle number.");
        return ASI_BAD;
    }

    htype   = handlType(i);         /* check handle type */
    if (htype != type) {
        ads_printf ("\nWrong handle type.");
        return ASI_BAD;
    }

    if (type < COM_H)               /* whether there are correspondent
                                       CONNECTION and/or COMMUNICATION
                                       handles */
    for (j = 0; j < n && ret == ASI_GOOD; j++) {
                                    /* search in 'hid' array from
                                       start to end exept current */
        htype                           = SHRT_MAX;
        if (hid[j].pt != NULL && j != i)
        if (hid[j].NCh == i)            htype   = COM_H;
        else if(hid[j].NBh == i)        htype   = CON_H;
        if (htype != SHRT_MAX)      /* correspondent handle is found */
            ret &= eraseHandlId (j, htype);
    }


    switch (type) {
        case COM_H:                 /* closes communication handle */
        ret &= asi_chdl (hid[i].pt);
        break;

        case CON_H:                 /* logs off from data base */
        ret &= asi_lof (hid[i].pt);
        break;

        case DRV_H:
        if (searchDrvHandle (i) == SHRT_MAX)
                                    /* whether no more such DBMS handles
                                       detected */
        ret &= asi_termdrv (hid[i].pt);
    }

    if (hid[i].pt != NULL) {        /* wheter current handle not free */
        free(hid[i].pt);            /* free current handle */
        hid[i].pt   = NULL;         /* set current handle structure pointer 
                                       to NULL */
    }

    return ret;                     /* returns error code */
}                                   /* end of eraseHandleId () */

/****************************************************************************/
/*.doc termAll (internal) */
/*+
    Terms all handles.
-*/
/****************************************************************************/
void
/*FCN*/termAll()
{
    short   i;                      /* work index */

    for (i = 0; i < n; i++)         /* loop by all DBMS handle from 'hid'
                                       array */ 
    if (hid[i].pt != NULL && handlType (i) == DRV_H)
        eraseHandlId (i, DRV_H);    /* free DBMS handle and all correspondent
                                       handles */

    if (hid != NULL)    free (hid); /* free lobal 'hid' array */
    hid     = NULL;
    n       = 0;

}                                   /* end of termAll() */

/****************************************************************************/
/*.doc searchDrvHandle (internal) */
/*+
    Search DBMS handle with driver equal to current DBMS handle one.
-*/
/****************************************************************************/
short
/*FCN*/searchDrvHandle (current)
short               current;        /* current handle number in 'hid' array */
{
    short           i;              /* work index */

    for (i = 0; i < n; i++)         /* for all handle identifiers in 'hid' */
    if (i != current && 
        hid[i].pt->drv_id == hid[current].pt->drv_id) 
        break;                    /* there is any such handle */

    return i < n                  /* returns */
        ? i                       /* number of such handle */  
        : SHRT_MAX;
}                                 /* end of searchDrvHandle () */

/****************************************************************************/
/*.doc enttail (internal) */
/*+
    Search an tail of entity buffer.
-*/
/****************************************************************************/
struct resbuf *
/*FCN*/enttail (rchain)
  struct resbuf   *rchain;          /* pointer to current result buffer */
{
    while (rchain->rbnext != NULL)  /* go by buffer step by step */
    rchain = rchain->rbnext;

    return rchain;                  /* returns pointer to last group */
}                                   /* end of enttail() */

/****************************************************************************/
/*.doc rbColDsc (internal) */
/*+
    Builds column description result buffer.
-*/
/****************************************************************************/
struct resbuf *
/*FCN*/rbColDsc (nhandl, ncol)
  short             nhandl,         /* COMMUNICATION handle humber */
                    ncol;           /* column number */
{
    COLDSC          cd;             /* column description structure */
    struct resbuf   *rb = NULL;     /* work result buffer */

    if (asi_cds (hid[nhandl].pt, (int)ncol, &cd) == ASI_GOOD) {
                                    /* gets column description */
        rb = ads_buildlist (RTSTR, cd.colname,
                            RTSHORT, cd.precision,
                            RTSHORT, cd.scale,
                            NULL);
    }

    return rb;                      /* returns column description result 
                                       buffer */

}                                   /* end of rbColDsc () */

/****************************************************************************/
/*.doc findName(internal) */
/*+
    Search column by defined name.
-*/
/****************************************************************************/
int
/*FCN*/findName ( Hnumb, cd, Name)
  short           Hnumb;            /* COMMUNICATION handle humber */
  COLDSCPTR       cd;               /* column description */
  char            *Name;            /* column name */
{
    int         flag = FALSE,       /* find flag (off by default) */
                i;                  /* work index */

    for (i = 0; asi_cds (hid[Hnumb].pt, (int)i, cd) == ASI_GOOD; i++)
                                    /* loop by all columns */
    if (!stricmp (Name, cd->colname)) {
                                    /* names are equal */
        flag = TRUE;                /* flag on */
        break;                      /* stop search */
    }

    return flag;                    /* returns find flag */

}                                   /* end of findName () */

/****************************************************************************/
/*.doc rbcmp (internal) */
/*+
    Compares result buffers for length and types.
-*/
/****************************************************************************/
int
/*FCN*/rbcmp (rb, shab)
  struct resbuf   *rb,              /* result buffer to be checked */
                  *shab;            /* result buffer to be compared with */
{
    int         flag = TRUE;        /* continue flag */

    while (flag && shab != NULL)    /* loop by both buffers */
    if (rb == NULL &&               /* end of checked buffer */
        ads_printf ("\nerror: too few arguments.") == RTNORM ||
        rb->restype != shab->restype &&
                                    /* types mismatch */
        ads_printf ("\nerror: bad argument type.")== RTNORM)
        flag    = FALSE;            /* continue flag off */
    else {                          /* go to next groups */
        rb      = rb->rbnext;
        shab    = shab->rbnext;
    }

    if (flag && rb != NULL) {       /* end of second buffer */
        ads_printf ("\nerror: too many arguments.");
        flag = FALSE;
    }
    if (shab != NULL)       ads_relrb(shab);

    return flag;                    
}                                   /* end of rbcmp () */

/****************************************************************************/
/*.doc testHandle (internal) */
/*+
    Checks handle identifier in 'hid' array by number and type.
-*/
/****************************************************************************/
int
/*FCN*/testHandle (Numb, type)
  short       Numb;                 /* handle identifier number */
  int         type;                 /* handle type */
{
    int     ret = FALSE;            /* returns code (FALSE by default) */

    if (!testHandlNumb(Numb))       /*  check handle number  */
        ads_prompt ("\nerror: bad handle.");
    else if (handlType (Numb) != type)
                                    /* check handle type */
        ads_prompt ("\nerror: wrong handle type.");
    else ret = TRUE;

    return ret;

}                                   /* end of testHandle () */

/****************************************************************************/
/*.doc getVal (internal) */
/*+
    Returns value from Ncol column and current row as result buffer.
-*/
/****************************************************************************/
struct resbuf *
/*FCN*/getVal(Nhandle, NCol)
  short               Nhandle,      /* COMMUNICATION handle number */
                      NCol;         /* column number */
{
    struct resbuf   *rb = NULL;     /* work buffer */
    COLDSC          cd;             /* column description */
    int             type;           /* work variables */
    char            buff[256];

    if (asi_cds (hid[Nhandle].pt, (int)NCol, &cd) == ASI_GOOD &&
                                    /* gets column description */
        asi_cvl (hid[Nhandle].pt, (int)NCol, buff, cd.precision, &type)
        == ASI_GOOD)                /* gets column value */
        switch (type) {
        case ASI_SCHAR:             /* string value */
            buff[cd.precision] = '\0';
            rb = ads_buildlist (RTSTR, buff, NULL);
            break;

        case ASI_SNULL:             /* empty value */
            rb = ads_buildlist (RTSTR, "", NULL);
            break;

        case ASI_SINT:              /* integer value */
            rb = ads_buildlist(RTLONG, *(int *)buff, NULL);
            break;

        case ASI_SREAL:             /* float value */
            rb = ads_buildlist(RTREAL, *(ads_real *)buff, NULL);
        }

    return rb;                      /* returns value as result buffer */

}                                   /* end of getVal () */

/****************************************************************************/
/*.doc getRowVal (internal) */
/*+
    Gets values from current row as result buffer.
-*/
/****************************************************************************/
struct resbuf *
/*FCN*/getRowVal (Nhandle)
  short             Nhandle;        /* COMMUNICATION handle number in 'hid'
                                       array */
{
    struct resbuf   *root = NULL,   /* work pointers */
                    *rb;
    short           i;              /* work index */

    for (root = NULL, i = 0; (rb = getVal(Nhandle, i)) != NULL; i++)
                                    /* loop by all columns */
        if (i == 0)     root = rb;  /* first column */
        else            enttail(root)->rbnext = rb;
                                    /* next column */
    return root;                    /* returns pointer to first group */

}                                   /* end of getRowVal () */

/****************************************************************************/
/*.doc print_sql_err(internal) */
/*+
    Prints SQL statement and marcs error.
-*/
/****************************************************************************/
void print_sql_err (str, pos)
  char        *str;                 /* SQL statement */
  int         pos;                  /* error position */
{
    char    c   = '\0';             /* work variable */
    int     l   = (int)strlen (str);/* work variable */

    str[pos]    = ' ';              /* put ' ' to error position */

    while (l > 0) {                 /* devide str by parts of
                                       length MAX_ACAD_PRINT */
        if (l > MAX_ACAD_PRINT)  {  /* string is longer then MAX_ACAD_PRINT */
            c                   = str[MAX_ACAD_PRINT];
                                    /* csore symbol at MAX_ACAD_PRINT
                                       position in 'c' */
            str[MAX_ACAD_PRINT] = '\0';
                                    /* put end_of_line symbol to
                                       MAX_ACAD_PRINT position */
        }
        ads_printf ("%s", str);     /* prints part of str */
        l -= MAX_ACAD_PRINT;        /* recalculate length of unprinted part */
        str += MAX_ACAD_PRINT;      /* recalculate start of unprinted part */
        if (l > 0)              *str = c;
                                    /* put 'c' instead of end_of_file symbol */
    }
}                                   /* end of print_sql_err () */ 

/****************************************************************************/
/*.doc lsp_initdrv (external) */
/*+
    Driver initialization.
-*/
/****************************************************************************/
void
/*FCN*/lsp_initdrv ()
{
    short         i;                /* work variable */
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSTR);        /* for driver name */

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, rb1)) {          /* wheter arguments are right */
        i = putHandlId (SHRT_MAX, SHRT_MAX);
                                    /* put new DBMS handle to 'hid' */
        if (asi_initdrv (hid[i].pt, S(rb)) == ASI_GOOD)
                                    /* driver initialization */
            ads_retint(i);          /* returns handle number to AutoLISP */
    } 
}                                   /* end of lsp_initdrv () */

/****************************************************************************/
/*.doc lsp_termdrv (external) */
/*+
    Terminate driver and free all correspondent CONNECTION and
    COMMUNICATION drivers.
-*/
/****************************************************************************/
void
/*FCN*/lsp_termdrv ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb ,rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), DRV_H) &&/* whether handle number is right */
        eraseHandlId (I(rb), DRV_H) == ASI_GOOD)
                                    /* whether free successfull */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else   ads_retnil();            /* returns nil to AutoLISP */

}                                   /* end of lsp_termdrv () */

/****************************************************************************/
/*.doc lsp_termsql (external) */
/*+
    Term SQL and free all handles.
-*/
/****************************************************************************/
void
/*FCN*/lsp_termsql ()
{
    struct resbuf *rb;              /* result buffer to get input from
                                       AutoLISP */  

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, NULL)) {         /* wheter arguments are right */
        termAll();                  /* free all handles and 'hid' array */
        if (asi_termsql () == ASI_GOOD) 
                                    /* whether term successfull */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    }
}                                   /* end of lsp_termsql () */

/****************************************************************************/
/*.doc lsp_lon (external) */
/*+
    Logon to data base.
-*/
/****************************************************************************/
void
/*FCN*/lsp_lon ()
{
    short         i;                /* work variable */
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSTR, "",  /* for table name */
                        RTSTR, "",  /* for user name */
                        RTSTR, "",  /* for password */
                        NULL);

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), DRV_H)) {/* whether handle number is right */
        i = putHandlId (I(rb), SHRT_MAX);
                                    /* put new DBMS handle to 'hid' */
        if (asi_lon (hid[i].pt, hid[I(rb)].pt, S(rb->rbnext),
                     S(rb->rbnext->rbnext),
                     S(rb->rbnext->rbnext->rbnext)) == ASI_GOOD)
                                    /* lof on successful */
            ads_retint (i);         /* returns handle number to AutoLISP */
        else {                      /* logon to data base failed */
            free (hid[i].pt);       /* free new handle in 'hid' */
            hid[i].pt = NULL;
        }
    }
}                                   /* end of lsp_lon () */

/****************************************************************************/
/*.doc lsp_lof (external) */
/*+
    Log off from data base.
-*/

/****************************************************************************/
void
/*FCN*/lsp_lof ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), CON_H) &&/* whether handle number is right */
        eraseHandlId (I(rb), CON_H) == ASI_GOOD)
                                    /* whether free succsessfull */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else   ads_retnil();            /* returns nil to AutoLISP */

}                                   /* end of lsp_lof () */

/****************************************************************************/
/*.doc lsp_ohdl (external) */
/*+
    Open COMMUNICATION handle.
-*/
/****************************************************************************/
void
/*FCN*/lsp_ohdl ()
{
    short         i;                /* work variable */
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle(I(rb), CON_H)) { /* whether handle number is right */
        i = putHandlId (SHRT_MAX, I(rb));
                                    /* put new COMMUNICATION handle to 'hid' */
        if (asi_ohdl (hid[i].pt, hid[I(rb)].pt) == ASI_GOOD)
            ads_retint (i);         /* returns handle number to AutoLISP */
        else  {                     /* open handle failed */
            free(hid[i].pt);        /* free new handle in 'hid' */
            hid[i].pt = NULL;
        }
    }
}                                   /* end of lsp_ohdl () */

/****************************************************************************/
/*.doc lsp_chdl (external) */
/*+
    Close COMMUNICATION handle and free handle in 'hid' array.
-*/
/****************************************************************************/
void
/*FCN*/lsp_chdl ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle(I(rb), COM_H) && /* whether handle number is right */
        eraseHandlId (I(rb), COM_H) == ASI_GOOD)
                                    /* whether free successful */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else   ads_retnil();            /* returns nil to AutoLISP */

}                                   /* end of lsp_chdl () */


/****************************************************************************/
/*.doc lsp_com (external) */
/*+
    Compiles SQL statement.
-*/
/****************************************************************************/
void
/*FCN*/lsp_com ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSTR, "",  /* for SQL statement */
                        NULL);

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle(I(rb), COM_H))   /* whether handle number is right */
    if (asi_com (hid[I(rb)].pt, S(rb->rbnext)) == ASI_BAD) {
                                    /* compilation failed */
        ads_prompt ("\n");
        print_sql_err (S(rb->rbnext),asi_synerrpos(hid[I(rb)].pt));
                                    /* prints statement with arrow symbol
                                       att error position */
        ads_printf ("\n%s", asi_errmsg(hid[I(rb)].pt));
                                    /* prints error message */
    }  else ads_retint(1);          /* returns TRUE to AutoLISP */

}                                   /* end of lsp_com () */


/****************************************************************************/
/*.doc lsp_bnd (external) */
/*+
    Binds host variable into the SQL statement.
-*/
/****************************************************************************/
void
/*FCN*/lsp_bnd ()
{
    int             ret = ASI_BAD;  /*return code (ASI_BAD by default) */
    ads_real        rbuff;          /* work buffer */
    integer         ibuff;          /* work buffer */
    long            lbuff;          /* work buffer */
    struct resbuf   *rb,            /* result buffer to input from
                                       AutoLISP */
                    *rb1;           /* result buffer to compare input */


    rb  = ads_getargs();            /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSTR, "",  /* for host variable name */
                        RTSTR, "",  /* for value to be bind or '\0' for
                                       empty value */
                        RTSTR, "",  /* for value type */
                        RTSHORT, 0, /* for length */
                        NULL);

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H))  /* whether handle number is right */

    if (*S(rb->rbnext->rbnext) == '\0')
                                    /* empty value */
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), NULL,
                      I(rb->rbnext->rbnext->rbnext->rbnext), 0);
    else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hchar"))
                                    /* string value */
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), S(rb->rbnext->rbnext),
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HCHAR);
    else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hint")) {
                                    /* integer value */
        ibuff = atoi(S(rb->rbnext->rbnext));
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), &ibuff,
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HINT);
    } else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hshort")) {
                                    /* short value */
        ibuff = atoi(S(rb->rbnext->rbnext));
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), &ibuff,
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HSHORT);
    } else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hlong")) {
                                    /* long value */
        lbuff = (long)atoi(S(rb->rbnext->rbnext));
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), &lbuff,
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HLONG);
    } else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hreal")) {
                                    /* real value */
        rbuff = atof(S(rb->rbnext->rbnext));
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), &rbuff,
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HREAL);
    } else if (!stricmp(S(rb->rbnext->rbnext->rbnext), "asi_hfloat")) {
                                    /* float value */
        rbuff = atof(S(rb->rbnext->rbnext));
        ret = asi_bnd(hid[I(rb)].pt, S(rb->rbnext), &rbuff,
                      I(rb->rbnext->rbnext->rbnext->rbnext), ASI_HFLOAT);
    }

    if (ret == ASI_GOOD)            /* whether bind successful */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else                            /* bind failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /*  end of lsp_bnd () */

/****************************************************************************/
/*.doc lsp_exe (external) */
/*+
    Executes compiled SQL statement.
-*/
/****************************************************************************/

void lsp_exe ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H) &&/* whether handle number is right */
        asi_exe (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether compilation successful */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_exe () */

/****************************************************************************/
/*.doc lsp_cex (external) */
/*+
    Compiles and executes SQL statement.
-*/
/****************************************************************************/

void lsp_cex ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSTR, "",  /* for SQL statement */
                        NULL);

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle(I(rb), COM_H) && /* whether handle number is right */
        asi_cex (hid[I(rb)].pt, S(rb->rbnext)) == ASI_GOOD)
                                    /* whether compilation and execution OK */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else                            /* compilation or execution failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_cex () */

/****************************************************************************/
/*.doc lsp_fet (external) */
/*+
    Fetches forwards.
-*/
/****************************************************************************/

void
/*FCN*/lsp_fet ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H) &&/* whether handle number is right */
        asi_fet (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether fetch OK */
        ads_retint (1);             /* returns TRUE to AutoLISP */
    else                            /* fetch failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_fet () */

/****************************************************************************/
/*.doc lsp_fbk (external) */
/*+
    Fetches backs.
-*/
/****************************************************************************/

void
/*FCN*/lsp_fbk ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_fbk (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether fetch OK */
        ads_retint (1);             /* returns TRUE to AutoLISP */
    else                            /* fetch failed */ 
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_fbk () */

/****************************************************************************/
/*.doc lsp_ftr (external) */
/*+
    Fetches top row.
-*/
/****************************************************************************/

void
/*FCN*/lsp_ftr ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_ftr (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether fetch OK */
        ads_retint (1);             /* returns TRUE to AutoLISP */
    else                            /* fetch failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_ftr () */

/****************************************************************************/
/*.doc lsp_fbr (external) */
/*+
    Fetches bottom row.
-*/
/****************************************************************************/
void
/*FCN*/lsp_fbr ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_fbr (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether fetch OK */
        ads_retint (1);             /* returns TRUE to AutoLISP */
    else                            /* fetch failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_fbr () */

/****************************************************************************/
/* .doc lsp_del (extrnal) */
/*+
    Delates current row.
-*/
/****************************************************************************/
void
/*FCN*/lsp_del ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_del (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether delate OK */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else                            /* delate failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_del () */

/****************************************************************************/
/* .doc lsp_upd (extrnal) */
/*+
    Updates value in defined column at current row.
-*/
/****************************************************************************/

void
/*FCN*/lsp_upd ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */
    int           flag = ASI_BAD;   /* return value (ASI_BAD by default) */
    COLDSC        cd;               /* column description */
    ads_real      rbuff;            /* real buffer */
    integer       ibuff;            /* integer buffer */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSTR, "",  /* for column name */
                        RTSTR, "",  /* for new value or '\0' for NULL value */
                        NULL);

    if (rbcmp(rb, rb1) &&           /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        findName (I(rb), &cd, S(rb->rbnext)))
                                    /* whether there is column with 
                                       defined name */

    if (*S(rb->rbnext->rbnext) == '\0')
                                    /* NULL value */
        flag = asi_upd (hid[I(rb)].pt, cd.colname, 0, NULL, "");
    else
        switch (cd.type) {
            case ASI_CHAR:          /* string value */
            flag = asi_upd (hid[I(rb)].pt, cd.colname, ASI_HCHAR,
                            S(rb->rbnext->rbnext), "");
            break;

            case ASI_DECIMAL:
            case ASI_INTEGER:
            case ASI_SMALLINT:      /* integer value */
            ibuff = atoi(S(rb->rbnext->rbnext));
            flag = asi_upd (hid[I(rb)].pt, cd.colname, ASI_HINT, &ibuff,
                            "");
            break;

            case ASI_NUMERIC:
            case ASI_FLOAT:
            case ASI_REAL:
            case ASI_DOUBLE:        /* double value */
            rbuff = atof(S(rb->rbnext->rbnext));
            flag = asi_upd (hid[I(rb)].pt, cd.colname, ASI_HREAL, &rbuff,
                            "");
        }

    if (flag == ASI_GOOD)           /* update OK */
        ads_retint(1);              /* returns TRUE to AutoLISP */
    else                            /* update failed */
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_upd () */

/****************************************************************************/
/*.doc lsp_currow (external) */
/*+
    Returns current row number to AutoLISP as RTLONG.
-*/
/****************************************************************************/
void
/*FCN*/lsp_currow ()
{

    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H))  /* whether handle number is right */
        ads_retval(ads_buildlist(RTLONG, asi_currow (hid[I(rb)].pt), NULL));
                                    /* returns current row number to 
                                       AutoLISP */
    else 
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_currow */

/****************************************************************************/
/*.doc lsp_rowqty (external) */
/*+
    Returns total row number to AutoLISP as RTLONG.
-*/
/****************************************************************************/

void lsp_rowqty ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H))  /* whether handle number is right */
        ads_retval(ads_buildlist (RTLONG, asi_rowqty (hid[I(rb)].pt), NULL));
                                    /* Returns total row number to AutoLISP */
    else
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_rowqty */

/****************************************************************************/
/*.doc lsp_cds (external) */
/*+
    Returns column description as the result buffer to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_cds ()
{

    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSHORT, 0, /* for column number */
                        NULL);

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        (asi_stm (hid[I(rb)].pt) == ASI_CURSOR ||
         asi_stm (hid[I(rb)].pt) == ASI_UPDATE ||
         asi_stm (hid[I(rb)].pt) == ASI_DELETE ||
         asi_stm (hid[I(rb)].pt) == ASI_ALTER ||
         asi_stm (hid[I(rb)].pt) == ASI_INSERT) &&
                                    /* whether cursor operations */
        (rb1 = rbColDsc (I(rb), I(rb->rbnext))) != NULL)
        ads_retlist(rb1);           /* returns column description
                                       result buffer to AutoLISP */  
    else  ads_retnil();             /* returns nil to AutoLISP */

}                                   /* end of lsp_cds () */

/****************************************************************************/
/*.doc lsp_colsdsc (external) */
/*+
    Returns list of column descriptions as result buffer to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_colsdsc ()
{
    short         i;                /* work index */
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1,             /* result buffer to compare input */
                  *root,            /* work result buffers pointers */
                  *tail;

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        (asi_stm (hid[I(rb)].pt) == ASI_CURSOR ||
         asi_stm (hid[I(rb)].pt) == ASI_UPDATE ||
         asi_stm (hid[I(rb)].pt) == ASI_DELETE ||
         asi_stm (hid[I(rb)].pt) == ASI_ALTER ||
         asi_stm (hid[I(rb)].pt) == ASI_INSERT))
                                    /* whether cursor operations */
    for (root = NULL, i = 0; (rb1 = rbColDsc (I(rb), i)) != NULL; i++) {
                                    /* loop by all columns */
                   
        tail                    = ads_newrb (RTLB);
                                    /* open paren */
        tail->rbnext            = rb1;
                                    /* put current column description
                                       into result buffer */
        enttail(tail)->rbnext   = ads_newrb(RTLE);
                                    /* close paren */
        if (root == NULL)       root = tail;
                                    /* score root result buffer */
        else enttail(root)->rbnext = tail;
                                    /* put current result buffer to
                                       the tail */ 
    }

    if (root != NULL)      
        ads_retlist(root);          /* returns result buffer to AutoLISP */
    else              
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_coldsc () */

/****************************************************************************/
/*.doc lsp_cmt (external) */
/*+
    Saves changes into data base.
-*/
/****************************************************************************/
void
/*FCN*/lsp_cmt ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_cmt (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether save OK */
        ads_retint (1);             /* return TRUE to AutoLISP */
    else 
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_cmt () */

/****************************************************************************/
/*.doc lsp_rbk (external) */
/*+
    Rollback of all changes not yet commited to the data base.
-*/
/****************************************************************************/
void
/*FCN*/lsp_rbk ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H) &&/* whether handle number is right */
        asi_rbk (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether rollback OK */
        ads_retint (1);             /* returns TRUE to AutoLISP */
    else       
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_rbk () */

/****************************************************************************/
/*.doc lsp_cvl (external) */
/*+
    Returns value in defined column of current row to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_cvl ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_buildlist(RTSHORT, 0, /* for handle number */
                        RTSHORT, 0, /* for column number */
                        NULL);

    ads_retnil();                   /* returns nil to AutoLISP by default */
    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H) &&/* whether handle number is right */
        (rb1 = getVal(I(rb), I(rb->rbnext))) != NULL)
                                    /* whether value gets OK */
        ads_retval(rb1);            /* returns value to AutoLISP */

}                                   /* end of lsp_cvl () */

/****************************************************************************/
/*.doc lsp_stm (external) */
/*+
    Returns type of last operation to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_stm ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */
    char          state[10];

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    *state          = '/0';

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H))  /* whether handle number is right */
        switch (asi_stm (hid[I(rb)].pt)) {

        case ASI_DELETE:            /* delete statement */
            strcpy (state, "ASI_DELETE");
            break;

        case ASI_CREATE:            /* create statement */
            strcpy (state, "ASI_CREATE");
            break;

        case ASI_CURSOR:            /* cursor statement */
            strcpy (state, "ASI_CURSOR");
            break;

        case ASI_GRANT:             /* grant statement */
            strcpy (state, "ASI_GRANT");
            break;

        case ASI_REVOKE:            /* revoke statement */
            strcpy (state, "ASI_REVOKE");
            break;

        case ASI_INSERT:            /* insert statement */
            strcpy (state, "ASI_INSERT");
            break;

        case ASI_DROP:              /* drop statement */
            strcpy (state, "ASI_DROP");
            break;

        case ASI_UPDATE:            /* update statement */
            strcpy (state, "ASI_UPDATE");
            break;

        case ASI_ALTER:             /* alter statement */
            strcpy (state, "ASI_ALTER");
        }

    if (*state == '\0')        
        ads_retnil();               /* returns nil to AutoLISP */
    else                    
        ads_retstr(state);          /* returns state to AutoLISP */

}                                   /* end of lsp_stm () */


/****************************************************************************/
/*.doc lsp_opr (external) */
/*+
    Returns type of last operation to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_opr ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */
    char          state[10];

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    *state = '/0';

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb) ,COM_H))  /* whether handle number is right */
        switch (asi_opr (hid[I(rb)].pt)) {

        case ASI_COM:               /* compile operation */
            strcpy (state, "ASI_COM");
            break;

        case ASI_EXE:               /* execute operation */
            strcpy (state, "ASI_EXE");
            break;

        case ASI_DEL:               /* delete operation */
            strcpy (state, "ASI_DEL");
            break;

        case ASI_FET:               /* fetch forward operation */
            strcpy (state, "ASI_FET");
            break;

        case ASI_FBK:               /* fetch backward operation */
            strcpy (state, "ASI_FBK");
            break;

        case ASI_UPD:               /* update operation */
            strcpy (state, "ASI_UPD");
            break;

        case ASI_CMT:               /* commit operation */
            strcpy (state, "ASI_CMT");
            break;

        case ASI_RBK:               /* roll back changes operation */
            strcpy (state, "ASI_RBK");
            break;

        case ASI_FFR:               /* fetch top operation */
            strcpy (state, "ASI_FFR");
            break;

        case ASI_FLR:               /* fetch bottom operation */
            strcpy (state, "ASI_FLR");
        }

    if (*state == '\0')  
        ads_retnil();               /* returns nil to AutoLISP */
    else             
        ads_retstr(state);          /* returns operation name to AutoLISP */

}                                   /* end of lsp_opr () */

/****************************************************************************/
/*.doc lsp_err (external) */
/*+
    Returns error position in SQL statement to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_err ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandlNumb (I(rb)))      /* whether handle number is right */
        ads_retint (asi_err (hid[I(rb)].pt));
                                    /* returns error position in SQL
                                       statement to AutoLISP */
    else    
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_err () */

/****************************************************************************/
/*.doc lsp_errmsg (external) */
/*+
    Returns error message to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_errmsg ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandlNumb (I(rb)))      /* whether handle number is right */
        ads_retstr (asi_errmsg (hid[I(rb)].pt));
                                    /* returns error messge to AutoLISP */
    else
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_errmsg () */

/****************************************************************************/
/*.doc lsp_grow () */
/*+
    Returns current row values to AutoLISP as result buffers.
-*/
/****************************************************************************/
void
/*FCN*/lsp_grow ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1;             /* result buffer to compare input */

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H) &&/* whether handle number is right */
        (asi_stm (hid[I(rb)].pt) == ASI_CURSOR ||
         asi_stm (hid[I(rb)].pt) == ASI_UPDATE ||
         asi_stm (hid[I(rb)].pt) == ASI_DELETE ||
         asi_stm (hid[I(rb)].pt) == ASI_ALTER ||
         asi_stm (hid[I(rb)].pt) == ASI_INSERT) &&
                                    /* whether cursor operation */
        (rb1 = getRowVal(I(rb))) != NULL)
                                    /* whether gets values OK */
        ads_retlist(rb1);           /* returns current row values to AutoLISP
                                       as result buffers */
    else 
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_grow () */

/****************************************************************************/
/*.doc lsp_gettable () */
/*+
    Returns values of current cursor to AutoLISP.
-*/
/****************************************************************************/
void
/*FCN*/lsp_gettable ()
{
    struct resbuf *rb,              /* result buffer to get input from
                                       AutoLISP */  
                  *rb1,             /* result buffer to compare input */
                  *root = NULL,     /* work result buffers */
                  *tail;

    rb = ads_getargs();             /* gets arguments from AutoLISP
                                       as result buffer */
    rb1 = ads_newrb (RTSHORT);      /* for handle number */

    if (rbcmp (rb, rb1) &&          /* wheter arguments are right */
        testHandle (I(rb), COM_H) &&/* whether handle number is right */
        (asi_stm (hid[I(rb)].pt) == ASI_CURSOR ||
         asi_stm (hid[I(rb)].pt) == ASI_UPDATE ||
         asi_stm (hid[I(rb)].pt) == ASI_DELETE ||
         asi_stm (hid[I(rb)].pt) == ASI_ALTER ||
         asi_stm (hid[I(rb)].pt) == ASI_INSERT) &&
                                    /* whether cursor operation */
        asi_ftr (hid[I(rb)].pt) == ASI_GOOD)
                                    /* whether fetch top OK */

    do                              /* loop by all rows */
    if ((rb1 = getRowVal(I(rb))) != NULL) {
                                    /* get current row values */
        tail                    = ads_newrb (RTLB);
                                    /* open paren */
        tail->rbnext            = rb1;
                                    /* put current row buffer to list */
        enttail(tail)->rbnext   = ads_newrb(RTLE);
                                    /* close paren */
        if (root == NULL)       root = tail;
                                    /* score pointer to root group */
        else enttail(root)->rbnext  
                                = tail;
                                /* put next geroup to the tail of list */
    }
    while (asi_fet(hid[I(rb)].pt) == ASI_GOOD);

    if (root != NULL)       
        ads_retlist(root);          /* returns list to AutoLISP */
    else                    
        ads_retnil();               /* returns nil to AutoLISP */

}                                   /* end of lsp_gettable () */
/*EOF*/

