/*
    DXF.C  --  Parse a DXF file.
               This program segment is used to parse ASCII
               and binary DXF(tm) files produced by AutoCAD.
               The DXF file format as documented in the Release 10
               AutoCAD User Reference is supported by this code.
               In addition, supplementary routines for performing
               coordinate transformation are supported in ARBAXIS.C
               and are required by this program.

               This code currently supports parsing of the "ENTITIES" 
               section of DXF files only.

               This code is in the public domain, and has no implied
               guarantees of fitness for purpose, reliability, etc.

               This code has been tested with Borland's Turbo C(tm)
               and the Sun Workstation's C compiler.

               1988 Autodesk, Inc.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#ifdef unix
#include <malloc.h>
#endif

#define Boolean short
#define real double
#define TRUE  1
#define FALSE 0
#define EOS   0  

/*  Globals imported  */
extern void trans();

/*  Forward procedures  */

static Boolean inent(), ingroup(), getline();
static void dxferr(), dumpgroup(), getstring();
static char *alloc();
static void line(), circ(), pline(), vert(), seqend();

#ifdef mc68000
static void cvshort(), cvreal();
#endif

/*  Local variables  */

static FILE *fp;                   /* Input file pointer */
static char *tbfr;                 /* General purpose text buffer */
static char *gtext;                /* Group text value */
static char *tgroup[10];           /* Group store table for text */
static real *rgroup;               /* Group store table for reals */
static short *igroup;              /* Group store table for integers */
static real normal[3];             /* Extrusion direction array */

static int binary;                 /* Binary type DXF file flag */

static short gcode;                /* Group code */
static short gint;                 /* Group integer value */
static real greal;                 /* Group real value */

static char binsent[20];           /* Binary DXF file sentinel buffer */
static char ename[20];             /* Entity type name */
static real plsw, plew;            /* Polyline default start and end width */
static char stemp1[40];            /* String input temporary */

static Boolean errored;            /* Error occurred flag */

#ifdef unix
#define OPENMODE "r"
#else
#define OPENMODE "rb"              /* Most MS-DOS compilers */
#endif

/* Read a dxf file */
void readxf(fname)   
   char *fname;
{
        short i, j;
        unsigned long l;

        if ((fp = fopen(fname, "r")) == NULL) {
           fprintf(stderr, "Cannot open DXF file %s\n", fname);
           exit(1);
        }

        binsent[18] = 0;

        /*  Peek at the header to see if it's binary  */
        if ((fread(binsent, 1, 18, fp) != 18) ||
            (strcmp(binsent, "AutoCAD Binary DXF") != 0)) {
           rewind(fp);
           binary = FALSE;
        } else {
           /* This is a binary DXF file.  To insure correct processing
              of binary information we must close the file and re-open
              it with the binary I/O attribute. */
           fclose(fp);
           if ((fp = fopen(fname, OPENMODE)) == NULL) {
              fprintf(stderr, "Cannot open binary DXF file %s\n", fname);
              exit(1);
           }

           /* Skip past the header */

           fseek(fp, 22L, 0);
           binary = TRUE;
        }

        /*  Allocate dynamic storage to read file  */

        tbfr = (char *) alloc(200);
        gtext = (char *) alloc(200);
        for (i = 0; i < 10; i++)
           tgroup[i] = alloc(80);
        rgroup = (real *) alloc(60 * sizeof(real));
        igroup = (short *) alloc(80 * sizeof(short));

        /*  Ignore all of DXF file before ENTITIES section  */
        /*  This code could be modified to recognized other sections */

        errored = FALSE;
        while (!errored && ingroup()) {
           if (gcode == 0 && !strcmp(gtext, "SECTION")) {
              if (!ingroup()) {
                 dxferr();
                 break;
              }
              if (gcode != 2) {
                 dxferr();
              }
              if (!strcmp(gtext, "ENTITIES"))
                 break;
           }
        }

        /*  Now process entities and build entity items  */

        if (!errored && !ingroup())
           dxferr();
        while (!errored && inent()) {
           if (!strcmp(ename, "LINE")) {
                 line();
           } else if (!strcmp(ename, "POINT")) {
              printf("Point at %g,%g,%g\n", rgroup[10], rgroup[20], rgroup[30]);
           } else if (!strcmp(ename, "CIRCLE")) {
              circ();
           } else if (!strcmp(ename, "ARC")) {
              printf("Arc\n");
           } else if (!strcmp(ename, "TEXT")) {
              printf("Text\n");
           } else if (!strcmp(ename, "POLYLINE")) {
              pline();
           } else if (!strcmp(ename, "VERTEX")) {
              vert();
           } else if (!strcmp(ename, "SEQEND")) {
              seqend();
           } else
              printf("Ignored %s entity.\n", ename);
        }

        /* Release dynamic storage used whilst reading DXF file */

        free(tbfr);
        free(gtext);
        for (i = 0; i < 10; i++)
           free(tgroup[i]);
        free(rgroup);
        free(igroup);
        if (errored)
           fprintf(stderr, "\nError reading DXF file.\n");
        fclose(fp);
}


/*  INENT  --  Read in next entity  */

static short inent()
{
        short i;

        if (gcode != 0) {
           dxferr();
           return FALSE;
        }
        if (!strcmp(gtext, "ENDSEC"))
           return FALSE;           /* End of entity section */
        strcpy(ename, gtext);

        /* Supply defaults to fields  */

        igroup[62] = 0;
        rgroup[38] = 0.0;
        rgroup[39] = 0.0;
        normal[0] = normal[1] = normal[2] = 0.0;
        for (i = 0; i < 10; i++)
           tgroup[i][0] = EOS;
        if (!strcmp(ename, "TEXT")) {
           rgroup[50] = 0.0;
           rgroup[41] = 1.0;
           rgroup[51] = 0.0;
           igroup[71] = 0;
           igroup[72] = 0;
        } else if (!strcmp(ename, "SHAPE")) {
           rgroup[50] = 0.0;
           rgroup[40] = 1.0;
           rgroup[51] = 0.0;
        } else if (!strcmp(ename, "INSERT")) {
           igroup[66] = 0;
           rgroup[41] = 1.0;
           rgroup[42] = 1.0;
           rgroup[43] = 1.0;
           rgroup[50] = 0.0;
           igroup[70] = 1;
           igroup[71] = 1;
           rgroup[44] = 0.0;
           rgroup[45] = 0.0;
        } else if (!strcmp(ename, "ATTDEF")) {
           igroup[73] = 0;
           rgroup[50] = 0.0;
           rgroup[41] = 1.0;
           rgroup[51] = 0.0;
           igroup[71] = 0;
           igroup[72] = 0;
        } else if (!strcmp(ename, "ATTRIB")) {
           igroup[73] = 0;
           rgroup[50] = 0.0;
           rgroup[41] = 1.0;
           rgroup[51] = 0.0;
           igroup[71] = 0;
           igroup[72] = 0;
        } else if (!strcmp(ename, "POLYLINE")) {
           igroup[70] = 0;
           rgroup[40] = 0.0;
           rgroup[41] = 0.0;
        }  else if (!strcmp(ename, "VERTEX")) {
           rgroup[40] = plsw;
           rgroup[41] = plew;
           rgroup[42] = 0.0;
           igroup[70] = 0;
           rgroup[50] = 0.0;
        }

        while (TRUE) {
           if (!ingroup()) {
              dxferr();
              break;
           }
           if (gcode == 0)
              break;
           if (gcode < 10)
              strncpy(tgroup[gcode], gtext, 80); /* text */
           else if (gcode < 60)                  /* reals */
              rgroup[gcode] = greal;
           else if (gcode >= 210 && gcode < 240) /* extrusion dirs */
              normal[gcode / 10 - 21] = greal;
           else if (gcode >= 60 && gcode < 80)   /* ints */
              igroup[gcode] = gint;
        }   
     if (!strcmp(ename, "POLYLINE")) {
           plsw = rgroup[40];
           plew = rgroup[41];
        }
        return TRUE;
}

/*  INGROUP  --  Read in group from DXF file  */

static short ingroup()
{

    /* If we're reading a binary file */

    if (binary) {

        if (!gettype(&gcode)) {
           /* End of file */
           return FALSE;
        }
        if (gcode < 0) {
           errored = TRUE;
           return FALSE;
        }
        if (gcode < 10)
           getstring(gtext);
        else if (gcode < 60)
           getreal(&greal);
        else if (gcode >= 210 && gcode < 240)
           getreal(&greal);
        else if (gcode >= 60 && gcode < 80)
           getshort(&gint);
        else {              /* unknown gcode */
           errored = TRUE;
           return FALSE;
        }

        return TRUE;

    } else {

    /* We're reading an ASCII DXF file */

        if (getline()) {
           if (sscanf(tbfr, "%hd", &gcode) != 1) {
              errored = TRUE;
              return FALSE;
           }
           if (gcode < 0) {
              errored = TRUE;
              return FALSE;
           }
           if (!getline()) {
              errored = TRUE;
              return FALSE;
           }
           if (gcode < 10) {
              strcpy(gtext, tbfr);
           } else if (gcode < 60) {
              if (sscanf(tbfr, "%lf", &greal) != 1) {
                 errored = TRUE;
                 return FALSE;
              }
           } else if (gcode >= 210 && gcode < 240) {
              if (sscanf(tbfr, "%lf", &greal) != 1) {
                 errored = TRUE;
                 return FALSE;
              }
           } else if (gcode >= 60 && gcode < 80) {
              if (sscanf(tbfr, "%hd", &gint) != 1) {
                 errored = TRUE;
                 return FALSE;
              }
           } else if (gcode == 999)
              fprintf (stderr, tbfr);   /* comment line  */
           else {                       /* unknown gcode */
              errored = TRUE;
              return FALSE;
           }
           return TRUE;
        }
        return FALSE;              /* End of file */
    }
}

/*  DXFERR  --  Diagnose unspecific error in DXF file  */

static void dxferr()
{
        errored = TRUE;
        fprintf(stderr, "\nError in DXF file!\n");
        dumpgroup();
}

/*  DUMPGROUP  --  Dump current group for debugging  */

static void dumpgroup()
{
        fprintf(stderr, "\ngcode = %2d: ", gcode);
        if (gcode < 10)
           fprintf(stderr, "%s", gtext);
        else if (gcode < 60)
           fprintf(stderr, "%9.4f", greal);
        else if (gcode >=60 && gcode < 80)
           fprintf(stderr, "%hd", gint);
        else if (gcode >= 210 && gcode < 240)
           fprintf(stderr, "%9.4f", greal);
        else if (gcode == 999)
           fprintf(stderr, "%s", gtext);
        else
           fprintf(stderr, "Unknown type");
}

/*  GETLINE  --  Obtain next line from input file  */

static Boolean getline()
{
        if (fgets(tbfr, 132, fp)) {
           tbfr[strlen(tbfr) - 1] = EOS;
           return TRUE;
        }
        return FALSE;
}

/*  GETTYPE  --  Get the identifier (gcode) for the data item */

static int gettype(type)
  short *type;
{
   unsigned char c;

   if (fread(&c, 1, 1, fp)) {
      *type = (short) c;
      return TRUE;
   }

   return FALSE;
}

/*  GETSTRING  --  Get a string from the binary file */

static void getstring(str)
  char *str;
{
   char *s;

   s = str;
   while (*(s++) = getc(fp))
        ;
}

/*  GETSHORT  --  Get an integer (in 8086 order!) from the binary file */

static int getshort(ptr)
  short *ptr;
{
   if (fread(ptr, 2, 1, fp)) {
#ifdef mc68000
      cvshort(ptr, *ptr);
#endif
      return TRUE;
   }

   return FALSE;
}

/*  GETREAL  --  Get a real (double) from the binary file in IEEE */

static int getreal(ptr)
  real *ptr;
{
   if (fread(ptr, 8, 1, fp)) {
#ifdef mc68000
      cvreal(ptr);
#endif
      return TRUE;
   }

   return FALSE;
}


#ifdef mc68000

/*  CVSHORT  --  Reorder bytes of a short  */

static void cvshort(cp, u)
char *cp;
unsigned short u;
{
	*cp++ = u & 0xFF;
	*cp = u >> 8;
}

/*  CVREAL  --  Reorder byte of a real */

static void cvreal(rbuf)
  unsigned char *rbuf;
{
   register int i;
   unsigned char ptr[8];

	ptr[0] = rbuf[7];
	ptr[1] = rbuf[6];
	ptr[2] = rbuf[5];
	ptr[3] = rbuf[4];
	ptr[4] = rbuf[3];
	ptr[5] = rbuf[2];
	ptr[6] = rbuf[1];
	ptr[7] = rbuf[0];
        for(i=0;i<8;i++) rbuf[i] = ptr[i];
}
#endif /* mc68000 */

static char *
alloc(size)
  unsigned size;
{
   char *ptr;
   register int i;

   if ((ptr = (char *) malloc(size)) == NULL) {
      printf("Out of memory\n");
      exit(1);
   }
   for (i=0; i<size; i++)
      ptr[i] = 0;
   return ptr;
}
/* ------------- below this line lie your local routines ----- */

static double pnorm[3];
static int inpoly;


/*  Main calling routine */

void main(argc, argv)
char *argv[];
int argc;
{
   char fname[20];

   if (argc < 2) {
      printf("usage: dxf filename\n");
      exit(1);
   }

   strcpy(fname, argv[1]);
   strcat(fname, ".dxf");

   readxf(fname);

}

static void line ()
{

   printf("\nLINE: x1=%g y1=%g z1=%g\n", rgroup[10], rgroup[20], rgroup[30]);
   printf("      x2=%g y2=%g z2=%g\n", rgroup[11], rgroup[21], rgroup[31]);

}

static void pline ()
{
   int i;

   printf("\nPolyline: %s", (igroup[70] & 1) ? "Closed" : "Open");
   printf("\n          normal dir = %g %g %g",
          normal[0], normal[1], normal[2]);
   printf("\n          extrusion distance = %g", rgroup[39]);

   for (i=0;i<3;i++) pnorm[i] = normal[i];
   inpoly = TRUE;
}

static void vert ()
{
   double pt[3];

   pt[0] = rgroup[10]; pt[1] = rgroup[20]; pt[2] = rgroup[30];

   if (!normtest(pnorm))
      trans(pt, pt, pnorm);

   printf("\nVertex: x = %g, y = %g, z = %g",
          pt[0], pt[1], pt[2]);
   if (rgroup[42] > 0.0)
      printf("        arc, bulge = %g", rgroup[42]);

}



static void circ()
{
   double pt[3];

   pt[0] = rgroup[10]; pt[1] = rgroup[20]; pt[2] = rgroup[30];

   if (!normtest(normal))
      trans(pt, pt, normal);

   printf("\nCircle (cylinder): center at x = %g y = %g z = %g",
          pt[0], pt[1], pt[2]);
   printf(" radius = %g", rgroup[40]);
   printf("\n                   normal dir = %g %g %g",
          normal[0], normal[1], normal[2]);
   printf("\n                   height = %g", rgroup[39]);

}

static void seqend ()
{
   if (inpoly) {
     inpoly = FALSE;
     printf("\nEnd of Polyline.\n");
   } else
     printf("\nEnd of Attribute.\n");

}

/*  NORMTEST  --  Test a normal to see if it's been set.
                  Note that this uses an ABSOLUTE comparison
                  to 0.0; normally a naughty thing, but with
                  this code it's guaranteed to be 0.0 if no
                  (210,220,230) group existed.  */

static int normtest (norm)
real *norm;
{

   return (norm[0] == 0.0 && norm[1] == 0.0 && norm[2] == 0.0);

}
