// Code for converting BMPs to strings and strings to Bitmaps.
//     (C) 1993, Balboa Technical Services.
//     
//  
//

// Header files.
#include <windows.h>
#include "str2bmp.h"
#pragma hdrstop

// DESCRIPTION:
//     DLL Initialization code.
// PARAMETERS:
//     Windows stuff.
// RETURN VALUE:
//     1 if succesfully initialization, else 0.
int FAR PASCAL LibMain (HANDLE hInstance, WORD wDataSeg, WORD cbHeapSize, LPSTR lpCmdLine)
 { return 1;
 }

// PARAMETERS: Windows stuff.
// RETURN VALUE: 1 if succesfully initialization, else 0.
//int FAR PASCAL WEP (int nParameter)
// { // Dll removal successful.
//   return 1;
// }

//    DESCRIPTION:
//        Function for converting VB bitmap to string.
//    PARAMETERS:
//        hdc : Picture.hdc
//	       himage : Picture.image
//	       string : VB string. Must be large enough to hold data. See
//			             Bmp_GetSize for more info.
//    RETURN VALUE:
//        Number of pixels in the bitmap.
//
DLL_LONG BmpToString( HDC hdc, HBITMAP hImage, STRINGPTR string)
 { // Local variables.
   WORD bw, bh;	       // Byte width and byte height.
   HBITMAP hOldBitmap;

   // Get information about bitmap size.
   BITMAP bm;
   GetObject( hImage, sizeof( BITMAP), (LPSTR) &bm);

   // Select VB's picture...
   hOldBitmap = (HBITMAP) SelectObject( hdc, hImage);

   // Store the width and height in a header at beginning of string
   StringBmp_SetDimensions( string, bm.bmWidth, bm.bmHeight, bm.bmWidthBytes);

   // Copy bitmap data into string.
   GetBitmapBits( hImage, bm.bmWidthBytes * bm.bmHeight, string+sizeof( SB));

   // DeSelect VB's picture.
   SelectObject( hdc, hOldBitmap);

   // Return size in pixels of the bitmap.
   return (bw*bh);
 }

//    DESCRIPTION:
//       Function for converting string to a bitmap.
//    PARAMETERS:
//	      string : VB string. Must contain valid bitmap info created using
//			            BmpToString
//       hdcSrc : VB Source Picture.hdc
//	      hbmpSrc : VB Source Picture.image
//    RETURN VALUE:
//       ON SUCCESS: Number of pixels in the image.
//       ON FAILURE: -1 if string does not contain valid bitmap info.
//
DLL_LONG StringToBmp( STRINGPTR string, HDC hdcSrc, HBITMAP hbmpSrc)
 { // Local variables.
   HBITMAP hOldBitmapNew;
   HBITMAP hOldBitmapSrc;
   HBITMAP hNewBitmap;
   HDC hdcNew;
   WORD bw, bh, bwb;
   int ret;

   // Make sure parameter is bitmap string.
   //   Return -1 if string does not contain valid bitmap info.
   if( (ret = StringBmp_ValidateStringBmp( string))<0)
    { return ret;                   
    }

   // Get width and height from string header.
   bw = StringBmp_GetWidth(string);
   bh = StringBmp_GetHeight(string);
   bwb = StringBmp_GetWidthBytes(string);
      
   // Create a new bitmap.
   hdcNew = CreateCompatibleDC( hdcSrc);
   hOldBitmapSrc = (HBITMAP) SelectObject( hdcSrc, hbmpSrc);
   hNewBitmap = CreateCompatibleBitmap( hdcSrc, bw, bh);

   // Select the new bitmap.
   hOldBitmapNew = (HBITMAP) SelectObject( hdcNew, hNewBitmap);

   // Copy bitmap data from string.
   SetBitmapBits( hNewBitmap, bwb*bh, string+sizeof( SB));

   // Copy the new bitmap to VB's
   BitBlt( hdcSrc, 0,0,bw,bh, hdcNew, 0,0, SRCCOPY);

   // DeSelect bitmaps.
   SelectObject( hdcSrc, hOldBitmapSrc);
   SelectObject( hdcNew, hOldBitmapNew);

   // Delete the new bitmap.
   DeleteObject( hNewBitmap);
   DeleteDC( hdcNew);

   // Return bitmap size in pixels.
   return (bwb*bh);
 }

//    DESCRIPTION:
//        Return required size in bytes for a string to hold a bitmap.
//    PARAMETERS:
//	       hImage : VB Picture.image
//    RETURN VALUE:
//        ON SUCCESS: Number of bytes to initialize a string size. Use the
//			   following code in VB:
//			     Dim bytes as long, a$
//			     bytes = Bmp_GetSize( Picture1.Image)
//			     if bytes = 0 then stop ' Error! Picture too big.
//			     a$ = spaces( bytes) ' Initialize string size.
//        ON FAILURE: 0 if bitmap has more than 32000 pixels.
//
DLL_LONG Bmp_GetSize( HBITMAP hImage)
 { // Local variables.
   HBITMAP hOldBitmap;
   DWORD bwB, hB;

   // Get information about bitmap size.
   BITMAP bm;
   GetObject( hImage, sizeof( BITMAP), (LPSTR) &bm);

   bwB = bm.bmWidthBytes;
   hB = bm.bmHeight;


   // Check for error condition
   if( (bwB * hB + sizeof( SB) )>32000L)
    { return 0;
    }

   // Calculate size of string
   return( sizeof( SB) + hB * bwB);

 }

//    DESCRIPTION:
//        Returns the height in pixels of a bitmap.
//    PARAMETERS:
//	       hImage : VB Picture.image
//    RETURN VALUE:
//	       Height in pixels.
//
DLL_LONG Bmp_GetHeight( HBITMAP hImage)
 { // Get information about bitmap size.
   BITMAP bm;
   GetObject( hImage, sizeof( BITMAP), (LPSTR) &bm);
   return ( bm.bmHeight);
 }

//    DESCRIPTION:
//        Returns the width in pixels of a bitmap.
//    PARAMETERS:
//	       hImage : VB Picture.image
//    RETURN VALUE:
//	       Width in pixels.
//
DLL_LONG Bmp_GetWidth( HBITMAP hImage)
 { // Get information about bitmap size.
   BITMAP bm;
   GetObject( hImage, sizeof( BITMAP), (LPSTR) &bm);
   return( bm.bmWidthBytes);
 }


//    DESCRIPTION:
//        Store the dimensions of the bitmap in a header at the beginning
//              of the string.
//    PARAMETERS:
//	       A : VB string
//        w : width of VB bitmap to be stored in the string.
//        h : height of VB bitmap to be stored in the string.
//        wb : width in bytes (always even) of VB bitmap to be stored.
//    RETURN VALUE:
//        ON SUCCESS: 1
//        ON FAILURE: -1 = Illegal format bitmap string.
DLL_INT StringBmp_SetDimensions( STRINGPTR A, int w, int h, int wb)
 { union StringBmpType temp;
   temp.vbstring = A;
   temp.stringbmp->identifier = StringBmpIdentifier;
   temp.stringbmp->width = w;
   temp.stringbmp->height = h;
   temp.stringbmp->widthbytes = wb;
   return 1;
 }


//    DESCRIPTION:
//        Examine the header of a string containing bitmap info and
//              determine the bitmap width.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//        width in pixels of the stored bitmap.
DLL_INT StringBmp_GetWidth( STRINGPTR A)
 { union StringBmpType temp;
   temp.vbstring = A;
   return temp.stringbmp->width;
 }

//    DESCRIPTION:
//        Examine the header of a string containing bitmap info and
//              determine the bitmap height.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//        Height in pixels of the stored bitmap.
DLL_INT StringBmp_GetHeight( LPSTR A)
 { union StringBmpType temp;
   temp.vbstring = A;
   return temp.stringbmp->height;
 }

//    DESCRIPTION:
//        Examine the header of a string containing bitmap info and
//              determine the bitmap width in bytes. (Always even).
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//        Width in bytes of the stored bitmap.
DLL_INT StringBmp_GetWidthBytes( LPSTR A)
 { union StringBmpType temp;
   temp.vbstring = A;
   return temp.stringbmp->widthbytes;
  }

//    DESCRIPTION:
//        Examine the header of a string containing bitmap info and
//              return the "identifier word". This word is used to check
//              whether the string contains valid bitmap info.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//        Identifier word. Should be 'SB' (0x5342)
DLL_INT StringBmp_GetIdentifier( STRINGPTR A)
 { union StringBmpType temp;
   temp.vbstring = A;
   return temp.stringbmp->identifier;
  }

//    DESCRIPTION:
//        Examine the header of a string containing bitmap info and
//              determine whether the bitmap contains valid bitmap info.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//        ON SUCCESS: Return 0.
//        ON FAILURE: -1 = invalid bitmap info.
DLL_LONG StringBmp_ValidateStringBmp( STRINGPTR A)
 { int wA, hA, wbA;

   // Check bitmap A:
   if (A != NULL)
    { if (StringBmp_GetIdentifier( A) != StringBmpIdentifier)
       { return -1;
       }
      wA = StringBmp_GetWidth( A);
      hA = StringBmp_GetHeight( A);
      wbA = StringBmp_GetWidthBytes( A);
      if( (wA== -1)||(hA==-1)||(wbA==-1) )
       { return -1; // Illegal format bitmap string.
       }
    }

   // Success! The string contains a valid StringBmp.
   return 0;
 }

//    DESCRIPTION:
//        Examine the header of two strings containing bitmap info and
//              make sure they are the same size.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//	       B : VB string containing bitmap info.
//    RETURN VALUE:
//        ON SUCCESS: 0 = bitmaps are the same size.
//        ON FAILURE: -2 = Bitmaps are not the same size.
DLL_LONG StringBmp_CheckSameSize( STRINGPTR A, STRINGPTR B)
 { int wA, hA, wbA;
   int wB, hB, wbB;

   // Check bitmap A:
   wA = StringBmp_GetWidth( A);
   hA = StringBmp_GetHeight( A);
   wbA = StringBmp_GetWidthBytes( A);

   // Get size of bitmap B.
   wB = StringBmp_GetWidth( B);
   hB = StringBmp_GetHeight( B);
   wbB = StringBmp_GetWidthBytes( B);
   

   // Make sure both bitmaps the same size.
   if ( !( (wbA == wbB) && (hA == hB) && (wA == wB) ) )
    { return -2; // Bitmaps not the same size.
    }

   // Everything OK.
   return 0;
 }

//    DESCRIPTION:
//        Return number of pixels differing between two equal sized bitmaps.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//	       B : VB string containing bitmap info.
//    RETURN VALUE:
//      ON SUCCESS: Return # of pixels differing (zero if no difference).
//      ON FAILURE: -1 = Illegal format bitmap string.
//                  -2 = Bitmaps not the same width/height.
DLL_LONG StringBmp_COMPARE( STRINGPTR A, STRINGPTR B)
 { // Local variables.
   int wA, hA, wbA;
   int h,w, ret;
   DWORD DifferenceCount;
   LPSTR ptrA, ptrB, ptrRESULT, ptrLINEA, ptrLINEB;

   // Validate parameters.
   if( (ret = StringBmp_ValidateStringBmp( A))<0)
    { return ret;
    }
   if( (ret = StringBmp_ValidateStringBmp( B))<0)
    { return ret;
    }

   // Make sure both are the same size
   if( (ret = StringBmp_CheckSameSize( A, B))<0)
    { return ret;
    }

   // Get bitmap dimensions.
   wA = StringBmp_GetWidth( A);
   hA = StringBmp_GetHeight( A);
   wbA = StringBmp_GetWidthBytes( A);


   // Set Pointers to beginning of data in string.
   ptrLINEA = A + sizeof( SB);
   ptrLINEB = B + sizeof( SB);

   // Compare the strings.
   DifferenceCount = 0L; 
   for( h=0; h<hA; h++)
    { ptrA = ptrLINEA;
      ptrB = ptrLINEB;
      ptrLINEA += wbA;
      ptrLINEB += wbA;
      for( w=0; w<wA; w++)
       { if (*(ptrA++) !=  *(ptrB++))
          { DifferenceCount++;   // Count # of deltas.
          }
       }
    }

   return DifferenceCount;
 }


//    DESCRIPTION:
//        For each pixel in strings A and B, perform an OR function and
//           store the result in string RESULT. A and B must be the same size.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//	       B : VB string containing bitmap info.
//    RETURN VALUE:
//      ON SUCCESS: Return # of white pixels in the result string.
//      ON FAILURE: -1 = Illegal format bitmap string.
//                  -2 = Bitmaps not the same width/height.
DLL_LONG StringBmp_OR( STRINGPTR A, STRINGPTR B, STRINGPTR RESULT)
 { // Local variables.
   int wbA, hA, wA;
   int i, ret, h, w;
   char d;
   DWORD WhiteCount;
   LPSTR ptrA, ptrB, ptrRESULT;
   LPSTR ptrLINEA, ptrLINEB, ptrLINERESULT;

   // Validate parameters.
   if( (ret = StringBmp_ValidateStringBmp( A))<0)
    { return ret;
    }
   if( (ret = StringBmp_ValidateStringBmp( B))<0)
    { return ret;
    }

   // Make sure both are the same size
   if( (ret = StringBmp_CheckSameSize( A, B))<0)
    { return ret;
    }

   // Set the header for the result.
   StringBmp_SetDimensions( RESULT, StringBmp_GetWidth(A), StringBmp_GetHeight(A), StringBmp_GetWidthBytes(A));

   // Get bitmap dimensions.
   wA = StringBmp_GetWidth( A);
   hA = StringBmp_GetHeight( A);
   wbA = StringBmp_GetWidthBytes( A);


   // Set Pointers to beginning of data in string.
   ptrLINEA = A + sizeof( SB);
   ptrLINEB = B + sizeof( SB);
   ptrLINERESULT = RESULT + sizeof( SB);

   // OR the strings.
   WhiteCount = 0; // Count the # of BLACK results.
   for( h=0; h<hA; h++)
    { ptrA = ptrLINEA;
      ptrB = ptrLINEB;
      ptrRESULT = ptrLINERESULT;
      ptrLINEA += wbA;
      ptrLINEB += wbA;
      ptrLINERESULT += wbA;
      for( w=0; w<wA; w++)
       { if ( WHITE ==(*(ptrRESULT++) = (*(ptrA++) & *(ptrB++))))
          { WhiteCount++;
          }
       }

    }
   return WhiteCount;
 }

//    DESCRIPTION:
//        For each pixel in strings A and B, perform an AND function and
//           store the result in string RESULT. A and B must be the same size.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//	       B : VB string containing bitmap info.
//    RETURN VALUE:
//      ON SUCCESS: Return # of black pixels in the result string.
//      ON FAILURE: -1 = Illegal format bitmap string.
//                  -2 = Bitmaps not the same width/height.
DLL_LONG StringBmp_AND( STRINGPTR A, STRINGPTR B, STRINGPTR RESULT)
 { // Local variables.
   int wbA, hA, wA;
   int i, ret, h, w;
   unsigned char d;
   DWORD BlackCount;
   LPSTR ptrA, ptrB, ptrRESULT;
   LPSTR ptrLINEA, ptrLINEB, ptrLINERESULT;

   // Validate parameters.
   if( (ret = StringBmp_ValidateStringBmp( A))<0)
    { return ret;
    }
   if( (ret = StringBmp_ValidateStringBmp( B))<0)
    { return ret;
    }

   // Make sure both are the same size
   if( (ret = StringBmp_CheckSameSize( A, B))<0)
    { return ret;
    }

   // Set the header for the result.
   StringBmp_SetDimensions( RESULT, StringBmp_GetWidth(A), StringBmp_GetHeight(A), StringBmp_GetWidthBytes(A));

   // Get bitmap dimensions.
   wA = StringBmp_GetWidth( A);
   hA = StringBmp_GetHeight( A);
   wbA = StringBmp_GetWidthBytes( A);

   // Set Pointers to beginning of data in string.
   ptrLINEA = A + sizeof( SB);
   ptrLINEB = B + sizeof( SB);
   ptrLINERESULT = RESULT + sizeof( SB);

   // OR the strings.
   BlackCount = 0; // Count the # of BLACK results.
   for( h=0; h<hA; h++)
    { ptrA = ptrLINEA;
      ptrLINEA += wbA;

      ptrB = ptrLINEB;
      ptrLINEB += wbA;

      ptrRESULT = ptrLINERESULT;
      ptrLINERESULT += wbA;

      for( w=0; w<wA; w++)
       { if( BLACK == (*(ptrRESULT++) = (*(ptrA++) | *(ptrB++))))
          { BlackCount++;
          }
       }
    }
   return BlackCount;
 }

//    DESCRIPTION:
//        For each pixel in string A, perform a NOT function and
//           store the result in string RESULT.
//    PARAMETERS:
//	       A : VB string containing bitmap info.
//    RETURN VALUE:
//      ON SUCCESS: Return # of pixels in A.
//      ON FAILURE: -1 = Illegal format bitmap string.
//                  -2 = Bitmaps not the same width/height.
DLL_LONG StringBmp_INVERT( LPSTR A, LPSTR RESULT)
 { // Local variables.
   int wA, hA;
   int ret;
   LONG i, size;
   LPSTR ptrA, ptrRESULT;

   // Validate parameters.
   if( (ret = StringBmp_ValidateStringBmp( A))<0) 
    { return ret;
    }

   // Set the header for the result.
   StringBmp_SetDimensions( RESULT, StringBmp_GetWidth(A), StringBmp_GetHeight(A), StringBmp_GetWidthBytes(A));

   // Calculate size in bytes.
   size = StringBmp_GetWidthBytes(A)*StringBmp_GetHeight(A); 

   // Set Pointers to beginning of data in string.
   ptrA = A + sizeof( SB);
   ptrRESULT = RESULT + sizeof( SB);

   // Invert the string.
   for( i=0; i<size; i++)
    { *(ptrRESULT++) = ~*(ptrA++);
    }

   return (size);
 }

