Option Explicit

Sub Main ()

    Dim sCurrent As String
    Dim sUser As String
    
    'set the error branch
    On Error GoTo MainError

    'init the interactive flag
    ngInter = False

    'show the status box (label) on the
    'frmMain form
    frmMain.Show

    'update the status box
    frmMain!lblStatus = "Gathering INI update target information..."
    frmMain!lblStatus.Refresh

    'gather target info
    If Not nGetAppInfo() Then
	Error 32767
    End If

    'update the status box
    frmMain!lblStatus = "Gathering current INI setting..."
    frmMain!lblStatus.Refresh

    'get the current value
    If sgPrivate Then
	sCurrent = sPGetINI(sgFileName, sgApp, sgKey, "")
    Else
	sCurrent = sGetINI(sgApp, sgKey, "")
    End If
    
    'update the status box
    frmMain!lblStatus = "Verifying current INI setting..."
    frmMain!lblStatus.Refresh

    'is ini already correct
    If sCurrent = sgDefault Then
	'ini matches...not blank
	'nothing to update
	End
    Else
	'make sure default isn't blank...
	If sgDefault = "" Then
	    'update the status box
	    frmMain!lblStatus = "Deleting specified INI entry..."
	    frmMain!lblStatus.Refresh
	    
	    If sgPrivate Then
		If Not nPDeleteINI(sgFileName, sgApp, sgKey) Then Error 32766
	    Else
		If Not nDeleteINI(sgApp, sgKey) Then Error 32766
	    End If
	Else
	    'update the status box
	    frmMain!lblStatus = "Updating specified INI entry..."
	    frmMain!lblStatus.Refresh
    
	    If sgPrivate Then
		If Not nPWriteINI(sgFileName, sgApp, sgKey, sgDefault) Then Error 32765
	    Else
		If Not nWriteINI(sgApp, sgKey, sgDefault) Then Error 32765
	    End If
	End If
	
    End If

    'if we got this far we updated the ini...let's log it
    'update the status box
    frmMain!lblStatus = "Updating log file..."
    frmMain!lblStatus.Refresh

    'get the user name
    sUser = Environ$("USER")

    'update the ini
    If Not nLogIt(sUser & " UPDATED AT " & Format$(Now, "General Date")) Then
	Error 32764
    End If

    'update the status box
    frmMain!lblStatus = "Done!"
    frmMain!lblStatus.Refresh

    'end the app
    End
    
    Exit Sub

MainError:

    Select Case Err
	Case 32767
	    MsgBox "Could not gather target INI information.", MB_ICONSTOP, "INI Update"
	Case 32766
	    If ngInter Then
		MsgBox "Could not delete target INI information.", MB_ICONSTOP, "INI Update"
	    End If
	Case 32765
	    If ngInter Then
		MsgBox "Could not update target INI information.", MB_ICONSTOP, "INI Update"
	    End If
	Case 32764
	    If ngInter Then
		MsgBox "Could not log update information.", MB_ICONSTOP, "INI Update"
	    End If
	Case Else
	    If ngInter Then
		MsgBox "Unknown error occurred :" & CStr(Err), MB_ICONSTOP, "INI Update"
	    End If
    End Select
    
    'end the app
    End

End Sub

Function nDeleteINI (ByVal lpsAppName As String, ByVal lpsKeyName As String) As Integer
    '
    'deletes a win.ini entry specified by...
    '
    'lpsAppName=the target win.ini application name
    'lpsKeyname=the target win.ini key name
    '
    'returns...
    'True=if ini entry deleted
    'False=if error occured during delete
    
    'setup the error stuff
    On Error GoTo nDeleteINIError
    
    'delete the entry...pass a null string
    nDeleteINI = WriteProfileString(lpsAppName, lpsKeyName, 0&)
    
    'return true...
    nDeleteINI = True

    Exit Function

nDeleteINIError:

    'return false...something bad happened
    nDeleteINI = False

End Function

Function nGetAppInfo () As Integer
    '
    'reads the iniupd.ini file to get the application specific
    'info...
    '
    'returns...
    'true=if all goes well
    'false=if something happened while reading the iniupd.ini file

    Dim nX As Integer

    'setup error branch
    On Error GoTo AppError

    'get the interactive flag
    ngInter = nPGetINI(app.Path & "\INIUPD.INI", "General", "Interactive", 0)

    'get the target ini file
    sgFileName = Trim$(sPGetINI(app.Path & "\INIUPD.INI", "Update_Info", "File", ""))

    'make sure we got something
    If sgFileName = "" Then
	Error 32767
    Else
	'is this the win.ini
	If UCase$(Right$(sgFileName, 7)) = "WIN.INI" Then
	    'yes...the win.ini is the target
	    sgPrivate = False
	Else
	    'no...we're targeting a private ini file
	    sgPrivate = True
	End If
    
	'add path if necessary
	nX = InStr(sgFileName, "\")
	If nX = 0 Then
	    'add the current windows path to file name
	    sgFileName = sGetWindowsDir() & sgFileName
	End If
    End If

    'get the target application (section)
    sgApp = Trim$(sPGetINI(app.Path & "\INIUPD.INI", "Update_Info", "Application", ""))

    'make sure we got something
    If sgApp = "" Then
	Error 32767
    End If

    'get the target keyname (left side of x=x entry)
    sgKey = Trim$(sPGetINI(app.Path & "\INIUPD.INI", "Update_Info", "Key", ""))

    'make sure we got something
    If sgKey = "" Then
	Error 32767
    End If

    'get the target value (right side of x=x entry)
    'no need to verify... if it's blank we delete the entry
    sgDefault = Trim$(sPGetINI(app.Path & "\INIUPD.INI", "Update_Info", "Default", ""))

    'get the appls log file
    sgLogFile = Trim$(sPGetINI(app.Path & "\INIUPD.INI", "Log_File", "FileName", ""))

    'make sure we got something
    If sgLogFile = "" Then
	Error 32767
    End If

    'set good return
    nGetAppInfo = True
    
    Exit Function

AppError:

    'set bad return
    nGetAppInfo = False

End Function

Function nGetINI (ByVal lpsgAppName As String, ByVal lpsKeyName As String, nDefault As Integer) As Integer
    '
    'get's an integer value from the win.ini file for the specified...
    'lpsAppName=target application in the win.ini file
    'lpsKeyName=target keyname in the win.ini file
    'nDefault=the default of the target key name
    '
    'returns...
    'no problems=the integer value found
    'problems=0
    
    'here's the error handler stuff
    On Error GoTo nGetINIError
    
    'do it to it
    nGetINI = GetProfileInt(lpsgAppName, lpsKeyName, nDefault)
    
    Exit Function

nGetINIError:
    
    'set bad return code
    nGetINI = 0
    
    Exit Function

End Function

Function nLogIt (ByVal sString As String) As Integer
    '
    'logs the specified string in the application logfile
    '
    'sString=string to log in the log file
    '
    'returns...
    'true=if the log worked
    'false=if the log failed for some reason

    Dim nFile As Integer

    'setup error branch
    On Error GoTo LogError

    'get next available file handle
    nFile = FreeFile

    'open the log file to append the sString
    Open app.Path & "\" & sgLogFile For Append As #nFile

    'add the sString to the file
    Print #nFile, sString

    'close the file
    Close #nFile

    'set good return
    nLogIt = True

    Exit Function

LogError:

    'something happened...set bad return
    nLogIt = False

End Function

Function nPDeleteINI (ByVal lpsFileName As String, ByVal lpsAppName As String, ByVal lpsKeyName As String) As Integer
    '
    'deletes a ini entry specified by...
    '
    'lpsFileName=the target private ini file
    'lpsAppName=the target ini application name
    'lpsKeyname=the target ini key name
    '
    'returns...
    'True=if ini entry deleted
    'False=if error occured during delete

    'setup the error stuff
    On Error GoTo nPDeleteINIError
   
    'delete it...pass a null string
    nPDeleteINI = WritePrivateProfileString(lpsAppName, lpsKeyName, 0&, lpsFileName)
    
    'set good return
    nPDeleteINI = True
    
    Exit Function

nPDeleteINIError:

    'set bad return
    nPDeleteINI = False

End Function

Function nPGetINI (ByVal lpsFileName As String, ByVal lpsAppName As String, ByVal lpsKeyName As String, nDefault As Integer) As Integer
    '
    'get's an integer value from the win.ini file for the specified...
    'lpsFileName=target private ini file
    'lpsAppName=target application in the ini file
    'lpsKeyName=target keyname in the ini file
    'nDefault=the default of the target key name
    '
    'returns...
    'no problems=the integer value found
    'problems=0
    
    'here's the error handler stuff
    On Error GoTo nPGetINIError
    
    'do it to it
    nPGetINI = GetPrivateProfileInt(lpsAppName, lpsKeyName, nDefault, lpsFileName)
    
    Exit Function

nPGetINIError:

    'set bad return code
    nPGetINI = 0

End Function

Function nPWriteINI (ByVal lpsFileName As String, ByVal lpsAppName As String, ByVal lpsKeyName As String, ByVal lpsAppString As String) As Integer
    '
    'writes the specified value to the target ini...
    'lpsFileName=target private ini file
    'lpsAppName=target application name in the ini file
    'lpsKeyName=target key name in the ini file
    'lpsAppString=the value to store
    '
    'returns
    'true=no problems
    'false=something happened while trying to write the value

    'setup the error stuff
    On Error GoTo nPWriteINIError
   
    'do it
    nPWriteINI = WritePrivateProfileString(lpsAppName, lpsKeyName, lpsAppString, lpsFileName)
    
    'set good return value
    nPWriteINI = True
    
    Exit Function

nPWriteINIError:

    'set bad return value
    nPWriteINI = False

End Function

Function nWriteINI (ByVal lpsAppName As String, ByVal lpsKeyName As String, ByVal lpsAppString As String) As Integer
    '
    'writes the specified value to the win.ini...
    'lpsAppName=target application name in the win.ini file
    'lpsKeyName=target key name in the win.ini file
    'lpsAppString=the value to store
    '
    'returns
    'true=no problems
    'false=something happened while trying to write the value
    
    'setup the error stuff
    On Error GoTo nWriteINIError
    
    'doit
    nWriteINI = WriteProfileString(lpsAppName, lpsKeyName, lpsAppString)
    
    'set good return value
    nWriteINI = True

    Exit Function

nWriteINIError:

    'set bad return value
    nWriteINI = False

End Function

Function sGetINI (ByVal lpsAppName As String, ByVal lpsKeyName As String, ByVal lpsDefault As String) As String
    '
    'gets the specified value from the win.ini file
    'lpsAppName=target application name in the win.ini file
    'lpsKeyName=target key name in the win.ini file
    'lpsDefault=the default entry if nothing found
    '
    'returns...
    'good=the found value or the passed lpsDefault
    'bad=empty string (i.e. "")

    Dim nSize As Integer
    Dim nValid As Integer
    Dim lpsReturnString As String
    
    'here's the error handler stuff
    On Error GoTo sGetINIError
    
    'need to initialize the return string
    lpsReturnString = Space$(255)
    
    'initialize the size of the return string
    nSize = Len(lpsReturnString)
    
    'do it to it
    nValid = GetProfileString(lpsAppName, lpsKeyName, lpsDefault, lpsReturnString, nSize)
    
    'Discard the trailing spaces and null character...
    sGetINI = Left$(lpsReturnString, nValid)
    
    Exit Function

sGetINIError:

    'set bad return
    sGetINI = ""

End Function

Function sGetWindowsDir () As String
    '
    'gets the windows directory...
    'sGetWindowsDir=the windows directory string upon return

    Dim sTemp As String
    Dim nX As Integer

    'size the buffer
    sTemp = String$(145, 0)

    'call api
    nX = GetWindowsDirectory(sTemp, 145)
    
    'trim string
    sTemp = Left$(sTemp, nX)

    'check the path
    If Right$(sTemp, 1) <> "\" Then

	'set the return value
	sGetWindowsDir = sTemp + "\"
    Else
	
	'set the return value
	sGetWindowsDir = sTemp
    End If

End Function

Function sPGetINI (ByVal lpsFileName As String, ByVal lpsAppName As String, ByVal lpsKeyName As String, ByVal lpsDefault As String) As String
    '
    'gets the specified value from the target private ini file
    'lpsFileName=target private ini file
    'lpsAppName=target application name in the ini file
    'lpsKeyName=target key name in the ini file
    'lpsDefault=the default entry if nothing found
    '
    'returns...
    'good=the found value or the passed lpsDefault
    'bad=empty string (i.e. "")
    
    Dim nSize As Integer
    Dim nValid As Integer
    Dim lpsReturnString As String
    
    'here's the error handler stuff
    On Error GoTo sPGetINIError

    'need to initialize the return string
    lpsReturnString = Space$(255)
    
    'initialize the size of the return string
    nSize = Len(lpsReturnString)
    
    'do it to it
    nValid = GetPrivateProfileString(lpsAppName, lpsKeyName, lpsDefault, lpsReturnString, nSize, lpsFileName)
    
    'Discard the trailing spaces and null character...
    sPGetINI = Left$(lpsReturnString, nValid)
    
    Exit Function

sPGetINIError:

    'set bad return value
    sPGetINI = ""

End Function

