unit Placemnt;

(*
================================================================================
TPlacement Component version 1.0 by Barry L. Harkness - Public Domain

           Feel free to use this component in any application, in
           any context, without any warranty.  This component is
           donated to the public domain to help fellow Delphi
           programmers dominate windows software development.

What's up with this?

          The TPlacement component helps to automate the saving and restoring
          of TForm positions and WindowState properties.  You can drop the
          TPlacement component onto any form, and specify to which .INI file
          and section the TPlacement component is to store window positions &
          state.  Place calls to the component's Read method to set the Form's
          placement in either the Form's OnCreate or OnShow event handers.
          Call the Write method on the Form's OnDestroy method to save the
          Forms placement to the .INI file.

How to use:

       1) Choose Options|Install Components...|Add... and choose this file
          (the PLACEMNT.DCR file must also reside in the same directory).  The
          TPlacement component will be placed on the System page of the
          component palette.

       2) Drop the TPlacement component onto the desired form.  Use the Object
          Inspector to specify which .INI file to use (Windows will use your
          windows directory if none is specified; remember to include the .INI
          extension).  The Section property defaults to the Form's Name
          property, but you can change it to whatever you want (You may want
          several windows to use the same placement).

       3) Call the Placement component's Read method to read the .INI file and
          set the Form's placement (usually in the Form's OnCreate or OnShow
          event handler).

                procedure TMyForm.FormCreate(Sender: TObject);
                begin
                  Placement1.Read;
                end;

          Use the OnCreate event handler if the Form is to be shown immediately,
          or use the OnShow event if the Form is to be created but not shown
          until a later time (the Read method will set the Form's Visible
          property to its state when last saved.

       4) Call the Placement component's Write method to save the Form's
          placement into the .INI file (usually in the Form's OnDestroy event
          handler).

                procedure TMyForm.FormDestroy(Sender: TObject);
                begin
                  Placement1.Write;
                end;

       5) Test!  The Form will be restored to its saved window state (maximized,
          minimized, hidden, etc.).  It will even remember it's normal position
          and its iconized position (note: if the Form is the Main Form of an
          application, it will not be restored in a minimized state).

*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type
  TPlacement = class(TComponent)
  private
    { Private declarations }
    FIniFileName : string;
    FSection  : string;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    procedure Write; virtual;
    procedure Read; virtual;
  published
    { Published declarations }
    property IniFileName : string read FIniFileName write FIniFileName;
    property Section : string read FSection write FSection;
  end;

procedure Register;

implementation

uses IniFiles;

const
  pcFlags = 'PlaceFlags';
  pcShowCmd = 'PlaceShowCmd';
  pcMinPosX = 'PlaceMinPosX';
  pcMinPosY = 'PlaceMinPosY';
  pcMaxPosX = 'PlaceMaxPosX';
  pcMaxPosY = 'PlaceMaxPosY';
  pcNormPosL = 'PlaceNormPosL';
  pcNormPosT = 'PlaceNormPosT';
  pcNormPosR = 'PlaceNormPosR';
  pcNormPosB = 'PlaceNormPosB';


constructor TPlacement.Create(AOwner : TComponent);

begin
  inherited Create(AOwner);
  if Section = '' then
    Section :=AOwner.Name;
end;

procedure TPlacement.Read;

var
  IniFile : TIniFile;
  Placement : TWindowPlacement;

begin
  if IniFileName = '' then
    Exit;
  IniFile :=TIniFile.Create(IniFileName);
  try
    Placement.length :=SizeOf(TWindowPlacement);
    with Placement do
    begin
      Flags :=IniFile.ReadInteger(Section, pcFlags, 0);
      ShowCmd :=IniFile.ReadInteger(Section, pcShowCmd, sw_ShowNormal);
      ptMinPosition.X :=IniFile.ReadInteger(Section, pcMinPosX, 0);
      ptMinPosition.Y :=IniFile.ReadInteger(Section, pcMinPosY, 0);
      ptMaxPosition.X :=IniFile.ReadInteger(Section, pcMaxPosX, 0);
      ptMaxPosition.Y :=IniFile.ReadInteger(Section, pcMaxPosY, 0);
      rcNormalPosition.Left :=IniFile.ReadInteger(Section, pcNormPosL, 0);
      rcNormalPosition.Top :=IniFile.ReadInteger(Section, pcNormPosT, 0);
      rcNormalPosition.Right :=IniFile.ReadInteger(Section, pcNormPosR, 0);
      rcNormalPosition.Bottom :=IniFile.ReadInteger(Section, pcNormPosB, 0);
      if rcNormalPosition.Right > rcNormalPosition.Left then
      SetWindowPlacement(TForm(Owner).Handle, @Placement)
    end;
  finally
    IniFile.Free;
  end;
end;

procedure TPlacement.Write;

var
  IniFile : TIniFile;
  Placement : TWindowPlacement;

begin
  if IniFileName = '' then
    Exit;
  Placement.length :=SizeOf(TWindowPlacement);
  if not GetWindowPlacement(TForm(Owner).Handle, @Placement) then
  begin
    raise EAbort.Create('Unable to retrieve window placement');
    Exit;
  end;
  IniFile :=TIniFile.Create(IniFileName);
  try
    with Placement do
    begin
      IniFile.WriteInteger(Section, pcFlags, Flags);
      IniFile.WriteInteger(Section, pcShowCmd, ShowCmd);
      IniFile.WriteInteger(Section, pcMinPosX, ptMinPosition.X);
      IniFile.WriteInteger(Section, pcMinPosY, ptMinPosition.Y);
      IniFile.WriteInteger(Section, pcMaxPosX, ptMaxPosition.X);
      IniFile.WriteInteger(Section, pcMaxPosY, ptMaxPosition.Y);
      IniFile.WriteInteger(Section, pcNormPosL, rcNormalPosition.Left);
      IniFile.WriteInteger(Section, pcNormPosT, rcNormalPosition.Top);
      IniFile.WriteInteger(Section, pcNormPosR, rcNormalPosition.Right);
      IniFile.WriteInteger(Section, pcNormPosB, rcNormalPosition.Bottom);
    end;
  finally
    IniFile.Free;
  end;
end;

procedure Register;
begin
  RegisterComponents('System', [TPlacement]);
end;

end.
